/**
  ******************************************************************************
  * @file    kpm32xx_ddl_ebus.c
  * @author  Kiwi Software Team
  * @brief   EBUS DDL module driver.
  *          This file provides firmware functions to manage the following
  *          functionalities of EBUS peripheral:
  *          + Initialization and Configuration functions
  * @note
  *          V1.0.0, 2024/12/20.
  *
  * Copyright (c) 2024, Kiwi Instruments Co., Ltd.
  *
  * Redistribution and use in source and binary forms, with or without modification,
  * are permitted provided that the following conditions are met:
  *
  *   1. Redistributions of source code must retain the above copyright notice,
  *      this list of conditions and the following disclaimer.
  *
  *   2. Redistributions in binary form must reproduce the above copyright notice,
  *      this list of conditions and the following disclaimer in the documentation
  *      and/or other materials provided with the distribution.
  *
  *   3. Neither the name of the copyright holder nor the names of its contributors
  *      may be used to endorse or promote products derived from this software without
  *      specific prior written permission.
  *
  * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
  * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
  * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
  * THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
  * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE
  * GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
  * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
  * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
  * THE POSSIBILITY OF SUCH DAMAGE.
  ******************************************************************************
  */
#include "kpm32xx_ddl.h"
#include <string.h>



#ifdef DDL_EBUS_MODULE_ENABLED


/**
  * @brief  Make EBUS peripheral active.
  * @param  None.
  * @retval None.
  */
void DDL_EBUS_Instance_Active(void)
{
	/* Do Clock and Reset EBUS peripheral */
	__DDL_RCC_EBUS_ACTIVE();
	__DDL_RCC_EBUS_CLK_RELEASE();
}


/**
  * @brief  Configure the structure of EBUS_Init_T to default values.
  *
  * @param  eBusStruct Pointer to a EBUS_Init_T structure that contains
  *                        the configuration information for the given EBUS module.
  *
  * @retval None
  */
void DDL_EBUS_StructInit(EBUS_Init_T *eBusStruct)
{
    eBusStruct->channel      = EBUS_CHN_0;
    eBusStruct->syncMode     = EBUS_CHN_ASYNC_MODE;
	eBusStruct->adj2ChansAnd = EBUS_CHN_ADJ2CHN_ANDDIS;
	eBusStruct->adj2ChansOr  = EBUS_CHN_ADJ2CHN_ORDIS;
	eBusStruct->detSrc       = EBUS_CHN_DETSRC_RSORFL;
	eBusStruct->invOut       = EBUS_CHN_NOROUT;
	eBusStruct->out2Pad      = EBUS_CHN_OUT2PAD_DISABLE;
	eBusStruct->mstrSrc      = EBUS_CHN_MSTR_BTM2;
}


/**
  * @brief  Initializes the EBUS by the specified parameters in
  *         the EBUS_Handle_T and initialize the associated handle.
  * @param  EBUSx Pointer to DMA instance.
  * @param  eBusStruct Pointer to a EBUS_Init_T structure that contains
  *                  the configuration information for the given DMA module.
  * @retval None
  */
void DDL_EBUS_Init(EBUS_Type *EBUSx, EBUS_Init_T *eBusStruct)
{
    uint32_t tempValue = 0;

	tempValue = eBusStruct->syncMode     | eBusStruct->mstrSrc      | \
				eBusStruct->adj2ChansAnd | eBusStruct->adj2ChansOr  | \
				eBusStruct->invOut       | eBusStruct->detSrc       | eBusStruct->out2Pad;

	WRITE_REG(EBUSx->CHN[eBusStruct->channel], tempValue);
}


/**
  * @brief  DeInitializes the EBUS peripheral.
  * @param  EBUSx Pointer to DMA instance.
  * @param  eBusStruct Pointer to a EBUS_Init_T structure that contains
  *                  the configuration information for the given DMA module.
  * @retval None
  */
void DDL_EBUS_DeInit(EBUS_Type *EBUSx, EBUS_Init_T *eBusStruct)
{
	/* Caution !!: Do not reset EBus once more than one EBUS Channel is being used */
	__DDL_RCC_EBUS_DEACTIVE();
	__DDL_RCC_EBUS_CLK_HOLD();
}


/**
  * @brief  periapherals trigger EBUS event to CPU.
  * @param  EBUSx Pointer to DMA instance.
  * @param  eBusStruct Pointer to a EBUS_Init_T structure that contains
  *                  the configuration information for the given DMA module.
  *
  * @retval none
  */
void DDL_EBUS_PeriphEvtRoute2CPU(EBUS_Type *EBUSx, EBUS_Init_T *eBusStruct)
{
	MODIFY_REG(EBUSx->CPU, EBUS_CPU_CHNSEL_Msk, ((eBusStruct->channel) << EBUS_CPU_CHNSEL_Pos));
	__DDL_EBUS_EVENT2CPU_ENABLE(EBUSx);
}


/**
  * @brief  CPU triggers an event to Peripheral in form of Level or Pulse
  *         TrigOrClr makes sense:
  *         1. only when CPUEvtType is EBUS_CPUEVT_LEVEL;
  *         2. when TrigOrClr is 1, EBUS_CPUEVT_LEVEL asserts to given channel of chnIndex;
  *         3. when TrigOrClr is 0, EBUS_CPUEVT_LEVEL desserts.
  *
  * @param  EBUSx Pointer to DMA instance.
  * @param  eBusStruct Pointer to a EBUS_Init_T structure that contains
  *                  the configuration information for the given DMA module.
  * @param  CPUEvtType  CPU event types.
  * @param  TrigOrClr   refer to the brief description.
  *
  */
void DDL_EBUS_CPUEvtRoute2Periph(EBUS_Type *EBUSx, EBUS_Init_T *eBusStruct, EBUSCPUEvtLvl_T CPUEvtType, uint32_t TrigOrClr)
{
	uint32_t i = eBusStruct->channel;

	if (CPUEvtType == EBUS_CPUEVT_LEVEL)
	{
		if ( TrigOrClr )
		{
			SET_BIT(EBUSx->SWT, (0x1UL << (i + 16)));
		}
		else
		{
			CLEAR_BIT(EBUSx->SWT, (0x1UL << (i + 16)));
		}
	}
	else
	{
		/*Auto-Clear by EBUS hardware */
		SET_BIT(EBUSx->SWT, (0x1UL << (i)));
	}
}

#endif /** DDL_EBUS_MODULE_ENABLED */

