/**
  ******************************************************************************
  * @file    kpm32xx_ddl_crc.c
  * @author  Kiwi Software Team
  * @brief   CRC DDL module driver.
  *          This file provides firmware functions to manage the following
  *          functionalities of the CRC peripheral:
  *          + CRC init and de-init
  *          + Set init value and calculate crc
  * @note
  *          V1.0.0, 2024/12/20.
  *
  * Copyright (c) 2024, Kiwi Instruments Co,. Ltd.
  *
  * Redistribution and use in source and binary forms, with or without modification,
  * are permitted provided that the following conditions are met:
  *
  *   1. Redistributions of source code must retain the above copyright notice,
  *      this list of conditions and the following disclaimer.
  *
  *   2. Redistributions in binary form must reproduce the above copyright notice,
  *      this list of conditions and the following disclaimer in the documentation
  *      and/or other materials provided with the distribution.
  *
  *   3. Neither the name of the copyright holder nor the names of its contributors
  *      may be used to endorse or promote products derived from this software without
  *      specific prior written permission.
  *
  * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
  * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
  * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
  * THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
  * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE
  * GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
  * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
  * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
  * THE POSSIBILITY OF SUCH DAMAGE.
  ******************************************************************************
  */


#include "kpm32xx_ddl.h"

#ifdef DDL_CRC_MODULE_ENABLED



/**
  * @brief  Configure the elements of structure CRC_Init_T to default values.
  * @param  pCrc Pointer to a CRC_Init_T structure that contains
  *                  the configuration information for the given crc module.
  */
void DDL_CRC_StructInit(CRC_Init_T *pCrc)
{
	pCrc->inputInvert = CRC_INPUT_INVT_NONE;
	pCrc->outputInvert = CRC_OUTPUT_INVT_NONE;
}


/**
  * @brief  Initialize the CRC according to the specified
  *         parameters in the CRC_Init_T.
  * @param  pCrc  CRC_Init_T
  */
void DDL_CRC_Init(CRC_Init_T *pCrc)
{
	if (pCrc->outputInvert == CRC_OUTPUT_INVT_WORD)
	{
		SET_BIT(CRC->CTRL, CRC_CTRL_OUTINVT);
	}
	else
	{
		CLEAR_BIT(CRC->CTRL, CRC_CTRL_OUTINVT);
	}

	MODIFY_REG(CRC->CTRL, CRC_CTRL_ININVT_Msk, pCrc->inputInvert);
}


/**
  * @brief  DeInitialize the CRC peripheral.
  * @param  hcrc  CRC handle
  */
void DDL_CRC_DeInit(void)
{
	CLEAR_BIT(CRC->CTRL, CRC_CTRL_EN);
}


/**
  * @brief  Compute the CRC value of the specify data.
  * @brief  default: big endian
  * @param  pCrc          CRC_Init_T
  * @param  initValue     CRC init value.
  * @param  pBuffer       pointer to the input data buffer.
  * @param  bufferLength  input data buffer length.
  * @retval crc value
  */
uint32_t DDL_CRC_Calculate(CRC_Init_T *pCrc, uint32_t initValue, uint32_t *pBuffer, uint32_t bufferLength)
{
	uint32_t index;
	uint32_t temp = 0U;

	/* Configure Initial Value before CRC is enabled */
	CRC->DT = initValue;
	SET_BIT(CRC->CTRL, CRC_CTRL_EN);

	for (index = 0U; index < bufferLength; index++)
	{
		CRC->DT = pBuffer[index];
	}

	temp = CRC->DT;

	return temp;
}


/**
  * @param  initValue  CRC init value.
  */
void DDL_CRC_Set_InitValue(uint32_t initValue)
{
	CLEAR_BIT(CRC->CTRL, CRC_CTRL_EN);
	/* Configure Initial Value before CRC is enabled */
	CRC->DT = initValue;
	SET_BIT(CRC->CTRL, CRC_CTRL_EN);
}


/**
  * @brief  Compute the CRC value of the specify data.
  * @brief  default: big endian
  * @param  pCrc          CRC_Init_T
  * @param  pBuffer       pointer to the input data buffer.
  * @param  bufferLength  input data buffer length.
  * @retval crc value
  */
uint32_t DDL_CRC_Accumulate(CRC_Init_T *pCrc, uint32_t *pBuffer, uint32_t bufferLength)
{
	uint32_t index;
	uint32_t temp = 0U;

	for (index = 0U; index < bufferLength; index++)
	{
		CRC->DT = pBuffer[index];
	}

	temp = CRC->DT;
	return temp;
}

#endif /* DDL_CRC_MODULE_ENABLED */

