/**
  ******************************************************************************
  * @file    kpm32xx_ddl_cmp.h
  * @author  Kiwi Software Team
  * @brief   Header file of CMP DDL module.
  * @note
  *          V1.0.0, 2025/1/6.
  *
  * Copyright (c) 2024, Kiwi Instruments Co,. Ltd.
  *
  * Redistribution and use in source and binary forms, with or without modification,
  * are permitted provided that the following conditions are met:
  *
  *   1. Redistributions of source code must retain the above copyright notice,
  *      this list of conditions and the following disclaimer.
  *
  *   2. Redistributions in binary form must reproduce the above copyright notice,
  *      this list of conditions and the following disclaimer in the documentation
  *      and/or other materials provided with the distribution.
  *
  *   3. Neither the name of the copyright holder nor the names of its contributors
  *      may be used to endorse or promote products derived from this software without
  *      specific prior written permission.
  *
  * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
  * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
  * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
  * THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
  * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE
  * GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
  * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
  * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
  * THE POSSIBILITY OF SUCH DAMAGE.
  ******************************************************************************
  */


#ifndef KPM32xx_DDL_CMP_H
#define KPM32xx_DDL_CMP_H

#ifdef __cplusplus
extern "C" {
#endif


#include "kpm32xx_ddl_def.h"

struct __CMP_Handle_T;

/* voltage cmp P input pin select */
typedef enum
{
	CMP0_P_SEL_XP42         = 0x00U,
	CMP0_P_SEL_XP21         = 0x01U,
	CMP0_P_SEL_XP40         = 0x02U,
	CMP0_P_SEL_XP45         = 0x03U,

	CMP1_P_SEL_XP42         = 0x00U,
	CMP1_P_SEL_XP21         = 0x01U,
	CMP1_P_SEL_XP40         = 0x02U,
	CMP1_P_SEL_XP45         = 0x03U,

	CMP2_P_SEL_XP51         = 0x00U,
	CMP2_P_SEL_XP44         = 0x01U,
	CMP2_P_SEL_XP53         = 0x02U,
	CMP2_P_SEL_XP46         = 0x03U,

	CMP3_P_SEL_XP51         = 0x00U,
	CMP3_P_SEL_XP44         = 0x01U,
	CMP3_P_SEL_XP53         = 0x02U,
	CMP3_P_SEL_XP46         = 0x03U
} CMP_P_Select_T;

/* voltage cmp P input pin  enable select */
typedef enum
{
	CMP0_P_ASEL_XP42         = 0x01U,
	CMP0_P_ASEL_XP21         = 0x02U,
	CMP0_P_ASEL_XP40         = 0x04U,
	CMP0_P_ASEL_XP45         = 0x08U,

	CMP1_P_ASEL_XP42         = 0x01U,
	CMP1_P_ASEL_XP21         = 0x02U,
	CMP1_P_ASEL_XP40         = 0x04U,
	CMP1_P_ASEL_XP45         = 0x08U,

	CMP2_P_ASEL_XP51         = 0x01U,
	CMP2_P_ASEL_XP44         = 0x02U,
	CMP2_P_ASEL_XP53         = 0x04U,
	CMP2_P_ASEL_XP46         = 0x08U,

	CMP3_P_ASEL_XP51         = 0x01U,
	CMP3_P_ASEL_XP44         = 0x02U,
	CMP3_P_ASEL_XP53         = 0x04U,
	CMP3_P_ASEL_XP46         = 0x08U
} CMP_P_ASelect_T;

/* voltage cmp N input pin select */
typedef enum
{
	CMP0_N_SEL_XP42         = 0x00U,
	CMP0_N_SEL_XP20         = 0x01U,
	CMP0_N_SEL_XP40         = 0x02U,
	CMP0_N_SEL_DACOUT       = 0x03U,

	CMP1_N_SEL_XP42         = 0x00U,
	CMP1_N_SEL_XP20         = 0x01U,
	CMP1_N_SEL_XP40         = 0x02U,
	CMP1_N_SEL_DACOUT       = 0x03U,

	CMP2_N_SEL_XP21         = 0x00U,
	CMP2_N_SEL_XP44         = 0x01U,
	CMP2_N_SEL_XP53         = 0x02U,
	CMP2_N_SEL_DACOUT       = 0x03U,

	CMP3_N_SEL_XP21         = 0x00U,
	CMP3_N_SEL_XP44         = 0x01U,
	CMP3_N_SEL_XP53         = 0x02U,
	CMP3_N_SEL_DACOUT       = 0x03U,
} CMP_N_Select_T;

/* voltage cmp N input pin enable select */
typedef enum
{
	CMP0_N_ASEL_XP42         = 0x01U,
	CMP0_N_ASEL_XP20         = 0x02U,
	CMP0_N_ASEL_XP40         = 0x04U,
	CMP0_N_ASEL_DACOUT       = 0x00U,

	CMP1_N_ASEL_XP42         = 0x01U,
	CMP1_N_ASEL_XP20         = 0x02U,
	CMP1_N_ASEL_XP40         = 0x04U,
	CMP1_N_ASEL_DACOUT       = 0x00U,

	CMP2_N_ASEL_XP21    	  = 0x01U,
	CMP2_N_ASEL_XP44    	  = 0x02U,
	CMP2_N_ASEL_XP53    	  = 0x04U,
	CMP2_N_ASEL_DACOUT  	  = 0x00U,

	CMP3_N_ASEL_XP21    	  = 0x01U,
	CMP3_N_ASEL_XP44    	  = 0x02U,
	CMP3_N_ASEL_XP53    	  = 0x04U,
	CMP3_N_ASEL_DACOUT  	  = 0x00U
} CMP_N_ASelect_T;


/*dac mode select */
typedef enum
{
	DAC_MODE_NORMAL         = 0x00U,
	DAC_MODE_EBUS_TRIGGER   = 0x01U
} DAC_CHANGE_MODE_T;

/* dac ramp mode trigger source select */
typedef enum
{
	DAC_RAMP_MODE_SOFT_TRG  = 0x00U,
	DAC_RAMP_MODE_EBUS_TRG  = 0x01U
} DAC_RAMP_MODE_T;

/* cmp trigger edge select */
typedef enum
{
	CMP_EDGE_SINGLE        = 0x00U,
	CMP_EDGE_DOUBLE        = 0x01U
} CMP_EdgeDet_T;

/* cmp trigger polarity select */
typedef enum
{
	CMP_EDGE_POL_RISING    = 0x00U,
	CMP_EDGE_POL_FALLING   = 0x01U
} CMP_EdgePol_T;

/* cmp out polarity */
typedef enum
{
	CMP_OUT_POL_OUT    	    = 0x00U,
	CMP_OUT_POL_OUTREVERSE  = 0x01U
} CMP_OutPol_T;

/* cmp out polarity */
typedef enum
{
	CMP_HYSTERESIS_NO    	= 0x00U,
	CMP_HYSTERESIS_25MV     = 0x01U,
	CMP_HYSTERESIS_50MV     = 0x01U,
	CMP_HYSTERESIS_100MV    = 0x01U,
} CMP_HySel_T;

/* cmp filter config */
typedef struct
{
	uint32_t cmpFilterEnable;          /* cmp filter enable */
	uint32_t cmpFilterDiv;             /* cmp filter clock divide */
	uint32_t cmpFilterNum;             /* cmp filter num */
	uint32_t cmpFilterWind;            /* cmp filter wind */
} CMP_Filter_Handle_T;

/* dac base config */
typedef struct
{
	uint32_t ebusChannelSel;            /* dac ebus channel select */
	uint32_t dacRampTrigger;            /* dac ramp Trigger */
	uint32_t dacRampMode;               /* dac ramp Mode */
	uint32_t dacRampEnable;             /* dac ramp enable */
	uint32_t dacMD;                		/* dac mode change */
	uint32_t dacEnable;                 /* dac enable */
} DAC_InitTypeDef;


/**
  * @brief  CMP init
  * @param  CMPx     pointer to a ACMP_Type structure.
			outPol  cmp out polar
			edgeSel cmp edge detection
			edgePol cmp edge polar
			hySel   cmp hysteresis select
  * @retval None
  */
void DDL_CMP_Init(ACMP_Type *CMPx, CMP_OutPol_T outPol, CMP_EdgeDet_T edgeSel , CMP_EdgePol_T edgePol,CMP_HySel_T hySel);


/**
  * @brief  CMP init
  * @param  CMPx     pointer to a ACMP_Type structure.
			pSel  cmp p select pin
			pASel cmp p select pin enable
			nSel  cmp n select pin
			nASel cmp n select pin enable
  * @retval None
  */
void DDL_CMP_PNSel(ACMP_Type *CMPx, CMP_P_Select_T pSel, CMP_P_ASelect_T pASel , CMP_N_Select_T nSel,CMP_N_ASelect_T nASel);


/**
  * @brief  CMP enable
  * @param  CMPx     pointer to a ACMP_Type structure.
  * @retval None
  */
void DDL_CMP_Enable(ACMP_Type *CMPx);


/**
  * @brief  CMP disable
  * @param  CMPx     pointer to a ACMP_Type structure.
  * @retval None
  */
void DDL_CMP_Disable(ACMP_Type *CMPx);


/**
  * @brief  CMP interrupt enable
  * @param  CMPx     pointer to a ACMP_Type structure.
  * @retval None
  */
void DDL_CMP_IntEnable(ACMP_Type *CMPx);


/**
  * @brief  CMP interrupt disable
  * @param  CMPx     pointer to a ACMP_Type structure.
  * @retval None
  */
void DDL_CMP_IntDisable(ACMP_Type *CMPx);


/**
  * @brief  CMP filter config init
  * @param  hFilter     pointer to a CMP_Filter_Handle_T structure.
  * @retval None
  */
void DDL_CMP_Filter_StructInit(CMP_Filter_Handle_T *hFilter);


/**
  * @brief  CMP filter config
  * @param  CMPx     pointer to a ACMP_Type structure.
  * @retval None
  */
void DDL_CMP_Filter_Config(ACMP_Type *CMPx, CMP_Filter_Handle_T *hFilter);


/**
  * @brief  CMP output mode select
  * @param  CMPx   : pointer to a ACMP_Type structure.
  * @param  source: 0 EBUS
  *                 1 PIN
  *                 2 TIM
  * @param  mode:   0 async output
  *                 1 sync output
  *                 2 filter output
  *                 3 edge output
  */
void DDL_CMP_Output_Mode_Select(ACMP_Type *CMPx, uint8_t source, uint8_t mode);


/**
  * @brief  CMP filter config init
  * @param  hdac     pointer to a DAC_InitTypeDef structure.
  * @retval None
  */
void DDL_DAC_Init_StructInit(DAC_InitTypeDef *hdac);


/**
  * @brief  use to config counter mode para
  * @param  CMPx     pointer to a ACMP_Type structure.
  * @param  hdac	 pointer to a DAC_InitTypeDef to config parameter
  * @retval NULL
  */
void DDL_DAC_Init(ACMP_Type *CMPx, DAC_InitTypeDef *hdac);


/**
  * @brief  dac value write
  * @param  CMPx   : pointer to a ACMP_Type structure.
  * @retval dacValue: write data int dac value
  */
void DDL_DAC_Value(ACMP_Type *CMPx,uint32_t dacValue);


/**
  * @brief  dac value write
  * @param  CMPx   : pointer to a ACMP_Type structure.
  * @retval dacValue: write data int dac ramp value
  */
void DDL_DAC_Ramp_Value(ACMP_Type *CMPx,uint32_t dacValue);


/**
  * @brief  dac ramp para config
  * @param  hCMP   : pointer to a CMP_Handle_T structure.
  * @retval dly: write data int dac ramp dly  para
			step: write data int dac ramp step para
			slope: write data int dac ramp slope para
  */
void DDL_DAC_Ramp_Config(ACMP_Type *CMPx,uint32_t dly,uint32_t step,uint32_t slope);



#ifdef __cplusplus
}
#endif

#endif /* KPM32xx_DDL_CMP_H */

