/**
 ******************************************************************************
 * @file    kpm32xx_ddl_adc.h
 * @author  Kiwi Software Team
 * @brief   Header file of ADC DDL module.
 * @note
 *          V1.0.0, 2024/12/20.
 *
 * Copyright (c) 2024, Kiwi Instruments Co., Ltd.
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 *	1. Redistributions of source code must retain the above copyright notice,
 *	   this list of conditions and the following disclaimer.
 *
 *	2. Redistributions in binary form must reproduce the above copyright notice,
 *	   this list of conditions and the following disclaimer in the documentation
 *	   and/or other materials provided with the distribution.
 *
 *	3. Neither the name of the copyright holder nor the names of its contributors
 *	   may be used to endorse or promote products derived from this software without
 *	   specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
 * THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE
 * GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 ******************************************************************************
 */


#ifndef __KPM32xx_DDL_ADC_H
#define __KPM32xx_DDL_ADC_H

#ifdef __cplusplus
	extern "C" {
#endif


#include "kpm32xx_ddl_def.h"


/* The maximum channels supported */
#define CHANNEL_NUM_MAX       16

/* The maximum SOC supported */
#define SOC_NUM_MAX            16

/* The maximum PPB supported */
#define PPB_NUM_MAX            4


/* ADC different compare mode */
typedef enum
{
	BIGGER_THAN_HIGHTHRESHOLD      = 0x00U,       /* Conversion result bigger than high threshold */
	LESS_THAN_LOWTHRESHOLD         = 0x01U,       /* Conversion result less than low threshold */
} ADC_CompareMode_T;

/* ADC resolution setting */
typedef enum
{
	RESOLUTION_12BIT               = 0x00U,       /* 12 bit resolution */
	RESOLUTION_10BIT               = 0x01U,       /* 10 bit resolution */
	RESOLUTION_8BIT                = 0x02U,       /* 8 bit resolution */
} ADC_Resolution_T;


/* ADC sample time setting */
typedef enum
{
	SAMPLE_3_ADCCLK                = 0x00U,       /* Sample time : 3 adc clock */
	SAMPLE_4_ADCCLK                = 0x01U,       /* Sample time : 4 adc clock */
	SAMPLE_6_ADCCLK                = 0x02U,       /* Sample time : 6 adc clock */
	SAMPLE_8_ADCCLK                = 0x03U,       /* Sample time : 8 adc clock */
	SAMPLE_12_ADCCLK               = 0x04U,       /* Sample time : 12 adc clock */
	SAMPLE_16_ADCCLK               = 0x05U,       /* Sample time : 16 adc clock */
	SAMPLE_24_ADCCLK               = 0x06U,       /* Sample time : 24 adc clock */
	SAMPLE_32_ADCCLK               = 0x07U,       /* Sample time : 32 adc clock */
	SAMPLE_48_ADCCLK               = 0x08U,       /* Sample time : 48 adc clock */
	SAMPLE_64_ADCCLK               = 0x09U,       /* Sample time : 64 adc clock */
	SAMPLE_96_ADCCLK               = 0x0AU,       /* Sample time : 96 adc clock */
	SAMPLE_128_ADCCLK              = 0x0BU,       /* Sample time : 128 adc clock */
	SAMPLE_192_ADCCLK              = 0x0CU,       /* Sample time : 192 adc clock */
	SAMPLE_256_ADCCLK              = 0x0DU,       /* Sample time : 256 adc clock */
	SAMPLE_384_ADCCLK              = 0x0EU,       /* Sample time : 384 adc clock */
	SAMPLE_512_ADCCLK              = 0x0FU,       /* Sample time : 512 adc clock */
} ADC_SampleTime_T;

/* ADC Soft trigger difinition */
typedef enum
{
	ADC_SOC0_TRIGGER              = 0x00U,
	ADC_SOC1_TRIGGER              = 0x01U,
	ADC_SOC2_TRIGGER              = 0x02U,
	ADC_SOC3_TRIGGER              = 0x03U,
	ADC_SOC4_TRIGGER              = 0x04U,
	ADC_SOC5_TRIGGER              = 0x05U,
	ADC_SOC6_TRIGGER              = 0x06U,
	ADC_SOC7_TRIGGER              = 0x07U,
	ADC_SOC8_TRIGGER              = 0x08U,
	ADC_SOC9_TRIGGER              = 0x09U,
	ADC_SOC10_TRIGGER             = 0x0AU,
	ADC_SOC11_TRIGGER             = 0x0BU,
	ADC_SOC12_TRIGGER             = 0x0CU,
	ADC_SOC13_TRIGGER             = 0x0DU,
	ADC_SOC14_TRIGGER             = 0x0EU,
	ADC_SOC15_TRIGGER             = 0x0FU,
	ADC_BURST_TRIGGER             = 0x10U,
	ADC_STOP_TRIGGER              = 0x11U
} ADC_SoftTrigger_T;

/* ADC Burst length definition */
typedef enum
{
	ADC_BURSTLEN_1           = 0x0U,
	ADC_BURSTLEN_2           = 0x1U,
	ADC_BURSTLEN_3           = 0x2U,
	ADC_BURSTLEN_4           = 0x3U,
	ADC_BURSTLEN_5           = 0x4U,
	ADC_BURSTLEN_6           = 0x5U,
	ADC_BURSTLEN_7           = 0x6U,
	ADC_BURSTLEN_8           = 0x7U,
	ADC_BURSTLEN_9           = 0x8U,
	ADC_BURSTLEN_10          = 0x9U,
	ADC_BURSTLEN_11          = 0xAU,
	ADC_BURSTLEN_12          = 0xBU,
	ADC_BURSTLEN_13          = 0xCU,
	ADC_BURSTLEN_14          = 0xDU,
	ADC_BURSTLEN_15          = 0xEU,
	ADC_BURSTLEN_16          = 0xFU,
} ADC_BurstLen_T;

/* ADC SOC priority definition */
typedef enum
{
	ADC_PRI_ALL_ROUND_ROBIN   = 0,        /* Round robin mode is used for all      */
	ADC_PRI_SOC0_HIPRI,                   /* SOC 0 high priority, others in round robin   */
	ADC_PRI_THRU_SOC1_HIPRI,              /* SOC 0-1 high priority, others in round robin */
	ADC_PRI_THRU_SOC2_HIPRI,              /* SOC 0-2 high priority, others in round robin */
	ADC_PRI_THRU_SOC3_HIPRI,              /* SOC 0-3 high priority, others in round robin */
	ADC_PRI_THRU_SOC4_HIPRI,              /* SOC 0-4 high priority, others in round robin */
	ADC_PRI_THRU_SOC5_HIPRI,              /* SOC 0-5 high priority, others in round robin */
	ADC_PRI_THRU_SOC6_HIPRI,              /* SOC 0-6 high priority, others in round robin */
	ADC_PRI_THRU_SOC7_HIPRI,              /* SOC 0-7 high priority, others in round robin */
	ADC_PRI_THRU_SOC8_HIPRI,              /* SOC 0-8 high priority, others in round robin */
	ADC_PRI_THRU_SOC9_HIPRI,              /* SOC 0-9 high priority, others in round robin */
	ADC_PRI_THRU_SOC10_HIPRI,             /* SOC 0-10 high priority, others in round robin */
	ADC_PRI_THRU_SOC11_HIPRI,             /* SOC 0-11 high priority, others in round robin */
	ADC_PRI_THRU_SOC12_HIPRI,             /* SOC 0-12 high priority, others in round robin */
	ADC_PRI_THRU_SOC13_HIPRI,             /* SOC 0-13 high priority, others in round robin */
	ADC_PRI_THRU_SOC14_HIPRI,             /* SOC 0-14 high priority, SOC15 in round robin  */
	ADC_PRI_ALL_HIPRI                     /* All priorities based on SOC number     */
} ADC_SOCPriority_T;

/* ADC channel */
typedef enum
{
	ADC_CHANNEL_0        = 0x00U,
	ADC_CHANNEL_1,
	ADC_CHANNEL_2,
	ADC_CHANNEL_3,
	ADC_CHANNEL_4,
	ADC_CHANNEL_5,
	ADC_CHANNEL_6,
	ADC_CHANNEL_7,
	ADC_CHANNEL_8,
	ADC_CHANNEL_9,
	ADC_CHANNEL_10,
	ADC_CHANNEL_11,
	ADC_CHANNEL_12,
	ADC_CHANNEL_13,
	ADC_CHANNEL_14,
	ADC_CHANNEL_15
} ADC_Channel_T;

/* ADC SOC definition */
typedef enum
{
	ADC_SOC0       = 0x00U,
	ADC_SOC1,
	ADC_SOC2,
	ADC_SOC3,
	ADC_SOC4,
	ADC_SOC5,
	ADC_SOC6,
	ADC_SOC7,
	ADC_SOC8,
	ADC_SOC9,
	ADC_SOC10,
	ADC_SOC11,
	ADC_SOC12,
	ADC_SOC13,
	ADC_SOC14,
	ADC_SOC15
} ADC_SOCNumber_T;

/* ADC post-processing block definition */
typedef enum
{
	ADC_PPB0        = 0x00U,
	ADC_PPB1,
	ADC_PPB2,
	ADC_PPB3
} ADC_PPBNumber_T;

/* ADC callback event */
typedef enum
{
	ADC_EVENT_SOC0   = (0x1U << 0),             /* End of SOC0 conversion event */
	ADC_EVENT_SOC1   = (0x1U << 1),             /* End of SOC1 conversion event */
	ADC_EVENT_SOC2   = (0x1U << 2),             /* End of SOC2 conversion event */
	ADC_EVENT_SOC3   = (0x1U << 3),             /* End of SOC3 conversion event */
	ADC_EVENT_SOC4   = (0x1U << 4),             /* End of SOC4 conversion event */
	ADC_EVENT_SOC5   = (0x1U << 5),             /* End of SOC5 conversion event */
	ADC_EVENT_SOC6   = (0x1U << 6),             /* End of SOC6 conversion event */
	ADC_EVENT_SOC7   = (0x1U << 7),             /* End of SOC7 conversion event */
	ADC_EVENT_SOC8   = (0x1U << 8),             /* End of SOC8 conversion event */
	ADC_EVENT_SOC9   = (0x1U << 9),             /* End of SOC9 conversion event */
	ADC_EVENT_SOC10   = (0x1U << 10),            /* End of SOC10 conversion event */
	ADC_EVENT_SOC11   = (0x1U << 11),            /* End of SOC11 conversion event */
	ADC_EVENT_SOC12   = (0x1U << 12),            /* End of SOC12 conversion event */
	ADC_EVENT_SOC13   = (0x1U << 13),            /* End of SOC13 conversion event */
	ADC_EVENT_SOC14   = (0x1U << 14),            /* End of SOC14 conversion event */
	ADC_EVENT_SOC15   = (0x1U << 15),            /* End of SOC15 conversion event */
	ADC_EVENT_PPB0_COMPH   = (0x1U << 16),       /* PPB0 result bigger than high threshold event */
	ADC_EVENT_PPB0_COMPL   = (0x1U << 17),       /* PPB0 result less than low threshold event */
	ADC_EVENT_PPB0_ZERO    = (0x1U << 18),       /* PPB0 result zero detect event */
	ADC_EVENT_PPB1_COMPH   = (0x1U << 19),       /* PPB1 result bigger than high threshold event */
	ADC_EVENT_PPB1_COMPL   = (0x1U << 20),       /* PPB1 result less than low threshold event */
	ADC_EVENT_PPB1_ZERO    = (0x1U << 21),       /* PPB1 result zero detect event */
	ADC_EVENT_PPB2_COMPH   = (0x1U << 22),       /* PPB2 result bigger than high threshold event */
	ADC_EVENT_PPB2_COMPL   = (0x1U << 23),       /* PPB2 result less than low threshold event */
	ADC_EVENT_PPB2_ZERO    = (0x1U << 24),       /* PPB2 result zero detect event */
	ADC_EVENT_PPB3_COMPH   = (0x1U << 25),       /* PPB3 result bigger than high threshold event */
	ADC_EVENT_PPB3_COMPL   = (0x1U << 26),       /* PPB3 result less than low threshold event */
	ADC_EVENT_PPB3_ZERO    = (0x1U << 27),       /* PPB3 result zero detect event */
	ADC_EVENT_SOC_ANY      = (0x1U << 28),       /* Any SOC convert finish event */
	ADC_EVENT_SOC_OVF      = (0x1U << 29),       /* Any SOC overflow event */
	ADC_EVENT_ERROR        = 0x1FFFFFFF,
} ADC_Event_T;

/* Configure ADC PPB */
typedef struct
{
	uint8_t            enableTwosComplement;   /* If enable : ADCPPBxRESULT = ADCSOCxOFFREF - ADCRESULTx, else : ADCPPBxRESULT = ADCRESULTx - ADCSOCxOFFREF*/
	ADC_SOCNumber_T    soc;                    /* Specify which SOC to use */
	uint16_t           refOffset;              /* Specify the reference offset value */
	int16_t            calOffset;              /* Specify the Calibration Offset value */
	int16_t            highThreshold;          /* Set the compare high threshold value, only 13-bit valid, from -4095 to 4095. */
	int16_t            lowThreshold;           /* Set the compare low threshold value, only 13-bit valid, from -4095 to 4095. */
} ADC_PPBConfig_T;

/* Burst mode configuration */
typedef struct
{
	uint32_t        burstLen;          /* Specify the conversion length */
	uint32_t        hwTrigger;         /* Enable hardware tigger or not */
	uint32_t        triggerSrc;        /* Specify the hardware trigger source */
} ADC_BurstConfig_T;

/* Configure ADC init parameters */
typedef struct
{
	int8_t                   zeroOffsetCal;       /* Zero offset calibration, only 7-bit valid, from -64 ~ 63. */
	ADC_Resolution_T         resolution;          /* Configure the ADC resolution */
	ADC_SOCPriority_T        socPriority;         /* Specify the high priority soc */
	uint32_t                 adcClkPrescaler;     /* Specify ADC clock prescaler */
} ADC_Init_T;

/* Configure ADC SOC */
typedef struct
{
	ADC_Channel_T            channel;          /* Specify the channels to use */
	ADC_SampleTime_T         sampleTime;       /* Configure ADC channel sample time */
	uint8_t                  hwTrigger;        /* Enable hardware trigger */
	uint32_t                 ebusChannel;      /* Configure hardware trigger source */
} ADC_SOCConfig_T;


/* Enable the ADC peripheral. */
#define __DDL_ADC_ENABLE(__INSTANCE__) ((__INSTANCE__)->CTRL |= ADC_CTRL_ENABLE)

/* Disable the ADC peripheral. */
#define __DDL_ADC_DISABLE(__INSTANCE__) ((__INSTANCE__)->CTRL &= ~ADC_CTRL_ENABLE)

/* Start software trigger */
#define __DDL_ADC_SOFT_TRIGGER_START(__INSTANCE__, __SOC__)  SET_BIT((__INSTANCE__)->TRIG, 0x1 << __SOC__)

/* Start burst trigger */
#define __DDL_ADC_BURST_TRIGGER_START(__INSTANCE__)  SET_BIT((__INSTANCE__)->TRIG, ADC_TRIG_BURST)

/* Stop software trigger */
#define __DDL_ADC_SOFT_TRIGGER_STOP(__INSTANCE__)   SET_BIT((__INSTANCE__)->TRIG, ADC_TRIG_STOP)

/* Enable ADC interrupt. */
#define __DDL_ADC_ENABLE_IT(__INSTANCE__, __INTERRUPT__) (((__INSTANCE__)->INT) |= (__INTERRUPT__))

/* Disable ADC interrupt. */
#define __DDL_ADC_DISABLE_IT(__INSTANCE__, __INTERRUPT__) (((__INSTANCE__)->INT) &= ~(__INTERRUPT__))

/* Check if the specified ADC interrupt source is enabled or disabled */
#define __DDL_ADC_GET_IT_SOURCE(__INSTANCE__, __INTERRUPT__)  (((__INSTANCE__)->INT & (__INTERRUPT__)) == (__INTERRUPT__))

/* Clear ADC interrupt flag. */
#define __DDL_ADC_CLEAR_IT_FLAG(__INSTANCE__, __FLAG__) WRITE_REG((__INSTANCE__)->CLEAR, __FLAG__)

/* Get ADC interrupt flag. */
#define __DDL_ADC_GET_FLAG(__INSTANCE__, __FLAG__) ((((__INSTANCE__)->STATUS) & (__FLAG__)) == (__FLAG__))

/* GET SOC overflow flag. */
#define __DDL_ADC_GET_SOC_OVERFLOW_FLAG(__INSTANCE__, __FLAG__) ((((__INSTANCE__)->TRGOVF) & (__FLAG__)) == (__FLAG__))

/* Clear SOC overflow flag. */
#define __DDL_ADC_CLEAR_SOC_OVERFLOW_FLAG(__INSTANCE__, __FLAG__) WRITE_REG((__INSTANCE__)->OVFCLR, __FLAG__)

/* Check ADC is busy */
#define IS_ADC_BUSY(__INSTANCE__)  ((((__INSTANCE__)->STATUS) & (ADC_STATUS_BUSY)) == (ADC_STATUS_BUSY))

/* Reset RRP to default value : 0x10000 */
#define __DDL_ADC_RESET_RRP(__INSTANCE__)   SET_BITMASK((__INSTANCE__)->PRIO, ADC_PRIO_HIGHSEL_Msk, 0)


/**
  * @brief  Make specified ADC active.
  * @param  ADCx        ADC instance.
  * @retval None.
  */
void DDL_ADC_Instance_Active(ADC_Type *ADCx);

/**
  * @brief  Make specified ADC deactive.
  * @param  ADCx        ADC instance.
  * @retval None.
  */
void DDL_ADC_Instance_Deactive(ADC_Type *ADCx);

/**
  * @brief  Configure the elements of structure ADC_Init_T to default values.
  * @param  pADCStruct Pointer to a ADC_Init_T structure that contains
  *         the configuration information for the given ADC module.
  * @retval None
  */
void DDL_ADC_StructInit(ADC_Init_T *pADCStruct);

/**
  * @brief  Initialize the ADC peripheral according to the specified parameters
  *         in the ADC_Init_T.
  * @param  ADCx        ADC instance.
  * @param  pADCStruct  pointer to a ADC_Init_T structure that contains
  *         the configuration information for the specified ADC.
  * @retval None.
  */
void DDL_ADC_Init(ADC_Type *ADCx, ADC_Init_T *pADCStruct);

/**
  * @brief  Configures the SOC to use.
  * @param  ADCx     ADC instance.
  * @param  SOCx     ADC SOC index.
  * @param  pConfig  pointer to a ADC_SOCConfig_T that contains
  *                  the SOC information.
  * @retval None
  */
void DDL_ADC_ConfigSOC(ADC_Type *ADCx, ADC_SOCNumber_T SOCx, ADC_SOCConfig_T *pConfig);

/**
  * @brief  Configures the post processing block.
  * @param  ADCx        ADC instance.
  * @param  PPBx        PPB index.
  * @param  pConfig     pointer to an ADC_PPBConfig_T structure
  *                     that contains the configuration information of ADC PPB.
  * @retval DDL status
  */
void DDL_ADC_ConfigPPB(ADC_Type *ADCx, ADC_PPBNumber_T PPBx, ADC_PPBConfig_T *pConfig);

/**
  * @brief  Enable post processing block (PPB).
  * @param  ADCx        ADC instance.
  * @param  PPBx        PPB index.
  *
  * @retval None.
  */
void DDL_ADC_EnablePPBMode(ADC_Type* ADCx, ADC_PPBNumber_T PPBx);

/**
  * @brief  Disable post processing block (PPB).
  * @param  ADCx        ADC instance.
  * @param  PPBx        PPB index.
  *
  * @retval None.
  */
void DDL_ADC_DisablePPBMode(ADC_Type* ADCx, ADC_PPBNumber_T PPBx);

/**
  * @brief  Configure Burst to use.
  * @param  ADCx     ADC instance.
  * @param  pConfig  pointer to a ADC_BurstConfig_T that contains
  *                  the SOC information.
  * @retval None.
  */
void DDL_ADC_BurstConfig(ADC_Type *ADCx, ADC_BurstConfig_T *pConfig);

/**
  * @brief  Poll for conversion event
  * @param  ADCx       ADC instance.
  * @param  eventType  the ADC event type, ref to ADC_Event_T
  * @param  timeout    timeout value in millisecond.
  * @retval DDL status.
  */
DDL_Status_T DDL_ADC_PollForEvent(ADC_Type *ADCx, uint32_t eventType, uint32_t timeout);

/**
  * @brief  Poll for ADC idle
  * @param  ADCx     ADC instance.
  * @param  timeout  timeout value in millisecond.
  * @retval DDL status.
  */
DDL_Status_T DDL_ADC_PollForIdle(ADC_Type *ADCx, uint32_t timeout);

/**
  * @brief  Enable ADC interrupt with specified type.
  * @param  ADCx      ADC instance.
  * @param  intrMask  ADC interrupt type.
  * @retval None.
  */
void DDL_ADC_IntEnable(ADC_Type *ADCx, uint32_t intrMask);

/**
  * @brief  Disable ADC interrupt with specified type.
  * @param  ADCx      ADC instance.
  * @param  intrMask  ADC interrupt type.
  * @retval None.
  */
void DDL_ADC_IntDisable(ADC_Type *ADCx, uint32_t intrMask);

/**
  * @brief  Enable ADC DMA function.
  * @param  ADCx    ADC instance.
  * @retval None
  */
void DDL_ADC_DmaEnable(ADC_Type *ADCx);

/**
  * @brief  Disable ADC DMA function.
  * @param  ADCx    ADC instance.
  * @retval None
  */
void DDL_ADC_DmaDisable(ADC_Type *ADCx);

/**
  * @brief  Gets the converted value from the specify SOC.
  * @param  ADCx   ADC instance.
  * @param  SOCx   Specify which SOC to read, from 0~15.
  * @retval converted value
  */
uint16_t DDL_ADC_ReadSOCResult(ADC_Type *ADCx, ADC_SOCNumber_T SOCx);

/**
  * @brief  Gets the converted value from the specify PPB.
  * @param  ADCx    ADC instance.
  * @param  PPBx    Specify which PPB to read, from 0~3.
  * @retval converted value
  */
uint16_t DDL_ADC_ReadPPBResult(ADC_Type *ADCx, ADC_PPBNumber_T PPBx);

/**
  * @brief  Start specify SOC conversion.
  * @param  ADCx  ADC instance.
  * @param  SOCx  the SOC need to trigger.
  * @retval None
  */
void DDL_ADC_StartSOC_BySwTrig(ADC_Type *ADCx, ADC_SOCNumber_T SOCx);

/**
  * @brief  Trigger specified ADC SOC to sample by Hardware sources.
  * @param  ADCx     ADC instance.
  * @param  SOCx     ADC SOC index.
  * @retval None.
  */
void DDL_ADC_StartSOC_ByHwTrig(ADC_Type *ADCx, ADC_SOCNumber_T SOCx);

/**
  * @brief  Stop one ADC SOC to sample by Hardware sources.
  * @param  ADCx     ADC instance.
  * @param  SOCx     ADC SOC index.
  * @retval None.
  */
void DDL_ADC_StopSOC_ByHwTrig(ADC_Type *ADCx, ADC_SOCNumber_T SOCx);

/**
  * @brief  ADC Burst started by software trigger.
  * @param  ADCx     ADC instance.
  * @retval None.
  */
void DDL_ADC_StartBurst_BySwTrig(ADC_Type *ADCx);

/**
  * @brief  ADC Burst started by Hardware sources.
  * @param  ADCx     ADC instance.
  * @retval None.
  */
void DDL_ADC_StartBurst_ByHwTrig(ADC_Type *ADCx);

/**
  * @brief  Stop ADC Burst by Hardware sources.
  * @param  ADCx     ADC instance.
  * @retval None.
  */
void DDL_ADC_StopBurst_ByHwTrig(ADC_Type *ADCx);


#ifdef __cplusplus
}
#endif

#endif /*__KPM32xx_DDL_ADC_H */

