/**
  *****************************************************************************************************
  * @file    main.c
  * @author  Kiwi Software Team
  * @brief   This example code demonstrates how timg works.
  *
  * Copyright (c) 2024, Kiwi Instruments Co,. Ltd.
  *
  * Redistribution and use in source and binary forms, with or without modification,
  * are permitted provided that the following conditions are met:
  *
  *   1. Redistributions of source code must retain the above copyright notice,
  *      this list of conditions and the following disclaimer.
  *
  *   2. Redistributions in binary form must reproduce the above copyright notice,
  *      this list of conditions and the following disclaimer in the documentation
  *      and/or other materials provided with the distribution.
  *
  *   3. Neither the name of the copyright holder nor the names of its contributors
  *      may be used to endorse or promote products derived from this software without
  *      specific prior written permission.
  *
  * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
  * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
  * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
  * THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
  * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE
  * GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
  * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
  * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
  * THE POSSIBILITY OF SUCH DAMAGE.
  *****************************************************************************************************
  */

#include "main.h"
#include "kpm32xx_demoboard_ddl.h"


#if defined (__DEBUG_CONSOLE_PRINT__)
/**
  * @brief  This function is implemented for UART print PinMux configuration.
  * @param  None
  * @retval None
  */
void DebugConsole_PinMux_Config(void)
{
	/* SCI1 UART Tx */
	DDL_GPIO_Config2AltFunc(GPIOA, GPIO_PIN_7, GPIOA7_AF1_SCI1_TX_SDA_MOSI);
}
#endif


/**
  * @brief  This function is implemented for system clock configuration.
  *         where:
  *         System Clock source            = PLL (HIRC)
  *         System Clock frequency         = 100MHz
  * @param  None
  * @retval None
  */
void SystemClock_Config(void)
{
	RCC_PLLInit_T pllInitStruct    = {0};
	RCC_ClkInit_T rccClkInitStruct = {0};

	DDL_RCC_PllStructInit(&pllInitStruct);
	/* FCLK : 100MHz */
	pllInitStruct.pllPrescaler  = 4;
	if (DDL_RCC_PllConfig(&pllInitStruct) != DDL_OK)
	{
		while(1) ;
	}

	/* Select PLL as system clock source */
	rccClkInitStruct.sysClkSource  = RCC_SYSCLKSOURCE_PLLCLK;
	rccClkInitStruct.sysClkDivider = 0;
	if (DDL_RCC_ClkSrcConfig(&rccClkInitStruct) != DDL_OK)
	{
		while(1) ;
	}
}


/**
  * @brief  This function is implemented for led toggle.
  * @retval None
  */
void led_toggle(void)
{
	BSP_LEDx_Twinkle(LED_S1);
	BSP_LEDx_Twinkle(LED_S2);
	BSP_LEDx_Twinkle(LED_S3);
}


/**
  * @brief  This function is implemented for TIMG0 PWM PinMux configuration.
  * @param  None
  * @retval None
  */
void TIMG0_PinMux_Config(void)
{
	DDL_GPIO_Config2AltFunc(GPIOC, GPIO_PIN_8, GPIOC8_AF4_TIMG0_CH0);
}


/**
  * @brief  This function is implemented for timg capture pwm.
  * @retval None
  */
void timg_capture_init(void)
{
	TIMG_BaseInit_T tmrStruct;
	TIMG_CapturePara_T capStruct;

	DDL_TIMG_Capture_StructInit(&tmrStruct);
	DDL_TIMG_Capture_Init(TIMG0,&tmrStruct);

	/* ch0 init config */
	DDL_TIMG_Ch0_Capture_Polar_Config(TIMG0,0,0);
	DDL_TIMG_Capture_Channel0_StructInit(&capStruct);
	DDL_TIMG_Capture_Channel0_Config(TIMG0,&capStruct);

	/* ch1 init config */
	DDL_TIMG_Ch1_Capture_Polar_Config(TIMG0,1,0);
	DDL_TIMG_Capture_Channel1_StructInit(&capStruct);
	DDL_TIMG_Capture_Channel1_Config(TIMG0,&capStruct);

	/* slave mode */
	DDL_TIMG_SlaveMode_Config(TIMG0,TIMG_CLOCKSOURCE_TI0FPT,TIMG_SLAVEMODE_RESET);

	/* channel enable */
	DDL_TIMG_PWM_Enable(TIMG0, TIMG_CHANNEL0|TIMG_CHANNEL1);


	NVIC_SetPriority(TIMG0_IRQn, 0);
	NVIC_EnableIRQ(TIMG0_IRQn);
	DDL_TIMG_IntEnable(TIMG0,TIMG_DIER_UIE);
}


/**
  * @brief  The application entry point.
  * @retval int
  */
int main(void)
{
	uint16_t cycleCaptureValue = 0,dutyCaptureValue = 0;
	uint32_t cycleCaptureTime = 0,dutyCaptureTime = 0;
	/* Reset of all peripherals, Initializes the Flash interface and the Systick. */
	DDL_Init();
	SystemClock_Config();

	/* To make GPIO active */
	DDL_GPIO_Instance_Active();

	BSP_LEDx_Init(LED_S1);
	BSP_LEDx_Init(LED_S2);
	BSP_LEDx_Init(LED_S3);

#if defined (__DEBUG_CONSOLE_PRINT__)
	DebugConsole_PinMux_Config();
	DDL_SCI_Instance_Active(SCI1);
	DDL_SCIUart_DebugConsole_Init(SCI1);
#endif

    printf("It's a timg pwm capture demo(100M) ... \n");
	DDL_GPIO_Config2Output(GPIOD, GPIO_PIN_6, 1, GPIO_NOPULL);
	TIMG0_PinMux_Config();
	DDL_TIMG_Instance_Active(TIMG0);
	timg_capture_init();
	DDL_TIMG_Capture_Start(TIMG0);

	DDL_GPIO_WritePin(GPIOD, GPIO_PIN_6, GPIO_PIN_SET )  ;
	DDL_Delay(1);
	DDL_GPIO_WritePin(GPIOD, GPIO_PIN_6, GPIO_PIN_RESET )  ;
	DDL_Delay(1);
	DDL_GPIO_WritePin(GPIOD, GPIO_PIN_6, GPIO_PIN_SET )  ;
	DDL_Delay(1);
	DDL_GPIO_WritePin(GPIOD, GPIO_PIN_6, GPIO_PIN_RESET )  ;
	DDL_Delay(1);

	cycleCaptureValue = TIMG0->CCR0;
	cycleCaptureTime = cycleCaptureValue * 100 /100;
	printf("cycle capture time: %d us\n",cycleCaptureTime);

	dutyCaptureValue = TIMG0->CCR1;
	dutyCaptureTime = dutyCaptureValue * 100 /100;
	printf("duty capture time: %d us\n",dutyCaptureTime);
    while (1)
    {
		;
    }
}

