/**
  *****************************************************************************************************
  * @file    main.c
  * @author  Kiwi Software Team
  * @brief   This example code demonstrates how Flash works.
  *
  * Copyright (c) 2024, Kiwi Instruments Co,. Ltd.
  *
  * Redistribution and use in source and binary forms, with or without modification,
  * are permitted provided that the following conditions are met:
  *
  *   1. Redistributions of source code must retain the above copyright notice,
  *      this list of conditions and the following disclaimer.
  *
  *   2. Redistributions in binary form must reproduce the above copyright notice,
  *      this list of conditions and the following disclaimer in the documentation
  *      and/or other materials provided with the distribution.
  *
  *   3. Neither the name of the copyright holder nor the names of its contributors
  *      may be used to endorse or promote products derived from this software without
  *      specific prior written permission.
  *
  * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
  * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
  * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
  * THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
  * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE
  * GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
  * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
  * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
  * THE POSSIBILITY OF SUCH DAMAGE.
  *****************************************************************************************************
  */

#include "main.h"
#include "kpm32xx_demoboard_ddl.h"


#if defined (__DEBUG_CONSOLE_PRINT__)
/**
  * @brief  This function is implemented for UART print PinMux configuration.
  * @param  None
  * @retval None
  */
void DebugConsole_PinMux_Config(void)
{
	/* SCI1 UART Tx */
	DDL_GPIO_Config2AltFunc(GPIOA, GPIO_PIN_7, GPIOA7_AF1_SCI1_TX_SDA_MOSI);
}
#endif


/**
  * @brief  This function is implemented for system clock configuration.
  *         where:
  *         System Clock source            = PLL (HIRC)
  *         System Clock frequency         = 100MHz
  * @param  None
  * @retval None
  */
void SystemClock_Config(void)
{
	RCC_PLLInit_T pllInitStruct    = {0};
	RCC_ClkInit_T rccClkInitStruct = {0};

	DDL_RCC_PllStructInit(&pllInitStruct);
	/* FCLK : 100MHz */
	pllInitStruct.pllPrescaler  = 4;
	if (DDL_RCC_PllConfig(&pllInitStruct) != DDL_OK)
	{
		while(1) ;
	}

	/* Select PLL as system clock source */
	rccClkInitStruct.sysClkSource  = RCC_SYSCLKSOURCE_PLLCLK;
	rccClkInitStruct.sysClkDivider = 0;
	if (DDL_RCC_ClkSrcConfig(&rccClkInitStruct) != DDL_OK)
	{
		while(1) ;
	}
}


/**
  * @brief  Program 16 bytes(4 words) in the last page of Code Flash with 0xA5A5A5A5.
  * @param  None
  * @retval DDL_Status_T
  */
static DDL_Status_T Flash_Program_Demo(void)
{
	uint32_t i = 0;
	uint32_t err = 0;
	DDL_Status_T ret = DDL_OK;
	uint32_t readData = 0;
	uint32_t progData = 0xA5A5A5A5;
	uint32_t address = FLASH_CODE_BASE + FLASH_CODE_PAGESIZE * FLASH_CODE_MAX_SECTNUM;

	for (i = 0; i < 4; i++)
	{
		/* Program data with WORD type */
		ret = DDL_FLASH_ProgramWord(address, progData);
		if (ret != DDL_OK)
		{
			/* The following cases will cause program fail:
			 * 1. the address has been programed, need to erase the page first;
			 * 2. the page is protected, need to mass erase;
			 * 3. the chip is in security mode, need to mass erase;
			 */
			err = DDL_FLASH_GetErrorCode();
			printf("err = 0x%x.\n", err);
			return DDL_ERROR;
		}

		/* Read the 4 bytes back and check */
		DDL_FLASH_Read(address, (uint8_t *)&readData, 4);
		if (readData != progData)
		{
			return DDL_ERROR;
		}
		else
		{
			address += 4;
		}
	}

	return DDL_OK;
}


/**
  * @brief  Erase the last page of the Code Flash.
  * @param  None
  * @retval DDL_Status_T
  */
static DDL_Status_T Flash_Erase_Demo(void)
{
	DDL_Status_T ret = DDL_OK;

	ret = DDL_FLASH_EraseSector(FLASH_CODE_MAX_SECTNUM);

	return ret;
}


/**
  * @brief  The application entry point.
  * @retval int
  */
int main(void)
{
	DDL_Status_T ret = DDL_OK;

	/* Reset of all peripherals, Initializes the Flash interface and the Systick. */
	DDL_Init();
	SystemClock_Config();

	/* To make GPIO active */
	DDL_GPIO_Instance_Active();

#if defined (__DEBUG_CONSOLE_PRINT__)
	DebugConsole_PinMux_Config();
	DDL_SCI_Instance_Active(SCI1);
	DDL_SCIUart_DebugConsole_Init(SCI1);
#endif

    printf("Flash demo start:\n");

	ret = Flash_Program_Demo();
	if (ret != DDL_OK)
	{
		printf("	Flash program demo fail.\n");
	}
	else
	{
		printf("	Flash program demo success.\n");
	}

	ret = Flash_Erase_Demo();
	if (ret != DDL_OK)
	{
		printf("	Flash page erase demo fail.\n");
	}
	else
	{
		printf("	Flash page erase demo success.\n");
	}

    printf("Flash demo end.\n");
}

