/**
  *****************************************************************************************************
  * @file    kpm32xx_ddl_tims.c
  * @author  Kiwi Software Team
  * @brief   TIMS DDL module driver.
  *          This file provides firmware functions to manage the following
  *          functionalities:
  *          1. TIMS as basetimer.
  *          2. TIMS pwm output.
  *			     3. TIMS pwm capture.
  *			     4. TIMS pwm brake.
  *          5. TIMS as master/slave mode
  * @note
  *          V1.0.0, 2025/1/6.
  *
  * Copyright (c) 2024, Kiwi Instruments Co,. Ltd.
  *
  * Redistribution and use in source and binary forms, with or without modification,
  * are permitted provided that the following conditions are met:
  *
  *   1. Redistributions of source code must retain the above copyright notice,
  *      this list of conditions and the following disclaimer.
  *
  *   2. Redistributions in binary form must reproduce the above copyright notice,
  *      this list of conditions and the following disclaimer in the documentation
  *      and/or other materials provided with the distribution.
  *
  *   3. Neither the name of the copyright holder nor the names of its contributors
  *      may be used to endorse or promote products derived from this software without
  *      specific prior written permission.
  *
  * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
  * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
  * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
  * THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
  * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE
  * GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
  * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
  * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
  * THE POSSIBILITY OF SUCH DAMAGE.
  *****************************************************************************************************
  */
  
  
#include "kpm32xx_ddl.h"


#ifdef DDL_TIMS_MODULE_ENABLED


/**
  * @brief  make TIMS active.
  * @param  TIMSx  TIMS instance.
  *
  */
void DDL_TIMS_Instance_Active(TIMS_Type *TIMSx)
{
	if (TIMSx == TIMS0)
	{
		__DDL_RCC_TIMS0_ACTIVE();
		__DDL_RCC_TIMS0_CLK_RELEASE();
	}
	else if (TIMSx == TIMS1)
	{
		__DDL_RCC_TIMS1_ACTIVE();
		__DDL_RCC_TIMS1_CLK_RELEASE();
	}
	else if (TIMSx == TIMS2)
	{
		__DDL_RCC_TIMS2_ACTIVE();
		__DDL_RCC_TIMS2_CLK_RELEASE();
	}
	else if (TIMSx == TIMS3)
	{
		__DDL_RCC_TIMS3_ACTIVE();
		__DDL_RCC_TIMS3_CLK_RELEASE();
	}
}


/**
  * @brief  Configure the elements of structure TIMS_BaseInit_T to default values.
  *			use for timer 
  * @param  TIMS_BaseInit_T Pointer to a TIMS_BaseInit_T structure that contains
  *                  the configuration information for the given TIMS module.
  *
  * @retval TIMS
  */
void DDL_TIMS_TMR_StructInit(TIMS_BaseInit_T *pTmrStruct)
{
	pTmrStruct->countermode = TIMS_COUNTERMODE_COUNTER_UP;
	pTmrStruct->period = 19999;
	pTmrStruct->prescaler = 9;
	pTmrStruct->repetitionCounter = 0;
}


/**
  * @brief  Initializes the TIMS by the specified parameters in the TIMS_BaseInit_T.
  *			use for timer 
  * @param  TIMSx  TIMS instance.
  *			pTmrStruct Pointer to a TIMS_BaseInit_T structure that contains
  *                  the configuration information for the given TIMS module.
  *
  * @retval TIMS
  */
void DDL_TIMS_TMR_Init(TIMS_Type *TIMSx, TIMS_BaseInit_T *pTmrStruct)
{
	uint32_t tmpcr = 0;
	
	tmpcr = TIMSx->CR;
	tmpcr &= ~TIMS_COUNTERMODE_COUNTER_CENTER_3;
	tmpcr |= pTmrStruct->countermode;
	TIMSx->CR = tmpcr;
	TIMSx->ARR = pTmrStruct->period;
	TIMSx->PSC = pTmrStruct->prescaler;
	TIMSx->RCR = pTmrStruct->repetitionCounter;
}


/**
  * @brief  use for timer start.
  * @param  TIMSx  TIMS instance.
  * @retval None
  */
void DDL_TIMS_TMR_Start(TIMS_Type *TIMSx)
{
	MODIFY_REG(TIMSx->BDTR, TIMS_BDTR_MOE , 1 << TIMS_BDTR_MOE_Pos);
	MODIFY_REG(TIMSx->CR,  TIMS_CR_CEN ,  1 << TIMS_CR_CEN_Pos);
}


/**
  * @brief  use for timer stop.
  * @param  TIMSx  TIMS instance.
  * @retval None
  */
void DDL_TIMS_TMR_Stop(TIMS_Type *TIMSx)
{
	MODIFY_REG(TIMSx->BDTR, TIMS_BDTR_MOE , 0 << TIMS_BDTR_MOE_Pos);
	MODIFY_REG(TIMSx->CR,  TIMS_CR_CEN ,  0 << TIMS_CR_CEN_Pos);
}


/**
  * @brief  use for TIMS interrupt enable.
  * @param  intrMask : COMIE/UIE/C0IE/C1IE/C2IE/TIE/BIE
  *				
  * @retval None
  */
void DDL_TIMS_IntEnable(TIMS_Type *TIMSx, uint32_t intrMask)
{
	SET_BIT(TIMSx->IER,  intrMask);
}


/**
  * @brief  use for TIMS interrupt disable.
  * @param  intrMask : COMIE/UIE/C0IE/C1IE/C2IE/TIE/BIE
  *				
  * @retval None
  */
void DDL_TIMS_IntDisable(TIMS_Type *TIMSx, uint32_t intrMask)
{
	CLEAR_BIT(TIMSx->IER,  intrMask);
}


/**
  * @brief  Configure the elements of structure TIMS_BaseInit_T to default values.
  *			use for pwm output 
  * @param  pPwmStruct Pointer to a TIMS_BaseInit_T structure that contains
  *                  the configuration information for the given TIMS module.
  *  	
  * @retval None
  */
void DDL_TIMS_PWM_StructInit(TIMS_BaseInit_T *pPwmStruct)
{
	pPwmStruct->countermode = TIMS_COUNTERMODE_COUNTER_CENTER_3;
	pPwmStruct->period = 6249;
	pPwmStruct->prescaler = 0;
	pPwmStruct->repetitionCounter = 1;
}


/**
  * @brief  Initializes the TIMS by the specified parameters in the TIMS_BaseInit_T.
  *			use for pwm output 
  * @param  pTmrStruct Pointer to a TIMS_BaseInit_T structure that contains
  *                  the configuration information for the given TIMS module.
  *			TIMSx  TIMS instance.
  * @retval None
  */
void DDL_TIMS_PWM_Init(TIMS_Type *TIMSx, TIMS_BaseInit_T *pPwmStruct)
{
	uint32_t tmpcr = 0;
	
	tmpcr = TIMSx->CR;
	tmpcr &= ~TIMS_COUNTERMODE_COUNTER_CENTER_3;
	tmpcr |= pPwmStruct->countermode;
	TIMSx->CR = tmpcr;
	TIMSx->ARR = pPwmStruct->period;
	TIMSx->PSC = pPwmStruct->prescaler;
	TIMSx->RCR = pPwmStruct->repetitionCounter;
}


/**
  * @brief  use for pwm output channel1 High Resistsnce State.
  *			
  * @param  TIMSx  TIMS instance.
  *			chHighRes  : channel0 High Resistsnce State
  *         chnHighRes : channel0n High Resistsnce State
  *         1: high level
  * 		0: low level
  * @retval None
  */
void DDL_TIMS_Ch1_HighResistsnceState_Config(TIMS_Type *TIMSx, uint8_t chHighRes, uint8_t chnHighRes)
{
	MODIFY_REG(TIMSx->CR,(3<<TIMS_CR_HIZ1_Pos) , (chHighRes|(chnHighRes<<1)) << TIMS_CR_HIZ1_Pos);
}


/**
  * @brief  use for pwm output channel2 High Resistsnce State.
  *			
  * @param  TIMSx  TIMS instance.
  *			chHighRes  : channel1 High Resistsnce State
  *         1: high level
  * 		0: low level
  * @retval None
  */
void DDL_TIMS_Ch2_HighResistsnceState_Config(TIMS_Type *TIMSx, uint8_t chHighRes)
{
	MODIFY_REG(TIMSx->CR,(1<<TIMS_CR_HIZ2_Pos) , (chHighRes) << TIMS_CR_HIZ2_Pos);
}


/**
  * @brief  use for pwm output channel1 brake idle state.
  *			
  * @param  TIMSx  TIMS instance.
  *			chIdel  : channel1 idle state
  *         chnIdel : channel1n idle state
  *         1: high level
  * 		0: low level
  * @retval None
  */
void DDL_TIMS_Ch1_BrkIdelState_Config(TIMS_Type *TIMSx, uint8_t chIdel, uint8_t chnIdel)
{
	MODIFY_REG(TIMSx->CR,(3<<TIMS_CR_OIS1_Pos) , (chIdel|(chnIdel<<1)) << TIMS_CR_OIS1_Pos);
}


/**
  * @brief  use for pwm output channel2 brake idle state.
  *			
  * @param  TIMSx  TIMS instance.
  *			chIdel  : channel2 idle state
  *         1: high level
  * 		0: low level
  * @retval None
  */
void DDL_TIMS_Ch2_BrkIdelState_Config(TIMS_Type *TIMSx, uint8_t chIdel)
{
	MODIFY_REG(TIMSx->CR,(1<<TIMS_CR_OIS2_Pos) , (chIdel) << TIMS_CR_OIS2_Pos);
}


/**
  * @brief  use for pwm output channel1 duty set.
  *			
  * @param  TIMSx  TIMS instance.
			preload  : 
					0: effective immediately
					1: effective for the next cycle
*			chDuty: channel1 ccr set
  * @retval None
  */
void DDL_TIMS_Ch1_Duty_Config(TIMS_Type *TIMSx, uint8_t preload, uint16_t chDuty)
{
	MODIFY_REG(TIMSx->CCMR,TIMS_CCMR_OC1PE , preload << TIMS_CCMR_OC1PE_Pos);
	WRITE_REG(TIMSx->CCR1, chDuty );
}


/**
  * @brief  use for pwm output channel2 duty set.
  *			
  * @param  TIMSx  TIMS instance.
			preload  : 
					0: effective immediately
					1: effective for the next cycle
  *			chDuty: channel2 ccr set
  * @retval None
  */
void DDL_TIMS_Ch2_Duty_Config(TIMS_Type *TIMSx, uint8_t preload, uint16_t chDuty)
{
	MODIFY_REG(TIMSx->CCMR,TIMS_CCMR_OC2PE , preload << TIMS_CCMR_OC2PE_Pos);
	WRITE_REG(TIMSx->CCR2, chDuty );
}



/**
  * @brief  use for pwm channel1 output mode config.
  *			
  * @param  TIMSx  TIMS instance.
			mode  : 
				0: TIMS_OCMODE_FREEZE_CH1					
				1: TIMS_OCMODE_COMPARE_FORCED_HIGH_CH1   
				2: TIMS_OCMODE_COMPARE_FORCED_LOW_CH1    
				3: TIMS_OCMODE_COMPARE_TOGGLE_CH1			
				4: TIMS_OCMODE_FORCED_LOW_CH1    	  		
				5: TIMS_OCMODE_FORCED_HIGH_CH1       		
				6: TIMS_OCMODE_PWM0_CH1                  
				7: TIMS_OCMODE_PWM1_CH1                             

  * @retval None
  */
void DDL_TIMS_Ch1_PWM_Mode_Config(TIMS_Type *TIMSx,uint8_t mode)
{
	MODIFY_REG(TIMSx->CCMR,TIMS_CCMR_OC1M , mode << TIMS_CCMR_OC1M_Pos);
}


/**
  * @brief  use for pwm channel2 output mode config.
  *			
  * @param  TIMSx  TIMS instance.
			mode  : 
				0: TIMS_OCMODE_FREEZE_CH2					
				1: TIMS_OCMODE_COMPARE_FORCED_HIGH_CH2    
				2: TIMS_OCMODE_COMPARE_FORCED_LOW_CH2     
				3: TIMS_OCMODE_COMPARE_TOGGLE_CH2			
				4: TIMS_OCMODE_FORCED_HIGH_CH2      		
				5: TIMS_OCMODE_FORCED_LOW_CH2       		
				6: TIMS_OCMODE_PWM0_CH2                   
				7: TIMS_OCMODE_PWM1_CH2                           

  * @retval None
  */
void DDL_TIMS_Ch2_PWM_Mode_Config(TIMS_Type *TIMSx,uint8_t mode)
{
	MODIFY_REG(TIMSx->CCMR,TIMS_CCMR_OC2M , mode << TIMS_CCMR_OC2M_Pos);
}


/**
  * @brief  use for pwm channel1 output polar config.
  *			
  * @param  TIMSx  TIMS instance.
			chPolar: channel0 polar config  
			chnPolar: channel0n polar config 
  * @retval None
  */
void DDL_TIMS_Ch1_PWM_Polar_Config(TIMS_Type *TIMSx, uint8_t chPolar, uint8_t chnPolar)
{
	MODIFY_REG(TIMSx->CCER,5 << TIMS_CCER_CC1P_Pos , (chPolar|(chnPolar<<2)) << TIMS_CCER_CC1P_Pos);
}


/**
  * @brief  use for pwm channel2 output polar config.
  *			
  * @param  TIMSx  TIMS instance.
			chPolar: channel0 polar config  
			chnPolar: channel0n polar config 
  * @retval None
  */
void DDL_TIMS_Ch2_PWM_Polar_Config(TIMS_Type *TIMSx, uint8_t chPolar, uint8_t chnPolar)
{
	MODIFY_REG(TIMSx->CCER,5 << TIMS_CCER_CC2P_Pos , (chPolar|(chnPolar<<2)) << TIMS_CCER_CC2P_Pos);
}


/**
  * @brief  use for pwm output channel enable.
  *			
  * @param  TIMSx  TIMS instance.
			chEnable: channel enable select  
  *
  * @retval None
  */
void DDL_TIMS_PWM_Enable(TIMS_Type *TIMSx, uint32_t chEnable)
{
	TIMSx->CCER |= chEnable;
}


/**
  * @brief  use for pwm output start.
  * @param  TIMSx  TIMS instance.
  * @retval None
  */
void DDL_TIMS_PWM_Start(TIMS_Type *TIMSx)
{
	MODIFY_REG(TIMSx->BDTR, TIMS_BDTR_MOE , 1 << TIMS_BDTR_MOE_Pos);
	MODIFY_REG(TIMSx->CR,  TIMS_CR_CEN ,  1 << TIMS_CR_CEN_Pos);
}


/**
  * @brief  use for pwm output stop.
  * @param  TIMSx  TIMS instance.
  * @retval None
  */
void DDL_TIMS_PWM_Stop(TIMS_Type *TIMSx)
{
	MODIFY_REG(TIMSx->BDTR, TIMS_BDTR_MOE , 0 << TIMS_BDTR_MOE_Pos);
	MODIFY_REG(TIMSx->CR,  TIMS_CR_CEN ,  0 << TIMS_CR_CEN_Pos);
}


/**
  * @brief  Configure the deadtime value
  * @param  TIMSx  TIMS instance.
  *			deadTime:  config deadtime value 
  *
  * @retval None
  */
void DDL_TIMS_DeadTime_Config(TIMS_Type *TIMSx, uint8_t deadTime)
{
	MODIFY_REG(TIMSx->BDTR, 0xff , deadTime);
}


/**
  * @brief  use for pwm output brake config.
  * @param  brkSrc : 
				 TIMS_BRK_SOURCE_EBUS_CHx	
                 TIMS_BRAKE_IO_SRC         
                 TIMS_CPU_HALT_SRC         
                 TIMS_SRAM_PARITY_ERR_SRC  
                 TIMS_PLL_LOSS_PEND_SRC    
                 TIMS_MCLK_LOSS_PEND_SRC   
			polar :
				0: low  level
				1: high level
  * @retval None
  */
void DDL_TIMS_PWM_Brake_Config(TIMS_Type *TIMSx, uint32_t brkSrc, uint8_t polar)
{
	if (TIMSx == TIMS0)
	{
		MODIFY_REG(TIMSx->BDTR, TIMS_BDTR_BKE , TIMS_BDTR_BKE);
		WRITE_REG(TIM_COMMON->TIMS0_SBCR ,  brkSrc|(polar<<TIMS0_SBCR_BRK_POL_Pos));
	}
	else if (TIMSx == TIMS1)
	{
		MODIFY_REG(TIMSx->BDTR, TIMS_BDTR_BKE , TIMS_BDTR_BKE);
		WRITE_REG(TIM_COMMON->TIMS1_SBCR ,  brkSrc|(polar<<TIMS1_SBCR_BRK_POL_Pos));
	}
	else if (TIMSx == TIMS2)
	{
		MODIFY_REG(TIMSx->BDTR, TIMS_BDTR_BKE , TIMS_BDTR_BKE);
		WRITE_REG(TIM_COMMON->TIMS2_SBCR ,  brkSrc|(polar<<TIMS2_SBCR_BRK_POL_Pos));
	}
	else if (TIMSx == TIMS3)
	{
		MODIFY_REG(TIMSx->BDTR, TIMS_BDTR_BKE , TIMS_BDTR_BKE);
		WRITE_REG(TIM_COMMON->TIMS3_SBCR ,  brkSrc|(polar<<TIMS3_SBCR_BRK_POL_Pos));
	}
}


/**
  * @brief  pwm auto output enable.
  * @param  TIMSx  TIMS instance.				
  * @retval None
  */
void DDL_TIMS_Auto_Output_Enable(TIMS_Type *TIMSx)
{
	SET_BIT(TIMSx->BDTR,TIMS_BDTR_AOE);
}


/**
  * @brief  pwm auto output disable.
  * @param  TIMSx  TIMS instance.				
  * @retval None
  */
void DDL_TIMS_Auto_Output_Disable(TIMS_Type *TIMSx)
{
	CLEAR_BIT(TIMSx->BDTR,TIMS_BDTR_AOE);
}


/**
  * @brief  TIMS as master trgo.
  * @param  TIMSx  TIMS instance.
  * @param  trgoSrc : RESET/ENABLE/UPDATE/COMPAREPULSE/COMPAREOCxREF
  *				
  * @retval None
  */
void DDL_TIMS_AsMaster_Trgo(TIMS_Type *TIMSx,uint32_t trgoSrc)
{
	MODIFY_REG(TIMSx->MCR, TIMS_MCR_MMS , trgoSrc<<TIMS_MCR_MMS_Pos);
}


/**
  * @brief  TIMS soft brake.
  * @param  TIMSx  TIMS instance.
  * @retval None
  */
void DDL_TIMS_SoftBrake(TIMS_Type *TIMSx)
{
	WRITE_REG(TIMSx->EGR ,  TIMS_EGR_BG);
}


/**
  * @brief  Configure the elements of structure TIMS_BaseInit_T to default values.
  *			use for pwm capture 
  * @param  pPwmStruct Pointer to a TIMS_BaseInit_T structure that contains
  *                  the configuration information for the given TIMS module.
  *  	
  * @retval None
  */
void DDL_TIMS_Capture_StructInit(TIMS_BaseInit_T *pPwmStruct)
{
	pPwmStruct->countermode = TIMS_COUNTERMODE_COUNTER_UP;
	pPwmStruct->period = 0xfffe;
	pPwmStruct->prescaler = 99;
	pPwmStruct->repetitionCounter = 0;
}


/**
  * @brief  Initializes the TIMS by the specified parameters in the TIMS_BaseInit_T.
  *			use for pwm capture 
  * @param  pTmrStruct Pointer to a TIMS_BaseInit_T structure that contains
  *                  the configuration information for the given TIMS module.
  *			TIMSx  TIMS instance.
  * @retval None
  */
void DDL_TIMS_Capture_Init(TIMS_Type *TIMSx, TIMS_BaseInit_T *pPwmStruct)
{
	uint32_t tmpcr = 0;
	
	tmpcr = TIMSx->CR;
	tmpcr &= ~TIMS_COUNTERMODE_COUNTER_UP;
	tmpcr |= pPwmStruct->countermode;
	TIMSx->CR = tmpcr;
	TIMSx->ARR = pPwmStruct->period;
	TIMSx->PSC = pPwmStruct->prescaler;
	TIMSx->RCR = pPwmStruct->repetitionCounter;
}


/**
  * @brief  use for capture channel1 output polar config.
  *			
  * @param  TIMSx  TIMS instance.
			chPolar: chnPolar
	  rising   0   :   0	
	  failing  1   :   0	
	  both     1   :   1	
  * @retval None
  */
void DDL_TIMS_Ch1_Capture_Polar_Config(TIMS_Type *TIMSx, uint8_t chPolar, uint8_t chnPolar)
{
	MODIFY_REG(TIMSx->CCER,TIMS_CCER_CC1P , chPolar << TIMS_CCER_CC1P_Pos);
	MODIFY_REG(TIMSx->CCER,TIMS_CCER_CC1NP , chnPolar << TIMS_CCER_CC1NP_Pos);
}


/**
  * @brief  use for capture channel2 output polar config.
  *			
  * @param  TIMSx  TIMS instance.
			chPolar: chnPolar
	  rising   0   :   0	
	  failing  1   :   0	
	  both     1   :   1	
  * @retval None
  */
void DDL_TIMS_Ch2_Capture_Polar_Config(TIMS_Type *TIMSx, uint8_t chPolar, uint8_t chnPolar)
{
	MODIFY_REG(TIMSx->CCER,TIMS_CCER_CC2P , chPolar << TIMS_CCER_CC2P_Pos);
	MODIFY_REG(TIMSx->CCER,TIMS_CCER_CC2NP , chnPolar << TIMS_CCER_CC2NP_Pos);
}


/**
  * @brief  Configure the elements of structure TIMS_CapturePara_T to default values.
  *			use for capture 
  * @param  pCapStruct Pointer to a TIMS_CapturePara_T structure that contains
  *                  the configuration information for the given TIMS module.
  *
  * @retval None
  */
void DDL_TIMS_Capture_Channel1_StructInit(TIMS_CapturePara_T *pCapStruct)
{
	pCapStruct->chxDirectSel = TIMS_IC1_DIRECTTI1;
	pCapStruct->chxFilter    = 0;
	pCapStruct->chxFilterEventNum = 0;
	pCapStruct->chxPsc       = 0;
}


/**
  * @brief  Initializes the TIMS by the specified parameters in the TIMS_CapturePara_T.
  *			use for capture 
  * @param  TIMSx  TIMS instance.
			pCapStruct Pointer to a TIMS_CapturePara_T structure that contains
  *                  the configuration information for the given TIMS module.
  *
  * @retval None
  */
void DDL_TIMS_Capture_Channel1_Config(TIMS_Type *TIMSx, TIMS_CapturePara_T *pCapStruct)
{
	uint32_t tmp = TIMSx->CCMR;
	tmp |= pCapStruct->chxDirectSel   << TIMS_CCMR_CC1S_Pos;
	tmp |= pCapStruct->chxFilterEventNum  << TIMS_CCMR_IC1FC_Pos;
	tmp |= pCapStruct->chxFilter      << TIMS_CCMR_IC1F_Pos;
	tmp |= pCapStruct->chxPsc         << TIMS_CCMR_IC1PSC_Pos;
	WRITE_REG(TIMSx->CCMR, tmp);       
}


/**
  * @brief  Configure the elements of structure TIMS_CapturePara_T to default values.
  *			use for capture 
  * @param  pCapStruct Pointer to a TIMS_CapturePara_T structure that contains
  *                  the configuration information for the given TIMS module.
  *
  * @retval None
  */
void DDL_TIMS_Capture_Channel2_StructInit(TIMS_CapturePara_T *pCapStruct)
{
	pCapStruct->chxDirectSel = TIMS_IC2_DIRECTTI1;
	pCapStruct->chxFilter    = 0;
	pCapStruct->chxFilterEventNum = 0;
	pCapStruct->chxPsc       = 0;
}


/**
  * @brief  Initializes the TIMS by the specified parameters in the TIMS_CapturePara_T.
  *			use for capture 
  * @param  TIMSx  TIMS instance.
			pCapStruct Pointer to a TIMS_CapturePara_T structure that contains
  *                  the configuration information for the given TIMS module.
  *
  * @retval None
  */
void DDL_TIMS_Capture_Channel2_Config(TIMS_Type *TIMSx, TIMS_CapturePara_T *pCapStruct)
{
	uint32_t tmp = TIMSx->CCMR;
	tmp |= pCapStruct->chxDirectSel   << TIMS_CCMR_CC2S_Pos;
	tmp |= pCapStruct->chxFilterEventNum  << TIMS_CCMR_IC2FC_Pos;
	tmp |= pCapStruct->chxFilter      << TIMS_CCMR_IC2F_Pos;
	tmp |= pCapStruct->chxPsc         << TIMS_CCMR_IC2PSC_Pos;
	WRITE_REG(TIMSx->CCMR, tmp);        
}


/**
  * @brief  use for TIMS slave mode config.
  *			
  * @param  TIMSx  TIMS instance.
			triggerSignal:  ITRx/TI0FED/TI0FPT/TI1FPT
			slaveMode:		reset/gate/trigger
  * @retval None
  */
void DDL_TIMS_SlaveMode_Config(TIMS_Type *TIMSx, uint8_t triggerSignal, uint8_t slaveMode)
{
	uint32_t tmp = 0;
	tmp = (triggerSignal<<20)|(slaveMode<<16);
	MODIFY_REG(TIMSx->MCR,0x3f,tmp );
}


/**
  * @brief  use for timer start.
  * @param  TIMSx  TIMS instance.
  * @retval None
  */
void DDL_TIMS_Capture_Start(TIMS_Type *TIMSx)
{
	SET_BIT(TIMSx->BDTR, TIMS_BDTR_MOE);
	SET_BIT(TIMSx->CR,   TIMS_CR_CEN);
}


/**
  * @brief  use for timer stop.
  * @param  TIMSx  TIMS instance.
  * @retval None
  */
void DDL_TIMS_Capture_Stop(TIMS_Type *TIMSx)
{
	CLEAR_BIT(TIMSx->BDTR, TIMS_BDTR_MOE);
	CLEAR_BIT(TIMSx->CR,   TIMS_CR_CEN);
}


#endif /* DDL_TIMS_MODULE_ENABLED */















