/**
  *****************************************************************************************************
  * @file    kpm32xx_ddl_timg.c
  * @author  Kiwi Software Team
  * @brief   TIMG DDL module driver.
  *          This file provides firmware functions to manage the following
  *          functionalities:
  *          1. TIMG as basetimer.
  *          2. TIMG pwm output.
  *			     3. TIMG pwm capture.
  *          4. TIMG as master/slave mode
  * @note
  *          V1.0.0, 2025/1/6.
  *
  * Copyright (c) 2024, Kiwi Instruments Co,. Ltd.
  *
  * Redistribution and use in source and binary forms, with or without modification,
  * are permitted provided that the following conditions are met:
  *
  *   1. Redistributions of source code must retain the above copyright notice,
  *      this list of conditions and the following disclaimer.
  *
  *   2. Redistributions in binary form must reproduce the above copyright notice,
  *      this list of conditions and the following disclaimer in the documentation
  *      and/or other materials provided with the distribution.
  *
  *   3. Neither the name of the copyright holder nor the names of its contributors
  *      may be used to endorse or promote products derived from this software without
  *      specific prior written permission.
  *
  * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
  * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
  * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
  * THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
  * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE
  * GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
  * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
  * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
  * THE POSSIBILITY OF SUCH DAMAGE.
  *****************************************************************************************************
  */
  
  
#include "kpm32xx_ddl.h"


#ifdef DDL_TIMG_MODULE_ENABLED


/**
  * @brief  make TIMG active.
  * @param  TIMGx  TIMG instance.
  *
  */
void DDL_TIMG_Instance_Active(TIMG_Type *TIMGx)
{
	if (TIMGx == TIMG0)
	{
		__DDL_RCC_TIMG0_ACTIVE();
		__DDL_RCC_TIMG0_CLK_RELEASE();
	}
	else if (TIMGx == TIMG1)
	{
		__DDL_RCC_TIMG1_ACTIVE();
		__DDL_RCC_TIMG1_CLK_RELEASE();
	}
	else if (TIMGx == TIMG2)
	{
		__DDL_RCC_TIMG2_ACTIVE();
		__DDL_RCC_TIMG2_CLK_RELEASE();
	}
}


/**
  * @brief  Configure the elements of structure TIMG_BaseInit_T to default values.
  *			use for timer 
  * @param  TIMG_BaseInit_T Pointer to a TIMG_BaseInit_T structure that contains
  *                  the configuration information for the given TIMG module.
  *
  * @retval None
  */
void DDL_TIMG_TMR_StructInit(TIMG_BaseInit_T *pTmrStruct)
{
	pTmrStruct->countermode = TIMG_COUNTERMODE_COUNTER_UP;
	pTmrStruct->period = 19999;
	pTmrStruct->prescaler = 9;
}


/**
  * @brief  Initializes the TIMG by the specified parameters in the TIMG_BaseInit_T.
  *			use for timer 
  * @param  TIMGx  TIMG instance.
  *			pTmrStruct Pointer to a TIMG_BaseInit_T structure that contains
  *                  the configuration information for the given TIMG module.
  *
  * @retval None
  */
void DDL_TIMG_TMR_Init(TIMG_Type *TIMGx, TIMG_BaseInit_T *pTmrStruct)
{
	uint32_t tmpcr = 0;
	
	tmpcr = TIMGx->CR0;
	tmpcr &= ~TIMG_COUNTERMODE_COUNTER_CENTER_3;
	tmpcr |= pTmrStruct->countermode;
	TIMGx->CR0 = tmpcr;
	TIMGx->ARR = pTmrStruct->period;
	TIMGx->PSC = pTmrStruct->prescaler;
}


/**
  * @brief  use for timer start.
  * @param  TIMGx  TIMG instance.
  * @retval None
  */
void DDL_TIMG_TMR_Start(TIMG_Type *TIMGx)
{
	SET_BIT(TIMGx->CR0,   TIMG_CR0_CEN);;
}


/**
  * @brief  use for timer stop.
  * @param  TIMGx  TIMG instance.
  * @retval None
  */
void DDL_TIMG_TMR_Stop(TIMG_Type *TIMGx)
{
	CLEAR_BIT(TIMGx->CR0,   TIMG_CR0_CEN);
}


/**
  * @brief  use for TIMG interrupt enable.
  * @param  intrMask : COMIE/UIE/C0IE/C1IE/C2IE/TIE/BIE
  *				
  * @retval None
  */
void DDL_TIMG_IntEnable(TIMG_Type *TIMGx, uint32_t intrMask)
{
	SET_BIT(TIMGx->DIER,  intrMask);
}


/**
  * @brief  use for TIMG interrupt disable.
  * @param  intrMask : COMIE/UIE/C0IE/C1IE/C2IE/TIE/BIE
  *				
  * @retval None
  */
void DDL_TIMG_IntDisable(TIMG_Type *TIMGx, uint32_t intrMask)
{
	CLEAR_BIT(TIMGx->DIER,  intrMask);
}


/**
  * @brief  Configure the elements of structure TIMG_BaseInit_T to default values.
  *			use for pwm output 
  * @param  pPwmStruct Pointer to a TIMG_BaseInit_T structure that contains
  *                  the configuration information for the given TIMG module.
  *  	
  * @retval None
  */
void DDL_TIMG_PWM_StructInit(TIMG_BaseInit_T *pPwmStruct)
{
	pPwmStruct->countermode = TIMG_COUNTERMODE_COUNTER_CENTER_3;
	pPwmStruct->period = 6249;
	pPwmStruct->prescaler = 0;
}


/**
  * @brief  Initializes the TIMG by the specified parameters in the TIMG_BaseInit_T.
  *			use for pwm output 
  * @param  pTmrStruct Pointer to a TIMG_BaseInit_T structure that contains
  *                  the configuration information for the given TIMG module.
  *			TIMGx  TIMG instance.
  * @retval None
  */
void DDL_TIMG_PWM_Init(TIMG_Type *TIMGx, TIMG_BaseInit_T *pPwmStruct)
{
	uint32_t tmpcr = 0;
	
	tmpcr = TIMGx->CR0;
	tmpcr &= ~TIMG_COUNTERMODE_COUNTER_CENTER_3;
	tmpcr |= pPwmStruct->countermode;
	TIMGx->CR0 = tmpcr;
	TIMGx->ARR = pPwmStruct->period;
	TIMGx->PSC = pPwmStruct->prescaler;
}


/**
  * @brief  use for pwm output channel0 duty set.
  *			
  * @param  TIMGx  TIMG instance.
			preload  : 
					0: effective immediately
					1: effective for the next cycle
  *         chDuty  : channel0 ccr set
  * @retval None
  */
void DDL_TIMG_Ch0_Duty_Config(TIMG_Type *TIMGx, uint8_t preload, uint32_t chDuty)
{
	MODIFY_REG(TIMGx->CCMR0,TIMG_CCMR0_OC0PE , preload << TIMG_CCMR0_OC0PE_Pos);
	WRITE_REG(TIMGx->CCR0, chDuty );
}


/**
  * @brief  use for pwm output channel1 duty set.
  *			
  * @param  TIMGx  TIMG instance.
			preload  : 
					0: effective immediately
					1: effective for the next cycle
  *			chDuty  : channel1 ccr set
  * @retval None
  */
void DDL_TIMG_Ch1_Duty_Config(TIMG_Type *TIMGx, uint8_t preload, uint32_t chDuty)
{
	MODIFY_REG(TIMGx->CCMR0,TIMG_CCMR0_OC1PE , preload << TIMG_CCMR0_OC1PE_Pos);
	WRITE_REG(TIMGx->CCR1, chDuty );
}


/**
  * @brief  use for pwm output channel2 duty set.
  *			
  * @param  TIMGx  TIMG instance.
			preload  : 
					0: effective immediately
					1: effective for the next cycle
  *			chDuty  : channel2 ccr set
  * @retval None
  */
void DDL_TIMG_Ch2_Duty_Config(TIMG_Type *TIMGx, uint8_t preload, uint32_t chDuty)
{
	MODIFY_REG(TIMGx->CCMR1,TIMG_CCMR1_OC2PE , preload << TIMG_CCMR1_OC2PE_Pos);
	WRITE_REG(TIMGx->CCR2, chDuty );
}


/**
  * @brief  use for pwm output channel3 duty set.
  *			
  * @param  TIMGx  TIMG instance.
			preload  : 
					0: effective immediately
					1: effective for the next cycle
  *			chDuty  : channel3 ccr set
  * @retval None
  */
void DDL_TIMG_Ch3_Duty_Config(TIMG_Type *TIMGx, uint8_t preload, uint32_t chDuty)
{
	MODIFY_REG(TIMGx->CCMR1,TIMG_CCMR1_OC3PE , preload << TIMG_CCMR1_OC3PE_Pos);
	WRITE_REG(TIMGx->CCR3, chDuty );
}


/**
  * @brief  use for pwm channel0 output mode config.
  *			
  * @param  TIMGx  TIMG instance.
			mode  : 
				0: TIMG_OCMODE_FREEZE_CH0					
				1: TIMG_OCMODE_COMPARE_FORCED_HIGH_CH0   
				2: TIMG_OCMODE_COMPARE_FORCED_LOW_CH0    
				3: TIMG_OCMODE_COMPARE_TOGGLE_CH0			
				4: TIMG_OCMODE_FORCED_LOW_CH0    	  		
				5: TIMG_OCMODE_FORCED_HIGH_CH0       		
				6: TIMG_OCMODE_PWM0_CH0                  
				7: TIMG_OCMODE_PWM1_CH0                  
				C: TIMG_OCMODE_COMBINATION_MODE_PWM0_CH0 
				D: TIMG_OCMODE_COMBINATION_MODE_PWM1_CH0 
				E: TIMG_OCMODE_ASYM_PWM0_CH0             
				F: TIMG_OCMODE_ASYM_PWM1_CH0             

  * @retval None
  */
void DDL_TIMG_Ch0_PWM_Mode_Config(TIMG_Type *TIMGx,uint8_t mode)
{
	MODIFY_REG(TIMGx->CCMR0,TIMG_CCMR0_OC0M , mode << TIMG_CCMR0_OC0M_Pos);
}


/**
  * @brief  use for pwm channel1 output mode config.
  *			
  * @param  TIMGx  TIMG instance.
			mode  : 
				0: TIMG_OCMODE_FREEZE_CH1					
				1: TIMG_OCMODE_COMPARE_FORCED_HIGH_CH1   
				2: TIMG_OCMODE_COMPARE_FORCED_LOW_CH1    
				3: TIMG_OCMODE_COMPARE_TOGGLE_CH1			
				4: TIMG_OCMODE_FORCED_LOW_CH1    	  		
				5: TIMG_OCMODE_FORCED_HIGH_CH1       		
				6: TIMG_OCMODE_PWM0_CH1                  
				7: TIMG_OCMODE_PWM1_CH1                  
				C: TIMG_OCMODE_COMBINATION_MODE_PWM0_CH1 
				D: TIMG_OCMODE_COMBINATION_MODE_PWM1_CH1 
				E: TIMG_OCMODE_ASYM_PWM0_CH1             
				F: TIMG_OCMODE_ASYM_PWM1_CH1             

  * @retval None
  */
void DDL_TIMG_Ch1_PWM_Mode_Config(TIMG_Type *TIMGx,uint8_t mode)
{
	MODIFY_REG(TIMGx->CCMR0,TIMG_CCMR0_OC1M , mode << TIMG_CCMR0_OC1M_Pos);
}


/**
  * @brief  use for pwm channel2 output mode config.
  *			
  * @param  TIMGx  TIMG instance.
			mode  : 
				0: TIMG_OCMODE_FREEZE_CH2					
				1: TIMG_OCMODE_COMPARE_FORCED_HIGH_CH2    
				2: TIMG_OCMODE_COMPARE_FORCED_LOW_CH2     
				3: TIMG_OCMODE_COMPARE_TOGGLE_CH2			
				4: TIMG_OCMODE_FORCED_HIGH_CH2      		
				5: TIMG_OCMODE_FORCED_LOW_CH2       		
				6: TIMG_OCMODE_PWM0_CH2                   
				7: TIMG_OCMODE_PWM1_CH2                   
				C: TIMG_OCMODE_COMBINATION_MODE_PWM0_CH2  
				D: TIMG_OCMODE_COMBINATION_MODE_PWM1_CH2  
				E: TIMG_OCMODE_ASYM_PWM0_CH2              
				F: TIMG_OCMODE_ASYM_PWM1_CH2              

  * @retval None
  */
void DDL_TIMG_Ch2_PWM_Mode_Config(TIMG_Type *TIMGx,uint8_t mode)
{
	MODIFY_REG(TIMGx->CCMR1,TIMG_CCMR1_OC2M , mode << TIMG_CCMR1_OC2M_Pos);
}


/**
  * @brief  use for pwm channel3 output mode config.
  *			
  * @param  TIMGx  TIMG instance.
			mode  : 
				0: TIMG_OCMODE_FREEZE_CH3					
				1: TIMG_OCMODE_COMPARE_FORCED_HIGH_CH3     
				2: TIMG_OCMODE_COMPARE_FORCED_LOW_CH3      
				3: TIMG_OCMODE_COMPARE_TOGGLE_CH3			
				4: TIMG_OCMODE_FORCED_HIGH_CH3      		
				5: TIMG_OCMODE_FORCED_LOW_CH3       		
				6: TIMG_OCMODE_PWM0_CH3                    
				7: TIMG_OCMODE_PWM1_CH3                    
				C: TIMG_OCMODE_COMBINATION_MODE_PWM0_CH3   
				D: TIMG_OCMODE_COMBINATION_MODE_PWM1_CH3   
				E: TIMG_OCMODE_ASYM_PWM0_CH3               
				F: TIMG_OCMODE_ASYM_PWM1_CH3               

  * @retval None
  */
void DDL_TIMG_Ch3_PWM_Mode_Config(TIMG_Type *TIMGx,uint8_t mode)
{
	MODIFY_REG(TIMGx->CCMR1,TIMG_CCMR1_OC3M , mode << TIMG_CCMR1_OC3M_Pos);
}


/**
  * @brief  use for pwm channel0 output polar config.
  *			
  * @param  TIMGx  TIMG instance.
			chPolar: channel0 polar config  
			chnPolar: channel0n polar config 
  * @retval None
  */
void DDL_TIMG_Ch0_PWM_Polar_Config(TIMG_Type *TIMGx, uint8_t chPolar, uint8_t chnPolar)
{
	MODIFY_REG(TIMGx->CCER,5 << TIMG_CCER_CC0P_Pos , (chPolar|(chnPolar<<2)) << TIMG_CCER_CC0P_Pos);
}


/**
  * @brief  use for pwm channel1 output polar config.
  *			
  * @param  TIMGx  TIMG instance.
			chPolar: channel0 polar config  
			chnPolar: channel0n polar config 
  * @retval None
  */
void DDL_TIMG_Ch1_PWM_Polar_Config(TIMG_Type *TIMGx, uint8_t chPolar, uint8_t chnPolar)
{
	MODIFY_REG(TIMGx->CCER,5 << TIMG_CCER_CC1P_Pos , (chPolar|(chnPolar<<2)) << TIMG_CCER_CC1P_Pos);
}


/**
  * @brief  use for pwm channel2 output polar config.
  *			
  * @param  TIMGx  TIMG instance.
			chPolar: channel0 polar config  
			chnPolar: channel0n polar config 
  * @retval None
  */
void DDL_TIMG_Ch2_PWM_Polar_Config(TIMG_Type *TIMGx, uint8_t chPolar, uint8_t chnPolar)
{
	MODIFY_REG(TIMGx->CCER,5 << TIMG_CCER_CC2P_Pos , (chPolar|(chnPolar<<2)) << TIMG_CCER_CC2P_Pos);
}


/**
  * @brief  use for pwm channel3 output polar config.
  *			
  * @param  TIMGx  TIMG instance.
			chPolar: channel3 polar config  
			chnPolar: channel3n polar config 
  * @retval None
  */
void DDL_TIMG_Ch3_PWM_Polar_Config(TIMG_Type *TIMGx, uint8_t chPolar, uint8_t chnPolar)
{
	MODIFY_REG(TIMGx->CCER,5 << TIMG_CCER_CC3P_Pos , (chPolar|(chnPolar<<2)) << TIMG_CCER_CC3P_Pos);
}


/**
  * @brief  use for pwm output channel enable.
  *			
  * @param  TIMGx  TIMG instance.
			chEnable: channel enable select  
  *
  * @retval None
  */
void DDL_TIMG_PWM_Enable(TIMG_Type *TIMGx, uint32_t chEnable)
{
	TIMGx->CCER |= chEnable;
}


/**
  * @brief  use for pwm output start.
  * @param  TIMGx  TIMG instance.
  * @retval None
  */
void DDL_TIMG_PWM_Start(TIMG_Type *TIMGx)
{
	MODIFY_REG(TIMGx->CR0,  TIMG_CR0_CEN ,  1 << TIMG_CR0_CEN_Pos);
}


/**
  * @brief  use for pwm output stop.
  * @param  TIMGx  TIMG instance.
  * @retval None
  */
void DDL_TIMG_PWM_Stop(TIMG_Type *TIMGx)
{
	MODIFY_REG(TIMGx->CR0,  TIMG_CR0_CEN ,  0 << TIMG_CR0_CEN_Pos);
}



/**
  * @brief  Configure the elements of structure TIMG_BaseInit_T to default values.
  *			use for pwm capture 
  * @param  pPwmStruct Pointer to a TIMG_BaseInit_T structure that contains
  *                  the configuration information for the given TIMG module.
  *  	
  * @retval None
  */
void DDL_TIMG_Capture_StructInit(TIMG_BaseInit_T *pPwmStruct)
{
	pPwmStruct->countermode = TIMG_COUNTERMODE_COUNTER_UP;
	pPwmStruct->period = 0xfffe;
	pPwmStruct->prescaler = 99;
}


/**
  * @brief  Initializes the TIMG by the specified parameters in the TIMG_BaseInit_T.
  *			use for pwm capture 
  * @param  pTmrStruct Pointer to a TIMG_BaseInit_T structure that contains
  *                  the configuration information for the given TIMG module.
  *			TIMGx  TIMG instance.
  * @retval None
  */
void DDL_TIMG_Capture_Init(TIMG_Type *TIMGx, TIMG_BaseInit_T *pPwmStruct)
{
	uint32_t tmpcr = 0;
	
	tmpcr = TIMGx->CR0;
	tmpcr &= ~TIMG_COUNTERMODE_COUNTER_CENTER_3;
	tmpcr |= pPwmStruct->countermode;
	TIMGx->CR0 = tmpcr;
	TIMGx->ARR = pPwmStruct->period;
	TIMGx->PSC = pPwmStruct->prescaler;
}


/**
  * @brief  use for capture channel0 output polar config.
  *			
  * @param  TIMGx  TIMG instance.
			chPolar: chnPolar
	  rising   0   :   0	
	  failing  1   :   0	
	  both     1   :   1	
  * @retval None
  */
void DDL_TIMG_Ch0_Capture_Polar_Config(TIMG_Type *TIMGx, uint8_t chPolar, uint8_t chnPolar)
{
	MODIFY_REG(TIMGx->CCER,TIMG_CCER_CC0P , chPolar << TIMG_CCER_CC0P_Pos);
	MODIFY_REG(TIMGx->CCER,TIMG_CCER_CC0NP , chnPolar << TIMG_CCER_CC0NP_Pos);
}


/**
  * @brief  use for capture channel1 output polar config.
  *			
  * @param  TIMGx  TIMG instance.
			chPolar: chnPolar
	  rising   0   :   0	
	  failing  1   :   0	
	  both     1   :   1	
  * @retval None
  */
void DDL_TIMG_Ch1_Capture_Polar_Config(TIMG_Type *TIMGx, uint8_t chPolar, uint8_t chnPolar)
{
	MODIFY_REG(TIMGx->CCER,TIMG_CCER_CC1P , chPolar << TIMG_CCER_CC1P_Pos);
	MODIFY_REG(TIMGx->CCER,TIMG_CCER_CC1NP , chnPolar << TIMG_CCER_CC1NP_Pos);
}


/**
  * @brief  use for capture channel2 output polar config.
  *			
  * @param  TIMGx  TIMG instance.
			chPolar: chnPolar
	  rising   0   :   0	
	  failing  1   :   0	
	  both     1   :   1	
  * @retval None
  */
void DDL_TIMG_Ch2_Capture_Polar_Config(TIMG_Type *TIMGx, uint8_t chPolar, uint8_t chnPolar)
{
	MODIFY_REG(TIMGx->CCER,TIMG_CCER_CC2P , chPolar << TIMG_CCER_CC2P_Pos);
	MODIFY_REG(TIMGx->CCER,TIMG_CCER_CC2NP , chnPolar << TIMG_CCER_CC2NP_Pos);
}


/**
  * @brief  use for capture channel3 output polar config.
  *			
  * @param  TIMGx  TIMG instance.
			chPolar: chnPolar
	  rising   0   :   0	
	  failing  1   :   0	
	  both     1   :   1	
  * @retval None
  */
void DDL_TIMG_Ch3_Capture_Polar_Config(TIMG_Type *TIMGx, uint8_t chPolar, uint8_t chnPolar)
{
	MODIFY_REG(TIMGx->CCER,TIMG_CCER_CC3P , chPolar << TIMG_CCER_CC3P_Pos);
	MODIFY_REG(TIMGx->CCER,TIMG_CCER_CC3NP , chnPolar << TIMG_CCER_CC3NP_Pos);
}


/**
  * @brief  Configure the elements of structure TIMG_CapturePara_T to default values.
  *			use for capture 
  * @param  pCapStruct Pointer to a TIMG_CapturePara_T structure that contains
  *                  the configuration information for the given TIMG module.
  *
  * @retval None
  */
void DDL_TIMG_Capture_Channel0_StructInit(TIMG_CapturePara_T *pCapStruct)
{
	pCapStruct->chxDirectSel = TIMG_IC0_DIRECTTI0;
	pCapStruct->chxFilter    = 0;
	pCapStruct->chxPsc       = 0;
}


/**
  * @brief  Initializes the TIMG by the specified parameters in the TIMG_CapturePara_T.
  *			use for capture 
  * @param  TIMGx  TIMG instance.
			pCapStruct Pointer to a TIMG_CapturePara_T structure that contains
  *                  the configuration information for the given TIMG module.
  *
  * @retval None
  */
void DDL_TIMG_Capture_Channel0_Config(TIMG_Type *TIMGx, TIMG_CapturePara_T *pCapStruct)
{
	uint32_t tmp = TIMGx->CCMR0;
	tmp |= pCapStruct->chxDirectSel   << TIMG_CCMR0_CC0S_Pos;
	tmp |= pCapStruct->chxFilter      << TIMG_CCMR0_IC0F_Pos;
	tmp |= pCapStruct->chxPsc         << TIMG_CCMR0_IC0PSC_Pos;
	WRITE_REG(TIMGx->CCMR0,  tmp);        
}


/**
  * @brief  Configure the elements of structure TIMG_CapturePara_T to default values.
  *			use for capture 
  * @param  pCapStruct Pointer to a TIMG_CapturePara_T structure that contains
  *                  the configuration information for the given TIMG module.
  *
  * @retval None
  */
void DDL_TIMG_Capture_Channel1_StructInit(TIMG_CapturePara_T *pCapStruct)
{
	pCapStruct->chxDirectSel = TIMG_IC1_DIRECTTI0;
	pCapStruct->chxFilter    = 0;
	pCapStruct->chxPsc       = 0;
}


/**
  * @brief  Initializes the TIMG by the specified parameters in the TIMG_CapturePara_T.
  *			use for capture 
  * @param  TIMGx  TIMG instance.
			pCapStruct Pointer to a TIMG_CapturePara_T structure that contains
  *                  the configuration information for the given TIMG module.
  *
  * @retval None
  */
void DDL_TIMG_Capture_Channel1_Config(TIMG_Type *TIMGx, TIMG_CapturePara_T *pCapStruct)
{
	uint32_t tmp = TIMGx->CCMR0;
	tmp |= pCapStruct->chxDirectSel   << TIMG_CCMR0_CC1S_Pos;
	tmp |= pCapStruct->chxFilter      << TIMG_CCMR0_IC1F_Pos;
	tmp |= pCapStruct->chxPsc         << TIMG_CCMR0_IC1PSC_Pos;
	WRITE_REG(TIMGx->CCMR0,  tmp);        
}


/**
  * @brief  Configure the elements of structure TIMG_CapturePara_T to default values.
  *			use for capture 
  * @param  pCapStruct Pointer to a TIMG_CapturePara_T structure that contains
  *                  the configuration information for the given TIMG module.
  *
  * @retval None
  */
void DDL_TIMG_Capture_Channel2_StructInit(TIMG_CapturePara_T *pCapStruct)
{
	pCapStruct->chxDirectSel = TIMG_IC2_DIRECTTI2;
	pCapStruct->chxFilter    = 0;
	pCapStruct->chxPsc       = 0;
}


/**
  * @brief  Initializes the TIMG by the specified parameters in the TIMG_CapturePara_T.
  *			use for capture 
  * @param  TIMGx  TIMG instance.
			pCapStruct Pointer to a TIMG_CapturePara_T structure that contains
  *                  the configuration information for the given TIMG module.
  *
  * @retval None
  */
void DDL_TIMG_Capture_Channel2_Config(TIMG_Type *TIMGx, TIMG_CapturePara_T *pCapStruct)
{
	uint32_t tmp = TIMGx->CCMR1;
	tmp |= pCapStruct->chxDirectSel   << TIMG_CCMR1_CC2S_Pos;
	tmp |= pCapStruct->chxFilter      << TIMG_CCMR1_IC2F_Pos;
	tmp |= pCapStruct->chxPsc         << TIMG_CCMR1_IC2PSC_Pos;
	WRITE_REG(TIMGx->CCMR1,  tmp);        
}


/**
  * @brief  Configure the elements of structure TIMG_CapturePara_T to default values.
  *			use for capture 
  * @param  pCapStruct Pointer to a TIMG_CapturePara_T structure that contains
  *                  the configuration information for the given TIMG module.
  *
  * @retval None
  */
void DDL_TIMG_Capture_Channel3_StructInit(TIMG_CapturePara_T *pCapStruct)
{
	pCapStruct->chxDirectSel = TIMG_IC3_DIRECTTI3;
	pCapStruct->chxFilter    = 0;
	pCapStruct->chxPsc       = 0;
}


/**
  * @brief  Initializes the TIMG by the specified parameters in the TIMG_CapturePara_T.
  *			use for capture 
  * @param  TIMGx  TIMG instance.
			pCapStruct Pointer to a TIMG_CapturePara_T structure that contains
  *                  the configuration information for the given TIMG module.
  *
  * @retval None
  */
void DDL_TIMG_Capture_Channel3_Config(TIMG_Type *TIMGx, TIMG_CapturePara_T *pCapStruct)
{
	uint32_t tmp = TIMGx->CCMR1;
	tmp |= pCapStruct->chxDirectSel   << TIMG_CCMR1_CC3S_Pos;
	tmp |= pCapStruct->chxFilter      << TIMG_CCMR1_IC3F_Pos;
	tmp |= pCapStruct->chxPsc         << TIMG_CCMR1_IC3PSC_Pos;
	WRITE_REG(TIMGx->CCMR1,  tmp);        
}


/**
  * @brief  use for TIMG slave mode config.
  *			
  * @param  TIMGx  TIMG instance.
			triggerSignal:  ITRx/TI0FED/TI0FPT/TI1FPT
			slaveMode:		reset/gate/trigger
  * @retval None
  */
void DDL_TIMG_SlaveMode_Config(TIMG_Type *TIMGx, uint8_t triggerSignal, uint8_t slaveMode)
{
	uint32_t tmp = 0;
	tmp = (triggerSignal<<4)|slaveMode;
	MODIFY_REG(TIMGx->SMCR,0x3f,tmp );
}


/**
  * @brief  use for timer start.
  * @param  TIMGx  TIMG instance.
  * @retval None
  */
void DDL_TIMG_Capture_Start(TIMG_Type *TIMGx)
{
	SET_BIT(TIMGx->CR0,   TIMG_CR0_CEN);
}


/**
  * @brief  use for timer stop.
  * @param  TIMGx  TIMG instance.
  * @retval None
  */
void DDL_TIMG_Capture_Stop(TIMG_Type *TIMGx)
{
	CLEAR_BIT(TIMGx->CR0,   TIMG_CR0_CEN);
}



#endif /* DDL_TIMG_MODULE_ENABLED */















