/**
  ******************************************************************************
  * @file    kpm32xx_ddl_iow.c
  * @author  Kiwi Software Team
  * @brief   IOW DDL module driver.
  *          This file provides firmware functions to manage the following
  *          functionalities of the IOW peripheral:
  *           + peripheral initializes and deInitializes
  *           + different channel of IOW operation
  *           + interrupt, IOW
  * @note
  *          V1.0.0, 2024/12/20.
  *
  * Copyright (c) 2024, Kiwi Instruments Co,. Ltd.
  *
  * Redistribution and use in source and binary forms, with or without modification,
  * are permitted provided that the following conditions are met:
  *
  *   1. Redistributions of source code must retain the above copyright notice,
  *      this list of conditions and the following disclaimer.
  *
  *   2. Redistributions in binary form must reproduce the above copyright notice,
  *      this list of conditions and the following disclaimer in the documentation
  *      and/or other materials provided with the distribution.
  *
  *   3. Neither the name of the copyright holder nor the names of its contributors
  *      may be used to endorse or promote products derived from this software without
  *      specific prior written permission.
  *
  * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
  * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
  * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
  * THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
  * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE
  * GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
  * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
  * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
  * THE POSSIBILITY OF SUCH DAMAGE.
  ******************************************************************************
  */
#include "kpm32xx_ddl.h"

#ifdef DDL_IOW_MODULE_ENABLED


/**
  * @brief  Makes IOW active.
  * @param  none
  * 
  * @retval none
  */
void DDL_IOW_Instance_Active(void)
{
	/* Clock and Reset */
	__DDL_RCC_IOW_ACTIVE();
	__DDL_RCC_IOW_CLK_RELEASE();
}

/**
  * @brief  IOW interrupt enable.
  * @param  IOWx IOW instance.
  * @param  iowInit  pointer to a IOW_Init_T structure.
  * 
  * @retval none
  */
void DDL_IOW_IntEnable(IOW_Type *IOWx, IOW_Init_T *iowInit)
{
	/* Enable One IOW channel Interrupt */
	__DDL_IOW_INTRCHN_ENABLE(IOWx, iowInit->krx);
	__DDL_IOW_INTR_ENABLE(IOWx);
}

/**
  * @brief  IOW interrupt disable.
  * @param  IOWx IOW instance.
  * @param  iowInit  pointer to a IOW_Init_T structure.
  * 
  * @retval none
  */
void DDL_IOW_IntDisable(IOW_Type *IOWx, IOW_Init_T *iowInit)
{
	/* Enable One IOW channel Interrupt */
	__DDL_IOW_INTRCHN_DISABLE(IOWx, iowInit->krx);
	__DDL_IOW_INTR_DISABLE(IOWx);
}

/**
  * @brief  Initialize IOW_Init_T structure.
  * @param  iowInit  pointer to a IOW_Init_T structure.
  * 
  * @retval none
  */
void DDL_IOW_StructInit(IOW_Init_T *iowInit)
{
	iowInit->krx     = IOW_KR0;
	iowInit->edges   = IOW_EDGE_FALLING;
}


/**
  * @brief  configure IOW parameters.
  * @param  IOWx     IO PORT instance.
  * @param  iowInit  pointer to a IOW_Init_T structure.
  *
  * @retval none
  */
DDL_Status_T DDL_IOW_Init(IOW_Type *IOWx, IOW_Init_T *iowInit)
{
	uint8_t i = 0;

	/* Clear the interrupt state*/
	__DDL_IOW_CLEAR_CHNINTR(IOWx, iowInit->krx);

	while (i < IOW_KRX_NUMBER)
	{
		if (iowInit->krx & (1 << i))
		{
			CLEAR_BIT(IOWx->EDG, (1 << i));

			SET_BIT(IOWx->EDG, (iowInit->edges << i));
		}

		i ++;
	}

	return DDL_OK;
}


#endif /* DDL_IOW_MODULE_ENABLED */
