/**
  *****************************************************************************************************
  * @file    kpm32xx_ddl_timg.h
  * @author  Kiwi Software Team
  * @brief   Header file of TIMG DDL module..
  * @note
  *          V1.0.0, 2025/1/6.
  *
  * Copyright (c) 2024, Kiwi Instruments Co,. Ltd.
  *
  * Redistribution and use in source and binary forms, with or without modification,
  * are permitted provided that the following conditions are met:
  *
  *   1. Redistributions of source code must retain the above copyright notice,
  *      this list of conditions and the following disclaimer.
  *
  *   2. Redistributions in binary form must reproduce the above copyright notice,
  *      this list of conditions and the following disclaimer in the documentation
  *      and/or other materials provided with the distribution.
  *
  *   3. Neither the name of the copyright holder nor the names of its contributors
  *      may be used to endorse or promote products derived from this software without
  *      specific prior written permission.
  *
  * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
  * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
  * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
  * THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
  * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE
  * GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
  * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
  * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
  * THE POSSIBILITY OF SUCH DAMAGE.
  *****************************************************************************************************
  */


#ifndef __KPM32xx_DDL_TIMG_H
#define __KPM32xx_DDL_TIMG_H

#ifdef __cplusplus
extern "C" {
#endif

#include "kpm32xx_ddl_def.h"


typedef enum
{
	TIMG_CHANNEL0         = 0x00000001,
	TIMG_CHANNEL1         = 0x00000010,
	TIMG_CHANNEL2         = 0x00000100,
	TIMG_CHANNEL3         = 0x00001000,
}DDL_TIMG_Channel_Enable_T;

typedef struct
{
	uint32_t	prescaler;         						/* the prescaler value used to divide the FCLK */
	uint32_t	period;            						/* the period value to be loaded into the ARR Register   */
	uint32_t    countermode;        					/* Specifies the counter mode. */
} TIMG_BaseInit_T;

typedef struct
{
	uint32_t	chxDirectSel;
	uint32_t	chxPsc;
	uint32_t	chxFilter;
} TIMG_CapturePara_T;

/* Counter mode  */
#define TIMG_COUNTERMODE_COUNTER_UP          				0x00
#define TIMG_COUNTERMODE_COUNTER_DOWN						TIMG_CR0_DIR
#define TIMG_COUNTERMODE_COUNTER_CENTER_1    				1UL << TIMG_CR0_CMS_Pos
#define TIMG_COUNTERMODE_COUNTER_CENTER_2    				2UL << TIMG_CR0_CMS_Pos
#define TIMG_COUNTERMODE_COUNTER_CENTER_3    				3UL << TIMG_CR0_CMS_Pos

/* TIMG channelx output/compare mode select */
#define TIMG_OCMODE_FREEZE_CH0								(0UL)
#define TIMG_OCMODE_COMPARE_FORCED_HIGH_CH0      			(1UL)
#define TIMG_OCMODE_COMPARE_FORCED_LOW_CH0       			(2UL)
#define TIMG_OCMODE_COMPARE_TOGGLE_CH0						(3UL)
#define TIMG_OCMODE_FORCED_LOW_CH0    	  					(4UL)
#define TIMG_OCMODE_FORCED_HIGH_CH0       					(5UL)
#define TIMG_OCMODE_PWM0_CH0                     			(6UL)
#define TIMG_OCMODE_PWM1_CH0                     			(7UL)

#define TIMG_OCMODE_FREEZE_CH1								(0UL)
#define TIMG_OCMODE_COMPARE_FORCED_HIGH_CH1      			(1UL)
#define TIMG_OCMODE_COMPARE_FORCED_LOW_CH1       			(2UL)
#define TIMG_OCMODE_COMPARE_TOGGLE_CH1						(3UL)
#define TIMG_OCMODE_FORCED_HIGH_CH1      					(5UL)
#define TIMG_OCMODE_FORCED_LOW_CH1       					(4UL)
#define TIMG_OCMODE_PWM0_CH1                     			(6UL)
#define TIMG_OCMODE_PWM1_CH1                     			(7UL)

#define TIMG_OCMODE_FREEZE_CH2								(0UL)
#define TIMG_OCMODE_COMPARE_FORCED_HIGH_CH2      			(1UL)
#define TIMG_OCMODE_COMPARE_FORCED_LOW_CH2       			(2UL)
#define TIMG_OCMODE_COMPARE_TOGGLE_CH2						(3UL)
#define TIMG_OCMODE_FORCED_HIGH_CH2      					(5UL)
#define TIMG_OCMODE_FORCED_LOW_CH2       					(4UL)
#define TIMG_OCMODE_PWM0_CH2                     			(6UL)
#define TIMG_OCMODE_PWM1_CH2                     			(7UL)

#define TIMG_OCMODE_FREEZE_CH3								(0UL)
#define TIMG_OCMODE_COMPARE_FORCED_HIGH_CH3      			(1UL)
#define TIMG_OCMODE_COMPARE_FORCED_LOW_CH3       			(2UL)
#define TIMG_OCMODE_COMPARE_TOGGLE_CH3						(3UL)
#define TIMG_OCMODE_FORCED_HIGH_CH3      					(5UL)
#define TIMG_OCMODE_FORCED_LOW_CH3       					(4UL)
#define TIMG_OCMODE_PWM0_CH3                     			(6UL)
#define TIMG_OCMODE_PWM1_CH3                     			(7UL)

#define TIMG_MASTERMODE_RESET							   	(0x0 << TIMG_CR1_MMS_Pos)
#define TIMG_MASTERMODE_ENABLE							   	(0x1 << TIMG_CR1_MMS_Pos)
#define TIMG_MASTERMODE_UPDATE							   	(0x2 << TIMG_CR1_MMS_Pos)
#define TIMG_MASTERMODE_COMPAREPULSE					   	(0x3 << TIMG_CR1_MMS_Pos)
#define TIMG_MASTERMODE_COMPAREOC0REF					   	(0x4 << TIMG_CR1_MMS_Pos)
#define TIMG_MASTERMODE_COMPAREOC1REF					   	(0x5 << TIMG_CR1_MMS_Pos)
#define TIMG_MASTERMODE_COMPAREOC2REF					   	(0x6 << TIMG_CR1_MMS_Pos)
#define TIMG_MASTERMODE_COMPAREOC3REF					   	(0x7 << TIMG_CR1_MMS_Pos)

#define TIMG_IC0_DIRECTTI0           						(1)   /* TIMG Input 0 OR 1 is selected to be connected to IC0 OR IC1, respectively */
#define TIMG_IC0_DIRECTTI1         							(2)   /* TIMG Input 0 OR 1 is selected to be connected to IC0 OR IC1, respectively */
#define TIMG_IC0_SELECT_TRC                					(3)   /* TIMG Input 0, 1 is selected to be connected to TRC */
#define TIMG_IC1_DIRECTTI1           						(1)   /* TIMG Input 1 OR 0 is selected to be connected to IC1 OR IC0, respectively */
#define TIMG_IC1_DIRECTTI0         							(2)   /* TIMG Input 1 OR 0 is selected to be connected to IC1 OR IC0, respectively */
#define TIMG_IC1_SELECT_TRC                					(3)   /* TIMG Input 1, 0 is selected to be connected to TRC */
#define TIMG_IC2_DIRECTTI2           						(1)   /* TIMG Input 2 OR 3 is selected to be connected to IC2 OR IC3, respectively */
#define TIMG_IC2_DIRECTTI3         							(2)   /* TIMG Input 2 OR 3 is selected to be connected to IC2 OR IC3, respectively */
#define TIMG_IC2_SELECT_TRC                					(3)   /* TIMG Input 2, 3 is selected to be connected to TRC */
#define TIMG_IC3_DIRECTTI3           						(1)   /* TIMG Input 3 OR 2 is selected to be connected to IC3 OR IC2, respectively */
#define TIMG_IC3_DIRECTTI2         							(2)   /* TIMG Input 3 OR 2 is selected to be connected to IC3 OR IC2, respectively */
#define TIMG_IC3_SELECT_TRC                					(3)   /* TIMG Input 3, 2 is selected to be connected to TRC */

/*slave mode config*/
#define TIMG_SLAVEMODE_DISABLE								(0x0UL)
#define TIMG_SLAVEMODE_RESET								(0x4UL)
#define TIMG_SLAVEMODE_GATE									(0x5UL)
#define TIMG_SLAVEMODE_TRIGGER								(0x6UL)
#define TIMG_SLAVEMODE_EXTERNCLOCK							(0x7UL)

/* timg trigger selection */
#define TIMG_CLOCKSOURCE_ITR0								(0x0UL)
#define TIMG_CLOCKSOURCE_ITR1								(0x1UL)
#define TIMG_CLOCKSOURCE_ITR2								(0x2UL)
#define TIMG_CLOCKSOURCE_ITR3								(0x3UL)
#define	TIMG_CLOCKSOURCE_TI0FED      						(0x4UL)
#define	TIMG_CLOCKSOURCE_TI0FPT      						(0x5UL)
#define	TIMG_CLOCKSOURCE_TI1FPT      						(0x6UL)
#define	TIMG_CLOCKSOURCE_ETRF        						(0x7UL)



/**
  * @brief  make TIMG active.
  * @param  SCIx  TIMG instance.
  *
  */
void DDL_TIMG_Instance_Active(TIMG_Type *TIMGx);


/**
  * @brief  Configure the elements of structure TIMG_BaseInit_T to default values.
  *			use for timer
  * @param  TIMG_BaseInit_T Pointer to a TIMG_BaseInit_T structure that contains
  *                  the configuration information for the given TIMG module.
  *
  * @retval None
  */
void DDL_TIMG_TMR_StructInit(TIMG_BaseInit_T *pTmrStruct);


/**
  * @brief  Initializes the TIMG by the specified parameters in the TIMG_BaseInit_T.
  *			use for timer
  * @param  TIMGx  TIMG instance.
  *			pTmrStruct Pointer to a TIMG_BaseInit_T structure that contains
  *                  the configuration information for the given TIMG module.
  *
  * @retval None
  */
void DDL_TIMG_TMR_Init(TIMG_Type *TIMGx, TIMG_BaseInit_T *pTmrStruct);


/**
  * @brief  use for timer start.
  * @param  TIMGx  TIMG instance.
  * @retval None
  */
void DDL_TIMG_TMR_Start(TIMG_Type *TIMGx);


/**
  * @brief  use for timer stop.
  * @param  TIMGx  TIMG instance.
  * @retval None
  */
void DDL_TIMG_TMR_Stop(TIMG_Type *TIMGx);


/**
  * @brief  use for TIMG interrupt enable.
  * @param  intrMask : COMIE/UIE/C0IE/C1IE/C2IE/TIE/BIE
  *
  * @retval None
  */
void DDL_TIMG_IntEnable(TIMG_Type *TIMGx, uint32_t intrMask);


/**
  * @brief  use for TIMG interrupt disable.
  * @param  intrMask : COMIE/UIE/C0IE/C1IE/C2IE/TIE/BIE
  *
  * @retval None
  */
void DDL_TIMG_IntDisable(TIMG_Type *TIMGx, uint32_t intrMask);


/**
  * @brief  Configure the elements of structure TIMG_BaseInit_T to default values.
  *			use for pwm output
  * @param  pPwmStruct Pointer to a TIMG_BaseInit_T structure that contains
  *                  the configuration information for the given TIMG module.
  *
  * @retval None
  */
void DDL_TIMG_PWM_StructInit(TIMG_BaseInit_T *pPwmStruct);


/**
  * @brief  Initializes the TIMG by the specified parameters in the TIMG_BaseInit_T.
  *			use for pwm output
  * @param  pTmrStruct Pointer to a TIMG_BaseInit_T structure that contains
  *                  the configuration information for the given TIMG module.
  *			TIMGx  TIMG instance.
  * @retval None
  */
void DDL_TIMG_PWM_Init(TIMG_Type *TIMGx, TIMG_BaseInit_T *pPwmStruct);


/**
  * @brief  use for pwm output channel0 duty set.
  *
  * @param  TIMGx  TIMG instance.
			preload  :
					0: effective immediately
					1: effective for the next cycle
  *         chDuty  : channel0 ccr set
  * @retval None
  */
void DDL_TIMG_Ch0_Duty_Config(TIMG_Type *TIMGx, uint8_t preload, uint32_t chDuty);


/**
  * @brief  use for pwm output channel1 duty set.
  *
  * @param  TIMGx  TIMG instance.
			preload  :
					0: effective immediately
					1: effective for the next cycle
  *			chDuty  : channel1 ccr set
  * @retval None
  */
void DDL_TIMG_Ch1_Duty_Config(TIMG_Type *TIMGx, uint8_t preload, uint32_t chDuty);


/**
  * @brief  use for pwm output channel2 duty set.
  *
  * @param  TIMGx  TIMG instance.
			preload  :
					0: effective immediately
					1: effective for the next cycle
  *			chDuty  : channel2 ccr set
  * @retval None
  */
void DDL_TIMG_Ch2_Duty_Config(TIMG_Type *TIMGx, uint8_t preload, uint32_t chDuty);


/**
  * @brief  use for pwm output channel3 duty set.
  *
  * @param  TIMGx  TIMG instance.
			preload  :
					0: effective immediately
					1: effective for the next cycle
  *			chDuty  : channel3 ccr set
  * @retval None
  */
void DDL_TIMG_Ch3_Duty_Config(TIMG_Type *TIMGx, uint8_t preload, uint32_t chDuty);


/**
  * @brief  use for pwm channel0 output mode config.
  *
  * @param  TIMGx  TIMG instance.
			mode  :
				0: TIMG_OCMODE_FREEZE_CH0
				1: TIMG_OCMODE_COMPARE_FORCED_HIGH_CH0
				2: TIMG_OCMODE_COMPARE_FORCED_LOW_CH0
				3: TIMG_OCMODE_COMPARE_TOGGLE_CH0
				4: TIMG_OCMODE_FORCED_LOW_CH0
				5: TIMG_OCMODE_FORCED_HIGH_CH0
				6: TIMG_OCMODE_PWM0_CH0
				7: TIMG_OCMODE_PWM1_CH0
				C: TIMG_OCMODE_COMBINATION_MODE_PWM0_CH0
				D: TIMG_OCMODE_COMBINATION_MODE_PWM1_CH0
				E: TIMG_OCMODE_ASYM_PWM0_CH0
				F: TIMG_OCMODE_ASYM_PWM1_CH0

  * @retval None
  */
void DDL_TIMG_Ch0_PWM_Mode_Config(TIMG_Type *TIMGx,uint8_t mode);


/**
  * @brief  use for pwm channel1 output mode config.
  *
  * @param  TIMGx  TIMG instance.
			mode  :
				0: TIMG_OCMODE_FREEZE_CH1
				1: TIMG_OCMODE_COMPARE_FORCED_HIGH_CH1
				2: TIMG_OCMODE_COMPARE_FORCED_LOW_CH1
				3: TIMG_OCMODE_COMPARE_TOGGLE_CH1
				4: TIMG_OCMODE_FORCED_LOW_CH1
				5: TIMG_OCMODE_FORCED_HIGH_CH1
				6: TIMG_OCMODE_PWM0_CH1
				7: TIMG_OCMODE_PWM1_CH1
				C: TIMG_OCMODE_COMBINATION_MODE_PWM0_CH1
				D: TIMG_OCMODE_COMBINATION_MODE_PWM1_CH1
				E: TIMG_OCMODE_ASYM_PWM0_CH1
				F: TIMG_OCMODE_ASYM_PWM1_CH1

  * @retval None
  */
void DDL_TIMG_Ch1_PWM_Mode_Config(TIMG_Type *TIMGx,uint8_t mode);


/**
  * @brief  use for pwm channel2 output mode config.
  *
  * @param  TIMGx  TIMG instance.
			mode  :
				0: TIMG_OCMODE_FREEZE_CH2
				1: TIMG_OCMODE_COMPARE_FORCED_HIGH_CH2
				2: TIMG_OCMODE_COMPARE_FORCED_LOW_CH2
				3: TIMG_OCMODE_COMPARE_TOGGLE_CH2
				4: TIMG_OCMODE_FORCED_HIGH_CH2
				5: TIMG_OCMODE_FORCED_LOW_CH2
				6: TIMG_OCMODE_PWM0_CH2
				7: TIMG_OCMODE_PWM1_CH2
				C: TIMG_OCMODE_COMBINATION_MODE_PWM0_CH2
				D: TIMG_OCMODE_COMBINATION_MODE_PWM1_CH2
				E: TIMG_OCMODE_ASYM_PWM0_CH2
				F: TIMG_OCMODE_ASYM_PWM1_CH2

  * @retval None
  */
void DDL_TIMG_Ch2_PWM_Mode_Config(TIMG_Type *TIMGx,uint8_t mode);


/**
  * @brief  use for pwm channel3 output mode config.
  *
  * @param  TIMGx  TIMG instance.
			mode  :
				0: TIMG_OCMODE_FREEZE_CH3
				1: TIMG_OCMODE_COMPARE_FORCED_HIGH_CH3
				2: TIMG_OCMODE_COMPARE_FORCED_LOW_CH3
				3: TIMG_OCMODE_COMPARE_TOGGLE_CH3
				4: TIMG_OCMODE_FORCED_HIGH_CH3
				5: TIMG_OCMODE_FORCED_LOW_CH3
				6: TIMG_OCMODE_PWM0_CH3
				7: TIMG_OCMODE_PWM1_CH3
				C: TIMG_OCMODE_COMBINATION_MODE_PWM0_CH3
				D: TIMG_OCMODE_COMBINATION_MODE_PWM1_CH3
				E: TIMG_OCMODE_ASYM_PWM0_CH3
				F: TIMG_OCMODE_ASYM_PWM1_CH3

  * @retval None
  */
void DDL_TIMG_Ch3_PWM_Mode_Config(TIMG_Type *TIMGx,uint8_t mode);


/**
  * @brief  use for pwm channel0 output polar config.
  *
  * @param  TIMGx  TIMG instance.
			chPolar: channel0 polar config
			chnPolar: channel0n polar config
  * @retval None
  */
void DDL_TIMG_Ch0_PWM_Polar_Config(TIMG_Type *TIMGx, uint8_t chPolar, uint8_t chnPolar);


/**
  * @brief  use for pwm channel1 output polar config.
  *
  * @param  TIMGx  TIMG instance.
			chPolar: channel0 polar config
			chnPolar: channel0n polar config
  * @retval None
  */
void DDL_TIMG_Ch1_PWM_Polar_Config(TIMG_Type *TIMGx, uint8_t chPolar, uint8_t chnPolar);


/**
  * @brief  use for pwm channel2 output polar config.
  *
  * @param  TIMGx  TIMG instance.
			chPolar: channel0 polar config
			chnPolar: channel0n polar config
  * @retval None
  */
void DDL_TIMG_Ch2_PWM_Polar_Config(TIMG_Type *TIMGx, uint8_t chPolar, uint8_t chnPolar);


/**
  * @brief  use for pwm channel3 output polar config.
  *
  * @param  TIMGx  TIMG instance.
			chPolar: channel3 polar config
			chnPolar: channel3n polar config
  * @retval None
  */
void DDL_TIMG_Ch3_PWM_Polar_Config(TIMG_Type *TIMGx, uint8_t chPolar, uint8_t chnPolar);


/**
  * @brief  use for pwm output channel enable.
  *
  * @param  TIMGx  TIMG instance.
			chEnable: channel enable select
  *
  * @retval None
  */
void DDL_TIMG_PWM_Enable(TIMG_Type *TIMGx, uint32_t chEnable);


/**
  * @brief  use for pwm output start.
  * @param  TIMGx  TIMG instance.
  * @retval None
  */
void DDL_TIMG_PWM_Start(TIMG_Type *TIMGx);


/**
  * @brief  use for pwm output stop.
  * @param  TIMGx  TIMG instance.
  * @retval None
  */
void DDL_TIMG_PWM_Stop(TIMG_Type *TIMGx);


/**
  * @brief  Configure the elements of structure TIMG_BaseInit_T to default values.
  *			use for pwm capture
  * @param  pPwmStruct Pointer to a TIMG_BaseInit_T structure that contains
  *                  the configuration information for the given TIMG module.
  *
  * @retval None
  */
void DDL_TIMG_Capture_StructInit(TIMG_BaseInit_T *pPwmStruct);


/**
  * @brief  Initializes the TIMG by the specified parameters in the TIMG_BaseInit_T.
  *			use for pwm capture
  * @param  pTmrStruct Pointer to a TIMG_BaseInit_T structure that contains
  *                  the configuration information for the given TIMG module.
  *			TIMGx  TIMG instance.
  * @retval None
  */
void DDL_TIMG_Capture_Init(TIMG_Type *TIMGx, TIMG_BaseInit_T *pPwmStruct);


/**
  * @brief  use for capture channel0 output polar config.
  *
  * @param  TIMGx  TIMG instance.
			chPolar: chnPolar
	  rising   0   :   0
	  failing  1   :   0
	  both     1   :   1
  * @retval None
  */
void DDL_TIMG_Ch0_Capture_Polar_Config(TIMG_Type *TIMGx, uint8_t chPolar, uint8_t chnPolar);


/**
  * @brief  use for capture channel1 output polar config.
  *
  * @param  TIMGx  TIMG instance.
			chPolar: chnPolar
	  rising   0   :   0
	  failing  1   :   0
	  both     1   :   1
  * @retval None
  */
void DDL_TIMG_Ch1_Capture_Polar_Config(TIMG_Type *TIMGx, uint8_t chPolar, uint8_t chnPolar);


/**
  * @brief  use for capture channel2 output polar config.
  *
  * @param  TIMGx  TIMG instance.
			chPolar: chnPolar
	  rising   0   :   0
	  failing  1   :   0
	  both     1   :   1
  * @retval None
  */
void DDL_TIMG_Ch2_Capture_Polar_Config(TIMG_Type *TIMGx, uint8_t chPolar, uint8_t chnPolar);


/**
  * @brief  use for capture channel3 output polar config.
  *
  * @param  TIMGx  TIMG instance.
			chPolar: chnPolar
	  rising   0   :   0
	  failing  1   :   0
	  both     1   :   1
  * @retval None
  */
void DDL_TIMG_Ch3_Capture_Polar_Config(TIMG_Type *TIMGx, uint8_t chPolar, uint8_t chnPolar);


/**
  * @brief  Configure the elements of structure TIMG_CapturePara_T to default values.
  *			use for capture
  * @param  pCapStruct Pointer to a TIMG_CapturePara_T structure that contains
  *                  the configuration information for the given TIMG module.
  *
  * @retval None
  */
void DDL_TIMG_Capture_Channel0_StructInit(TIMG_CapturePara_T *pCapStruct);


/**
  * @brief  Initializes the TIMG by the specified parameters in the TIMG_CapturePara_T.
  *			use for capture
  * @param  TIMGx  TIMG instance.
			pCapStruct Pointer to a TIMG_CapturePara_T structure that contains
  *                  the configuration information for the given TIMG module.
  *
  * @retval None
  */
void DDL_TIMG_Capture_Channel0_Config(TIMG_Type *TIMGx, TIMG_CapturePara_T *pCapStruct);


/**
  * @brief  Configure the elements of structure TIMG_CapturePara_T to default values.
  *			use for capture
  * @param  pCapStruct Pointer to a TIMG_CapturePara_T structure that contains
  *                  the configuration information for the given TIMG module.
  *
  * @retval None
  */
void DDL_TIMG_Capture_Channel1_StructInit(TIMG_CapturePara_T *pCapStruct);


/**
  * @brief  Initializes the TIMG by the specified parameters in the TIMG_CapturePara_T.
  *			use for capture
  * @param  TIMGx  TIMG instance.
			pCapStruct Pointer to a TIMG_CapturePara_T structure that contains
  *                  the configuration information for the given TIMG module.
  *
  * @retval None
  */
void DDL_TIMG_Capture_Channel1_Config(TIMG_Type *TIMGx, TIMG_CapturePara_T *pCapStruct);


/**
  * @brief  Configure the elements of structure TIMG_CapturePara_T to default values.
  *			use for capture
  * @param  pCapStruct Pointer to a TIMG_CapturePara_T structure that contains
  *                  the configuration information for the given TIMG module.
  *
  * @retval None
  */
void DDL_TIMG_Capture_Channel2_StructInit(TIMG_CapturePara_T *pCapStruct);


/**
  * @brief  Initializes the TIMG by the specified parameters in the TIMG_CapturePara_T.
  *			use for capture
  * @param  TIMGx  TIMG instance.
			pCapStruct Pointer to a TIMG_CapturePara_T structure that contains
  *                  the configuration information for the given TIMG module.
  *
  * @retval None
  */
void DDL_TIMG_Capture_Channel2_Config(TIMG_Type *TIMGx, TIMG_CapturePara_T *pCapStruct);


/**
  * @brief  Configure the elements of structure TIMG_CapturePara_T to default values.
  *			use for capture
  * @param  pCapStruct Pointer to a TIMG_CapturePara_T structure that contains
  *                  the configuration information for the given TIMG module.
  *
  * @retval None
  */
void DDL_TIMG_Capture_Channel3_StructInit(TIMG_CapturePara_T *pCapStruct);


/**
  * @brief  Initializes the TIMG by the specified parameters in the TIMG_CapturePara_T.
  *			use for capture
  * @param  TIMGx  TIMG instance.
			pCapStruct Pointer to a TIMG_CapturePara_T structure that contains
  *                  the configuration information for the given TIMG module.
  *
  * @retval None
  */
void DDL_TIMG_Capture_Channel3_Config(TIMG_Type *TIMGx, TIMG_CapturePara_T *pCapStruct);


/**
  * @brief  use for TIMG slave mode config.
  *
  * @param  TIMGx  TIMG instance.
			triggerSignal:  ITRx/TI0FED/TI0FPT/TI1FPT
			slaveMode:		reset/gate/trigger
  * @retval None
  */
void DDL_TIMG_SlaveMode_Config(TIMG_Type *TIMGx, uint8_t triggerSignal, uint8_t slaveMode);


/**
  * @brief  use for timer start.
  * @param  TIMGx  TIMG instance.
  * @retval None
  */
void DDL_TIMG_Capture_Start(TIMG_Type *TIMGx);


/**
  * @brief  use for timer stop.
  * @param  TIMGx  TIMG instance.
  * @retval None
  */
void DDL_TIMG_Capture_Stop(TIMG_Type *TIMGx);



#ifdef __cplusplus
}
#endif

#endif /* __KPM32xx_DDL_TIMG_H */










