/**
  *****************************************************************************************************
  * @file    kpm32xx_ddl_tima.h
  * @author  Kiwi Software Team
  * @brief   Header file of TIMA DDL module..
  * @note
  *          V1.0.0, 2025/1/6.
  *
  * Copyright (c) 2024, Kiwi Instruments Co,. Ltd.
  *
  * Redistribution and use in source and binary forms, with or without modification,
  * are permitted provided that the following conditions are met:
  *
  *   1. Redistributions of source code must retain the above copyright notice,
  *      this list of conditions and the following disclaimer.
  *
  *   2. Redistributions in binary form must reproduce the above copyright notice,
  *      this list of conditions and the following disclaimer in the documentation
  *      and/or other materials provided with the distribution.
  *
  *   3. Neither the name of the copyright holder nor the names of its contributors
  *      may be used to endorse or promote products derived from this software without
  *      specific prior written permission.
  *
  * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
  * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
  * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
  * THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
  * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE
  * GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
  * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
  * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
  * THE POSSIBILITY OF SUCH DAMAGE.
  *****************************************************************************************************
  */


#ifndef __KPM32xx_DDL_TIMA_H
#define __KPM32xx_DDL_TIMA_H

#ifdef __cplusplus
extern "C" {
#endif

#include "kpm32xx_ddl_def.h"


typedef enum
{
	TIMA_CHANNEL0         = 0x00000001,
	TIMA_CHANNEL0N        = 0x00000004,
	TIMA_CHANNEL1         = 0x00000010,
	TIMA_CHANNEL1N        = 0x00000040,
	TIMA_CHANNEL2         = 0x00000100,
	TIMA_CHANNEL2N        = 0x00000400,
	TIMA_CHANNEL3         = 0x00001000,
	TIMA_CHANNEL4         = 0x00010000,
	TIMA_CHANNEL5         = 0x00100000,
}DDL_TIMA_Channel_Enable_T;

/* tima0 trgo enum definition */
typedef enum
{
	TIMA0_SLAVE_TIMA_TIMA1_TRGO           = 0x01UL << TIMA0_ITCR_ITR_SEL_Pos,        /* ITR: TIMA0 as slave and TIMA1 as master */
	TIMA0_SLAVE_TIMA_TIMH_TRGO            = 0x02UL << TIMA0_ITCR_ITR_SEL_Pos,        /* ITR: TIMA0 as slave and TIMH  as master */
	TIMA0_SLAVE_TIMG_TIMG0_TRGO           = 0x04UL << TIMA0_ITCR_ITR_SEL_Pos,        /* ITR: TIMA0 as slave and TIMG0 as master */
	TIMA0_SLAVE_TIMG_TIMG1_TRGO           = 0x05UL << TIMA0_ITCR_ITR_SEL_Pos,        /* ITR: TIMA0 as slave and TIMG1 as master */
	TIMA0_SLAVE_TIMG_TIMG2_TRGO           = 0x06UL << TIMA0_ITCR_ITR_SEL_Pos,        /* ITR: TIMA0 as slave and TIMG2 as master */
	TIMA0_SLAVE_TIMS_TIMS0_TRGO           = 0x08UL << TIMA0_ITCR_ITR_SEL_Pos,        /* ITR: TIMA0 as slave and TIMS0 as master */
	TIMA0_SLAVE_TIMS_TIMS1_TRGO           = 0x09UL << TIMA0_ITCR_ITR_SEL_Pos,        /* ITR: TIMA0 as slave and TIMS1 as master */
	TIMA0_SLAVE_TIMS_TIMS2_TRGO           = 0x0AUL << TIMA0_ITCR_ITR_SEL_Pos,        /* ITR: TIMA0 as slave and TIMS2 as master */
	TIMA0_SLAVE_TIMS_TIMS3_TRGO 		  = 0x0BUL << TIMA0_ITCR_ITR_SEL_Pos		 /* ITR: TIMA0 as slave and TIMS3 as master */
}DDL_TIMA0_AsMaster_T;

/* tima1 trgo enum definition */
typedef enum
{
	TIMA1_SLAVE_TIMA_TIMA0_TRGO           = 0x00UL << TIMA1_ITCR_ITR_SEL_Pos,        /* ITR: TIMA1 as slave and TIMA0 as master */
	TIMA1_SLAVE_TIMA_TIMH_TRGO            = 0x02UL << TIMA1_ITCR_ITR_SEL_Pos,        /* ITR: TIMA1 as slave and TIMH  as master */
	TIMA1_SLAVE_TIMG_TIMG0_TRGO           = 0x04UL << TIMA1_ITCR_ITR_SEL_Pos,        /* ITR: TIMA1 as slave and TIMG0 as master */
	TIMA1_SLAVE_TIMG_TIMG1_TRGO           = 0x05UL << TIMA1_ITCR_ITR_SEL_Pos,        /* ITR: TIMA1 as slave and TIMG1 as master */
	TIMA1_SLAVE_TIMG_TIMG2_TRGO           = 0x06UL << TIMA1_ITCR_ITR_SEL_Pos,        /* ITR: TIMA1 as slave and TIMG2 as master */
	TIMA1_SLAVE_TIMS_TIMS0_TRGO           = 0x08UL << TIMA1_ITCR_ITR_SEL_Pos,        /* ITR: TIMA1 as slave and TIMS0 as master */
	TIMA1_SLAVE_TIMS_TIMS1_TRGO           = 0x09UL << TIMA1_ITCR_ITR_SEL_Pos,        /* ITR: TIMA1 as slave and TIMS1 as master */
	TIMA1_SLAVE_TIMS_TIMS2_TRGO           = 0x0AUL << TIMA1_ITCR_ITR_SEL_Pos,        /* ITR: TIMA1 as slave and TIMS2 as master */
	TIMA1_SLAVE_TIMS_TIMS3_TRGO 		  = 0x0BUL << TIMA1_ITCR_ITR_SEL_Pos         /* ITR: TIMA1 as slave and TIMS3 as master */
}DDL_TIMA1_AsMaster_T;

typedef struct
{
	uint32_t	prescaler;         						/* the prescaler value used to divide the FCLK */
	uint32_t	period;            						/* the period value to be loaded into the ARR Register   */
	uint32_t    countermode;        					/* Specifies the counter mode. */
	uint32_t	repetitionCounter;  					/* the repetition counter value */
} TIMA_BaseInit_T;

typedef struct
{
	uint32_t	preload;         						/* the preload deciding when to load value */
	uint32_t	modeSelect;            					/* the config select deadtime mode */
	uint32_t    risingDeadtime;        					/* config rising deadtime */
	uint32_t	fallingDeadtime;  					    /* config falling deadtime */
} TIMA_Deadtime_T;

typedef struct
{
	uint32_t	chxDirectSel;
	uint32_t	chxPsc;
	uint32_t	chxFilter;
} TIMA_CapturePara_T;


/* Counter mode  */
#define TIMA_COUNTERMODE_COUNTER_UP          				0x00
#define TIMA_COUNTERMODE_COUNTER_DOWN						TIMA_CR0_DIR
#define TIMA_COUNTERMODE_COUNTER_CENTER_1    				1UL << TIMA_CR0_CMS_Pos
#define TIMA_COUNTERMODE_COUNTER_CENTER_2    				2UL << TIMA_CR0_CMS_Pos
#define TIMA_COUNTERMODE_COUNTER_CENTER_3    				3UL << TIMA_CR0_CMS_Pos

/* tima channelx output/compare mode select */
#define TIMA_OCMODE_FREEZE_CH0								(0UL)
#define TIMA_OCMODE_COMPARE_FORCED_HIGH_CH0      			(1UL)
#define TIMA_OCMODE_COMPARE_FORCED_LOW_CH0       			(2UL)
#define TIMA_OCMODE_COMPARE_TOGGLE_CH0						(3UL)
#define TIMA_OCMODE_FORCED_LOW_CH0    	  					(4UL)
#define TIMA_OCMODE_FORCED_HIGH_CH0       					(5UL)
#define TIMA_OCMODE_PWM0_CH0                     			(6UL)
#define TIMA_OCMODE_PWM1_CH0                     			(7UL)
#define TIMA_OCMODE_COMBINATION_MODE_PWM0_CH0               (0xCUL)
#define TIMA_OCMODE_COMBINATION_MODE_PWM1_CH0               (0xDUL)
#define TIMA_OCMODE_ASYM_PWM0_CH0                			(0xEUL)
#define TIMA_OCMODE_ASYM_PWM1_CH0                			(0xFUL)

#define TIMA_OCMODE_FREEZE_CH1								(0UL)
#define TIMA_OCMODE_COMPARE_FORCED_HIGH_CH1      			(1UL)
#define TIMA_OCMODE_COMPARE_FORCED_LOW_CH1       			(2UL)
#define TIMA_OCMODE_COMPARE_TOGGLE_CH1						(3UL)
#define TIMA_OCMODE_FORCED_HIGH_CH1      					(5UL)
#define TIMA_OCMODE_FORCED_LOW_CH1       					(4UL)
#define TIMA_OCMODE_PWM0_CH1                     			(6UL)
#define TIMA_OCMODE_PWM1_CH1                     			(7UL)
#define TIMA_OCMODE_COMBINATION_MODE_PWM0_CH1               (0xCUL)
#define TIMA_OCMODE_COMBINATION_MODE_PWM1_CH1               (0xDUL)
#define TIMA_OCMODE_ASYM_PWM0_CH1                			(0xEUL)
#define TIMA_OCMODE_ASYM_PWM1_CH1                			(0xFUL)

#define TIMA_OCMODE_FREEZE_CH2								(0UL)
#define TIMA_OCMODE_COMPARE_FORCED_HIGH_CH2      			(1UL)
#define TIMA_OCMODE_COMPARE_FORCED_LOW_CH2       			(2UL)
#define TIMA_OCMODE_COMPARE_TOGGLE_CH2						(3UL)
#define TIMA_OCMODE_FORCED_HIGH_CH2      					(5UL)
#define TIMA_OCMODE_FORCED_LOW_CH2       					(4UL)
#define TIMA_OCMODE_PWM0_CH2                     			(6UL)
#define TIMA_OCMODE_PWM1_CH2                     			(7UL)
#define TIMA_OCMODE_COMBINATION_MODE_PWM0_CH2               (0xCUL)
#define TIMA_OCMODE_COMBINATION_MODE_PWM1_CH2               (0xDUL)
#define TIMA_OCMODE_ASYM_PWM0_CH2                			(0xEUL)
#define TIMA_OCMODE_ASYM_PWM1_CH2                			(0xFUL)

#define TIMA_OCMODE_FREEZE_CH3								(0UL)
#define TIMA_OCMODE_COMPARE_FORCED_HIGH_CH3      			(1UL)
#define TIMA_OCMODE_COMPARE_FORCED_LOW_CH3       			(2UL)
#define TIMA_OCMODE_COMPARE_TOGGLE_CH3						(3UL)
#define TIMA_OCMODE_FORCED_HIGH_CH3      					(5UL)
#define TIMA_OCMODE_FORCED_LOW_CH3       					(4UL)
#define TIMA_OCMODE_PWM0_CH3                     			(6UL)
#define TIMA_OCMODE_PWM1_CH3                     			(7UL)
#define TIMA_OCMODE_COMBINATION_MODE_PWM0_CH3               (0xCUL)
#define TIMA_OCMODE_COMBINATION_MODE_PWM1_CH3               (0xDUL)
#define TIMA_OCMODE_ASYM_PWM0_CH3                			(0xEUL)
#define TIMA_OCMODE_ASYM_PWM1_CH3                			(0xFUL)

#define TIMA_OCMODE_FREEZE_CH4								(0UL)
#define TIMA_OCMODE_COMPARE_FORCED_HIGH_CH4      			(1UL)
#define TIMA_OCMODE_COMPARE_FORCED_LOW_CH4       			(2UL)
#define TIMA_OCMODE_COMPARE_TOGGLE_CH4						(3UL)
#define TIMA_OCMODE_FORCED_HIGH_CH4      					(5UL)
#define TIMA_OCMODE_FORCED_LOW_CH4       					(4UL)
#define TIMA_OCMODE_PWM0_CH4                     			(6UL)
#define TIMA_OCMODE_PWM1_CH4                     			(7UL)


#define TIMA_OCMODE_FREEZE_CH5								(0UL)
#define TIMA_OCMODE_COMPARE_FORCED_HIGH_CH5      			(1UL)
#define TIMA_OCMODE_COMPARE_FORCED_LOW_CH5       			(2UL)
#define TIMA_OCMODE_COMPARE_TOGGLE_CH5						(3UL)
#define TIMA_OCMODE_FORCED_HIGH_CH5      					(5UL)
#define TIMA_OCMODE_FORCED_LOW_CH5       					(4UL)
#define TIMA_OCMODE_PWM0_CH5                     			(6UL)
#define TIMA_OCMODE_PWM1_CH5                     			(7UL)


/* brake source */
#define TIMA_BRK_SOURCE_EBUS_CH0	     					((1 << 0)  << 16)
#define TIMA_BRK_SOURCE_EBUS_CH1							((1 << 1)  << 16)
#define TIMA_BRK_SOURCE_EBUS_CH2							((1 << 2)  << 16)
#define TIMA_BRK_SOURCE_EBUS_CH3							((1 << 3)  << 16)
#define TIMA_BRK_SOURCE_EBUS_CH4							((1 << 4)  << 16)
#define TIMA_BRK_SOURCE_EBUS_CH5							((1 << 5)  << 16)
#define TIMA_BRK_SOURCE_EBUS_CH6							((1 << 6)  << 16)
#define TIMA_BRK_SOURCE_EBUS_CH7							((1 << 7)  << 16)
#define TIMA_BRK_SOURCE_EBUS_CH8							((1 << 8)  << 16)
#define TIMA_BRK_SOURCE_EBUS_CH9							((1 << 9)  << 16)
#define TIMA_BRK_SOURCE_EBUS_CH10							((1 << 10)  << 16)
#define TIMA_BRK_SOURCE_EBUS_CH11							((1 << 11)  << 16)
#define TIMA_BRK_SOURCE_EBUS_CH12							((1 << 12)  << 16)
#define TIMA_BRK_SOURCE_EBUS_CH13							((1 << 13)  << 16)
#define TIMA_BRK_SOURCE_EBUS_CH14							((1 << 14)  << 16)
#define TIMA_BRK_SOURCE_EBUS_CH15							((1 << 15)  << 16)
#define TIMA_BRAKE_IO_SRC                             		(1UL  << 4)
#define TIMA_CPU_HALT_SRC                             		(1UL  << 3)
#define TIMA_SRAM_PARITY_ERR_SRC                            (1UL  << 2)
#define TIMA_PLL_LOSS_PEND_SRC                              (1UL  << 1)
#define TIMA_MCLK_LOSS_PEND_SRC                             (1UL  << 0)


#define TIMA_MASTERMODE_RESET							   		(0x0 << TIMA_CR1_MMS_Pos)
#define TIMA_MASTERMODE_ENABLE							   		(0x1 << TIMA_CR1_MMS_Pos)
#define TIMA_MASTERMODE_UPDATE							   		(0x2 << TIMA_CR1_MMS_Pos)
#define TIMA_MASTERMODE_COMPAREPULSE					   		(0x3 << TIMA_CR1_MMS_Pos)
#define TIMA_MASTERMODE_COMPAREOC0REF					   		(0x4 << TIMA_CR1_MMS_Pos)
#define TIMA_MASTERMODE_COMPAREOC1REF					   		(0x5 << TIMA_CR1_MMS_Pos)
#define TIMA_MASTERMODE_COMPAREOC2REF					   		(0x6 << TIMA_CR1_MMS_Pos)
#define TIMA_MASTERMODE_COMPAREOC3REF					   		(0x7 << TIMA_CR1_MMS_Pos)

#define TIMA_MASTERMODE_RESET_TRGO1							   	(0x0 << TIMA_CR1_MMS1_Pos)
#define TIMA_MASTERMODE_Enable_TRGO1							(0x1 << TIMA_CR1_MMS1_Pos)
#define TIMA_MASTERMODE_Update_TRGO1							(0x2 << TIMA_CR1_MMS1_Pos)
#define TIMA_MASTERMODE_COMPAREPULSE_TRGO1					   	(0x3 << TIMA_CR1_MMS1_Pos)
#define TIMA_MASTERMODE_COMPAREOC0REF_TRGO1					   	(0x4 << TIMA_CR1_MMS1_Pos)
#define TIMA_MASTERMODE_COMPAREOC1REF_TRGO1					   	(0x5 << TIMA_CR1_MMS1_Pos)
#define TIMA_MASTERMODE_COMPAREOC2REF_TRGO1					   	(0x6 << TIMA_CR1_MMS1_Pos)
#define TIMA_MASTERMODE_COMPAREOC3REF_TRGO1					   	(0x7 << TIMA_CR1_MMS1_Pos)
#define TIMA_MASTERMODE_COMPAREOC4REF_TRGO1					   	(0x8 << TIMA_CR1_MMS1_Pos)
#define TIMA_MASTERMODE_COMPAREOC5REF_TRGO1					   	(0x9 << TIMA_CR1_MMS1_Pos)
#define TIMA_MASTERMODE_COMPAREOC3PULSE_TRGO1					(0xA << TIMA_CR1_MMS1_Pos)
#define TIMA_MASTERMODE_COMPAREOC5PULSE_TRGO1					(0xB << TIMA_CR1_MMS1_Pos)
#define TIMA_MASTERMODE_COMPAREOC3OR5RISINGPULSE_TRGO1			(0xC << TIMA_CR1_MMS1_Pos)
#define TIMA_MASTERMODE_COMPAREOC3RisingOROC5FAILINGPULSE_TRGO1	(0xD << TIMA_CR1_MMS1_Pos)
#define TIMA_MASTERMODE_COMPAREOC4OR5RISINGPULSE_TRGO1			(0xE << TIMA_CR1_MMS1_Pos)
#define TIMA_MASTERMODE_COMPAREOC4RISINGOROC5FAILINGPULSE_TRGO1	(0xF << TIMA_CR1_MMS1_Pos)

#define TIMA_IC0_DIRECTTI0           					(1)   /* TIMA Input 0 OR 1 is selected to be connected to IC0 OR IC1, respectively */
#define TIMA_IC0_DIRECTTI1         						(2)   /* TIMA Input 0 OR 1 is selected to be connected to IC0 OR IC1, respectively */
#define TIMA_IC0_SELECT_TRC                				(3)   /* TIMA Input 0, 1 is selected to be connected to TRC */
#define TIMA_IC1_DIRECTTI1           					(1)   /* TIMA Input 1 OR 0 is selected to be connected to IC1 OR IC0, respectively */
#define TIMA_IC1_DIRECTTI0         						(2)   /* TIMA Input 1 OR 0 is selected to be connected to IC1 OR IC0, respectively */
#define TIMA_IC1_SELECT_TRC                				(3)   /* TIMA Input 1, 0 is selected to be connected to TRC */
#define TIMA_IC2_DIRECTTI2           					(1)   /* TIMA Input 2 OR 3 is selected to be connected to IC2 OR IC3, respectively */
#define TIMA_IC2_DIRECTTI3         						(2)   /* TIMA Input 2 OR 3 is selected to be connected to IC2 OR IC3, respectively */
#define TIMA_IC2_SELECT_TRC                				(3)   /* TIMA Input 2, 3 is selected to be connected to TRC */
#define TIMA_IC3_DIRECTTI3           					(1)   /* TIMA Input 3 OR 2 is selected to be connected to IC3 OR IC2, respectively */
#define TIMA_IC3_DIRECTTI2         						(2)   /* TIMA Input 3 OR 2 is selected to be connected to IC3 OR IC2, respectively */
#define TIMA_IC3_SELECT_TRC                				(3)   /* TIMA Input 3, 2 is selected to be connected to TRC */

/*slave mode config*/
#define TIMA_SLAVEMODE_DISABLE								(0x0UL)
#define TIMA_SLAVEMODE_RESET								(0x4UL)
#define TIMA_SLAVEMODE_GATE									(0x5UL)
#define TIMA_SLAVEMODE_TRIGGER								(0x6UL)
#define TIMA_SLAVEMODE_EXTERNCLOCK							(0x7UL)

/* tima trigger selection */
#define TIMA_CLOCKSOURCE_ITR0								(0x0UL)
#define TIMA_CLOCKSOURCE_ITR1								(0x1UL)
#define TIMA_CLOCKSOURCE_ITR2								(0x2UL)
#define TIMA_CLOCKSOURCE_ITR3								(0x3UL)
#define	TIMA_CLOCKSOURCE_TI0FED      						(0x4UL)
#define	TIMA_CLOCKSOURCE_TI0FPT      						(0x5UL)
#define	TIMA_CLOCKSOURCE_TI1FPT      						(0x6UL)
#define	TIMA_CLOCKSOURCE_ETRF        						(0x7UL)

/**
  * @brief  make TIMA active.
  * @param  SCIx  TIMA instance.
  *
  */
void DDL_TIMA_Instance_Active(TIMA_Type *TIMAx);


/**
  * @brief  Configure the elements of structure TIMA_BaseInit_T to default values.
  *			use for timer
  * @param  TIMA_BaseInit_T Pointer to a TIMA_BaseInit_T structure that contains
  *                  the configuration information for the given TIMA module.
  *
  * @retval None
  */
void DDL_TIMA_TMR_StructInit(TIMA_BaseInit_T *pTmrStruct);


/**
  * @brief  Initializes the TIMA by the specified parameters in the TIMA_BaseInit_T.
  *			use for timer
  * @param  TIMAx  TIMA instance.
  *			pTmrStruct Pointer to a TIMA_BaseInit_T structure that contains
  *                  the configuration information for the given TIMA module.
  *
  * @retval TIMA
  */
void DDL_TIMA_TMR_Init(TIMA_Type *TIMAx, TIMA_BaseInit_T *pTmrStruct);


/**
  * @brief  use for timer start.
  * @param  TIMAx  TIMA instance.
  * @retval None
  */
void DDL_TIMA_TMR_Start(TIMA_Type *TIMAx);


/**
  * @brief  use for timer stop.
  * @param  TIMAx  TIMA instance.
  * @retval None
  */
void DDL_TIMA_TMR_Stop(TIMA_Type *TIMAx);


/**
  * @brief  use for tima interrupt enable.
  * @param  intrMask : COMIE/UIE/C0IE/C1IE/C2IE/TIE/BIE
  *
  * @retval None
  */
void DDL_TIMA_IntEnable(TIMA_Type *TIMAx, uint32_t intrMask);


/**
  * @brief  use for tima interrupt disable.
  * @param  intrMask : COMIE/UIE/C0IE/C1IE/C2IE/TIE/BIE
  *
  * @retval None
  */
void DDL_TIMA_IntDisable(TIMA_Type *TIMAx, uint32_t intrMask);


/**
  * @brief  Configure the elements of structure TIMA_BaseInit_T to default values.
  *			use for pwm output
  * @param  TIMA_BaseInit_T Pointer to a TIMA_BaseInit_T structure that contains
  *                  the configuration information for the given TIMA module.
  *
  * @retval None
  */
void DDL_TIMA_PWM_StructInit(TIMA_BaseInit_T *pPwmStruct);


/**
  * @brief  Initializes the TIMA by the specified parameters in the TIMA_BaseInit_T.
  *			use for pwm output
  * @param  pTmrStruct Pointer to a TIMA_BaseInit_T structure that contains
  *                  the configuration information for the given TIMA module.
  *
  * @retval None
  */
void DDL_TIMA_PWM_Init(TIMA_Type *TIMAx, TIMA_BaseInit_T *pPwmStruct);


/**
  * @brief  use for pwm output channel0 High Impedance State.
  *
  * @param  chHighRes  : channel0 High Impedance State
  *         chnHighRes : channel0n High Impedance State
  *         1: high level
  * 		0: low level
  * @retval None
  */
void DDL_TIMA_Ch0_HighImpedanceState_Config(TIMA_Type *TIMAx, uint8_t chHighRes, uint8_t chnHighRes);


/**
  * @brief  use for pwm output channel1 High Impedance State.
  *
  * @param  chHighRes  : channel1 High Impedance State
  *         chnHighRes : channel1n High Impedance State
  *         1: high level
  * 		0: low level
  * @retval None
  */
void DDL_TIMA_Ch1_HighImpedanceState_Config(TIMA_Type *TIMAx, uint8_t chHighRes, uint8_t chnHighRes);


/**
  * @brief  use for pwm output channel2 High Impedance State.
  *
  * @param  chHighRes  : channel2 High Impedance State
  *         chnHighRes : channel2n High Impedance State
  *         1: high level
  * 		0: low level
  * @retval None
  */
void DDL_TIMA_Ch2_HighImpedanceState_Config(TIMA_Type *TIMAx, uint8_t chHighRes, uint8_t chnHighRes);


/**
  * @brief  use for pwm output channel3 High Impedance State.
  *
  * @param  chHighRes  : channel3 High Impedance State
  *         1: high level
  * 		0: low level
  * @retval None
  */
void DDL_TIMA_Ch3_HighImpedanceState_Config(TIMA_Type *TIMAx, uint8_t chHighRes);


/**
  * @brief  use for pwm output channel0 brake idle state.
  *
  * @param  chIdel  : channel0 idle state
  *         chnIdel : channel0n idle state
  *         1: high level
  * 		0: low level
  * @retval None
  */
void DDL_TIMA_Ch0_BrkIdelState_Config(TIMA_Type *TIMAx, uint8_t chIdel, uint8_t chnIdel);


/**
  * @brief  use for pwm output channel1 brake idle state.
  *
  * @param  chIdel  : channel1 idle state
  *         chnIdel : channel1n idle state
  *         1: high level
  * 		0: low level
  * @retval None
  */
void DDL_TIMA_Ch1_BrkIdelState_Config(TIMA_Type *TIMAx, uint8_t chIdel, uint8_t chnIdel);


/**
  * @brief  use for pwm output channel2 brake idle state.
  *
  * @param  chIdel  : channel2 idle state
  *         chnIdel : channel2n idle state
  *         1: high level
  * 		0: low level
  * @retval None
  */
void DDL_TIMA_Ch2_BrkIdelState_Config(TIMA_Type *TIMAx, uint8_t chIdel, uint8_t chnIdel);


/**
  * @brief  use for pwm output channel3 brake idle state.
  *
  * @param  chIdel  : channel3 idle state
  *         1: high level
  * 		0: low level
  *
  * @retval None
  */
void DDL_TIMA_Ch3_BrkIdelState_Config(TIMA_Type *TIMAx, uint8_t chIdel);


/**
  * @brief  use for pwm output channel0 duty set.
  *
  * @param  TIMAx  TIMA instance.
			preload  :
					0: effective immediately
					1: effective for the next cycle
  *			hChDuty  : channel0 acr set
  *         lChDuty  : channel0 ccr set
  * @retval None
  */
void DDL_TIMA_Ch0_Duty_Config(TIMA_Type *TIMAx, uint8_t preload, uint16_t hChDuty, uint16_t lChDuty);


/**
  * @brief  use for pwm output channel1 duty set.
  *
  * @param  TIMAx  TIMA instance.
			preload  :
					0: effective immediately
					1: effective for the next cycle
  *			hChDuty  : channel1 acr set
  *         lChDuty  : channel1 ccr set
  * @retval None
  */
void DDL_TIMA_Ch1_Duty_Config(TIMA_Type *TIMAx, uint8_t preload, uint16_t hChDuty, uint16_t lChDuty);


/**
  * @brief  use for pwm output channel2 duty set.
  *
  * @param  TIMAx  TIMA instance.
			preload  :
					0: effective immediately
					1: effective for the next cycle
  *			hChDuty  : channel2 acr set
  *         lChDuty  : channel2 ccr set
  * @retval None
  */
void DDL_TIMA_Ch2_Duty_Config(TIMA_Type *TIMAx, uint8_t preload, uint16_t hChDuty, uint16_t lChDuty);


/**
  * @brief  use for pwm output channel3 duty set.
  *
  * @param  TIMAx  TIMA instance.
			preload  :
					0: effective immediately
					1: effective for the next cycle
  *			hChDuty  : channel3 acr set
  *         lChDuty  : channel3 ccr set
  * @retval None
  */
void DDL_TIMA_Ch3_Duty_Config(TIMA_Type *TIMAx, uint8_t preload, uint16_t hChDuty, uint16_t lChDuty);


/**
  * @brief  use for pwm output channel4 duty set.
  *
  * @param  TIMAx  TIMA instance.
			preload  :
					0: effective immediately
					1: effective for the next cycle
  *         lChDuty  : channel4 ccr set
  * @retval None
  */
void DDL_TIMA_Ch4_Duty_Config(TIMA_Type *TIMAx, uint8_t preload, uint16_t lChDuty);


/**
  * @brief  use for pwm output channel5 duty set.
  *
  * @param  TIMAx  TIMA instance.
			preload  :
					0: effective immediately
					1: effective for the next cycle
  *         lChDuty  : channel5 ccr set
  * @retval None
  */
void DDL_TIMA_Ch5_Duty_Config(TIMA_Type *TIMAx, uint8_t preload, uint16_t lChDuty);


/**
  * @brief  use for pwm channel0 output mode config.
  *
  * @param  TIMAx  TIMA instance.
			mode  :
				0: TIMA_OCMODE_FREEZE_CH0
				1: TIMA_OCMODE_COMPARE_FORCED_HIGH_CH0
				2: TIMA_OCMODE_COMPARE_FORCED_LOW_CH0
				3: TIMA_OCMODE_COMPARE_TOGGLE_CH0
				4: TIMA_OCMODE_FORCED_LOW_CH0
				5: TIMA_OCMODE_FORCED_HIGH_CH0
				6: TIMA_OCMODE_PWM0_CH0
				7: TIMA_OCMODE_PWM1_CH0
				C: TIMA_OCMODE_COMBINATION_MODE_PWM0_CH0
				D: TIMA_OCMODE_COMBINATION_MODE_PWM1_CH0
				E: TIMA_OCMODE_ASYM_PWM0_CH0
				F: TIMA_OCMODE_ASYM_PWM1_CH0

  * @retval None
  */
void DDL_TIMA_Ch0_PWM_Mode_Config(TIMA_Type *TIMAx,uint8_t mode);


/**
  * @brief  use for pwm channel1 output mode config.
  *
  * @param  TIMAx  TIMA instance.
			mode  :
				0: TIMA_OCMODE_FREEZE_CH1
				1: TIMA_OCMODE_COMPARE_FORCED_HIGH_CH1
				2: TIMA_OCMODE_COMPARE_FORCED_LOW_CH1
				3: TIMA_OCMODE_COMPARE_TOGGLE_CH1
				4: TIMA_OCMODE_FORCED_LOW_CH1
				5: TIMA_OCMODE_FORCED_HIGH_CH1
				6: TIMA_OCMODE_PWM0_CH1
				7: TIMA_OCMODE_PWM1_CH1
				C: TIMA_OCMODE_COMBINATION_MODE_PWM0_CH1
				D: TIMA_OCMODE_COMBINATION_MODE_PWM1_CH1
				E: TIMA_OCMODE_ASYM_PWM0_CH1
				F: TIMA_OCMODE_ASYM_PWM1_CH1

  * @retval None
  */
void DDL_TIMA_Ch1_PWM_Mode_Config(TIMA_Type *TIMAx,uint8_t mode);


/**
  * @brief  use for pwm channel2 output mode config.
  *
  * @param  TIMAx  TIMA instance.
			mode  :
				0: TIMA_OCMODE_FREEZE_CH2
				1: TIMA_OCMODE_COMPARE_FORCED_HIGH_CH2
				2: TIMA_OCMODE_COMPARE_FORCED_LOW_CH2
				3: TIMA_OCMODE_COMPARE_TOGGLE_CH2
				4: TIMA_OCMODE_FORCED_HIGH_CH2
				5: TIMA_OCMODE_FORCED_LOW_CH2
				6: TIMA_OCMODE_PWM0_CH2
				7: TIMA_OCMODE_PWM1_CH2
				C: TIMA_OCMODE_COMBINATION_MODE_PWM0_CH2
				D: TIMA_OCMODE_COMBINATION_MODE_PWM1_CH2
				E: TIMA_OCMODE_ASYM_PWM0_CH2
				F: TIMA_OCMODE_ASYM_PWM1_CH2

  * @retval None
  */
void DDL_TIMA_Ch2_PWM_Mode_Config(TIMA_Type *TIMAx,uint8_t mode);


/**
  * @brief  use for pwm channel3 output mode config.
  *
  * @param  TIMAx  TIMA instance.
			mode  :
				0: TIMA_OCMODE_FREEZE_CH3
				1: TIMA_OCMODE_COMPARE_FORCED_HIGH_CH3
				2: TIMA_OCMODE_COMPARE_FORCED_LOW_CH3
				3: TIMA_OCMODE_COMPARE_TOGGLE_CH3
				4: TIMA_OCMODE_FORCED_HIGH_CH3
				5: TIMA_OCMODE_FORCED_LOW_CH3
				6: TIMA_OCMODE_PWM0_CH3
				7: TIMA_OCMODE_PWM1_CH3
				C: TIMA_OCMODE_COMBINATION_MODE_PWM0_CH3
				D: TIMA_OCMODE_COMBINATION_MODE_PWM1_CH3
				E: TIMA_OCMODE_ASYM_PWM0_CH3
				F: TIMA_OCMODE_ASYM_PWM1_CH3

  * @retval None
  */
void DDL_TIMA_Ch3_PWM_Mode_Config(TIMA_Type *TIMAx,uint8_t mode);


/**
  * @brief  use for pwm channel4 output mode config.
  *
  * @param  TIMAx  TIMA instance.
			mode  :
				0: TIMA_OCMODE_FREEZE_CH4
				1: TIMA_OCMODE_COMPARE_FORCED_HIGH_CH4
				2: TIMA_OCMODE_COMPARE_FORCED_LOW_CH4
				3: TIMA_OCMODE_COMPARE_TOGGLE_CH4
				4: TIMA_OCMODE_FORCED_HIGH_CH4
				5: TIMA_OCMODE_FORCED_LOW_CH4
				6: TIMA_OCMODE_PWM0_CH4
				7: TIMA_OCMODE_PWM1_CH4

  * @retval None
  */
void DDL_TIMA_Ch4_PWM_Mode_Config(TIMA_Type *TIMAx,uint8_t mode);


/**
  * @brief  use for pwm channel5 output mode config.
  *
  * @param  TIMAx  TIMA instance.
			mode  :
				0: TIMA_OCMODE_FREEZE_CH5
				1: TIMA_OCMODE_COMPARE_FORCED_HIGH_CH5
				2: TIMA_OCMODE_COMPARE_FORCED_LOW_CH5
				3: TIMA_OCMODE_COMPARE_TOGGLE_CH5
				4: TIMA_OCMODE_FORCED_HIGH_CH5
				5: TIMA_OCMODE_FORCED_LOW_CH5
				6: TIMA_OCMODE_PWM0_CH5
				7: TIMA_OCMODE_PWM1_CH5

  * @retval None
  */
void DDL_TIMA_Ch5_PWM_Mode_Config(TIMA_Type *TIMAx,uint8_t mode);


/**
  * @brief  use for pwm channel0 output polar config.
  *
  * @param  TIMAx  TIMA instance.
			chPolar: channel0 polar config
			chnPolar: channel0n polar config
  * @retval None
  */
void DDL_TIMA_Ch0_PWM_Polar_Config(TIMA_Type *TIMAx, uint8_t chPolar, uint8_t chnPolar);


/**
  * @brief  use for pwm channel1 output polar config.
  *
  * @param  TIMAx  TIMA instance.
			chPolar: channel0 polar config
			chnPolar: channel0n polar config
  * @retval None
  */
void DDL_TIMA_Ch1_PWM_Polar_Config(TIMA_Type *TIMAx, uint8_t chPolar, uint8_t chnPolar);


/**
  * @brief  use for pwm channel2 output polar config.
  *
  * @param  TIMAx  TIMA instance.
			chPolar: channel0 polar config
			chnPolar: channel0n polar config
  * @retval None
  */
void DDL_TIMA_Ch2_PWM_Polar_Config(TIMA_Type *TIMAx, uint8_t chPolar, uint8_t chnPolar);


/**
  * @brief  use for pwm channel3 output polar config.
  *
  * @param  TIMAx  TIMA instance.
			chPolar: channel3 polar config
			chnPolar: channel3n polar config
  * @retval None
  */
void DDL_TIMA_Ch3_PWM_Polar_Config(TIMA_Type *TIMAx, uint8_t chPolar, uint8_t chnPolar);


/**
  * @brief  use for pwm output channel enable.
  *
  * @param  TIMAx  TIMA instance.
			chEnable: channel enable select
  *
  * @retval None
  */
void DDL_TIMA_PWM_Enable(TIMA_Type *TIMAx, uint32_t chEnable);


/**
  * @brief  use for pwm output start.
  * @param  TIMAx  TIMA instance.
  * @retval None
  */
void DDL_TIMA_PWM_Start(TIMA_Type *TIMAx);


/**
  * @brief  use for pwm output stop.
  * @param  TIMAx  TIMA instance.
  * @retval None
  */
void DDL_TIMA_PWM_Stop(TIMA_Type *TIMAx);


/**
  * @brief  Configure the elements of structure TIMA_Deadtime_T to default values.
  *			use for timer
  * @param  pDeadtimeStruct Pointer to a TIMA_Deadtime_T structure that contains
  *                  the configuration information for the given TIMA module.
  *
  * @retval None
  */
void DDL_TIMA_DeadTime_StructInit(TIMA_Deadtime_T *pDeadtimeStruct);


/**
  * @brief  Initializes the TIMA by the specified parameters in the TIMA_Deadtime_T.
  *			use for timer
  * @param  TIMAx  TIMA instance.
  *			pDeadtimeStruct Pointer to a TIMA_Deadtime_T structure that contains
  *                  the configuration information for the given TIMA module.
  *
  * @retval TIMA
  */
void DDL_TIMA_DeadTime_Init(TIMA_Type *TIMAx, TIMA_Deadtime_T *pDeadtimeStruct);


/**
  * @brief  use for pwm output brake config.
  * @param  brkSrc :
				 TIMA_BRK_SOURCE_EBUS_CHx
                 TIMA_BRAKE_IO_SRC
                 TIMA_CPU_HALT_SRC
                 TIMA_SRAM_PARITY_ERR_SRC
                 TIMA_PLL_LOSS_PEND_SRC
                 TIMA_MCLK_LOSS_PEND_SRC
			polar :
				0: low  level
				1: high level
  * @retval None
  */
void DDL_TIMA_PWM_Brake_Config(TIMA_Type *TIMAx, uint32_t brkSrc, uint8_t polar);


/**
  * @brief  pwm auto output enable.
  * @param  TIMAx  TIMA instance.
  * @retval None
  */
void DDL_TIMA_Auto_Output_Enable(TIMA_Type *TIMAx);


/**
  * @brief  pwm auto output disable.
  * @param  TIMAx  TIMA instance.
  * @retval None
  */
void DDL_TIMA_Auto_Output_Disable(TIMA_Type *TIMAx);


/**
  * @brief  tima as master trgo.
  * @param  TIMAx  TIMA instance.
  * @param  trgoSrc : RESET/ENABLE/UPDATE/COMPAREPULSE/COMPAREOCxREF
  *
  * @retval None
  */
void DDL_TIMA_AsMaster_Trgo(TIMA_Type *TIMAx,uint32_t trgoSrc);


/**
  * @brief  tima as master trg1.
  * @param  TIMAx  TIMA instance.
  * @param  trgo1Src : RESET/ENABLE/UPDATE/COMPAREPULSE/COMPAREOCxREF/COMPAREOCxPULSE...
  *
  * @retval None
  */
void DDL_TIMA_AsMaster_Trgo1(TIMA_Type *TIMAx,uint32_t trgo1Src);


/**
  * @brief  timA soft brake.
  * @param  TIMAx  TIMA instance.
  * @retval None
  */
void DDL_TIMA_SoftBrake(TIMA_Type *TIMAx);


/**
  * @brief  TIMA event trigger config
  * @param  TIMAx  TIMA instance.
  *			event0Psc: event0 psc
				prescaler = event0Psc + 1
  *			event0Edge: event0 config
				0: OC4REF Rising
				1: OC4REF Falling
				2: OC4REF both edge
  * @retval None
  */
void DDL_TIMA_Event0_Trigger_Config(TIMA_Type *TIMAx, uint32_t event0Psc, uint32_t event0Edge);


/**
  * @brief  TIMA event trigger config
  * @param  TIMAx  TIMA instance.
  *			event1Psc: event1 psc
				prescaler = event1Psc + 1
  *			event1Edge: event1 config
				0: OC5REF Rising
				1: OC5REF Falling
				2: OC5REF both edge
  * @retval None
  */
void DDL_TIMA_Event1_Trigger_Config(TIMA_Type *TIMAx, uint32_t event1Psc, uint32_t event1Edge);


/**
  * @brief  use to clear event product counter

  * @param  TIMAx  TIMA instance.
			clearEvent0Result  clear event product counter(the bit auto-clear)
			0:  no
			1:  yes(clear)
  */
void DDL_TIMA_Event_ClearEvent0_Config(TIMA_Type *TIMAx,uint8_t clearEvent0Result);


/**
  * @brief  use to clear event product counter

  * @param  TIMAx  TIMA instance.
			clearEvent1Result  clear event product counter(the bit auto-clear)
			0:  no
			1:  yes(clear)
  */
void DDL_TIMA_Event_ClearEvent1_Config(TIMA_Type *TIMAx, uint8_t clearEvent1Result);


/**
  * @brief  Configure the elements of structure TIMA_BaseInit_T to default values.
  *			use for pwm capture
  * @param  pPwmStruct Pointer to a TIMA_BaseInit_T structure that contains
  *                  the configuration information for the given TIMA module.
  *
  * @retval None
  */
void DDL_TIMA_Capture_StructInit(TIMA_BaseInit_T *pPwmStruct);


/**
  * @brief  Initializes the TIMA by the specified parameters in the TIMA_BaseInit_T.
  *			use for pwm capture
  * @param  pTmrStruct Pointer to a TIMA_BaseInit_T structure that contains
  *                  the configuration information for the given TIMA module.
  *			TIMAx  TIMA instance.
  * @retval None
  */
void DDL_TIMA_Capture_Init(TIMA_Type *TIMAx, TIMA_BaseInit_T *pPwmStruct);


/**
  * @brief  use for capture channel0 output polar config.
  *
  * @param  TIMAx  TIMA instance.
			chPolar: chnPolar
	  rising   0   :   0
	  failing  1   :   0
	  both     1   :   1
  * @retval None
  */
void DDL_TIMA_Ch0_Capture_Polar_Config(TIMA_Type *TIMAx, uint8_t chPolar, uint8_t chnPolar);


/**
  * @brief  use for capture channel1 output polar config.
  *
  * @param  TIMAx  TIMA instance.
			chPolar: chnPolar
	  rising   0   :   0
	  failing  1   :   0
	  both     1   :   1
  * @retval None
  */
void DDL_TIMA_Ch1_Capture_Polar_Config(TIMA_Type *TIMAx, uint8_t chPolar, uint8_t chnPolar);


/**
  * @brief  use for capture channel2 output polar config.
  *
  * @param  TIMAx  TIMA instance.
			chPolar: chnPolar
	  rising   0   :   0
	  failing  1   :   0
	  both     1   :   1
  * @retval None
  */
void DDL_TIMA_Ch2_Capture_Polar_Config(TIMA_Type *TIMAx, uint8_t chPolar, uint8_t chnPolar);


/**
  * @brief  use for capture channel3 output polar config.
  *
  * @param  TIMAx  TIMA instance.
			chPolar: chnPolar
	  rising   0   :   0
	  failing  1   :   0
	  both     1   :   1
  * @retval None
  */
void DDL_TIMA_Ch3_Capture_Polar_Config(TIMA_Type *TIMAx, uint8_t chPolar, uint8_t chnPolar);


/**
  * @brief  Configure the elements of structure TIMA_CapturePara_T to default values.
  *			use for capture
  * @param  pCapStruct Pointer to a TIMA_CapturePara_T structure that contains
  *                  the configuration information for the given tima module.
  *
  * @retval None
  */
void DDL_TIMA_Capture_Channel0_StructInit(TIMA_CapturePara_T *pCapStruct);


/**
  * @brief  Initializes the tima by the specified parameters in the TIMA_CapturePara_T.
  *			use for capture
  * @param  TIMAx  TIMA instance.
			pCapStruct Pointer to a TIMA_CapturePara_T structure that contains
  *                  the configuration information for the given tima module.
  *
  * @retval None
  */
void DDL_TIMA_Capture_Channel0_Config(TIMA_Type *TIMAx, TIMA_CapturePara_T *pCapStruct);


/**
  * @brief  Configure the elements of structure TIMA_CapturePara_T to default values.
  *			use for capture
  * @param  pCapStruct Pointer to a TIMA_CapturePara_T structure that contains
  *                  the configuration information for the given tima module.
  *
  * @retval None
  */
void DDL_TIMA_Capture_Channel1_StructInit(TIMA_CapturePara_T *pCapStruct);


/**
  * @brief  Initializes the tima by the specified parameters in the TIMA_CapturePara_T.
  *			use for capture
  * @param  TIMAx  TIMA instance.
			pCapStruct Pointer to a TIMA_CapturePara_T structure that contains
  *                  the configuration information for the given tima module.
  *
  * @retval None
  */
void DDL_TIMA_Capture_Channel1_Config(TIMA_Type *TIMAx, TIMA_CapturePara_T *pCapStruct);


/**
  * @brief  Configure the elements of structure TIMA_CapturePara_T to default values.
  *			use for capture
  * @param  pCapStruct Pointer to a TIMA_CapturePara_T structure that contains
  *                  the configuration information for the given tima module.
  *
  * @retval None
  */
void DDL_TIMA_Capture_Channel2_StructInit(TIMA_CapturePara_T *pCapStruct);


/**
  * @brief  Initializes the tima by the specified parameters in the TIMA_CapturePara_T.
  *			use for capture
  * @param  TIMAx  TIMA instance.
			pCapStruct Pointer to a TIMA_CapturePara_T structure that contains
  *                  the configuration information for the given tima module.
  *
  * @retval None
  */
void DDL_TIMA_Capture_Channel2_Config(TIMA_Type *TIMAx, TIMA_CapturePara_T *pCapStruct);


/**
  * @brief  Configure the elements of structure TIMA_CapturePara_T to default values.
  *			use for capture
  * @param  pCapStruct Pointer to a TIMA_CapturePara_T structure that contains
  *                  the configuration information for the given tima module.
  *
  * @retval None
  */
void DDL_TIMA_Capture_Channel3_StructInit(TIMA_CapturePara_T *pCapStruct);


/**
  * @brief  Initializes the tima by the specified parameters in the TIMA_CapturePara_T.
  *			use for capture
  * @param  TIMAx  TIMA instance.
			pCapStruct Pointer to a TIMA_CapturePara_T structure that contains
  *                  the configuration information for the given tima module.
  *
  * @retval None
  */
void DDL_TIMA_Capture_Channel3_Config(TIMA_Type *TIMAx, TIMA_CapturePara_T *pCapStruct);


/**
  * @brief  use for tima slave mode config.
  *
  * @param  TIMAx  TIMA instance.
			triggerSignal:  ITRx/TI0FED/TI0FPT/TI1FPT
			slaveMode:		reset/gate/trigger
  * @retval None
  */
void DDL_TIMA_SlaveMode_Config(TIMA_Type *TIMAx, uint8_t triggerSignal, uint8_t slaveMode);


/**
  * @brief  use for timer start.
  * @param  TIMAx  TIMA instance.
  * @retval None
  */
void DDL_TIMA_Capture_Start(TIMA_Type *TIMAx);


/**
  * @brief  use for timer stop.
  * @param  TIMAx  TIMA instance.
  * @retval None
  */
void DDL_TIMA_Capture_Stop(TIMA_Type *TIMAx);




#ifdef __cplusplus
}
#endif

#endif /* __KPM32xx_DDL_TIMA_H */










