/**
  *****************************************************************************************************
  * @file    kpm32xx_ddl.h
  * @author  Kiwi Software Team
  * @brief   Header file of common ddl.
  * @note
  *          V1.0.0, 2024/12/20.
  *
  * Copyright (c) 2024, Kiwi Instruments Co,. Ltd.
  *
  * Redistribution and use in source and binary forms, with or without modification,
  * are permitted provided that the following conditions are met:
  *
  *   1. Redistributions of source code must retain the above copyright notice,
  *      this list of conditions and the following disclaimer.
  *
  *   2. Redistributions in binary form must reproduce the above copyright notice,
  *      this list of conditions and the following disclaimer in the documentation
  *      and/or other materials provided with the distribution.
  *
  *   3. Neither the name of the copyright holder nor the names of its contributors
  *      may be used to endorse or promote products derived from this software without
  *      specific prior written permission.
  *
  * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
  * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
  * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
  * THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
  * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE
  * GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
  * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
  * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
  * THE POSSIBILITY OF SUCH DAMAGE.
  *****************************************************************************************************
  */


#ifndef __KPM32xx_DDL_H
#define __KPM32xx_DDL_H

#ifdef __cplusplus
extern "C" {
#endif

#include "kpm32xx_ddl_conf.h"


/**
  * Systick frequency.
  */
typedef enum
{
  DDL_TICKFREQ_10HZ         = 100U,
  DDL_TICKFREQ_100HZ        = 10U,
  DDL_TICKFREQ_1KHZ         = 1U,
  DDL_TICKFREQ_DEFAULT      = DDL_TICKFREQ_1KHZ
} DDL_TickFreq_T;


/**
  * @brief  Initialize systick.
  * @retval DDL status
  */
DDL_Status_T DDL_Init(void);

/**
  * @brief  This function configures time source to have 1ms 10ms or 100ms time base with a dedicated
  *         Tick interrupt priority.
  * @param  tickPriority Tick interrupt priority.
  * @retval DDL status
  */
DDL_Status_T DDL_InitTick(uint32_t tickPriority);

/**
  * @brief  This function is called to increment a software "Tick".
  * @retval None
  */
void DDL_IncTick(void);

/**
  * @brief  This function provides delay in milliseconds.
  * @note   Caution: The SysTick interrupt must have higher priority, in case that it's called
  *         in a certain peripheral ISR. Otherwise, the caller ISR process will be blocked.
  * @param  Delay  delay time, in milliseconds.
  * @retval None
  */
void DDL_Delay(uint32_t delay);

/**
  * @brief  Return tick value in millisecond.
  * @retval tick value
  */
uint32_t DDL_GetTick(void);

/**
  * @brief  This function returns Systick priority.
  * @retval tick priority
  */
uint32_t DDL_GetTickPrio(void);

/**
  * @brief  Set new tick tickFreq.
  * @retval status
  */
DDL_Status_T DDL_SetTickFreq(DDL_TickFreq_T tickFreq);

/**
  * @brief  Get new tick tickFreq.
  * @retval Tick frequency
  */
DDL_TickFreq_T DDL_GetTickFreq(void);

/**
  * @brief  Suspend Tick increment.
  * @note   DDL_SuspendTick() will disable Systick interrupt.
  *
  * @retval None
  */
void DDL_SuspendTick(void);

/**
  * @brief  Resume Tick increment.
  * @note   DDL_ResumeTick() will enable Systick interrupt.
  *
  * @retval None
  */
void DDL_ResumeTick(void);

/* incremented each 1ms in SysTick ISR. */
extern __IO uint32_t uwTicks;

/* Systick priority. */
extern uint32_t uwTickPriority;

/* Systick frequency. */
extern DDL_TickFreq_T uwTickFreq;


#define IS_TICKFREQ(FREQ) (((FREQ) == DDL_TICKFREQ_10HZ)  || \
                           ((FREQ) == DDL_TICKFREQ_100HZ) || \
                           ((FREQ) == DDL_TICKFREQ_1KHZ))


#ifdef __cplusplus
}
#endif

#endif /* __KPM32xx_DDL_H */

