/**
  *****************************************************************************************************
  * @file    main.c
  * @author  Kiwi Software Team
  * @brief   This example code demonstrates how SPI works.
  *
  * Copyright (c) 2024, Kiwi Instruments Co,. Ltd.
  *
  * Redistribution and use in source and binary forms, with or without modification,
  * are permitted provided that the following conditions are met:
  *
  *   1. Redistributions of source code must retain the above copyright notice,
  *      this list of conditions and the following disclaimer.
  *
  *   2. Redistributions in binary form must reproduce the above copyright notice,
  *      this list of conditions and the following disclaimer in the documentation
  *      and/or other materials provided with the distribution.
  *
  *   3. Neither the name of the copyright holder nor the names of its contributors
  *      may be used to endorse or promote products derived from this software without
  *      specific prior written permission.
  *
  * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
  * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
  * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
  * THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
  * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE
  * GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
  * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
  * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
  * THE POSSIBILITY OF SUCH DAMAGE.
  *****************************************************************************************************
  */

#include "main.h"
#include "kpm32xx_demoboard_ddl.h"


#define __MAX_SPI_DATA_VAR_LENGTH__        8


uint8_t wrBuf[__MAX_SPI_DATA_VAR_LENGTH__] = {'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h'};
uint8_t rdBuf[__MAX_SPI_DATA_VAR_LENGTH__] = {0x5A, 0x5A, 0x5A, 0x5A, 0xA5, 0xA5, 0xA5, 0xA5};
/* for data-comparison */
const uint8_t rdBufCmp[__MAX_SPI_DATA_VAR_LENGTH__] = {0x33, 0x35, 0x77, 0x89, 0x66, 0xf6, 0x1f, 0x5a};

uint8_t *pTxBuffPtr;
uint8_t *pRxBuffPtr;

volatile uint32_t remainSendCnt = 0;
volatile uint32_t remainReceiveCnt = 0;

volatile uint8_t spiTxCmplt = 0;
volatile uint8_t spiRxCmplt = 0;




#if defined (__DEBUG_CONSOLE_PRINT__)
/**
  * @brief  This function is implemented for UART print PinMux configuration.
  * @param  None
  * @retval None
  */
void DebugConsole_PinMux_Config(void)
{
	/* SCI1 UART Tx */
	DDL_GPIO_Config2AltFunc(GPIOA, GPIO_PIN_7, GPIOA7_AF1_SCI1_TX_SDA_MOSI);
}
#endif


void SPI_PinMux_Config(void)
{
	/* MOSI */
	DDL_GPIO_Config2AltFunc(GPIOC, GPIO_PIN_9, GPIOC9_AF2_SPI_MOSI);
	/* MISO */
	DDL_GPIO_Config2AltFunc(GPIOC, GPIO_PIN_10, GPIOC10_AF2_SPI_MISO);
	/* SCK */
	DDL_GPIO_Config2AltFunc(GPIOC, GPIO_PIN_8, GPIOC8_AF2_SPI_SCK);
	/* CS */
	//DDL_GPIO_Config2AltFunc(GPIOC, GPIO_PIN_7, GPIOC7_AF2_SPI_SS0);
	DDL_GPIO_Config2AltFunc(GPIOA, GPIO_PIN_10, GPIOA10_AF2_SPI_SS0);
}


/**
  * @brief  This function is implemented for system clock configuration.
  *         where:
  *         System Clock source            = PLL (HIRC)
  *         System Clock frequency         = 100MHz
  * @param  None
  * @retval None
  */
void SystemClock_Config(void)
{
	RCC_PLLInit_T pllInitStruct    = {0};
	RCC_ClkInit_T rccClkInitStruct = {0};

	DDL_RCC_PllStructInit(&pllInitStruct);
	/* FCLK : 100MHz */
	pllInitStruct.pllPrescaler  = 4;
	if (DDL_RCC_PllConfig(&pllInitStruct) != DDL_OK)
	{
		while(1) ;
	}

	/* Select PLL as system clock source */
	rccClkInitStruct.sysClkSource  = RCC_SYSCLKSOURCE_PLLCLK;
	rccClkInitStruct.sysClkDivider = 0;
	if (DDL_RCC_ClkSrcConfig(&rccClkInitStruct) != DDL_OK)
	{
		while(1) ;
	}
}


/**
  * @brief  The application entry point.
  * @retval int
  */
int main(void)
{
	uint32_t i = 0;
	uint32_t rx_thrshld = 0;
	SPI_Init_T pSPIStruct;

	/* Reset of all peripherals, Initializes the Flash interface and the Systick. */
	DDL_Init();
	SystemClock_Config();

	/* To make GPIO active */
	DDL_GPIO_Instance_Active();

#if defined (__DEBUG_CONSOLE_PRINT__)
	DebugConsole_PinMux_Config();
	DDL_SCI_Instance_Active(SCI1);
	DDL_SCIUart_DebugConsole_Init(SCI1);
#endif

	SPI_PinMux_Config();
	DDL_SPI_Instance_Active(SPI);

	DDL_SPI_StructInit(&pSPIStruct);
    if (DDL_SPI_Init(SPI, &pSPIStruct) != DDL_OK)
	{
		__asm volatile("bkpt 0");
	}

	DDL_Delay(1000);
	while(1)
	{
		printf("Start SPI transmission through interrupt ...\n");

		pTxBuffPtr = wrBuf;
		remainSendCnt = __MAX_SPI_DATA_VAR_LENGTH__;
		__DDL_SPI_CLEAR_TXFIFO(SPI);
		__DDL_SPI_MULTIDATATRANS_CONFIG(SPI, (__MAX_SPI_DATA_VAR_LENGTH__ - 1));
		__DDL_SPI_TXTHRD_CONFIG(SPI, 0);
		/* Clear all interrupt flags */
		WRITE_REG(SPI->INTST, 0xFFFFFFF);
		DDL_SPI_IntEnable(SPI, SPI_INTR_TXFFTHRWRN | SPI_INTR_SNDCMPLT);
		NVIC_SetPriority(SPI_IRQn, 3);
		NVIC_EnableIRQ(SPI_IRQn);

		while(spiTxCmplt == 0);
		spiTxCmplt = 0;

		printf("SPI transmitted %d byte(s) through interrupt completes \n", sizeof(wrBuf)/sizeof(uint8_t));

		DDL_Delay(500);

		pRxBuffPtr = rdBuf;
		remainReceiveCnt = __MAX_SPI_DATA_VAR_LENGTH__;
		__DDL_SPI_CLEAR_RXFIFO(SPI);
		__DDL_SPI_CLEAR_TXFIFO(SPI);
		__DDL_SPI_MULTIDATATRANS_CONFIG(SPI, (__MAX_SPI_DATA_VAR_LENGTH__ - 1));
		rx_thrshld = MIN(__SPI_FIFORX_THRESHOLD__, remainReceiveCnt);
		__DDL_SPI_RXTHRD_CONFIG(SPI, rx_thrshld);
		/* Clear all interrupt flags */
		WRITE_REG(SPI->INTST, 0xFFFFFFF);
		DDL_SPI_IntEnable(SPI, SPI_INTR_RXFFTHRWRN);
		NVIC_SetPriority(SPI_IRQn, 3);
		NVIC_EnableIRQ(SPI_IRQn);
		while (rx_thrshld)
		{
			SPI->DR = (uint8_t)__SPI_08BIT_DUMMY_DATA__;
			rx_thrshld--;
		}

		while(spiRxCmplt == 0);
		spiRxCmplt = 0;

		if (memcmp(rdBuf, rdBufCmp, __MAX_SPI_DATA_VAR_LENGTH__) != 0)
		{
			__asm volatile("bkpt 0");
		}

		printf("Received Data: ");
		for (i = 0; i < __MAX_SPI_DATA_VAR_LENGTH__; i++)
		{
			printf("0x%02x ", rdBuf[i]);
		}
		printf("\n");

		memset(rdBuf, 0xAA, __MAX_SPI_DATA_VAR_LENGTH__);
		/* wait slave ready */
		DDL_Delay(100);
	}
}

