/**
  *****************************************************************************************************
  * @file    main.c
  * @author  Kiwi Software Team
  * @brief   This example code demonstrates how IOW works.
  *
  * Copyright (c) 2024, Kiwi Instruments Co,. Ltd.
  *
  * Redistribution and use in source and binary forms, with or without modification,
  * are permitted provided that the following conditions are met:
  *
  *   1. Redistributions of source code must retain the above copyright notice,
  *      this list of conditions and the following disclaimer.
  *
  *   2. Redistributions in binary form must reproduce the above copyright notice,
  *      this list of conditions and the following disclaimer in the documentation
  *      and/or other materials provided with the distribution.
  *
  *   3. Neither the name of the copyright holder nor the names of its contributors
  *      may be used to endorse or promote products derived from this software without
  *      specific prior written permission.
  *
  * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
  * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
  * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
  * THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
  * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE
  * GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
  * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
  * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
  * THE POSSIBILITY OF SUCH DAMAGE.
  *****************************************************************************************************
  */

#include "main.h"
#include "kpm32xx_demoboard_ddl.h"



/* if want to detect rising edge */
//#define RISING_DET

static uint8_t iowKr2Flag = 0;



#if defined (__DEBUG_CONSOLE_PRINT__)
/**
  * @brief  This function is implemented for UART print PinMux configuration.
  * @param  None
  * @retval None
  */
void DebugConsole_PinMux_Config(void)
{
	/* SCI1 UART Tx */
	DDL_GPIO_Config2AltFunc(GPIOA, GPIO_PIN_7, GPIOA7_AF1_SCI1_TX_SDA_MOSI);
}
#endif


/**
  * @brief  This function is implemented for system clock configuration.
  *         where:
  *         System Clock source            = PLL (HIRC)
  *         System Clock frequency         = 100MHz
  * @param  None
  * @retval None
  */
void SystemClock_Config(void)
{
	RCC_PLLInit_T pllInitStruct    = {0};
	RCC_ClkInit_T rccClkInitStruct = {0};

	DDL_RCC_PllStructInit(&pllInitStruct);
	/* FCLK : 100MHz */
	pllInitStruct.pllPrescaler  = 4;
	if (DDL_RCC_PllConfig(&pllInitStruct) != DDL_OK)
	{
		while(1) ;
	}

	/* Select PLL as system clock source */
	rccClkInitStruct.sysClkSource  = RCC_SYSCLKSOURCE_PLLCLK;
	rccClkInitStruct.sysClkDivider = 0;
	if (DDL_RCC_ClkSrcConfig(&rccClkInitStruct) != DDL_OK)
	{
		while(1) ;
	}
}


/**
  * @brief  IOW Channel 0 interrupt callback.
  * @param  none
  *
  * @retval none
  */
void DDL_IOW_KR0Callback(IOW_Type *IOWx)
{
	iowKr2Flag = 1;

	/* clear flag of KR0 */
	__DDL_IOW_CLEAR_CHNINTR(IOWx,(1<<0));

	UNUSED(IOWx);
}


/**
  * @brief  The application entry point.
  * @retval int
  */
int main(void)
{
    IOW_Init_T iowInt;
	/* Reset of all peripherals, Initializes the Flash interface and the Systick. */
	DDL_Init();
	SystemClock_Config();

	/* To make GPIO active */
	DDL_GPIO_Instance_Active();

#if defined (__DEBUG_CONSOLE_PRINT__)
	DebugConsole_PinMux_Config();
	DDL_SCI_Instance_Active(SCI1);
	DDL_SCIUart_DebugConsole_Init(SCI1);
#endif
	printf("This demo demonstrates IOW mounted on board !!\n");

    /* To make GPIO active */
	DDL_GPIO_Instance_Active();
	/* SCI UART pinmux configuration */
	DebugConsole_PinMux_Config();
	/* Makes IOW active */
	DDL_IOW_Instance_Active();
    /* Init for IOW KR0, (PA_6, FALLING) */
	DDL_GPIO_Config2AltFunc(GPIOA, GPIO_PIN_6, GPIOA6_AF7_KR0);
	DDL_IOW_StructInit(&iowInt);
#ifdef RISING_DET
	iowInt.edges = IOW_EDGE_RISING;
#endif
	DDL_IOW_Init(IOW,&iowInt);

	/* Enable IOW Interrupt */
	DDL_IOW_IntEnable(IOW, &iowInt);
	NVIC_SetPriority(IOW_IRQn, __PREEMPT_INTRERRUPT_PRIORITY_0__);
	NVIC_EnableIRQ(IOW_IRQn);

	while(1)
	{
		if(iowKr2Flag)
		{
			iowKr2Flag = 0;

#ifdef RISING_DET
			printf("IOW KR0 Rising edge detected.\n");
			/*
			 * PA_0 connect to GND, then pull up to VCC.
			 * KR0 rising edge will be detected again.
			 */
#else
			printf("IOW KR0 Falling edge detected.\n");
			/*
			 * PA_0 connect to VCC, then pull down to GND.
			 * KR0 falling edge will be detected again.
			 */
#endif

			DDL_Delay(1000);
		}
	}

}

