/**
  *****************************************************************************************************
  * @file    kpm32xx_ddl_tima.c
  * @author  Kiwi Software Team
  * @brief   TIMA DDL module driver.
  *          This file provides firmware functions to manage the following
  *          functionalities:
  *          1. TIMA as basetimer.
  *          2. TIMA pwm output.
  *			     3. TIMA pwm capture.
  *			     4. TIMA pwm brake.
  *          5. TIMA as master/slave mode
  * @note
  *          V1.0.0, 2025/1/6.
  *
  * Copyright (c) 2024, Kiwi Instruments Co,. Ltd.
  *
  * Redistribution and use in source and binary forms, with or without modification,
  * are permitted provided that the following conditions are met:
  *
  *   1. Redistributions of source code must retain the above copyright notice,
  *      this list of conditions and the following disclaimer.
  *
  *   2. Redistributions in binary form must reproduce the above copyright notice,
  *      this list of conditions and the following disclaimer in the documentation
  *      and/or other materials provided with the distribution.
  *
  *   3. Neither the name of the copyright holder nor the names of its contributors
  *      may be used to endorse or promote products derived from this software without
  *      specific prior written permission.
  *
  * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
  * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
  * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
  * THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
  * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE
  * GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
  * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
  * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
  * THE POSSIBILITY OF SUCH DAMAGE.
  *****************************************************************************************************
  */


#include "kpm32xx_ddl.h"


#ifdef DDL_TIMA_MODULE_ENABLED


/**
  * @brief  make TIMA active.
  * @param  TIMAx  TIMA instance.
  *
  */
void DDL_TIMA_Instance_Active(TIMA_Type *TIMAx)
{
	if (TIMAx == TIMA0)
	{
		__DDL_RCC_TIMA0_ACTIVE();
		__DDL_RCC_TIMA0_CLK_RELEASE();
	}
	else if (TIMAx == TIMA1)
	{
		__DDL_RCC_TIMA1_ACTIVE();
		__DDL_RCC_TIMA1_CLK_RELEASE();
	}
}


/**
  * @brief  Configure the elements of structure TIMA_BaseInit_T to default values.
  *			use for timer
  * @param  TIMA_BaseInit_T Pointer to a TIMA_BaseInit_T structure that contains
  *                  the configuration information for the given TIMA module.
  *
  * @retval TIMA
  */
void DDL_TIMA_TMR_StructInit(TIMA_BaseInit_T *pTmrStruct)
{
	pTmrStruct->countermode = TIMA_COUNTERMODE_COUNTER_UP;
	pTmrStruct->period = 19999;
	pTmrStruct->prescaler = 9;
	pTmrStruct->repetitionCounter = 0;
}


/**
  * @brief  Initializes the TIMA by the specified parameters in the TIMA_BaseInit_T.
  *			use for timer
  * @param  TIMAx  TIMA instance.
  *			pTmrStruct Pointer to a TIMA_BaseInit_T structure that contains
  *                  the configuration information for the given TIMA module.
  *
  * @retval TIMA
  */
void DDL_TIMA_TMR_Init(TIMA_Type *TIMAx, TIMA_BaseInit_T *pTmrStruct)
{
	uint32_t tmpcr = 0;

	tmpcr = TIMAx->CR0;
	tmpcr &= ~TIMA_COUNTERMODE_COUNTER_CENTER_3;
	tmpcr |= pTmrStruct->countermode;
	TIMAx->CR0 = tmpcr;
	TIMAx->ARR = pTmrStruct->period;
	TIMAx->PSC = pTmrStruct->prescaler;
	TIMAx->RCR = pTmrStruct->repetitionCounter;
}


/**
  * @brief  use for timer start.
  * @param  TIMAx  TIMA instance.
  * @retval None
  */
void DDL_TIMA_TMR_Start(TIMA_Type *TIMAx)
{
	MODIFY_REG(TIMAx->BDTR, TIMA_BDTR_MOE , 1 << TIMA_BDTR_MOE_Pos);
	MODIFY_REG(TIMAx->CR0,  TIMA_CR0_CEN ,  1 << TIMA_CR0_CEN_Pos);
}


/**
  * @brief  use for timer stop.
  * @param  TIMAx  TIMA instance.
  * @retval None
  */
void DDL_TIMA_TMR_Stop(TIMA_Type *TIMAx)
{
	MODIFY_REG(TIMAx->BDTR, TIMA_BDTR_MOE , 0 << TIMA_BDTR_MOE_Pos);
	MODIFY_REG(TIMAx->CR0,  TIMA_CR0_CEN ,  0 << TIMA_CR0_CEN_Pos);
}


/**
  * @brief  use for tima interrupt enable.
  * @param  intrMask : COMIE/UIE/C0IE/C1IE/C2IE/TIE/BIE
  *			TIMAx  TIMA instance.
  * @retval None
  */
void DDL_TIMA_IntEnable(TIMA_Type *TIMAx, uint32_t intrMask)
{
	SET_BIT(TIMAx->DIER,  intrMask);
}


/**
  * @brief  use for tima interrupt disable.
  * @param  intrMask : COMIE/UIE/C0IE/C1IE/C2IE/TIE/BIE
  * 		TIMAx  TIMA instance.
  * @retval None
  */
void DDL_TIMA_IntDisable(TIMA_Type *TIMAx, uint32_t intrMask)
{
	CLEAR_BIT(TIMAx->DIER,  intrMask);
}


/**
  * @brief  Configure the elements of structure TIMA_BaseInit_T to default values.
  *			use for pwm output
  * @param  pPwmStruct Pointer to a TIMA_BaseInit_T structure that contains
  *                  the configuration information for the given TIMA module.
  *
  * @retval None
  */
void DDL_TIMA_PWM_StructInit(TIMA_BaseInit_T *pPwmStruct)
{
	pPwmStruct->countermode = TIMA_COUNTERMODE_COUNTER_CENTER_3;
	pPwmStruct->period = 6249;
	pPwmStruct->prescaler = 0;
	pPwmStruct->repetitionCounter = 1;
}


/**
  * @brief  Initializes the TIMA by the specified parameters in the TIMA_BaseInit_T.
  *			use for pwm output
  * @param  pTmrStruct Pointer to a TIMA_BaseInit_T structure that contains
  *                  the configuration information for the given TIMA module.
  *			TIMAx  TIMA instance.
  * @retval None
  */
void DDL_TIMA_PWM_Init(TIMA_Type *TIMAx, TIMA_BaseInit_T *pPwmStruct)
{
	uint32_t tmpcr = 0;

	tmpcr = TIMAx->CR0;
	tmpcr &= ~TIMA_COUNTERMODE_COUNTER_CENTER_3;
	tmpcr |= pPwmStruct->countermode;
	TIMAx->CR0 = tmpcr;
	TIMAx->ARR = pPwmStruct->period;
	TIMAx->PSC = pPwmStruct->prescaler;
	TIMAx->RCR = pPwmStruct->repetitionCounter;
}


/**
  * @brief  use for pwm output channel0 High Impedance State.
  *
  * @param  TIMAx  TIMA instance.
  *			chHighRes  : channel0 High Impedance State
  *         chnHighRes : channel0n High Impedance State
  *         1: high level
  * 		0: low level
  * @retval None
  */
void DDL_TIMA_Ch0_HighImpedanceState_Config(TIMA_Type *TIMAx, uint8_t chHighRes, uint8_t chnHighRes)
{
	MODIFY_REG(TIMAx->CR1,(3<<TIMA_CR1_HIZ0_Pos) , (chHighRes|(chnHighRes<<1)) << TIMA_CR1_HIZ0_Pos);
}


/**
  * @brief  use for pwm output channel1 High Impedance State.
  *
  * @param  TIMAx  TIMA instance.
  *			chHighRes  : channel1 High Impedance State
  *         chnHighRes : channel1n High Impedance State
  *         1: high level
  * 		0: low level
  * @retval None
  */
void DDL_TIMA_Ch1_HighImpedanceState_Config(TIMA_Type *TIMAx, uint8_t chHighRes, uint8_t chnHighRes)
{
	MODIFY_REG(TIMAx->CR1,(3<<TIMA_CR1_HIZ1_Pos) , (chHighRes|(chnHighRes<<1)) << TIMA_CR1_HIZ1_Pos);
}


/**
  * @brief  use for pwm output channel2 High Impedance State.
  *
  * @param  TIMAx  TIMA instance.
  *			chHighRes  : channel2 High Impedance State
  *         chnHighRes : channel2n High Impedance State
  *         1: high level
  * 		0: low level
  * @retval None
  */
void DDL_TIMA_Ch2_HighImpedanceState_Config(TIMA_Type *TIMAx, uint8_t chHighRes, uint8_t chnHighRes)
{
	MODIFY_REG(TIMAx->CR1,(3<<TIMA_CR1_HIZ2_Pos) , (chHighRes|(chnHighRes<<1)) << TIMA_CR1_HIZ2_Pos);
}


/**
  * @brief  use for pwm output channel3 High Impedance State.
  *
  * @param  TIMAx  TIMA instance.
  *	        chHighRes  : channel3 High Impedance State
  *         1: high level
  * 		0: low level
  * @retval None
  */
void DDL_TIMA_Ch3_HighImpedanceState_Config(TIMA_Type *TIMAx, uint8_t chHighRes)
{
	MODIFY_REG(TIMAx->CR1,(1<<TIMA_CR1_HIZ3_Pos) , chHighRes << TIMA_CR1_HIZ3_Pos);
}


/**
  * @brief  use for pwm output channel0 brake idle state.
  *
  * @param  TIMAx  TIMA instance.
  *			chIdel  : channel0 idle state
  *         chnIdel : channel0n idle state
  *         1: high level
  * 		0: low level
  * @retval None
  */
void DDL_TIMA_Ch0_BrkIdelState_Config(TIMA_Type *TIMAx, uint8_t chIdel, uint8_t chnIdel)
{
	MODIFY_REG(TIMAx->CR1,(3<<TIMA_CR1_OIS0_Pos) , (chIdel|(chnIdel<<1)) << TIMA_CR1_OIS0_Pos);
}


/**
  * @brief  use for pwm output channel1 brake idle state.
  *
  * @param  TIMAx  TIMA instance.
  *			chIdel  : channel1 idle state
  *         chnIdel : channel1n idle state
  *         1: high level
  * 		0: low level
  * @retval None
  */
void DDL_TIMA_Ch1_BrkIdelState_Config(TIMA_Type *TIMAx, uint8_t chIdel, uint8_t chnIdel)
{
	MODIFY_REG(TIMAx->CR1,(3<<TIMA_CR1_OIS1_Pos) , (chIdel|(chnIdel<<1)) << TIMA_CR1_OIS1_Pos);
}


/**
  * @brief  use for pwm output channel2 brake idle state.
  *
  * @param  TIMAx  TIMA instance.
  *			chIdel  : channel2 idle state
  *         chnIdel : channel2n idle state
  *         1: high level
  * 		0: low level
  * @retval None
  */
void DDL_TIMA_Ch2_BrkIdelState_Config(TIMA_Type *TIMAx, uint8_t chIdel, uint8_t chnIdel)
{
	MODIFY_REG(TIMAx->CR1,(3<<TIMA_CR1_OIS2_Pos) , (chIdel|(chnIdel<<1)) << TIMA_CR1_OIS2_Pos);
}


/**
  * @brief  use for pwm output channel3 brake idle state.
  *
  * @param  TIMAx  TIMA instance.
  *			chIdel  : channel3 idle state
  *         1: high level
  * 		0: low level
  *
  * @retval None
  */
void DDL_TIMA_Ch3_BrkIdelState_Config(TIMA_Type *TIMAx, uint8_t chIdel)
{
	MODIFY_REG(TIMAx->CR1,(1<<TIMA_CR1_OIS3_Pos) , chIdel << TIMA_CR1_OIS3_Pos);
}


/**
  * @brief  use for pwm output channel0 duty set.
  *
  * @param  TIMAx  TIMA instance.
			preload  :
					0: effective immediately
					1: effective for the next cycle
  *			hChDuty  : channel0 acr set
  *         lChDuty  : channel0 ccr set
  * @retval None
  */
void DDL_TIMA_Ch0_Duty_Config(TIMA_Type *TIMAx, uint8_t preload, uint16_t hChDuty, uint16_t lChDuty)
{
	MODIFY_REG(TIMAx->CCMR0,TIMA_CCMR0_OC0PE , preload << TIMA_CCMR0_OC0PE_Pos);
	WRITE_REG(TIMAx->CCR0, (hChDuty<<16)|lChDuty );
}


/**
  * @brief  use for pwm output channel1 duty set.
  *
  * @param  TIMAx  TIMA instance.
			preload  :
					0: effective immediately
					1: effective for the next cycle
  *			hChDuty  : channel1 acr set
  *         lChDuty  : channel1 ccr set
  * @retval None
  */
void DDL_TIMA_Ch1_Duty_Config(TIMA_Type *TIMAx, uint8_t preload, uint16_t hChDuty, uint16_t lChDuty)
{
	MODIFY_REG(TIMAx->CCMR0,TIMA_CCMR0_OC1PE , preload << TIMA_CCMR0_OC1PE_Pos);
	WRITE_REG(TIMAx->CCR1, (hChDuty<<16)|lChDuty );
}


/**
  * @brief  use for pwm output channel2 duty set.
  *
  * @param  TIMAx  TIMA instance.
			preload  :
					0: effective immediately
					1: effective for the next cycle
  *			hChDuty  : channel2 acr set
  *         lChDuty  : channel2 ccr set
  * @retval None
  */
void DDL_TIMA_Ch2_Duty_Config(TIMA_Type *TIMAx, uint8_t preload, uint16_t hChDuty, uint16_t lChDuty)
{
	MODIFY_REG(TIMAx->CCMR1,TIMA_CCMR1_OC2PE , preload << TIMA_CCMR1_OC2PE_Pos);
	WRITE_REG(TIMAx->CCR2, (hChDuty<<16)|lChDuty );
}


/**
  * @brief  use for pwm output channel3 duty set.
  *
  * @param  TIMAx  TIMA instance.
			preload  :
					0: effective immediately
					1: effective for the next cycle
  *			hChDuty  : channel3 acr set
  *         lChDuty  : channel3 ccr set
  * @retval None
  */
void DDL_TIMA_Ch3_Duty_Config(TIMA_Type *TIMAx, uint8_t preload, uint16_t hChDuty, uint16_t lChDuty)
{
	MODIFY_REG(TIMAx->CCMR1,TIMA_CCMR1_OC3PE , preload << TIMA_CCMR1_OC3PE_Pos);
	WRITE_REG(TIMAx->CCR3, (hChDuty<<16)|lChDuty );
}


/**
  * @brief  use for pwm output channel4 duty set.
  *
  * @param  TIMAx  TIMA instance.
			preload  :
					0: effective immediately
					1: effective for the next cycle
  *         lChDuty  : channel4 ccr set
  * @retval None
  */
void DDL_TIMA_Ch4_Duty_Config(TIMA_Type *TIMAx, uint8_t preload, uint16_t lChDuty)
{
	MODIFY_REG(TIMAx->CCMR2,TIMA_CCMR2_OC4PE , preload << TIMA_CCMR2_OC4PE_Pos);
	WRITE_REG(TIMAx->CCR4, lChDuty );
}


/**
  * @brief  use for pwm output channel5 duty set.
  *
  * @param  TIMAx  TIMA instance.
			preload  :
					0: effective immediately
					1: effective for the next cycle
  *         lChDuty  : channel5 ccr set
  * @retval None
  */
void DDL_TIMA_Ch5_Duty_Config(TIMA_Type *TIMAx, uint8_t preload, uint16_t lChDuty)
{
	MODIFY_REG(TIMAx->CCMR2,TIMA_CCMR2_OC5PE , preload << TIMA_CCMR2_OC5PE_Pos);
	WRITE_REG(TIMAx->CCR5, lChDuty );
}


/**
  * @brief  use for pwm channel0 output mode config.
  *
  * @param  TIMAx  TIMA instance.
			mode  :
				0: TIMA_OCMODE_FREEZE_CH0
				1: TIMA_OCMODE_COMPARE_FORCED_HIGH_CH0
				2: TIMA_OCMODE_COMPARE_FORCED_LOW_CH0
				3: TIMA_OCMODE_COMPARE_TOGGLE_CH0
				4: TIMA_OCMODE_FORCED_LOW_CH0
				5: TIMA_OCMODE_FORCED_HIGH_CH0
				6: TIMA_OCMODE_PWM0_CH0
				7: TIMA_OCMODE_PWM1_CH0
				C: TIMA_OCMODE_COMBINATION_MODE_PWM0_CH0
				D: TIMA_OCMODE_COMBINATION_MODE_PWM1_CH0
				E: TIMA_OCMODE_ASYM_PWM0_CH0
				F: TIMA_OCMODE_ASYM_PWM1_CH0

  * @retval None
  */
void DDL_TIMA_Ch0_PWM_Mode_Config(TIMA_Type *TIMAx,uint8_t mode)
{
	MODIFY_REG(TIMAx->CCMR0,TIMA_CCMR0_OC0M , mode << TIMA_CCMR0_OC0M_Pos);
}


/**
  * @brief  use for pwm channel1 output mode config.
  *
  * @param  TIMAx  TIMA instance.
			mode  :
				0: TIMA_OCMODE_FREEZE_CH1
				1: TIMA_OCMODE_COMPARE_FORCED_HIGH_CH1
				2: TIMA_OCMODE_COMPARE_FORCED_LOW_CH1
				3: TIMA_OCMODE_COMPARE_TOGGLE_CH1
				4: TIMA_OCMODE_FORCED_LOW_CH1
				5: TIMA_OCMODE_FORCED_HIGH_CH1
				6: TIMA_OCMODE_PWM0_CH1
				7: TIMA_OCMODE_PWM1_CH1
				C: TIMA_OCMODE_COMBINATION_MODE_PWM0_CH1
				D: TIMA_OCMODE_COMBINATION_MODE_PWM1_CH1
				E: TIMA_OCMODE_ASYM_PWM0_CH1
				F: TIMA_OCMODE_ASYM_PWM1_CH1

  * @retval None
  */
void DDL_TIMA_Ch1_PWM_Mode_Config(TIMA_Type *TIMAx,uint8_t mode)
{
	MODIFY_REG(TIMAx->CCMR0,TIMA_CCMR0_OC1M , mode << TIMA_CCMR0_OC1M_Pos);
}


/**
  * @brief  use for pwm channel2 output mode config.
  *
  * @param  TIMAx  TIMA instance.
			mode  :
				0: TIMA_OCMODE_FREEZE_CH2
				1: TIMA_OCMODE_COMPARE_FORCED_HIGH_CH2
				2: TIMA_OCMODE_COMPARE_FORCED_LOW_CH2
				3: TIMA_OCMODE_COMPARE_TOGGLE_CH2
				4: TIMA_OCMODE_FORCED_HIGH_CH2
				5: TIMA_OCMODE_FORCED_LOW_CH2
				6: TIMA_OCMODE_PWM0_CH2
				7: TIMA_OCMODE_PWM1_CH2
				C: TIMA_OCMODE_COMBINATION_MODE_PWM0_CH2
				D: TIMA_OCMODE_COMBINATION_MODE_PWM1_CH2
				E: TIMA_OCMODE_ASYM_PWM0_CH2
				F: TIMA_OCMODE_ASYM_PWM1_CH2

  * @retval None
  */
void DDL_TIMA_Ch2_PWM_Mode_Config(TIMA_Type *TIMAx,uint8_t mode)
{
	MODIFY_REG(TIMAx->CCMR1,TIMA_CCMR1_OC2M , mode << TIMA_CCMR1_OC2M_Pos);
}


/**
  * @brief  use for pwm channel3 output mode config.
  *
  * @param  TIMAx  TIMA instance.
			mode  :
				0: TIMA_OCMODE_FREEZE_CH3
				1: TIMA_OCMODE_COMPARE_FORCED_HIGH_CH3
				2: TIMA_OCMODE_COMPARE_FORCED_LOW_CH3
				3: TIMA_OCMODE_COMPARE_TOGGLE_CH3
				4: TIMA_OCMODE_FORCED_HIGH_CH3
				5: TIMA_OCMODE_FORCED_LOW_CH3
				6: TIMA_OCMODE_PWM0_CH3
				7: TIMA_OCMODE_PWM1_CH3
				C: TIMA_OCMODE_COMBINATION_MODE_PWM0_CH3
				D: TIMA_OCMODE_COMBINATION_MODE_PWM1_CH3
				E: TIMA_OCMODE_ASYM_PWM0_CH3
				F: TIMA_OCMODE_ASYM_PWM1_CH3

  * @retval None
  */
void DDL_TIMA_Ch3_PWM_Mode_Config(TIMA_Type *TIMAx,uint8_t mode)
{
	MODIFY_REG(TIMAx->CCMR1,TIMA_CCMR1_OC3M , mode << TIMA_CCMR1_OC3M_Pos);
}


/**
  * @brief  use for pwm channel4 output mode config.
  *
  * @param  TIMAx  TIMA instance.
			mode  :
				0: TIMA_OCMODE_FREEZE_CH4
				1: TIMA_OCMODE_COMPARE_FORCED_HIGH_CH4
				2: TIMA_OCMODE_COMPARE_FORCED_LOW_CH4
				3: TIMA_OCMODE_COMPARE_TOGGLE_CH4
				4: TIMA_OCMODE_FORCED_HIGH_CH4
				5: TIMA_OCMODE_FORCED_LOW_CH4
				6: TIMA_OCMODE_PWM0_CH4
				7: TIMA_OCMODE_PWM1_CH4

  * @retval None
  */
void DDL_TIMA_Ch4_PWM_Mode_Config(TIMA_Type *TIMAx,uint8_t mode)
{
	MODIFY_REG(TIMAx->CCMR2,TIMA_CCMR2_OC4M , mode << TIMA_CCMR2_OC4M_Pos);
}


/**
  * @brief  use for pwm channel5 output mode config.
  *
  * @param  TIMAx  TIMA instance.
			mode  :
				0: TIMA_OCMODE_FREEZE_CH5
				1: TIMA_OCMODE_COMPARE_FORCED_HIGH_CH5
				2: TIMA_OCMODE_COMPARE_FORCED_LOW_CH5
				3: TIMA_OCMODE_COMPARE_TOGGLE_CH5
				4: TIMA_OCMODE_FORCED_HIGH_CH5
				5: TIMA_OCMODE_FORCED_LOW_CH5
				6: TIMA_OCMODE_PWM0_CH5
				7: TIMA_OCMODE_PWM1_CH5

  * @retval None
  */
void DDL_TIMA_Ch5_PWM_Mode_Config(TIMA_Type *TIMAx,uint8_t mode)
{
	MODIFY_REG(TIMAx->CCMR2,TIMA_CCMR2_OC5M , mode << TIMA_CCMR2_OC5M_Pos);
}


/**
  * @brief  use for pwm channel0 output polar config.
  *
  * @param  TIMAx  TIMA instance.
			chPolar: channel0 polar config
			chnPolar: channel0n polar config
  * @retval None
  */
void DDL_TIMA_Ch0_PWM_Polar_Config(TIMA_Type *TIMAx, uint8_t chPolar, uint8_t chnPolar)
{
	MODIFY_REG(TIMAx->CCER,5 << TIMA_CCER_CC0P_Pos , (chPolar|(chnPolar<<2)) << TIMA_CCER_CC0P_Pos);
}


/**
  * @brief  use for pwm channel1 output polar config.
  *
  * @param  TIMAx  TIMA instance.
			chPolar: channel0 polar config
			chnPolar: channel0n polar config
  * @retval None
  */
void DDL_TIMA_Ch1_PWM_Polar_Config(TIMA_Type *TIMAx, uint8_t chPolar, uint8_t chnPolar)
{
	MODIFY_REG(TIMAx->CCER,5 << TIMA_CCER_CC1P_Pos , (chPolar|(chnPolar<<2)) << TIMA_CCER_CC1P_Pos);
}


/**
  * @brief  use for pwm channel2 output polar config.
  *
  * @param  TIMAx  TIMA instance.
			chPolar: channel0 polar config
			chnPolar: channel0n polar config
  * @retval None
  */
void DDL_TIMA_Ch2_PWM_Polar_Config(TIMA_Type *TIMAx, uint8_t chPolar, uint8_t chnPolar)
{
	MODIFY_REG(TIMAx->CCER,5 << TIMA_CCER_CC2P_Pos , (chPolar|(chnPolar<<2)) << TIMA_CCER_CC2P_Pos);
}


/**
  * @brief  use for pwm channel3 output polar config.
  *
  * @param  TIMAx  TIMA instance.
			chPolar: channel3 polar config
			chnPolar: channel3n polar config
  * @retval None
  */
void DDL_TIMA_Ch3_PWM_Polar_Config(TIMA_Type *TIMAx, uint8_t chPolar, uint8_t chnPolar)
{
	MODIFY_REG(TIMAx->CCER,5 << TIMA_CCER_CC3P_Pos , (chPolar|(chnPolar<<2)) << TIMA_CCER_CC3P_Pos);
}


/**
  * @brief  use for pwm output channel enable.
  *
  * @param  TIMAx  TIMA instance.
			chEnable: channel enable select
  *
  * @retval None
  */
void DDL_TIMA_PWM_Enable(TIMA_Type *TIMAx, uint32_t chEnable)
{
	TIMAx->CCER |= chEnable;
}


/**
  * @brief  use for pwm output start.
  * @param  TIMAx  TIMA instance.
  * @retval None
  */
void DDL_TIMA_PWM_Start(TIMA_Type *TIMAx)
{
	MODIFY_REG(TIMAx->BDTR, TIMA_BDTR_MOE , 1 << TIMA_BDTR_MOE_Pos);
	MODIFY_REG(TIMAx->CR0,  TIMA_CR0_CEN ,  1 << TIMA_CR0_CEN_Pos);
}


/**
  * @brief  use for pwm output stop.
  * @param  TIMAx  TIMA instance.
  * @retval None
  */
void DDL_TIMA_PWM_Stop(TIMA_Type *TIMAx)
{
	MODIFY_REG(TIMAx->BDTR, TIMA_BDTR_MOE , 0 << TIMA_BDTR_MOE_Pos);
	MODIFY_REG(TIMAx->CR0,  TIMA_CR0_CEN ,  0 << TIMA_CR0_CEN_Pos);
}


/**
  * @brief  Configure the elements of structure TIMA_Deadtime_T to default values.
  *			use for timer
  * @param  pDeadtimeStruct Pointer to a TIMA_Deadtime_T structure that contains
  *                  the configuration information for the given TIMA module.
  *
  * @retval None
  */
void DDL_TIMA_DeadTime_StructInit(TIMA_Deadtime_T *pDeadtimeStruct)
{
	pDeadtimeStruct->preload = 1;
	pDeadtimeStruct->modeSelect = 0;
	pDeadtimeStruct->risingDeadtime = 0xD2;
	pDeadtimeStruct->fallingDeadtime = 0xD2;
}


/**
  * @brief  Initializes the TIMA by the specified parameters in the TIMA_Deadtime_T.
  *			use for timer
  * @param  TIMAx  TIMA instance.
  *			pDeadtimeStruct Pointer to a TIMA_Deadtime_T structure that contains
  *                  the configuration information for the given TIMA module.
  *
  * @retval TIMA
  */
void DDL_TIMA_DeadTime_Init(TIMA_Type *TIMAx, TIMA_Deadtime_T *pDeadtimeStruct)
{
	uint32_t tmp = 0;
	tmp = (pDeadtimeStruct->preload<<TIMA_BDTR_DTGPE_Pos) |         \
			(pDeadtimeStruct->modeSelect<<TIMA_BDTR_DTGM_Pos) |      \
			(pDeadtimeStruct->risingDeadtime<<TIMA_BDTR_DTGP_Pos) |  \
			(pDeadtimeStruct->fallingDeadtime);

	MODIFY_REG(TIMAx->BDTR, 0xffff00ff , tmp);
}


/**
  * @brief  use for pwm output brake config.
  * @param  brkSrc :
				 TIMA_BRK_SOURCE_EBUS_CHx
                 TIMA_BRAKE_IO_SRC
                 TIMA_CPU_HALT_SRC
                 TIMA_SRAM_PARITY_ERR_SRC
                 TIMA_PLL_LOSS_PEND_SRC
                 TIMA_MCLK_LOSS_PEND_SRC
			polar :
				0: low  level
				1: high level
  * @retval None
  */
void DDL_TIMA_PWM_Brake_Config(TIMA_Type *TIMAx, uint32_t brkSrc, uint8_t polar)
{
	if (TIMAx == TIMA0)
	{
		MODIFY_REG(TIMAx->BDTR, TIMA_BDTR_BKE , TIMA_BDTR_BKE);
		WRITE_REG(TIM_COMMON->TIMA0_SBCR ,  brkSrc|(polar<<TIMA0_SBCR_BRK_POL_Pos));
	}
	else if (TIMAx == TIMA1)
	{
		MODIFY_REG(TIMAx->BDTR, TIMA_BDTR_BKE , TIMA_BDTR_BKE);
		WRITE_REG(TIM_COMMON->TIMA1_SBCR ,  brkSrc|(polar<<TIMA1_SBCR_BRK_POL_Pos));
	}
}


/**
  * @brief  pwm auto output enable.
  * @param  TIMAx  TIMA instance.
  * @retval None
  */
void DDL_TIMA_Auto_Output_Enable(TIMA_Type *TIMAx)
{
	SET_BIT(TIMAx->BDTR,TIMA_BDTR_AOE);
}


/**
  * @brief  pwm auto output disable.
  * @param  TIMAx  TIMA instance.
  * @retval None
  */
void DDL_TIMA_Auto_Output_Disable(TIMA_Type *TIMAx)
{
	CLEAR_BIT(TIMAx->BDTR,TIMA_BDTR_AOE);
}


/**
  * @brief  tima as master trgo.
  * @param  TIMAx  TIMA instance.
  * @param  trgoSrc : RESET/ENABLE/UPDATE/COMPAREPULSE/COMPAREOCxREF
  *
  * @retval None
  */
void DDL_TIMA_AsMaster_Trgo(TIMA_Type *TIMAx,uint32_t trgoSrc)
{
	MODIFY_REG(TIMAx->CR1, TIMA_CR1_MMS , trgoSrc);
}


/**
  * @brief  tima as master trg1.
  * @param  TIMAx  TIMA instance.
  * @param  trgo1Src : RESET/ENABLE/UPDATE/COMPAREPULSE/COMPAREOCxREF/COMPAREOCxPULSE...
  *
  * @retval None
  */
void DDL_TIMA_AsMaster_Trgo1(TIMA_Type *TIMAx,uint32_t trgo1Src)
{
	MODIFY_REG(TIMAx->CR1, TIMA_CR1_MMS1 , trgo1Src);
}


/**
  * @brief  timA soft brake.
  * @param  TIMAx  TIMA instance.
  * @retval None
  */
void DDL_TIMA_SoftBrake(TIMA_Type *TIMAx)
{
	WRITE_REG(TIMAx->EGR ,  TIMA_EGR_BG);
}


/**
  * @brief  TIMA event trigger config
  * @param  TIMAx  TIMA instance.
  *			event0Psc: event0 psc
				prescaler = event0Psc + 1
  *			event0Edge: event0 config
				0: OC4REF Rising
				1: OC4REF Falling
				2: OC4REF both edge
  * @retval None
  */
void DDL_TIMA_Event0_Trigger_Config(TIMA_Type *TIMAx, uint32_t event0Psc, uint32_t event0Edge)
{
	SET_BITS(TIMAx->ECR, (event0Psc<<4) | (event0Edge));
}


/**
  * @brief  TIMA event trigger config
  * @param  TIMAx  TIMA instance.
  *			event1Psc: event1 psc
				prescaler = event1Psc + 1
  *			event1Edge: event1 config
				0: OC5REF Rising
				1: OC5REF Falling
				2: OC5REF both edge
  * @retval None
  */
void DDL_TIMA_Event1_Trigger_Config(TIMA_Type *TIMAx, uint32_t event1Psc, uint32_t event1Edge)
{
	SET_BITS(TIMAx->ECR, (event1Psc<<12) | (event1Edge<<8));
}


/**
  * @brief  use to clear event product counter

  * @param  TIMAx  TIMA instance.
			clearEvent0Result  clear event product counter(the bit auto-clear)
			0:  no
			1:  yes(clear)
  */
void DDL_TIMA_Event_ClearEvent0_Config(TIMA_Type *TIMAx,uint8_t clearEvent0Result)
{
	SET_BIT(TIMAx->ECR, clearEvent0Result << 2);
}


/**
  * @brief  use to clear event product counter

  * @param  TIMAx  TIMA instance.
			clearEvent1Result  clear event product counter(the bit auto-clear)
			0:  no
			1:  yes(clear)
  */
void DDL_TIMA_Event_ClearEvent1_Config(TIMA_Type *TIMAx, uint8_t clearEvent1Result)
{
	SET_BIT(TIMAx->ECR, clearEvent1Result << 10);
}


/**
  * @brief  Configure the elements of structure TIMA_BaseInit_T to default values.
  *			use for pwm capture
  * @param  pPwmStruct Pointer to a TIMA_BaseInit_T structure that contains
  *                  the configuration information for the given TIMA module.
  *
  * @retval None
  */
void DDL_TIMA_Capture_StructInit(TIMA_BaseInit_T *pPwmStruct)
{
	pPwmStruct->countermode = TIMA_COUNTERMODE_COUNTER_UP;
	pPwmStruct->period = 0xfffe;
	pPwmStruct->prescaler = 99;
	pPwmStruct->repetitionCounter = 0;
}


/**
  * @brief  Initializes the TIMA by the specified parameters in the TIMA_BaseInit_T.
  *			use for pwm capture
  * @param  pTmrStruct Pointer to a TIMA_BaseInit_T structure that contains
  *                  the configuration information for the given TIMA module.
  *			TIMAx  TIMA instance.
  * @retval None
  */
void DDL_TIMA_Capture_Init(TIMA_Type *TIMAx, TIMA_BaseInit_T *pPwmStruct)
{
	uint32_t tmpcr = 0;

	tmpcr = TIMAx->CR0;
	tmpcr &= ~TIMA_COUNTERMODE_COUNTER_UP;
	tmpcr |= pPwmStruct->countermode;
	TIMAx->CR0 = tmpcr;
	TIMAx->ARR = pPwmStruct->period;
	TIMAx->PSC = pPwmStruct->prescaler;
	TIMAx->RCR = pPwmStruct->repetitionCounter;
}


/**
  * @brief  use for capture channel0 output polar config.
  *
  * @param  TIMAx  TIMA instance.
			chPolar: chnPolar
	  rising   0   :   0
	  failing  1   :   0
	  both     1   :   1
  * @retval None
  */
void DDL_TIMA_Ch0_Capture_Polar_Config(TIMA_Type *TIMAx, uint8_t chPolar, uint8_t chnPolar)
{
	MODIFY_REG(TIMAx->CCER,TIMA_CCER_CC0P , chPolar << TIMA_CCER_CC0P_Pos);
	MODIFY_REG(TIMAx->CCER,TIMA_CCER_CC0NP , chnPolar << TIMA_CCER_CC0NP_Pos);
}


/**
  * @brief  use for capture channel1 output polar config.
  *
  * @param  TIMAx  TIMA instance.
			chPolar: chnPolar
	  rising   0   :   0
	  failing  1   :   0
	  both     1   :   1
  * @retval None
  */
void DDL_TIMA_Ch1_Capture_Polar_Config(TIMA_Type *TIMAx, uint8_t chPolar, uint8_t chnPolar)
{
	MODIFY_REG(TIMAx->CCER,TIMA_CCER_CC1P , chPolar << TIMA_CCER_CC1P_Pos);
	MODIFY_REG(TIMAx->CCER,TIMA_CCER_CC1NP , chnPolar << TIMA_CCER_CC1NP_Pos);
}


/**
  * @brief  use for capture channel2 output polar config.
  *
  * @param  TIMAx  TIMA instance.
			chPolar: chnPolar
	  rising   0   :   0
	  failing  1   :   0
	  both     1   :   1
  * @retval None
  */
void DDL_TIMA_Ch2_Capture_Polar_Config(TIMA_Type *TIMAx, uint8_t chPolar, uint8_t chnPolar)
{
	MODIFY_REG(TIMAx->CCER,TIMA_CCER_CC2P , chPolar << TIMA_CCER_CC2P_Pos);
	MODIFY_REG(TIMAx->CCER,TIMA_CCER_CC2NP , chnPolar << TIMA_CCER_CC2NP_Pos);
}


/**
  * @brief  use for capture channel3 output polar config.
  *
  * @param  TIMAx  TIMA instance.
			chPolar: chnPolar
	  rising   0   :   0
	  failing  1   :   0
	  both     1   :   1
  * @retval None
  */
void DDL_TIMA_Ch3_Capture_Polar_Config(TIMA_Type *TIMAx, uint8_t chPolar, uint8_t chnPolar)
{
	MODIFY_REG(TIMAx->CCER,TIMA_CCER_CC3P , chPolar << TIMA_CCER_CC3P_Pos);
	MODIFY_REG(TIMAx->CCER,TIMA_CCER_CC3NP , chnPolar << TIMA_CCER_CC3NP_Pos);
}


/**
  * @brief  Configure the elements of structure TIMA_CapturePara_T to default values.
  *			use for capture
  * @param  pCapStruct Pointer to a TIMA_CapturePara_T structure that contains
  *                  the configuration information for the given tima module.
  *
  * @retval None
  */
void DDL_TIMA_Capture_Channel0_StructInit(TIMA_CapturePara_T *pCapStruct)
{
	pCapStruct->chxDirectSel = TIMA_IC0_DIRECTTI0;
	pCapStruct->chxFilter    = 0;
	pCapStruct->chxPsc       = 0;
}


/**
  * @brief  Initializes the tima by the specified parameters in the TIMA_CapturePara_T.
  *			use for capture
  * @param  TIMAx  TIMA instance.
			pCapStruct Pointer to a TIMA_CapturePara_T structure that contains
  *                  the configuration information for the given tima module.
  *
  * @retval None
  */
void DDL_TIMA_Capture_Channel0_Config(TIMA_Type *TIMAx, TIMA_CapturePara_T *pCapStruct)
{
	uint32_t tmp = TIMAx->CCMR0;
	tmp |= pCapStruct->chxDirectSel   << TIMA_CCMR0_CC0S_Pos;
	tmp |= pCapStruct->chxFilter      << TIMA_CCMR0_IC0F_Pos;
	tmp |= pCapStruct->chxPsc         << TIMA_CCMR0_IC0PSC_Pos;
	WRITE_REG(TIMAx->CCMR0,  tmp);
}


/**
  * @brief  Configure the elements of structure TIMA_CapturePara_T to default values.
  *			use for capture
  * @param  pCapStruct Pointer to a TIMA_CapturePara_T structure that contains
  *                  the configuration information for the given tima module.
  *
  * @retval None
  */
void DDL_TIMA_Capture_Channel1_StructInit(TIMA_CapturePara_T *pCapStruct)
{
	pCapStruct->chxDirectSel = TIMA_IC1_DIRECTTI0;
	pCapStruct->chxFilter    = 0;
	pCapStruct->chxPsc       = 0;
}


/**
  * @brief  Initializes the tima by the specified parameters in the TIMA_CapturePara_T.
  *			use for capture
  * @param  TIMAx  TIMA instance.
			pCapStruct Pointer to a TIMA_CapturePara_T structure that contains
  *                  the configuration information for the given tima module.
  *
  * @retval None
  */
void DDL_TIMA_Capture_Channel1_Config(TIMA_Type *TIMAx, TIMA_CapturePara_T *pCapStruct)
{
	uint32_t tmp = TIMAx->CCMR0;
	tmp |= pCapStruct->chxDirectSel   << TIMA_CCMR0_CC1S_Pos;
	tmp |= pCapStruct->chxFilter      << TIMA_CCMR0_IC1F_Pos;
	tmp |= pCapStruct->chxPsc         << TIMA_CCMR0_IC1PSC_Pos;
	WRITE_REG(TIMAx->CCMR0,  tmp);
}


/**
  * @brief  Configure the elements of structure TIMA_CapturePara_T to default values.
  *			use for capture
  * @param  pCapStruct Pointer to a TIMA_CapturePara_T structure that contains
  *                  the configuration information for the given tima module.
  *
  * @retval None
  */
void DDL_TIMA_Capture_Channel2_StructInit(TIMA_CapturePara_T *pCapStruct)
{
	pCapStruct->chxDirectSel = TIMA_IC2_DIRECTTI2;
	pCapStruct->chxFilter    = 0;
	pCapStruct->chxPsc       = 0;
}


/**
  * @brief  Initializes the tima by the specified parameters in the TIMA_CapturePara_T.
  *			use for capture
  * @param  TIMAx  TIMA instance.
			pCapStruct Pointer to a TIMA_CapturePara_T structure that contains
  *                  the configuration information for the given tima module.
  *
  * @retval None
  */
void DDL_TIMA_Capture_Channel2_Config(TIMA_Type *TIMAx, TIMA_CapturePara_T *pCapStruct)
{
	uint32_t tmp = TIMAx->CCMR1;
	tmp |= pCapStruct->chxDirectSel   << TIMA_CCMR1_CC2S_Pos;
	tmp |= pCapStruct->chxFilter      << TIMA_CCMR1_IC2F_Pos;
	tmp |= pCapStruct->chxPsc         << TIMA_CCMR1_IC2PSC_Pos;
	WRITE_REG(TIMAx->CCMR1,  tmp);
}


/**
  * @brief  Configure the elements of structure TIMA_CapturePara_T to default values.
  *			use for capture
  * @param  pCapStruct Pointer to a TIMA_CapturePara_T structure that contains
  *                  the configuration information for the given tima module.
  *
  * @retval None
  */
void DDL_TIMA_Capture_Channel3_StructInit(TIMA_CapturePara_T *pCapStruct)
{
	pCapStruct->chxDirectSel = TIMA_IC3_DIRECTTI3;
	pCapStruct->chxFilter    = 0;
	pCapStruct->chxPsc       = 0;
}


/**
  * @brief  Initializes the tima by the specified parameters in the TIMA_CapturePara_T.
  *			use for capture
  * @param  TIMAx  TIMA instance.
			pCapStruct Pointer to a TIMA_CapturePara_T structure that contains
  *                  the configuration information for the given tima module.
  *
  * @retval None
  */
void DDL_TIMA_Capture_Channel3_Config(TIMA_Type *TIMAx, TIMA_CapturePara_T *pCapStruct)
{
	uint32_t tmp = TIMAx->CCMR1;
	tmp |= pCapStruct->chxDirectSel   << TIMA_CCMR1_CC3S_Pos;
	tmp |= pCapStruct->chxFilter      << TIMA_CCMR1_IC3F_Pos;
	tmp |= pCapStruct->chxPsc         << TIMA_CCMR1_IC3PSC_Pos;
	WRITE_REG(TIMAx->CCMR1,  tmp);
}


/**
  * @brief  use for tima slave mode config.
  *
  * @param  TIMAx  TIMA instance.
			triggerSignal:  ITRx/TI0FED/TI0FPT/TI1FPT
			slaveMode:		reset/gate/trigger
  * @retval None
  */
void DDL_TIMA_SlaveMode_Config(TIMA_Type *TIMAx, uint8_t triggerSignal, uint8_t slaveMode)
{
	uint32_t tmp = 0;
	tmp = (triggerSignal<<4)|slaveMode;
	MODIFY_REG(TIMAx->SMCR,0x3f,tmp );
}


/**
  * @brief  use for timer start.
  * @param  TIMAx  TIMA instance.
  * @retval None
  */
void DDL_TIMA_Capture_Start(TIMA_Type *TIMAx)
{
	SET_BIT(TIMAx->BDTR, TIMA_BDTR_MOE);
	SET_BIT(TIMAx->CR0,   TIMA_CR0_CEN);
}


/**
  * @brief  use for timer stop.
  * @param  TIMAx  TIMA instance.
  * @retval None
  */
void DDL_TIMA_Capture_Stop(TIMA_Type *TIMAx)
{
	CLEAR_BIT(TIMAx->BDTR, TIMA_BDTR_MOE);
	CLEAR_BIT(TIMAx->CR0,   TIMA_CR0_CEN);
}

#endif /* DDL_TIMA_MODULE_ENABLED */















