/**
  ******************************************************************************
  * @file    kpm32xx_ddl_pwr.h
  * @author  Kiwi Software Team
  * @brief   Header file of  Power Controller DDL module.
  * @note
  *    	     V1.0.0, 2021/12/13.
  *
  * Copyright (c) 2024, Kiwi Instruments Co,. Ltd.
  *
  * Redistribution and use in source and binary forms, with or without modification,
  * are permitted provided that the following conditions are met:
  *
  *   1. Redistributions of source code must retain the above copyright notice,
  *      this list of conditions and the following disclaimer.
  *
  *   2. Redistributions in binary form must reproduce the above copyright notice,
  *      this list of conditions and the following disclaimer in the documentation
  *      and/or other materials provided with the distribution.
  *
  *   3. Neither the name of the copyright holder nor the names of its contributors
  *      may be used to endorse or promote products derived from this software without
  *      specific prior written permission.
  *
  * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
  * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
  * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
  * THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
  * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE
  * GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
  * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
  * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
  * THE POSSIBILITY OF SUCH DAMAGE.
  ******************************************************************************
  */



#ifndef __KPM32xx_DDL_PWR_H
#define __KPM32xx_DDL_PWR_H

#ifdef __cplusplus
extern "C" {
#endif


#include "kpm32xx_ddl_def.h"


/* Define the instructions for entering sleep mode */
#define PWR_ENTER_SLEEP_WFI           0UL    /* wake-up by interrupt request, debug request or reset. */
#define PWR_ENTER_SLEEP_WFE           1UL    /* wake-up by interrupt request, event input, debug request or reset. */


/* Configure PWR init parameters */
typedef struct
{
	uint32_t            sleepOnExitEnable;         /* When this bit is set, it will enter sleep mode when exit from interrupt to thread. */
	uint32_t            sevOnPendEnable;           /* When this bit is set, enabled events and all interrupts, including disabled ones can wakeup processor from WFE. */
	uint32_t            enterSleepInstruction;     /* Use WFI or WFE to enter sleep mode. */
	uint32_t            flashStandbyEnable;        /* Enable Flash low power mode, only for STOP sleep mode. */
} PWR_Init_T;


/* Enable EFLASH standby mode, only use for STOP mode */
#define __DDL_PWR_EFLASH_STANDBY_Enable()      SET_BIT(SYSCFG->PMC, SYSCONF_PMC_FLSTBY)
/* Disable EFLASH standby mode */
#define __DDL_PWR_EFLASH_STANDBY_Disable()     CLEAR_BIT(SYSCFG->PMC, SYSCONF_PMC_FLSTBY)


/**
  * @brief  Configure the elements of structure PWR_Init_T to default values.
  * @param  pPwrStruct  pointer to PWR_Init_T structure that contains
  *         the configuration information for Power Controller.
  * @retval None
  */
void DDL_PWR_StructInit(PWR_Init_T *pPwrStruct);

/**
  * @brief  Enter Sleep mode.
  * @param  pPwrStruct  pointer to PWR_Init_T structure that contains
  *         the configuration information for Power Controller.
  * @note   When WFI entry is used, systick interrupt have to be disabled if not
  *         desired as the interrupt wake up source.
  * @retval None
  */
void DDL_PWR_EnterSleepMode(PWR_Init_T *pPwrStruct);

/**
  * @brief  Enter Deep Sleep mode.
  * @param  pPwrStruct  pointer to PWR_Init_T structure that contains
  *         the configuration information for Power Controller.
  * @note   When WFI entry is used, systick interrupt have to be disabled if not
  *         desired as the interrupt wake up source.
  * @retval None
  */
void DDL_PWR_EnterDeepSleepMode(PWR_Init_T *pPwrStruct);
/**
  * @brief  Enter Stop mode.
  * @param  pPwrStruct  pointer to PWR_Init_T structure that contains
  *         the configuration information for Power Controller.
  * @note   When WFI entry is used, systick interrupt have to be disabled if not
  *         desired as the interrupt wake up source.
  *         Before entering stop mode, it will swich FCLK to HIRC, close pll and HOSC,
  *         so after calling this API, remember to reconfigure the clock.
  * @ret
  */
void DDL_PWR_EnterStopMode(PWR_Init_T *pPwrStruct);



#ifdef __cplusplus
}
#endif


#endif /* __KPM32xx_DDL_PWR_H */

