/**
  *****************************************************************************************************
  * @file    main.c
  * @author  Kiwi Software Team
  * @brief   This example code demonstrates how tims works.
  *
  * Copyright (c) 2024, Kiwi Instruments Co,. Ltd.
  *
  * Redistribution and use in source and binary forms, with or without modification,
  * are permitted provided that the following conditions are met:
  *
  *   1. Redistributions of source code must retain the above copyright notice,
  *      this list of conditions and the following disclaimer.
  *
  *   2. Redistributions in binary form must reproduce the above copyright notice,
  *      this list of conditions and the following disclaimer in the documentation
  *      and/or other materials provided with the distribution.
  *
  *   3. Neither the name of the copyright holder nor the names of its contributors
  *      may be used to endorse or promote products derived from this software without
  *      specific prior written permission.
  *
  * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
  * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
  * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
  * THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
  * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE
  * GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
  * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
  * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
  * THE POSSIBILITY OF SUCH DAMAGE.
  *****************************************************************************************************
  */

#include "main.h"
#include "kpm32xx_demoboard_ddl.h"


#if defined (__DEBUG_CONSOLE_PRINT__)
/**
  * @brief  This function is implemented for UART print PinMux configuration.
  * @param  None
  * @retval None
  */
void DebugConsole_PinMux_Config(void)
{
	/* SCI1 UART Tx */
	DDL_GPIO_Config2AltFunc(GPIOA, GPIO_PIN_7, GPIOA7_AF1_SCI1_TX_SDA_MOSI);
}
#endif


/**
  * @brief  This function is implemented for system clock configuration.
  *         where:
  *         System Clock source            = PLL (HIRC)
  *         System Clock frequency         = 100MHz
  * @param  None
  * @retval None
  */
void SystemClock_Config(void)
{
	RCC_PLLInit_T pllInitStruct    = {0};
	RCC_ClkInit_T rccClkInitStruct = {0};

	DDL_RCC_PllStructInit(&pllInitStruct);
	/* FCLK : 100MHz */
	pllInitStruct.pllPrescaler  = 4;
	if (DDL_RCC_PllConfig(&pllInitStruct) != DDL_OK)
	{
		while(1) ;
	}

	/* Select PLL as system clock source */
	rccClkInitStruct.sysClkSource  = RCC_SYSCLKSOURCE_PLLCLK;
	rccClkInitStruct.sysClkDivider = 0;
	if (DDL_RCC_ClkSrcConfig(&rccClkInitStruct) != DDL_OK)
	{
		while(1) ;
	}
}


/**
  * @brief  This function is implemented for led toggle.
  * @retval None
  */
void led_toggle(void)
{
	BSP_LEDx_Twinkle(LED_S1);
	BSP_LEDx_Twinkle(LED_S2);
	BSP_LEDx_Twinkle(LED_S3);
}


/**
  * @brief  This function is implemented for TIMS2 PWM PinMux configuration.
  * @param  None
  * @retval None
  */
void TIMS2_PinMux_Config(void)
{
	DDL_GPIO_Config2AltFunc(GPIOC, GPIO_PIN_8, GPIOC8_AF5_TIMS2_CH1);
	DDL_GPIO_Config2AltFunc(GPIOC, GPIO_PIN_10, GPIOC10_AF6_TIMS2_CH1N);
	DDL_GPIO_Config2AltFunc(GPIOC, GPIO_PIN_9, GPIOC9_AF5_TIMS2_CH2);
}


/**
  * @brief  This function is implemented for tims pwm output.
  * @retval None
  */
void tims_pwm_init(void)
{
	TIMS_BaseInit_T tmrStruct;

	DDL_TIMS_PWM_StructInit(&tmrStruct);
	tmrStruct.period = 3124;
	DDL_TIMS_PWM_Init(TIMS2,&tmrStruct);

	/* ch1 init config */
	DDL_TIMS_Ch1_HighResistsnceState_Config(TIMS2, 0, 0);
	DDL_TIMS_Ch1_BrkIdelState_Config(TIMS2, 0, 0);
	DDL_TIMS_Ch1_Duty_Config(TIMS2, 1, 3000);
	DDL_TIMS_Ch1_PWM_Mode_Config(TIMS2,TIMS_OCMODE_PWM0_CH1);

	/* ch2 init config */
	DDL_TIMS_Ch2_HighResistsnceState_Config(TIMS2, 0);
	DDL_TIMS_Ch2_BrkIdelState_Config(TIMS2, 0);
	DDL_TIMS_Ch2_Duty_Config(TIMS2, 1, 2000);
	DDL_TIMS_Ch2_PWM_Mode_Config(TIMS2,TIMS_OCMODE_PWM0_CH2);

	/* deadtime config */
	DDL_TIMS_DeadTime_Config(TIMS2,0xD2);

	/* channel enable */
	DDL_TIMS_PWM_Enable(TIMS2,TIMS_CHANNEL1|TIMS_CHANNEL1N|TIMS_CHANNEL2);

	NVIC_SetPriority(TIMS2_IRQn, 0);
	NVIC_EnableIRQ(TIMS2_IRQn);
	DDL_TIMS_IntEnable(TIMS2,TIMS_IER_UIE);
}


/**
  * @brief  The application entry point.
  * @retval int
  */
int main(void)
{
	/* Reset of all peripherals, Initializes the Flash interface and the Systick. */
	DDL_Init();
	SystemClock_Config();

	/* To make GPIO active */
	DDL_GPIO_Instance_Active();

	BSP_LEDx_Init(LED_S1);
	BSP_LEDx_Init(LED_S2);
	BSP_LEDx_Init(LED_S3);

#if defined (__DEBUG_CONSOLE_PRINT__)
	DebugConsole_PinMux_Config();
	DDL_SCI_Instance_Active(SCI1);
	DDL_SCIUart_DebugConsole_Init(SCI1);
#endif

    printf("It's a TIMS pwm output demo(100M 16K) ... \n");
	TIMS2_PinMux_Config();
	DDL_TIMS_Instance_Active(TIMS2);
	tims_pwm_init();
	DDL_TIMS_PWM_Start(TIMS2);

    while (1)
    {
		;
    }
}

