/**
  *****************************************************************************************************
  * @file    main.c
  * @author  Kiwi Software Team
  * @brief   This example code demonstrates how ADC works.
  *
  * Copyright (c) 2024, Kiwi Instruments Co,. Ltd.
  *
  * Redistribution and use in source and binary forms, with or without modification,
  * are permitted provided that the following conditions are met:
  *
  *   1. Redistributions of source code must retain the above copyright notice,
  *      this list of conditions and the following disclaimer.
  *
  *   2. Redistributions in binary form must reproduce the above copyright notice,
  *      this list of conditions and the following disclaimer in the documentation
  *      and/or other materials provided with the distribution.
  *
  *   3. Neither the name of the copyright holder nor the names of its contributors
  *      may be used to endorse or promote products derived from this software without
  *      specific prior written permission.
  *
  * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
  * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
  * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
  * THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
  * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE
  * GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
  * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
  * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
  * THE POSSIBILITY OF SUCH DAMAGE.
  *****************************************************************************************************
  */

#include "main.h"
#include "kpm32xx_demoboard_ddl.h"



#if defined (__DEBUG_CONSOLE_PRINT__)
/**
  * @brief  This function is implemented for UART print PinMux configuration.
  * @param  None
  * @retval None
  */
void DebugConsole_PinMux_Config(void)
{
	/* SCI1 UART Tx */
	DDL_GPIO_Config2AltFunc(GPIOA, GPIO_PIN_7, GPIOA7_AF1_SCI1_TX_SDA_MOSI);
}
#endif


/**
  * @brief  This function is implemented for system clock configuration.
  *         where:
  *         System Clock source            = PLL (HIRC)
  *         System Clock frequency         = 100MHz
  * @param  None
  * @retval None
  */
void SystemClock_Config(void)
{
	RCC_PLLInit_T pllInitStruct    = {0};
	RCC_ClkInit_T rccClkInitStruct = {0};

	DDL_RCC_PllStructInit(&pllInitStruct);
	/* FCLK : 100MHz */
	pllInitStruct.pllPrescaler  = 4;
	if (DDL_RCC_PllConfig(&pllInitStruct) != DDL_OK)
	{
		while(1) ;
	}

	/* Select PLL as system clock source */
	rccClkInitStruct.sysClkSource  = RCC_SYSCLKSOURCE_PLLCLK;
	rccClkInitStruct.sysClkDivider = 0;
	if (DDL_RCC_ClkSrcConfig(&rccClkInitStruct) != DDL_OK)
	{
		while(1) ;
	}
}


/**
  * @brief  This function is implemented for ADC0 PinMux configuration.
  * @param  None
  * @retval None
  */
void ADC_PinMux_Config(void)
{
	/* ADC0 channel 5 : GPIOD6 */
	DDL_GPIO_Config2Analog(GPIOD, GPIO_PIN_6);
}


/**
  * @brief  The application entry point.
  * @retval int
  */
int main(void)
{
	uint8_t cnt = 0;
	uint16_t res = 0;
	ADC_SOCConfig_T socConfig;
	ADC_Init_T adcStruct;

	/* Reset of all peripherals, Initializes the Flash interface and the Systick. */
	DDL_Init();
	SystemClock_Config();

	/* To make GPIO active */
	DDL_GPIO_Instance_Active();

#if defined (__DEBUG_CONSOLE_PRINT__)
	DebugConsole_PinMux_Config();
	DDL_SCI_Instance_Active(SCI1);
	DDL_SCIUart_DebugConsole_Init(SCI1);
#endif

	printf("ADC single channel polling demo start:\n");

	/* init ADC */
	DDL_ADC_Instance_Active(ADC0);
	ADC_PinMux_Config();

    DDL_ADC_StructInit(&adcStruct);
	DDL_ADC_Init(ADC0, &adcStruct);

	/* Use ADC_CHANNEL_6 which pinmux is PA6. */
	memset(&socConfig, 0, sizeof(ADC_SOCConfig_T));
	socConfig.channel    = ADC_CHANNEL_5;
	socConfig.sampleTime = SAMPLE_12_ADCCLK;
	socConfig.hwTrigger = 0;
	DDL_ADC_ConfigSOC(ADC0, ADC_SOC0, &socConfig);

	while (1)
	{
		DDL_ADC_StartSOC_BySwTrig(ADC0, ADC_SOC0);
		if (DDL_ADC_PollForEvent(ADC0, ADC_EVENT_SOC0, 100) != DDL_OK)
		{
			printf("	ADC poll for event fail.\n");
			return -1;
		}

		res = DDL_ADC_ReadSOCResult(ADC0, ADC_SOC0);
		printf("	SOC (%d) with channel (%d) convert value : 0x%x.\n", ADC_SOC0, ADC_CHANNEL_5, res);

		cnt++;
		if (cnt >= 100)
		{
			break;
		}
	}

	printf("ADC single channel polling demo end.\n");

	return 0;
}

