/**
  ******************************************************************************
  * @file    kpm32xx_ddl_wwdg.c
  * @author  Kiwi Software Team
  * @brief   WWDG DDL module driver.
  *          This file provides firmware functions to manage the following
  *          functionalities of WWDG peripheral:
  *          + Initialization and Configuration functions
  *          + WWDG feed function
  *          + WWDG early warning interrupt
  * @note
  *          V1.0.0, 2024/12/20.
  *
  * Copyright (c) 2024, Kiwi Instruments Co,. Ltd.
  *
  * Redistribution and use in source and binary forms, with or without modification,
  * are permitted provided that the following conditions are met:
  *
  *   1. Redistributions of source code must retain the above copyright notice,
  *      this list of conditions and the following disclaimer.
  *
  *   2. Redistributions in binary form must reproduce the above copyright notice,
  *      this list of conditions and the following disclaimer in the documentation
  *      and/or other materials provided with the distribution.
  *
  *   3. Neither the name of the copyright holder nor the names of its contributors
  *      may be used to endorse or promote products derived from this software without
  *      specific prior written permission.
  *
  * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
  * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
  * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
  * THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
  * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE
  * GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
  * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
  * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
  * THE POSSIBILITY OF SUCH DAMAGE.
  ******************************************************************************
  */
#include "kpm32xx_ddl.h"



#ifdef DDL_WWDG_MODULE_ENABLED

/**
  * @brief  Makes WWDG active.
  * @param  none
  * 
  * @retval none
  */
void DDL_WWDG_Instance_Active(void)
{
	/* Release WWDG clock */
	__DDL_RCC_WDT_CLK_RELEASE();
}


/**
  * @brief  Configure the elements of structure WWDG_Init_T to default values.
  *
  * @param  pWWDGStructInit Pointer to a WWDG_Init_T structure that contains
  *                         the configuration information for the given WWDG module.
  *
  * @retval None
  */
void DDL_WWDG_StructInit(WWDG_Init_T *pWWDGStructInit)
{
	pWWDGStructInit->mode      = WWDG_MODE_NORMAL;
	pWWDGStructInit->clkSrc    = WWDG_CLKSRC_MCLK;
	pWWDGStructInit->counter   = 0xFFFFFF; /* 0xFFFFFF 24bit */
	pWWDGStructInit->windowVal = 60000000; /* 0xFFFFFF 24bit */
	pWWDGStructInit->ewiEnable = 1;
	pWWDGStructInit->ewiThreshold = WWDG_WINLEN_1000H;
}


/**
  * @brief  Initialize the WWDG according to the specified.
  *         parameters in the WWDG_Init_T of  associated handle.
  * @param  WWDGx WWDG instance.
  * @param  pWWDGStructInit Pointer to a WWDG_Init_T structure that contains
  *                         the configuration information for the given WWDG module.
  * @retval None
  */
void DDL_WWDG_Init(WWDG_Type *WWDGx, WWDG_Init_T *pWWDGStructInit)
{
	RCC_WDTInit_T wwdgClkConfig;

	/* Set the source clock of WWDG */
	wwdgClkConfig.periphClkSrcSelection = RCC_PERIPHCLK_WDT;
	wwdgClkConfig.wdtClkSrc = pWWDGStructInit->clkSrc;

	DDL_RCC_WDT_PeriphCLKConfig(&wwdgClkConfig);

	__DDL_WWDG_REGS_UNLOCK__(WWDGx);

	/* Disable WWDG, need to wait real disable */
	__DDL_WWDG_DISABLE(WWDGx);
	while (__DDL_WWDG_IS_REAL_ENABLE(WWDGx))
	{
	}

	/* Set WWDG counter and window val */
	WRITE_REG(WWDGx->LD, pWWDGStructInit->counter);
	WRITE_REG(WWDGx->WLD, pWWDGStructInit->windowVal);

	/* Set WWDG work mode, EWI */
	WRITE_REG(WWDGx->CTRL, (pWWDGStructInit->ewiThreshold | pWWDGStructInit->mode | pWWDGStructInit->ewiEnable));

	__DDL_WWDG_ENABLE(WWDGx);
	__DDL_WWDG_REGS_LOCK__(WWDGx);
}


/**
  * @brief  Refresh the WWDG.
  * @param  WWDGx  WWDG instance.
  * @retval None
  */
void DDL_WWDG_Refresh(WWDG_Type *WWDGx)
{
	if (__DDL_WWDG_IS_REFRESH_BUSY(WWDGx) == 0)
	{
		WRITE_REG(WWDGx->KEY, WWDG_FEED_KEY);
	}
}


#endif /* DDL_WWDG_MODULE_ENABLED */

