/**
  ******************************************************************************
  * @file    kpm32xx_ddl_lvd.c
  * @author  Kiwi Software Team
  * @brief   Low voltage dector DDL module driver.
  *          This file provides firmware functions to manage the following
  *          functionalities of the LVD peripheral:
  *          + Configure LVD1 and LVD2
  * @note
  *          V1.0.0, 2024/12/20.
  *
  * Copyright (c) 2024, Kiwi Instruments Co,. Ltd.
  *
  * Redistribution and use in source and binary forms, with or without modification,
  * are permitted provided that the following conditions are met:
  *
  *   1. Redistributions of source code must retain the above copyright notice,
  *      this list of conditions and the following disclaimer.
  *
  *   2. Redistributions in binary form must reproduce the above copyright notice,
  *      this list of conditions and the following disclaimer in the documentation
  *      and/or other materials provided with the distribution.
  *
  *   3. Neither the name of the copyright holder nor the names of its contributors
  *      may be used to endorse or promote products derived from this software without
  *      specific prior written permission.
  *
  * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
  * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
  * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
  * THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
  * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE
  * GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
  * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
  * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
  * THE POSSIBILITY OF SUCH DAMAGE.
  ******************************************************************************
  */


#include "kpm32xx_ddl.h"

#ifdef DDL_LVD_MODULE_ENABLED

/**
  * @brief  Initialize the Low Voltage dector accoding to the specify parameters.
  * @param  index LVD index.
  * @param  init  Pointer to a LVD_Init_T structure that contains
  *               the configuration information for the specified LVD.
  * @retval DDL status
  */
DDL_Status_T DDL_LVD_Init(LVD_Index_T index, LVD_Init_T *init)
{
	if (index == LVD_INDEX_1)
	{
		__DDL_SYSREG_UNLOCK();
		CLEAR_BIT(SYSCFG->LVD1CTRL, SYSCONF_LVD1CTRL_EN);
		SET_BITMASK(SYSCFG->LVD1CTRL, SYSCONF_LVD1CTRL_SEL_Msk, init->detectLevel << SYSCONF_LVD1CTRL_SEL_Pos);
		SET_BIT(SYSCFG->LVD1CTRL, SYSCONF_LVD1CTRL_EN);
		__DDL_SYSREG_LOCK();
	}
	else if (index == LVD_INDEX_2)
	{
		__DDL_SYSREG_UNLOCK();
		CLEAR_BIT(SYSCFG->LVD2CTRL, SYSCONF_LVD2CTRL_EN);
		SET_BITMASK(SYSCFG->LVD2CTRL, SYSCONF_LVD2CTRL_SEL_Msk, init->detectLevel << SYSCONF_LVD2CTRL_SEL_Pos);
		SET_BITMASK(SYSCFG->LVD2CTRL, SYSCONF_LVD2CTRL_CLKPSC_Msk, init->filterClkPsc << SYSCONF_LVD2CTRL_CLKPSC_Pos);
		SET_BITMASK(SYSCFG->LVD2CTRL, SYSCONF_LVD2CTRL_THR_Msk, init->filterThreshold << SYSCONF_LVD2CTRL_THR_Pos);

		NVIC_EnableIRQ(LVD_IRQn);
		SET_BIT(SYSCFG->LVD2CTRL, SYSCONF_LVD2CTRL_IE);
		SET_BIT(SYSCFG->LVD2CTRL, SYSCONF_LVD2CTRL_EN);
		__DDL_SYSREG_LOCK();
	}
	else
	{
		return DDL_INVPARAM;
	}

	return DDL_OK;
}

#endif /* DDL_LVD_MODULE_ENABLED */

