/**
  ******************************************************************************
  * @file    kpm32xx_ddl_ioport.c
  * @author  Kiwi Software Team
  * @brief   IOPORT DDL module driver.
  *          This file provides firmware functions to manage the following
  *          functionalities of the IOPORT peripheral:
  *           + peripheral initializes and deInitializes
  *           + different mode of IOPORT operation
  *           + interrupt, IOPORT
  * @note
  *          V1.0.0, 2024/12/20.
  *
  * Copyright (c) 2024, Kiwi Instruments Co., Ltd.
  *
  * Redistribution and use in source and binary forms, with or without modification,
  * are permitted provided that the following conditions are met:
  *
  *   1. Redistributions of source code must retain the above copyright notice,
  *      this list of conditions and the following disclaimer.
  *
  *   2. Redistributions in binary form must reproduce the above copyright notice,
  *      this list of conditions and the following disclaimer in the documentation
  *      and/or other materials provided with the distribution.
  *
  *   3. Neither the name of the copyright holder nor the names of its contributors
  *      may be used to endorse or promote products derived from this software without
  *      specific prior written permission.
  *
  * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
  * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
  * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
  * THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
  * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE
  * GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
  * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
  * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
  * THE POSSIBILITY OF SUCH DAMAGE.
  ******************************************************************************
  */
#include "kpm32xx_ddl.h"



#ifdef DDL_IOPORT_MODULE_ENABLED


/**
  * @brief  configure IO Port parameters.
  * @param  ioPortx     IO PORT instance.
  * @param  ioPortInit  pointer to a IOPORT_Init_T structure.
  *
  * @retval none
  */
void DDL_IOPort_Config(IOPORT_Type *ioPortx, IOPORT_Init_T *ioPortInit)
{	
	if (ioPortInit->ioPortPin <= 7)
	{
		SET_BITS(ioPortx->SRCSEL1, ioPortInit->ioSelect << (ioPortInit->ioPortPin*4));
	}
	else
	{
		SET_BITS(ioPortx->SRCSEL2, ioPortInit->ioSelect << ((ioPortInit->ioPortPin-8)*4));
	}
	MODIFY_REG(ioPortx->IOD, (0x03UL << (2 * (ioPortInit->ioPortPin))), ((ioPortInit->edgeDect) << (2 * (ioPortInit->ioPortPin))));
}


/**
  * @brief  Change IO edge detection.
  * @param  ioPortx   IO PORT instance.
  * @param  ioPortPin IO Port Pin.
  * @param  edgeDect  Edge Detection Sources.
  *
  * @retval none
  */
void DDL_IOPort_ChangeEdgeDetection(IOPORT_Type *ioPortx, uint32_t ioPortPin, uint32_t edgeDect)
{
	MODIFY_REG(ioPortx->IOD, (0x03UL << (2 * ioPortPin)), (edgeDect << (2 * ioPortPin)));
}


/**
  * @brief  clear IO Port interrupt status.
  * @param  ioPortx    IO PORT instance.
  * @param  ioPortPin  the given IO Port Pin (IOPORT_PIN_0 ~ IOPORT_PIN_11).
  *
  * @retval none
  */
void DDL_IOPort_PinInt_Clear(IOPORT_Type *ioPortx, uint32_t ioPortPin)
{
	WRITE_REG(ioPortx->CLR, (1 << ioPortPin));
}



#endif /* DDL_IOPORT_MODULE_ENABLED */

