/**
  *****************************************************************************************************
  * @file    kpm32xx_ddl_tims.h
  * @author  Kiwi Software Team
  * @brief   Header file of TIMS DDL module..
  * @note
  *          V1.0.0, 2025/1/6.
  *
  * Copyright (c) 2024, Kiwi Instruments Co,. Ltd.
  *
  * Redistribution and use in source and binary forms, with or without modification,
  * are permitted provided that the following conditions are met:
  *
  *   1. Redistributions of source code must retain the above copyright notice,
  *      this list of conditions and the following disclaimer.
  *
  *   2. Redistributions in binary form must reproduce the above copyright notice,
  *      this list of conditions and the following disclaimer in the documentation
  *      and/or other materials provided with the distribution.
  *
  *   3. Neither the name of the copyright holder nor the names of its contributors
  *      may be used to endorse or promote products derived from this software without
  *      specific prior written permission.
  *
  * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
  * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
  * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
  * THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
  * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE
  * GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
  * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
  * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
  * THE POSSIBILITY OF SUCH DAMAGE.
  *****************************************************************************************************
  */


#ifndef __KPM32xx_DDL_TIMS_H
#define __KPM32xx_DDL_TIMS_H

#ifdef __cplusplus
extern "C" {
#endif

#include "kpm32xx_ddl_def.h"

typedef enum
{
	TIMS_CHANNEL1         = 0x00000001,
	TIMS_CHANNEL1N        = 0x00000004,
	TIMS_CHANNEL2         = 0x00000010,
}DDL_TIMS_Channel_Enable_T;

typedef struct
{
	uint32_t	prescaler;         						/* the prescaler value used to divide the FCLK */
	uint32_t	period;            						/* the period value to be loaded into the ARR Register   */
	uint32_t    countermode;        					/* Specifies the counter mode. */
	uint32_t	repetitionCounter;  					/* the repetition counter value */
} TIMS_BaseInit_T;

typedef struct
{
	uint32_t	chxDirectSel;
	uint32_t	chxPsc;
	uint32_t	chxFilterEventNum;
	uint32_t	chxFilter;
} TIMS_CapturePara_T;

/* Counter mode  */
#define TIMS_COUNTERMODE_COUNTER_UP          				0x00
#define TIMS_COUNTERMODE_COUNTER_DOWN						TIMS_CR_DIR
#define TIMS_COUNTERMODE_COUNTER_CENTER_1    				1UL << TIMS_CR_CMS_Pos
#define TIMS_COUNTERMODE_COUNTER_CENTER_2    				2UL << TIMS_CR_CMS_Pos
#define TIMS_COUNTERMODE_COUNTER_CENTER_3    				3UL << TIMS_CR_CMS_Pos

/* TIMS channelx output/compare mode select */
#define TIMS_OCMODE_FREEZE_CH1								(0UL)
#define TIMS_OCMODE_COMPARE_FORCED_HIGH_CH1      			(1UL)
#define TIMS_OCMODE_COMPARE_FORCED_LOW_CH1       			(2UL)
#define TIMS_OCMODE_COMPARE_TOGGLE_CH1						(3UL)
#define TIMS_OCMODE_FORCED_HIGH_CH1      					(5UL)
#define TIMS_OCMODE_FORCED_LOW_CH1       					(4UL)
#define TIMS_OCMODE_PWM0_CH1                     			(6UL)
#define TIMS_OCMODE_PWM1_CH1                     			(7UL)

#define TIMS_OCMODE_FREEZE_CH2								(0UL)
#define TIMS_OCMODE_COMPARE_FORCED_HIGH_CH2      			(1UL)
#define TIMS_OCMODE_COMPARE_FORCED_LOW_CH2       			(2UL)
#define TIMS_OCMODE_COMPARE_TOGGLE_CH2						(3UL)
#define TIMS_OCMODE_FORCED_HIGH_CH2      					(5UL)
#define TIMS_OCMODE_FORCED_LOW_CH2       					(4UL)
#define TIMS_OCMODE_PWM0_CH2                     			(6UL)
#define TIMS_OCMODE_PWM1_CH2                     			(7UL)

#define TIMS_IC1_DIRECTTI1           					(1)   /* TIMS Input 1 OR 0 is selected to be connected to IC1 OR IC0, respectively */
#define TIMS_IC1_DIRECTTI2         						(2)   /* TIMS Input 1 OR 0 is selected to be connected to IC1 OR IC0, respectively */
#define TIMS_IC1_SELECT_TRC                				(3)   /* TIMS Input 1, 0 is selected to be connected to TRC */
#define TIMS_IC2_DIRECTTI2           					(1)   /* TIMS Input 2 OR 3 is selected to be connected to IC2 OR IC3, respectively */
#define TIMS_IC2_DIRECTTI1         						(2)   /* TIMS Input 2 OR 3 is selected to be connected to IC2 OR IC3, respectively */
#define TIMS_IC2_SELECT_TRC                				(3)   /* TIMS Input 2, 3 is selected to be connected to TRC */

/* TIMS trigger selection */
#define TIMS_CLOCKSOURCE_ITR0								(0x0UL)
#define TIMS_CLOCKSOURCE_ITR1								(0x1UL)
#define TIMS_CLOCKSOURCE_ITR2								(0x2UL)
#define TIMS_CLOCKSOURCE_ITR3								(0x3UL)
#define	TIMS_CLOCKSOURCE_TI1FED      						(0x4UL)
#define	TIMS_CLOCKSOURCE_TI1FPT      						(0x5UL)
#define	TIMS_CLOCKSOURCE_TI2FPT      						(0x6UL)
#define	TIMS_CLOCKSOURCE_ETRF        						(0x7UL)

/*slave mode config*/
#define TIMS_SLAVEMODE_DISABLE								(0x0UL)
#define TIMS_SLAVEMODE_RESET								(0x4UL)
#define TIMS_SLAVEMODE_GATE									(0x5UL)
#define TIMS_SLAVEMODE_TRIGGER								(0x6UL)
#define TIMS_SLAVEMODE_EXTERNCLOCK							(0x7UL)

/* brake source */
#define TIMS_BRK_SOURCE_EBUS_CH0	     					((1 << 0)  << 16)
#define TIMS_BRK_SOURCE_EBUS_CH1							((1 << 1)  << 16)
#define TIMS_BRK_SOURCE_EBUS_CH2							((1 << 2)  << 16)
#define TIMS_BRK_SOURCE_EBUS_CH3							((1 << 3)  << 16)
#define TIMS_BRK_SOURCE_EBUS_CH4							((1 << 4)  << 16)
#define TIMS_BRK_SOURCE_EBUS_CH5							((1 << 5)  << 16)
#define TIMS_BRK_SOURCE_EBUS_CH6							((1 << 6)  << 16)
#define TIMS_BRK_SOURCE_EBUS_CH7							((1 << 7)  << 16)
#define TIMS_BRK_SOURCE_EBUS_CH8							((1 << 8)  << 16)
#define TIMS_BRK_SOURCE_EBUS_CH9							((1 << 9)  << 16)
#define TIMS_BRK_SOURCE_EBUS_CH10							((1 << 10)  << 16)
#define TIMS_BRK_SOURCE_EBUS_CH11							((1 << 11)  << 16)
#define TIMS_BRK_SOURCE_EBUS_CH12							((1 << 12)  << 16)
#define TIMS_BRK_SOURCE_EBUS_CH13							((1 << 13)  << 16)
#define TIMS_BRK_SOURCE_EBUS_CH14							((1 << 14)  << 16)
#define TIMS_BRK_SOURCE_EBUS_CH15							((1 << 15)  << 16)
#define TIMS_BRAKE_IO_SRC                             		(1UL  << 4)
#define TIMS_CPU_HALT_SRC                             		(1UL  << 3)
#define TIMS_SRAM_PARITY_ERR_SRC                            (1UL  << 2)
#define TIMS_PLL_LOSS_PEND_SRC                              (1UL  << 1)
#define TIMS_MCLK_LOSS_PEND_SRC                             (1UL  << 0)


/**
  * @brief  make TIMS active.
  * @param  SCIx  TIMS instance.
  *
  */
void DDL_TIMS_Instance_Active(TIMS_Type *TIMSx);


/**
  * @brief  Configure the elements of structure TIMS_BaseInit_T to default values.
  *			use for timer
  * @param  TIMS_BaseInit_T Pointer to a TIMS_BaseInit_T structure that contains
  *                  the configuration information for the given TIMS module.
  *
  * @retval None
  */
void DDL_TIMS_TMR_StructInit(TIMS_BaseInit_T *pTmrStruct);


/**
  * @brief  Initializes the TIMS by the specified parameters in the TIMS_BaseInit_T.
  *			use for timer
  * @param  TIMSx  TIMS instance.
  *			pTmrStruct Pointer to a TIMS_BaseInit_T structure that contains
  *                  the configuration information for the given TIMS module.
  *
  * @retval TIMS
  */
void DDL_TIMS_TMR_Init(TIMS_Type *TIMSx, TIMS_BaseInit_T *pTmrStruct);


/**
  * @brief  use for timer start.
  * @param  TIMSx  TIMS instance.
  * @retval None
  */
void DDL_TIMS_TMR_Start(TIMS_Type *TIMSx);


/**
  * @brief  use for timer stop.
  * @param  TIMSx  TIMS instance.
  * @retval None
  */
void DDL_TIMS_TMR_Stop(TIMS_Type *TIMSx);


/**
  * @brief  use for TIMS interrupt enable.
  * @param  intrMask : COMIE/UIE/C0IE/C1IE/C2IE/TIE/BIE
  *
  * @retval None
  */
void DDL_TIMS_IntEnable(TIMS_Type *TIMSx, uint32_t intrMask);


/**
  * @brief  use for TIMS interrupt disable.
  * @param  intrMask : COMIE/UIE/C0IE/C1IE/C2IE/TIE/BIE
  *
  * @retval None
  */
void DDL_TIMS_IntDisable(TIMS_Type *TIMSx, uint32_t intrMask);


/**
  * @brief  Configure the elements of structure TIMS_BaseInit_T to default values.
  *			use for pwm output
  * @param  pPwmStruct Pointer to a TIMS_BaseInit_T structure that contains
  *                  the configuration information for the given TIMS module.
  *
  * @retval None
  */
void DDL_TIMS_PWM_StructInit(TIMS_BaseInit_T *pPwmStruct);


/**
  * @brief  Initializes the TIMS by the specified parameters in the TIMS_BaseInit_T.
  *			use for pwm output
  * @param  pTmrStruct Pointer to a TIMS_BaseInit_T structure that contains
  *                  the configuration information for the given TIMS module.
  *			TIMSx  TIMS instance.
  * @retval None
  */
void DDL_TIMS_PWM_Init(TIMS_Type *TIMSx, TIMS_BaseInit_T *pPwmStruct);


/**
  * @brief  use for pwm output channel1 High Resistsnce State.
  *
  * @param  TIMSx  TIMS instance.
  *			chHighRes  : channel0 High Resistsnce State
  *         chnHighRes : channel0n High Resistsnce State
  *         1: high level
  * 		0: low level
  * @retval None
  */
void DDL_TIMS_Ch1_HighResistsnceState_Config(TIMS_Type *TIMSx, uint8_t chHighRes, uint8_t chnHighRes);


/**
  * @brief  use for pwm output channel2 High Resistsnce State.
  *
  * @param  TIMSx  TIMS instance.
  *			chHighRes  : channel1 High Resistsnce State
  *         1: high level
  * 		0: low level
  * @retval None
  */
void DDL_TIMS_Ch2_HighResistsnceState_Config(TIMS_Type *TIMSx, uint8_t chHighRes);


/**
  * @brief  use for pwm output channel1 brake idle state.
  *
  * @param  TIMSx  TIMS instance.
  *			chIdel  : channel1 idle state
  *         chnIdel : channel1n idle state
  *         1: high level
  * 		0: low level
  * @retval None
  */
void DDL_TIMS_Ch1_BrkIdelState_Config(TIMS_Type *TIMSx, uint8_t chIdel, uint8_t chnIdel);


/**
  * @brief  use for pwm output channel2 brake idle state.
  *
  * @param  TIMSx  TIMS instance.
  *			chIdel  : channel2 idle state
  *         1: high level
  * 		0: low level
  * @retval None
  */
void DDL_TIMS_Ch2_BrkIdelState_Config(TIMS_Type *TIMSx, uint8_t chIdel);


/**
  * @brief  use for pwm output channel1 duty set.
  *
  * @param  TIMSx  TIMS instance.
			preload  :
					0: effective immediately
					1: effective for the next cycle
*			chDuty: channel1 ccr set
  * @retval None
  */
void DDL_TIMS_Ch1_Duty_Config(TIMS_Type *TIMSx, uint8_t preload, uint16_t chDuty);


/**
  * @brief  use for pwm output channel2 duty set.
  *
  * @param  TIMSx  TIMS instance.
			preload  :
					0: effective immediately
					1: effective for the next cycle
  *			chDuty: channel2 ccr set
  * @retval None
  */
void DDL_TIMS_Ch2_Duty_Config(TIMS_Type *TIMSx, uint8_t preload, uint16_t chDuty);


/**
  * @brief  use for pwm channel1 output mode config.
  *
  * @param  TIMSx  TIMS instance.
			mode  :
				0: TIMS_OCMODE_FREEZE_CH1
				1: TIMS_OCMODE_COMPARE_FORCED_HIGH_CH1
				2: TIMS_OCMODE_COMPARE_FORCED_LOW_CH1
				3: TIMS_OCMODE_COMPARE_TOGGLE_CH1
				4: TIMS_OCMODE_FORCED_LOW_CH1
				5: TIMS_OCMODE_FORCED_HIGH_CH1
				6: TIMS_OCMODE_PWM0_CH1
				7: TIMS_OCMODE_PWM1_CH1

  * @retval None
  */
void DDL_TIMS_Ch1_PWM_Mode_Config(TIMS_Type *TIMSx,uint8_t mode);


/**
  * @brief  use for pwm channel2 output mode config.
  *
  * @param  TIMSx  TIMS instance.
			mode  :
				0: TIMS_OCMODE_FREEZE_CH2
				1: TIMS_OCMODE_COMPARE_FORCED_HIGH_CH2
				2: TIMS_OCMODE_COMPARE_FORCED_LOW_CH2
				3: TIMS_OCMODE_COMPARE_TOGGLE_CH2
				4: TIMS_OCMODE_FORCED_HIGH_CH2
				5: TIMS_OCMODE_FORCED_LOW_CH2
				6: TIMS_OCMODE_PWM0_CH2
				7: TIMS_OCMODE_PWM1_CH2

  * @retval None
  */
void DDL_TIMS_Ch2_PWM_Mode_Config(TIMS_Type *TIMSx,uint8_t mode);


/**
  * @brief  use for pwm channel1 output polar config.
  *
  * @param  TIMSx  TIMS instance.
			chPolar: channel0 polar config
			chnPolar: channel0n polar config
  * @retval None
  */
void DDL_TIMS_Ch1_PWM_Polar_Config(TIMS_Type *TIMSx, uint8_t chPolar, uint8_t chnPolar);


/**
  * @brief  use for pwm channel2 output polar config.
  *
  * @param  TIMSx  TIMS instance.
			chPolar: channel0 polar config
			chnPolar: channel0n polar config
  * @retval None
  */
void DDL_TIMS_Ch2_PWM_Polar_Config(TIMS_Type *TIMSx, uint8_t chPolar, uint8_t chnPolar);


/**
  * @brief  use for pwm output channel enable.
  *
  * @param  TIMSx  TIMS instance.
			chEnable: channel enable select
  *
  * @retval None
  */
void DDL_TIMS_PWM_Enable(TIMS_Type *TIMSx, uint32_t chEnable);


/**
  * @brief  use for pwm output start.
  * @param  TIMSx  TIMS instance.
  * @retval None
  */
void DDL_TIMS_PWM_Start(TIMS_Type *TIMSx);


/**
  * @brief  use for pwm output stop.
  * @param  TIMSx  TIMS instance.
  * @retval None
  */
void DDL_TIMS_PWM_Stop(TIMS_Type *TIMSx);


/**
  * @brief  Configure the deadtime value
  * @param  TIMSx  TIMS instance.
  *			deadTime:  config deadtime value
  *
  * @retval None
  */
void DDL_TIMS_DeadTime_Config(TIMS_Type *TIMSx, uint8_t deadTime);


/**
  * @brief  use for pwm output brake config.
  * @param  brkSrc :
				 TIMS_BRK_SOURCE_EBUS_CHx
                 TIMS_BRAKE_IO_SRC
                 TIMS_CPU_HALT_SRC
                 TIMS_SRAM_PARITY_ERR_SRC
                 TIMS_PLL_LOSS_PEND_SRC
                 TIMS_MCLK_LOSS_PEND_SRC
			polar :
				0: low  level
				1: high level
  * @retval None
  */
void DDL_TIMS_PWM_Brake_Config(TIMS_Type *TIMSx, uint32_t brkSrc, uint8_t polar);


/**
  * @brief  pwm auto output enable.
  * @param  TIMSx  TIMS instance.
  * @retval None
  */
void DDL_TIMS_Auto_Output_Enable(TIMS_Type *TIMSx);


/**
  * @brief  pwm auto output disable.
  * @param  TIMSx  TIMS instance.
  * @retval None
  */
void DDL_TIMS_Auto_Output_Disable(TIMS_Type *TIMSx);


/**
  * @brief  TIMS as master trgo.
  * @param  TIMSx  TIMS instance.
  * @param  trgoSrc : RESET/ENABLE/UPDATE/COMPAREPULSE/COMPAREOCxREF
  *
  * @retval None
  */
void DDL_TIMS_AsMaster_Trgo(TIMS_Type *TIMSx,uint32_t trgoSrc);


/**
  * @brief  TIMS soft brake.
  * @param  TIMSx  TIMS instance.
  * @retval None
  */
void DDL_TIMS_SoftBrake(TIMS_Type *TIMSx);


/**
  * @brief  Configure the elements of structure TIMS_BaseInit_T to default values.
  *			use for pwm capture
  * @param  pPwmStruct Pointer to a TIMS_BaseInit_T structure that contains
  *                  the configuration information for the given TIMS module.
  *
  * @retval None
  */
void DDL_TIMS_Capture_StructInit(TIMS_BaseInit_T *pPwmStruct);


/**
  * @brief  Initializes the TIMS by the specified parameters in the TIMS_BaseInit_T.
  *			use for pwm capture
  * @param  pTmrStruct Pointer to a TIMS_BaseInit_T structure that contains
  *                  the configuration information for the given TIMS module.
  *			TIMSx  TIMS instance.
  * @retval None
  */
void DDL_TIMS_Capture_Init(TIMS_Type *TIMSx, TIMS_BaseInit_T *pPwmStruct);


/**
  * @brief  use for capture channel1 output polar config.
  *
  * @param  TIMSx  TIMS instance.
			chPolar: chnPolar
	  rising   0   :   0
	  failing  1   :   0
	  both     1   :   1
  * @retval None
  */
void DDL_TIMS_Ch1_Capture_Polar_Config(TIMS_Type *TIMSx, uint8_t chPolar, uint8_t chnPolar);


/**
  * @brief  use for capture channel2 output polar config.
  *
  * @param  TIMSx  TIMS instance.
			chPolar: chnPolar
	  rising   0   :   0
	  failing  1   :   0
	  both     1   :   1
  * @retval None
  */
void DDL_TIMS_Ch2_Capture_Polar_Config(TIMS_Type *TIMSx, uint8_t chPolar, uint8_t chnPolar);


/**
  * @brief  Configure the elements of structure TIMS_CapturePara_T to default values.
  *			use for capture
  * @param  pCapStruct Pointer to a TIMS_CapturePara_T structure that contains
  *                  the configuration information for the given TIMS module.
  *
  * @retval None
  */
void DDL_TIMS_Capture_Channel1_StructInit(TIMS_CapturePara_T *pCapStruct);


/**
  * @brief  Initializes the TIMS by the specified parameters in the TIMS_CapturePara_T.
  *			use for capture
  * @param  TIMSx  TIMS instance.
			pCapStruct Pointer to a TIMS_CapturePara_T structure that contains
  *                  the configuration information for the given TIMS module.
  *
  * @retval None
  */
void DDL_TIMS_Capture_Channel1_Config(TIMS_Type *TIMSx, TIMS_CapturePara_T *pCapStruct);


/**
  * @brief  Configure the elements of structure TIMS_CapturePara_T to default values.
  *			use for capture
  * @param  pCapStruct Pointer to a TIMS_CapturePara_T structure that contains
  *                  the configuration information for the given TIMS module.
  *
  * @retval None
  */
void DDL_TIMS_Capture_Channel2_StructInit(TIMS_CapturePara_T *pCapStruct);


/**
  * @brief  Initializes the TIMS by the specified parameters in the TIMS_CapturePara_T.
  *			use for capture
  * @param  TIMSx  TIMS instance.
			pCapStruct Pointer to a TIMS_CapturePara_T structure that contains
  *                  the configuration information for the given TIMS module.
  *
  * @retval None
  */
void DDL_TIMS_Capture_Channel2_Config(TIMS_Type *TIMSx, TIMS_CapturePara_T *pCapStruct);


/**
  * @brief  use for TIMS slave mode config.
  *
  * @param  TIMSx  TIMS instance.
			triggerSignal:  ITRx/TI0FED/TI0FPT/TI1FPT
			slaveMode:		reset/gate/trigger
  * @retval None
  */
void DDL_TIMS_SlaveMode_Config(TIMS_Type *TIMSx, uint8_t triggerSignal, uint8_t slaveMode);


/**
  * @brief  use for timer start.
  * @param  TIMSx  TIMS instance.
  * @retval None
  */
void DDL_TIMS_Capture_Start(TIMS_Type *TIMSx);


/**
  * @brief  use for timer stop.
  * @param  TIMSx  TIMS instance.
  * @retval None
  */
void DDL_TIMS_Capture_Stop(TIMS_Type *TIMSx);



#ifdef __cplusplus
}
#endif

#endif /* __KPM32xx_DDL_TIMS_H */










