/**
  *****************************************************************************************************
  * @file    kpm32xx_ddl_rcc.h
  * @author  Kiwi Software Team
  * @brief   Header file of Reset and Clock Control module.
  * @note
  *          V1.0.0, 2024/12/20.
  *
  * Copyright (c) 2024, Kiwi Instruments Co,. Ltd.
  *
  * Redistribution and use in source and binary forms, with or without modification,
  * are permitted provided that the following conditions are met:
  *
  *   1. Redistributions of source code must retain the above copyright notice,
  *      this list of conditions and the following disclaimer.
  *
  *   2. Redistributions in binary form must reproduce the above copyright notice,
  *      this list of conditions and the following disclaimer in the documentation
  *      and/or other materials provided with the distribution.
  *
  *   3. Neither the name of the copyright holder nor the names of its contributors
  *      may be used to endorse or promote products derived from this software without
  *      specific prior written permission.
  *
  * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
  * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
  * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
  * THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
  * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE
  * GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
  * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
  * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
  * THE POSSIBILITY OF SUCH DAMAGE.
  *****************************************************************************************************
  */


#ifndef __KPM32xx_DDL_RCC_H
#define __KPM32xx_DDL_RCC_H

#ifdef __cplusplus
extern "C" {
#endif

#include "kpm32xx_ddl_def.h"

#if !defined  (HOSC_VALUE)
    #define HOSC_VALUE             (8000000UL)
#endif

#if !defined  (HIRC_VALUE)
    #define HIRC_VALUE             (8000000UL)
#endif

#if !defined  (MCLK_VALUE)
    #define MCLK_VALUE             (8000000UL)
#endif

#if !defined  (LIRC_VALUE)
    #define LIRC_VALUE             (32000UL)
#endif

#if !defined  (LOSC_VALUE)
    #define LOSC_VALUE             (32768UL)
#endif


/* First, unlock before write some special system registers */
#define __DDL_SYSREG_UNLOCK()               WRITE_REG(SYSCFG->KEY, 0x2EE1D879)
#define __DDL_SYSREG_LOCK()                 WRITE_REG(SYSCFG->KEY, 0x5A5AA5A5)

/* PLL sources */
#define IS_RCC_PLLSOURCE(__SOURCE__)     (((__SOURCE__) == RCC_PLLSOURCE_HIRC)  || ((__SOURCE__) == RCC_PLLSOURCE_HOSC))
/* System clock sources */
#define IS_RCC_SYSCLKSOURCE(__SOURCE__)  (((__SOURCE__) == RCC_SYSCLKSOURCE_MCLK) || ((__SOURCE__) == RCC_SYSCLKSOURCE_PLLCLK))

typedef struct
{
	uint32_t pllSource;
	uint32_t pllDIVM;        /* Divider(M) of PLL Clock Source (HOSC or HIRC), As Fin   */
	uint32_t pllMULN;        /* Multiplicator (N) of Fin (Fin: the source clock of PLL) */
	uint32_t pllPrescaler;   /* Division of PLL to FCLK (System Clock)                  */
} RCC_PLLInit_T;

typedef struct
{
	uint32_t sysClkSource;
	uint32_t sysClkDivider;
} RCC_ClkInit_T;


/* For M Clock and PLL Clock Source Setting */
#define RCC_PLLSOURCE_HIRC                      RCC_PLL_SOURCE_HIRC  /*HIRC clock selected as pll entry clock source */
#define RCC_PLLSOURCE_HOSC                      RCC_PLL_SOURCE_HOSC  /*HOSC clock selected as pll entry clock source */

#define RCC_MCLKSOURCE_HOSC                     (0x01UL << SYSCONF_CSEL0_MCLK_Pos)
#define RCC_MCLKSOURCE_HIRC                     (0x00UL << SYSCONF_CSEL0_MCLK_Pos)

/* For System Clock Setting */
#define RCC_SYSCLKSOURCE_PLLCLK                 (0x01UL << SYSCONF_CSEL0_FCLK_Pos)
#define RCC_SYSCLKSOURCE_MCLK                   (0x00UL << SYSCONF_CSEL0_FCLK_Pos)

/* System Clock Stable Status */
#define RCC_STR_PLLSTABLE                       SYSCONF_RSTS_PLLRDY
#define RCC_STR_LOSCSTABLE                      SYSCONF_RSTS_LOSCRDY
#define RCC_STR_HOSCSTABLE                      SYSCONF_RSTS_HOSCRDY
#define RCC_STR_LIRCSTABLE                      SYSCONF_RSTS_LIRCRDY
#define RCC_STR_HIRCSTABLE                      SYSCONF_RSTS_HIRCRDY

/* For M Clock and PLL Clock Status Checking */
#define RCC_PLL_SOURCE_HIRC                     (0x0UL << SYSCONF_PLL_CLKSRC_Pos)
#define RCC_PLL_SOURCE_HOSC                     (0x1UL << SYSCONF_PLL_CLKSRC_Pos)
#define RCC_MCLCK_SOURCE_HIRC                   (0x0UL << SYSCONF_CSEL0_MCLKST_Pos)
#define RCC_MCLCK_SOURCE_HOSC                   (0x1UL << SYSCONF_CSEL0_MCLKST_Pos)

/* Configure System Clock Divider */
#define __DDL_RCC_SYSCLKDIV_CONFIG(__CLKDIV__)  do { \
													__DDL_SYSREG_UNLOCK(); \
													MODIFY_REG(SYSCFG->CSEL0, SYSCONF_CSEL0_FDIV_Msk, (uint32_t)(__CLKDIV__)); \
													__DDL_SYSREG_LOCK();\
												}while(0)
/* Configure System Clock (FCLK) */
#define __DDL_RCC_SYSCLKSRC_CONFIG(__CLKSRC__)  do { \
													__DDL_SYSREG_UNLOCK(); \
													MODIFY_REG(SYSCFG->CSEL0, SYSCONF_CSEL0_FCLK_Msk, (uint32_t)(__CLKSRC__)); \
													__DDL_SYSREG_LOCK();\
												}while(0)

/* Configure M Clock (MCLK) */
#define __DDL_RCC_MCLKSRC_CONFIG(__CLKSRC__)  do { \
													__DDL_SYSREG_UNLOCK(); \
													MODIFY_REG(SYSCFG->CSEL0, SYSCONF_CSEL0_MCLK_Msk, (uint32_t)(__CLKSRC__)); \
													__DDL_SYSREG_LOCK();\
												}while(0)


/* For System Clock (FCLK) Status Checking */
#define RCC_SYSCLKSOURCE_STATUS_PLLCLK          (0x01UL << SYSCONF_CSEL0_FCLKST_Pos)
#define RCC_SYSCLKSOURCE_STATUS_MCLK            (0x00UL << SYSCONF_CSEL0_FCLKST_Pos)

/* Check whether system clock source is stable */
#define IS_RCC_LIRC_STABLE()                    ((SYSCFG->RSTS & RCC_STR_LIRCSTABLE) == RCC_STR_LIRCSTABLE)
#define IS_RCC_HIRC_STABLE()                    ((SYSCFG->RSTS & RCC_STR_HIRCSTABLE) == RCC_STR_HIRCSTABLE)
#define IS_RCC_HOSC_STABLE()                    ((SYSCFG->RSTS & RCC_STR_HOSCSTABLE) == RCC_STR_HOSCSTABLE)
#define IS_RCC_PLL_STABLE()                     ((SYSCFG->RSTS & RCC_STR_PLLSTABLE)  == RCC_STR_PLLSTABLE)

/* Get System Clock Source */
#define __DDL_RCC_GET_SYSCLK_SOURCE()           (SYSCFG->CSEL0 & SYSCONF_CSEL0_FCLKST_Msk)

/* Get M Clock Source */
#define __DDL_RCC_GET_MCLK_SOURCE()             (SYSCFG->CSEL0 & SYSCONF_CSEL0_MCLKST_Msk)

/* Macro to get the clock source divider, ranging from 1 to 256. */
#define __DDL_RCC_GET_SYSCLK_DIVIDER()          ((SYSCFG->CSEL0 & SYSCONF_CSEL0_FDIV_Msk) >> SYSCONF_CSEL0_FDIV_Pos)

/* Get PLL clock sources. !!! Caution: pll may only have HIRC and HOSC as clock sources !!! */
#define __DDL_RCC_GET_PLLCLK_SOURCE()           (SYSCFG->PLL & SYSCONF_PLL_CLKSRC_Msk)
#define __DDL_RCC_GET_PLL_M()                   ((SYSCFG->PLL & SYSCONF_PLL_PLLM_Msk) >> SYSCONF_PLL_PLLM_Pos)
#define __DDL_RCC_GET_PLL_N()                   ((SYSCFG->PLL & SYSCONF_PLL_PLLN_Msk) >> SYSCONF_PLL_PLLN_Pos)
#define __DDL_RCC_GET_PLL_P()                   ((SYSCFG->PLL & SYSCONF_PLL_DIV_Msk) >> SYSCONF_PLL_DIV_Pos)

/*********************** HOSC, LIRC, LOSC, HIRC Control ******************************/
#define __DDL_RCC_HOSC_ENABLE()                 do { \
													__DDL_SYSREG_UNLOCK(); \
													SET_BIT(SYSCFG->HOSC, SYSCONF_HOSC_EN); \
													__DDL_SYSREG_LOCK();\
												}while(0)

#define __DDL_RCC_HOSC_DISABLE()                do { \
													__DDL_SYSREG_UNLOCK(); \
													CLEAR_BIT(SYSCFG->HOSC, SYSCONF_HOSC_EN); \
													__DDL_SYSREG_LOCK();\
												}while(0)
/* pin as HOSC in HIN and HOUT */
#define __DDL_RCC_HOSC_HINOUT()                 do { \
													__DDL_SYSREG_UNLOCK(); \
													MODIFY_REG(SYSCFG->HOSC, SYSCONF_HOSC_SEL_Msk, SYSCONF_HOSC_SEL_HINOUT) ;\
													__DDL_SYSREG_LOCK();\
												}while(0)

#define __DDL_RCC_HOSC_NORM()                   do { \
													__DDL_SYSREG_UNLOCK(); \
													MODIFY_REG(SYSCFG->HOSC, SYSCONF_HOSC_SEL_Msk, SYSCONF_HOSC_SEL_NORM); \
													__DDL_SYSREG_LOCK();\
												}while(0)
/* PLL Enable and Disable */
#define __DDL_RCC_PLL_ENABLE()                  do { \
													__DDL_SYSREG_UNLOCK(); \
													SET_BIT(SYSCFG->PLL, SYSCONF_PLL_EN);\
													__DDL_SYSREG_LOCK();\
												}while(0)

#define __DDL_RCC_PLL_DISABLE()                 do { \
													__DDL_SYSREG_UNLOCK(); \
													CLEAR_BIT(SYSCFG->PLL, SYSCONF_PLL_EN); \
													__DDL_SYSREG_LOCK();\
												}while(0)

#define __DDL_RCC_PLL_SRCCLK_CONFIG(__PLLSRC__) do { \
													__DDL_SYSREG_UNLOCK(); \
													MODIFY_REG(SYSCFG->PLL, SYSCONF_PLL_CLKSRC_Msk, __PLLSRC__); \
													__DDL_SYSREG_LOCK();\
												}while(0)


#define __DDL_RCC_PLL_M_N_P_CONFIG(__PLLM__, __PLLN__, __PLLP__)   do{\
																		__DDL_SYSREG_UNLOCK(); \
																		MODIFY_REG(SYSCFG->PLL, (SYSCONF_PLL_DIV_Msk | SYSCONF_PLL_PLLN_Msk | SYSCONF_PLL_PLLM),\
																		(__PLLM__ << SYSCONF_PLL_PLLM_Pos | \
																	     __PLLN__ << SYSCONF_PLL_PLLN_Pos | \
																		 __PLLP__ << SYSCONF_PLL_DIV_Pos)); \
																		__DDL_SYSREG_LOCK();\
																   }while(0);

#define __DDL_RCC_HIRC_ENABLE()                 do{\
													__DDL_SYSREG_UNLOCK();\
													CLEAR_BIT(SYSCFG->IRC[0], SYSCONF_HIRC_PWRDN);\
													__DDL_SYSREG_LOCK();\
											    }while(0);

#define __DDL_RCC_HIRC_DISABLE()                do{\
													__DDL_SYSREG_UNLOCK();\
													SET_BIT(SYSCFG->IRC[0], SYSCONF_HIRC_PWRDN);\
													__DDL_SYSREG_LOCK();\
											    }while(0);

#define __DDL_RCC_LIRC_ENABLE()                 do{\
													__DDL_SYSREG_UNLOCK();\
													CLEAR_BIT(SYSCFG->IRC[0], SYSCONF_LIRC_PWRDN);\
													__DDL_SYSREG_LOCK();\
											    }while(0);

#define __DDL_RCC_LIRC_DISABLE() 				do{\
													__DDL_SYSREG_UNLOCK();\
													SET_BIT(SYSCFG->IRC[0], SYSCONF_LIRC_PWRDN);\
													__DDL_SYSREG_LOCK();\
											    }while(0);

#define __DDL_RCC_MCLK_MONINTOR()               do { \
												     SET_BIT(SYSCFG->CMNT, (SYSCONF_CMNT_MCLKEN | SYSCONF_CMNT_MCLKLPDEN)); \
   												   }while(0);

#define __DDL_RCC_PLL_MONINTOR()               do { \
												    SET_BIT(SYSCFG->CMNT, (SYSCONF_CMNT_PLLEN | SYSCONF_CMNT_PLLLPDEN)); \
   												  }while(0);

#define FREQMNT_OBJCLK_FCLK                    (0x00UL << SYSCONF_FMNT0_CLKOBJ_Pos)
#define FREQMNT_OBJCLK_HIRC                    (0x01UL << SYSCONF_FMNT0_CLKOBJ_Pos)
#define FREQMNT_OBJCLK_HIRC2                   (0x02UL << SYSCONF_FMNT0_CLKOBJ_Pos)
#define FREQMNT_OBJCLK_HOSC                    (0x03UL << SYSCONF_FMNT0_CLKOBJ_Pos)

#define FREQMNT_REFCLK_HIRC2                   (0x00UL << SYSCONF_FMNT0_CLKREF_Pos)
#define FREQMNT_REFCLK_HOSC                    (0x01UL << SYSCONF_FMNT0_CLKREF_Pos)
#define FREQMNT_REFCLK_LIRC                    (0x02UL << SYSCONF_FMNT0_CLKREF_Pos)
#define FREQMNT_REFCLK_EXTCLK                  (0x03UL << SYSCONF_FMNT0_CLKREF_Pos)

#define FREQMNT_REF_DIV_1                      (0x00UL << SYSCONF_FMNT0_CLKREFDIV_Pos)
#define FREQMNT_REF_DIV_32                     (0x01UL << SYSCONF_FMNT0_CLKREFDIV_Pos)
#define FREQMNT_REF_DIV_64                     (0x02UL << SYSCONF_FMNT0_CLKREFDIV_Pos)
#define FREQMNT_REF_DIV_128                    (0x03UL << SYSCONF_FMNT0_CLKREFDIV_Pos)
#define FREQMNT_REF_DIV_256                    (0x04UL << SYSCONF_FMNT0_CLKREFDIV_Pos)
#define FREQMNT_REF_DIV_512                    (0x05UL << SYSCONF_FMNT0_CLKREFDIV_Pos)
#define FREQMNT_REF_DIV_1024                   (0x06UL << SYSCONF_FMNT0_CLKREFDIV_Pos)
#define FREQMNT_REF_DIV_2048                   (0x07UL << SYSCONF_FMNT0_CLKREFDIV_Pos)

/* Configure Frequency Threshold */
#define __DDL_RCC_FREQTHRD_CONFIG(__UPPER__, __LOWER__)  MODIFY_REG(SYSCFG->FMNT1, 0xFFFFFFFF, (__UPPER__ << SYSCONF_FMNT1_UPP_Pos | __LOWER__ << SYSCONF_FMNT1_LOW_Pos))

#define __DDL_RCC_FREQMNT_CLKOBJREF_CONFIG(__OBJCLK__, __REFCLK__, __DIV__)  MODIFY_REG(SYSCFG->FMNT0, (SYSCONF_FMNT0_CLKOBJ | SYSCONF_FMNT0_CLKREFDIV | SYSCONF_FMNT0_CLKREF), \
                                                                                                       (__OBJCLK__ | \
																									    __REFCLK__  | \
																										__DIV__))

/* To Trigger NMI interrupt */
#define __DDL_RCC_FREQMNT_ERRPEND_ENABLE()      SET_BIT(SYSCFG->FMNT0, SYSCONF_FMNT0_FERRPNDEN)
#define __DDL_RCC_FREQMNT_ERRPEND_DISABLE()     CLEAR_BIT(SYSCFG->FMNT0, SYSCONF_FMNT0_FERRPNDEN)
#define __DDL_RCC_FREQMNT_MNT_ENABLE()          SET_BIT(SYSCFG->FMNT0, SYSCONF_FMNT0_FMNTEN)
#define __DDL_RCC_FREQMNT_MNT_DISABLE()         CLEAR_BIT(SYSCFG->FMNT0, SYSCONF_FMNT0_FMNTEN)

/* Peripheral Clock Source Selection */
#define RCC_PERIPHCLK_BUZ1                               0x00000001U
#define RCC_PERIPHCLK_BTIM                               0x00000008U
#define RCC_PERIPHCLK_WDT                                0x00000010U
#define RCC_PERIPHCLK_ADC0                               0x00000020U
#define RCC_PERIPHCLK_ADC1                               0x00000040U
#define RCC_PERIPHCLK_NF                                 0x00000080U


#define IS_WDT_PERIPHCLOCK(__WDTCLOCK__)                  (((__WDTCLOCK__) == RCC_WDT_CLKSRC_LIRC) || ((__WDTCLOCK__) == RCC_WDT_CLKSRC_MCLK))

#define IS_BUZ1_PERIPHCLOCK(__BUZ1CLOCK__)                (((__BUZ1CLOCK__) == RCC_BUZ1_CLKSRC_HIRC) || \
														   ((__BUZ1CLOCK__) == RCC_BUZ1_CLKSRC_HOSC) || \
														   ((__BUZ1CLOCK__) == RCC_BUZ1_CLKSRC_LIRC) || \
														   ((__BUZ1CLOCK__) == RCC_BUZ1_CLKSRC_PLL)  || \
														   ((__BUZ1CLOCK__) == RCC_BUZ1_CLKSRC_FCLK))


#define IS_NF_PERIPHCLOCK(__NFCLOCK__)                    (((__NFCLOCK__) == RCC_NF_CLKSRC_HIRC) || ((__NFCLOCK__) == RCC_NF_CLKSRC_FCLK))

#define IS_BTIM_PERIPHCLOCK(__BTIMCLOCK__)                (((__BTIMCLOCK__) == RCC_BTIM_CLKSRC_LIRC) ||  ((__BTIMCLOCK__) == RCC_BTIM_CLKSRC_MCLK))


/* WDT clock sources */
#define RCC_WDT_CLKSRC_LIRC                               SYSCONF_CSEL2_WDT_LIRC
#define RCC_WDT_CLKSRC_MCLK                               SYSCONF_CSEL2_WDT_MCLK

/* BUZ1 clock sources */
#define RCC_BUZ1_CLKSRC_HIRC                              SYSCONF_CSEL2_BUZ1_HIRC
#define RCC_BUZ1_CLKSRC_HOSC                              SYSCONF_CSEL2_BUZ1_HOSC
#define RCC_BUZ1_CLKSRC_LIRC                              SYSCONF_CSEL2_BUZ1_LIRC
#define RCC_BUZ1_CLKSRC_PLL                               SYSCONF_CSEL2_BUZ1_PLL
#define RCC_BUZ1_CLKSRC_FCLK                              SYSCONF_CSEL2_BUZ1_FCLK

/* BTIM clock sources */
#define RCC_BTIM_CLKSRC_LIRC                              SYSCONF_CSEL2_BTIM_LIRC
#define RCC_BTIM_CLKSRC_MCLK                              SYSCONF_CSEL2_BTIM_MCLK

/* NF clock sources */
#define RCC_NF_CLKSRC_HIRC                                SYSCONF_CSEL2_NF_HIRC
#define RCC_NF_CLKSRC_FCLK                                SYSCONF_CSEL2_NF_FCLK

#define __MAX_BUZ1_CLK_PRESCALER__                        14
#define __MAX_ADC_CLK_PRESCALER__                         128

#define __SYSCLK_DIVIDER_1                                (0x0UL << SYSCONF_CSEL0_FDIV_Pos)
#define __SYSCLK_DIVIDER_2                                (0x1UL << SYSCONF_CSEL0_FDIV_Pos)
#define __SYSCLK_DIVIDER_4                                (0x2UL << SYSCONF_CSEL0_FDIV_Pos)
#define __SYSCLK_DIVIDER_8                                (0x3UL << SYSCONF_CSEL0_FDIV_Pos)
#define __SYSCLK_DIVIDER_16                               (0x4UL << SYSCONF_CSEL0_FDIV_Pos)
#define __SYSCLK_DIVIDER_32                               (0x5UL << SYSCONF_CSEL0_FDIV_Pos)
#define __SYSCLK_DIVIDER_64                               (0x6UL << SYSCONF_CSEL0_FDIV_Pos)
#define __SYSCLK_DIVIDER_128                              (0x7UL << SYSCONF_CSEL0_FDIV_Pos)
#define __SYSCLK_DIVIDER_256                              (0x8UL << SYSCONF_CSEL0_FDIV_Pos)

#define IS_SYSCLK_DEVIDER(__SYSCLK_DEVIDER__)            (((__SYSCLK_DEVIDER__) == __SYSCLK_DIVIDER_1)  || \
														  ((__SYSCLK_DEVIDER__) == __SYSCLK_DIVIDER_2)  || \
														  ((__SYSCLK_DEVIDER__) == __SYSCLK_DIVIDER_4)  || \
														  ((__SYSCLK_DEVIDER__) == __SYSCLK_DIVIDER_8)  || \
														  ((__SYSCLK_DEVIDER__) == __SYSCLK_DIVIDER_16) || \
														  ((__SYSCLK_DEVIDER__) == __SYSCLK_DIVIDER_32) || \
														  ((__SYSCLK_DEVIDER__) == __SYSCLK_DIVIDER_64) || \
														  ((__SYSCLK_DEVIDER__) == __SYSCLK_DIVIDER_128)|| \
														  ((__SYSCLK_DEVIDER__) == __SYSCLK_DIVIDER_256))

/* Get WWDG Clock Source */
#define __DDL_RCC_GET_WDT_CLKSRC()                        READ_BIT(SYSCFG->CSEL2, SYSCONF_CSEL2_WDT_Msk)
/* Get ADC0/1 Clock Source */
#define __DDL_RCC_GET_ADC0_CLKSRC()                       READ_BIT(SYSCFG->CSEL2, SYSCONF_CSEL2_ADC0CLK_Msk)
#define __DDL_RCC_GET_ADC0_CLKPRESCALER()                 READ_BIT(SYSCFG->CSEL2, SYSCONF_CSEL2_ADC0DIV_Msk)
#define __DDL_RCC_GET_ADC1_CLKSRC()                       READ_BIT(SYSCFG->CSEL2, SYSCONF_CSEL2_ADC1CLK_Msk)
#define __DDL_RCC_GET_ADC1_CLKPRESCALER()                 READ_BIT(SYSCFG->CSEL2, SYSCONF_CSEL2_ADC1DIV_Msk)

/* Get Base-Timer Clock Source */
#define	__DDL_RCC_GET_BTIM_CLKSRC()                       READ_BIT(SYSCFG->CSEL2, SYSCONF_CSEL2_BTIM_Msk)


/* Get Noise-Filter Clock Source */
#define	__DDL_RCC_GET_NF_CLKSRC()                         READ_BIT(SYSCFG->CSEL2, SYSCONF_CSEL2_NF_Msk)
/* Get BUZ1 Clock Source */
#define	__DDL_RCC_GET_BUZ1_CLKSRC()                       READ_BIT(SYSCFG->CSEL2, SYSCONF_CSEL2_BUZ1_Msk)
/* Get BUZ0 Clock Prescaler */
#define	__DDL_RCC_GET_BUZ1_CLKPRESCALER()                 READ_BIT(SYSCFG->CSEL1, SYSCONF_CSEL1_BUZ1DIV_Msk)
/* Get NF Clock 0 Prescaler*/
#define	__DDL_RCC_GET_NF_CLK0PRESCALER()                  READ_BIT(SYSCFG->CSEL2, SYSCONF_CSEL2_NFCLK0_Msk)
/* Get NF Clock 1 Prescaler*/
#define	__DDL_RCC_GET_NF_CLK1PRESCALER()                  READ_BIT(SYSCFG->CSEL2, SYSCONF_CSEL2_NFCLK1_Msk)
/* Get NF Clock 2 Prescaler*/
#define	__DDL_RCC_GET_NF_CLK2PRESCALER()                  READ_BIT(SYSCFG->CSEL2, SYSCONF_CSEL2_NFCLK2_Msk)


#define NF_CLOCK0_PRESCALER_2                             0x00U
#define NF_CLOCK0_PRESCALER_4                             0x01U
#define NF_CLOCK0_PRESCALER_8                             0x02U
#define NF_CLOCK0_PRESCALER_16                            0x03U

#define NF_CLOCK1_PRESCALER_32                            0x00U
#define NF_CLOCK1_PRESCALER_64                            0x01U
#define NF_CLOCK1_PRESCALER_128                           0x02U
#define NF_CLOCK1_PRESCALER_256                           0x03U

#define NF_CLOCK2_PRESCALER_512                           0x00U
#define NF_CLOCK2_PRESCALER_1024                          0x01U
#define NF_CLOCK2_PRESCALER_2048                          0x02U
#define NF_CLOCK2_PRESCALER_4096                          0x03U

typedef struct
{
	uint32_t periphClkSrcSelection;
	uint32_t btimClkSRC;
} RCC_BTIMInit_T;

typedef struct
{
	uint32_t periphClkSrcSelection;
	uint32_t nfClkSRC;
	uint32_t nfClk0Prescaler;
	uint32_t nfClk1Prescaler;
	uint32_t nfClk2Prescaler;
} RCC_NFInit_T;

typedef struct
{
	uint32_t periphClkSrcSelection;
	uint32_t adcPrescalser;
} RCC_ADCInit_T;

typedef struct
{
	uint32_t periphClkSrcSelection;
	uint32_t buzClkSRC;
	uint32_t buzPrescalser;
} RCC_BUZInit_T;

typedef struct
{
	uint32_t periphClkSrcSelection;
	uint32_t wdtClkSrc;
} RCC_WDTInit_T;


/* Peripheral clock release & hold control and activate & deactivate */

/* Peripheral activated */
#define __DDL_RCC_DNU_ACTIVE()               SET_BIT(SYSCFG->RC0, SYSCONF_RC0_DNU)
#define __DDL_RCC_TIMG2_ACTIVE()             SET_BIT(SYSCFG->RC0, SYSCONF_RC0_TIMG2)
#define __DDL_RCC_TIMG1_ACTIVE()             SET_BIT(SYSCFG->RC0, SYSCONF_RC0_TIMG1)
#define __DDL_RCC_TIMG0_ACTIVE()             SET_BIT(SYSCFG->RC0, SYSCONF_RC0_TIMG0)
#define __DDL_RCC_TIMH_ACTIVE()              SET_BIT(SYSCFG->RC0, SYSCONF_RC0_TIMH)
#define __DDL_RCC_TIMA1_ACTIVE()             SET_BIT(SYSCFG->RC0, SYSCONF_RC0_TIMA1)
#define __DDL_RCC_TIMA0_ACTIVE()             SET_BIT(SYSCFG->RC0, SYSCONF_RC0_TIMA0)
#define __DDL_RCC_BTIM_ACTIVE()        		 SET_BIT(SYSCFG->RC0, SYSCONF_RC0_BTIM)
#define __DDL_RCC_ADC1_ACTIVE()              SET_BIT(SYSCFG->RC0, SYSCONF_RC0_ADC1)
#define __DDL_RCC_ADC0_ACTIVE()              SET_BIT(SYSCFG->RC0, SYSCONF_RC0_ADC0)
#define __DDL_RCC_EBUS_ACTIVE()              SET_BIT(SYSCFG->RC0, SYSCONF_RC0_EBUS)
#define __DDL_RCC_TMU_ACTIVE()               SET_BIT(SYSCFG->RC0, SYSCONF_RC0_TMU)
#define __DDL_RCC_DMA_ACTIVE()               SET_BIT(SYSCFG->RC0, SYSCONF_RC0_DMA)
#define __DDL_RCC_GPIO_ACTIVE()              SET_BIT(SYSCFG->RC0, SYSCONF_RC0_GPIO)
#define __DDL_RCC_I2C1_ACTIVE()              SET_BIT(SYSCFG->RC0, SYSCONF_RC0_I2C1)
#define __DDL_RCC_I2C0_ACTIVE()              SET_BIT(SYSCFG->RC0, SYSCONF_RC0_I2C0)
#define __DDL_RCC_TIMS3_ACTIVE()             SET_BIT(SYSCFG->RC1, SYSCONF_RC1_TIMS3)
#define __DDL_RCC_TIMS2_ACTIVE()             SET_BIT(SYSCFG->RC1, SYSCONF_RC1_TIMS2)
#define __DDL_RCC_TIMS1_ACTIVE()             SET_BIT(SYSCFG->RC1, SYSCONF_RC1_TIMS1)
#define __DDL_RCC_TIMS0_ACTIVE()             SET_BIT(SYSCFG->RC1, SYSCONF_RC1_TIMS0)
#define __DDL_RCC_CAN_ACTIVE()               SET_BIT(SYSCFG->RC1, SYSCONF_RC1_CAN)
#define __DDL_RCC_ACMP_ACTIVE()              SET_BIT(SYSCFG->RC1, SYSCONF_RC1_ACMP)
#define __DDL_RCC_DAC_ACTIVE()               SET_BIT(SYSCFG->RC1, SYSCONF_RC1_DAC)
#define __DDL_RCC_SCI2_ACTIVE()              SET_BIT(SYSCFG->RC1, SYSCONF_RC1_SCI2)
#define __DDL_RCC_SCI1_ACTIVE()              SET_BIT(SYSCFG->RC1, SYSCONF_RC1_SCI1)
#define __DDL_RCC_SCI0_ACTIVE()              SET_BIT(SYSCFG->RC1, SYSCONF_RC1_SCI0)
#define __DDL_RCC_SPI_ACTIVE()               SET_BIT(SYSCFG->RC1, SYSCONF_RC1_SPI)
#define __DDL_RCC_IOW_ACTIVE()               SET_BIT(SYSCFG->RC1, SYSCONF_RC1_IOW)

/* Peripheral deactivated */
#define __DDL_RCC_DNU_DEACTIVE()             CLEAR_BIT(SYSCFG->RC0, SYSCONF_RC0_DNU)
#define __DDL_RCC_TIMG2_DEACTIVE()           CLEAR_BIT(SYSCFG->RC0, SYSCONF_RC0_TIMG2)
#define __DDL_RCC_TIMG1_DEACTIVE()           CLEAR_BIT(SYSCFG->RC0, SYSCONF_RC0_TIMG1)
#define __DDL_RCC_TIMG0_DEACTIVE()           CLEAR_BIT(SYSCFG->RC0, SYSCONF_RC0_TIMG0)
#define __DDL_RCC_TIMH_DEACTIVE()            CLEAR_BIT(SYSCFG->RC0, SYSCONF_RC0_TIMH)
#define __DDL_RCC_TIMA1_DEACTIVE()           CLEAR_BIT(SYSCFG->RC0, SYSCONF_RC0_TIMA1)
#define __DDL_RCC_TIMA0_DEACTIVE()           CLEAR_BIT(SYSCFG->RC0, SYSCONF_RC0_TIMA0)
#define __DDL_RCC_BTIM_DEACTIVE()       	 CLEAR_BIT(SYSCFG->RC0, SYSCONF_RC0_BTIM)
#define __DDL_RCC_ADC1_DEACTIVE()            CLEAR_BIT(SYSCFG->RC0, SYSCONF_RC0_ADC1)
#define __DDL_RCC_ADC0_DEACTIVE()            CLEAR_BIT(SYSCFG->RC0, SYSCONF_RC0_ADC0)
#define __DDL_RCC_EBUS_DEACTIVE()            CLEAR_BIT(SYSCFG->RC0, SYSCONF_RC0_EBUS)
#define __DDL_RCC_TMU_DEACTIVE()             CLEAR_BIT(SYSCFG->RC0, SYSCONF_RC0_TMU)
#define __DDL_RCC_DMA_DEACTIVE()             CLEAR_BIT(SYSCFG->RC0, SYSCONF_RC0_DMA)
#define __DDL_RCC_GPIO_DEACTIVE()            CLEAR_BIT(SYSCFG->RC0, SYSCONF_RC0_GPIO)
#define __DDL_RCC_I2C1_DEACTIVE()            CLEAR_BIT(SYSCFG->RC0, SYSCONF_RC0_I2C1)
#define __DDL_RCC_I2C0_DEACTIVE()            CLEAR_BIT(SYSCFG->RC0, SYSCONF_RC0_I2C0)
#define __DDL_RCC_TIMS3_DEACTIVE()           CLEAR_BIT(SYSCFG->RC1, SYSCONF_RC1_TIMS3)
#define __DDL_RCC_TIMS2_DEACTIVE()           CLEAR_BIT(SYSCFG->RC1, SYSCONF_RC1_TIMS2)
#define __DDL_RCC_TIMS1_DEACTIVE()           CLEAR_BIT(SYSCFG->RC1, SYSCONF_RC1_TIMS1)
#define __DDL_RCC_TIMS0_DEACTIVE()           CLEAR_BIT(SYSCFG->RC1, SYSCONF_RC1_TIMS0)
#define __DDL_RCC_CAN_DEACTIVE()             CLEAR_BIT(SYSCFG->RC1, SYSCONF_RC1_CAN)
#define __DDL_RCC_ACMP_DEACTIVE()            CLEAR_BIT(SYSCFG->RC1, SYSCONF_RC1_ACMP)
#define __DDL_RCC_DAC_DEACTIVE()             CLEAR_BIT(SYSCFG->RC1, SYSCONF_RC1_DAC)
#define __DDL_RCC_SCI2_DEACTIVE()            CLEAR_BIT(SYSCFG->RC1, SYSCONF_RC1_SCI2)
#define __DDL_RCC_SCI1_DEACTIVE()            CLEAR_BIT(SYSCFG->RC1, SYSCONF_RC1_SCI1)
#define __DDL_RCC_SCI0_DEACTIVE()            CLEAR_BIT(SYSCFG->RC1, SYSCONF_RC1_SCI0)
#define __DDL_RCC_SPI_DEACTIVE()             CLEAR_BIT(SYSCFG->RC1, SYSCONF_RC1_SPI)
#define __DDL_RCC_IOW_DEACTIVE()             CLEAR_BIT(SYSCFG->RC1, SYSCONF_RC1_IOW)

/* Peripheral Clock Release */
#define __DDL_RCC_WDT_CLK_RELEASE()          do{\
											 		__DDL_SYSREG_UNLOCK();\
											 		SET_BIT(SYSCFG->CG0, SYSCONF_CG0_WDT);\
											 		__DDL_SYSREG_LOCK();\
											 } while (0)

#define __DDL_RCC_BUZ1_CLK_RELEASE()         SET_BIT(SYSCFG->CG0, SYSCONF_CG0_BUZ1)
#define __DDL_RCC_DNU_CLK_RELEASE()          SET_BIT(SYSCFG->CG0, SYSCONF_CG0_DNU)
#define __DDL_RCC_TIMG2_CLK_RELEASE()        SET_BIT(SYSCFG->CG0, SYSCONF_CG0_TIMG2)
#define __DDL_RCC_TIMG1_CLK_RELEASE()        SET_BIT(SYSCFG->CG0, SYSCONF_CG0_TIMG1)
#define __DDL_RCC_TIMG0_CLK_RELEASE()        SET_BIT(SYSCFG->CG0, SYSCONF_CG0_TIMG0)
#define __DDL_RCC_TIMH_CLK_RELEASE()         SET_BIT(SYSCFG->CG0, SYSCONF_CG0_TIMH)
#define __DDL_RCC_TIMA1_CLK_RELEASE()        SET_BIT(SYSCFG->CG0, SYSCONF_CG0_TIMA1)
#define __DDL_RCC_TIMA0_CLK_RELEASE()        SET_BIT(SYSCFG->CG0, SYSCONF_CG0_TIMA0)
#define __DDL_RCC_BTIM_CLK_RELEASE()    	 SET_BIT(SYSCFG->CG0, SYSCONF_CG0_BTIM)
#define __DDL_RCC_ADC1_CLK_RELEASE()         SET_BIT(SYSCFG->CG0, SYSCONF_CG0_ADC1)
#define __DDL_RCC_ADC0_CLK_RELEASE()         SET_BIT(SYSCFG->CG0, SYSCONF_CG0_ADC0)
#define __DDL_RCC_EBUS_CLK_RELEASE()         SET_BIT(SYSCFG->CG0, SYSCONF_CG0_EBUS)
#define __DDL_RCC_TMU_CLK_RELEASE()          SET_BIT(SYSCFG->CG0, SYSCONF_CG0_TMU)
#define __DDL_RCC_DMA_CLK_RELEASE()          SET_BIT(SYSCFG->CG0, SYSCONF_CG0_DMA)
#define __DDL_RCC_GPIO_CLK_RELEASE()         SET_BIT(SYSCFG->CG0, SYSCONF_CG0_GPIO)
#define __DDL_RCC_I2C1_CLK_RELEASE()         SET_BIT(SYSCFG->CG0, SYSCONF_CG0_I2C1)
#define __DDL_RCC_I2C0_CLK_RELEASE()         SET_BIT(SYSCFG->CG0, SYSCONF_CG0_I2C0)
#define __DDL_RCC_TIMS3_CLK_RELEASE()        SET_BIT(SYSCFG->CG1, SYSCONF_CG1_TIMS3)
#define __DDL_RCC_TIMS2_CLK_RELEASE()        SET_BIT(SYSCFG->CG1, SYSCONF_CG1_TIMS2)
#define __DDL_RCC_TIMS1_CLK_RELEASE()        SET_BIT(SYSCFG->CG1, SYSCONF_CG1_TIMS1)
#define __DDL_RCC_TIMS0_CLK_RELEASE()        SET_BIT(SYSCFG->CG1, SYSCONF_CG1_TIMS0)
#define __DDL_RCC_CAN_CLK_RELEASE()          SET_BIT(SYSCFG->CG1, SYSCONF_CG1_CAN)
#define __DDL_RCC_ACMP_CLK_RELEASE()         SET_BIT(SYSCFG->CG1, SYSCONF_CG1_ACMP)
#define __DDL_RCC_DAC_CLK_RELEASE()          SET_BIT(SYSCFG->CG1, SYSCONF_CG1_DAC)
#define __DDL_RCC_SCI2_CLK_RELEASE()         SET_BIT(SYSCFG->CG1, SYSCONF_CG1_SCI2)
#define __DDL_RCC_SCI1_CLK_RELEASE()         SET_BIT(SYSCFG->CG1, SYSCONF_CG1_SCI1)
#define __DDL_RCC_SCI0_CLK_RELEASE()         SET_BIT(SYSCFG->CG1, SYSCONF_CG1_SCI0)
#define __DDL_RCC_SPI_CLK_RELEASE()          SET_BIT(SYSCFG->CG1, SYSCONF_CG1_SPI)
#define __DDL_RCC_IOW_CLK_RELEASE()          SET_BIT(SYSCFG->CG1, SYSCONF_CG1_IOW)


/*  Peripheral Clock Hold */
#define __DDL_RCC_WDT_CLK_HOLD()             do{\
											 		__DDL_SYSREG_UNLOCK();\
											 		CLEAR_BIT(SYSCFG->CG0, SYSCONF_CG0_WDT);\
											 		__DDL_SYSREG_LOCK();\
											 } while (0)

#define __DDL_RCC_BUZ1_CLK_HOLD()            CLEAR_BIT(SYSCFG->CG0, SYSCONF_CG0_BUZ1)
#define __DDL_RCC_DNU_CLK_HOLD()             CLEAR_BIT(SYSCFG->CG0, SYSCONF_CG0_DNU)
#define __DDL_RCC_TIMG2_CLK_HOLD()           CLEAR_BIT(SYSCFG->CG0, SYSCONF_CG0_TIMG2)
#define __DDL_RCC_TIMG1_CLK_HOLD()           CLEAR_BIT(SYSCFG->CG0, SYSCONF_CG0_TIMG1)
#define __DDL_RCC_TIMG0_CLK_HOLD()           CLEAR_BIT(SYSCFG->CG0, SYSCONF_CG0_TIMG0)
#define __DDL_RCC_TIMH_CLK_HOLD()            CLEAR_BIT(SYSCFG->CG0, SYSCONF_CG0_TIMH)
#define __DDL_RCC_TIMA1_CLK_HOLD()           CLEAR_BIT(SYSCFG->CG0, SYSCONF_CG0_TIMA1)
#define __DDL_RCC_TIMA0_CLK_HOLD()           CLEAR_BIT(SYSCFG->CG0, SYSCONF_CG0_TIMA0)
#define __DDL_RCC_BTIM_CLK_HOLD()      		 CLEAR_BIT(SYSCFG->CG0, SYSCONF_CG0_BTIM)
#define __DDL_RCC_ADC1_CLK_HOLD()            CLEAR_BIT(SYSCFG->CG0, SYSCONF_CG0_ADC1)
#define __DDL_RCC_ADC0_CLK_HOLD()            CLEAR_BIT(SYSCFG->CG0, SYSCONF_CG0_ADC0)
#define __DDL_RCC_EBUS_CLK_HOLD()            CLEAR_BIT(SYSCFG->CG0, SYSCONF_CG0_EBUS)
#define __DDL_RCC_TMU_CLK_HOLD()             CLEAR_BIT(SYSCFG->CG0, SYSCONF_CG0_TMU)
#define __DDL_RCC_DMA_CLK_HOLD()             CLEAR_BIT(SYSCFG->CG0, SYSCONF_CG0_DMA)
#define __DDL_RCC_GPIO_CLK_HOLD()            CLEAR_BIT(SYSCFG->CG0, SYSCONF_CG0_GPIO)
#define __DDL_RCC_I2C1_CLK_HOLD()            CLEAR_BIT(SYSCFG->CG0, SYSCONF_CG0_I2C1)
#define __DDL_RCC_I2C0_CLK_HOLD()            CLEAR_BIT(SYSCFG->CG0, SYSCONF_CG0_I2C0)
#define __DDL_RCC_TIMS3_CLK_HOLD()           CLEAR_BIT(SYSCFG->CG1, SYSCONF_CG1_TIMS3)
#define __DDL_RCC_TIMS2_CLK_HOLD()           CLEAR_BIT(SYSCFG->CG1, SYSCONF_CG1_TIMS2)
#define __DDL_RCC_TIMS1_CLK_HOLD()           CLEAR_BIT(SYSCFG->CG1, SYSCONF_CG1_TIMS1)
#define __DDL_RCC_TIMS0_CLK_HOLD()           CLEAR_BIT(SYSCFG->CG1, SYSCONF_CG1_TIMS0)
#define __DDL_RCC_CAN_CLK_HOLD()             CLEAR_BIT(SYSCFG->CG1, SYSCONF_RC1_CAN)
#define __DDL_RCC_ACMP_CLK_HOLD()            CLEAR_BIT(SYSCFG->CG1, SYSCONF_RC1_ACMP)
#define __DDL_RCC_DAC_CLK_HOLD()             CLEAR_BIT(SYSCFG->CG1, SYSCONF_RC1_DAC)
#define __DDL_RCC_SCI2_CLK_HOLD()            CLEAR_BIT(SYSCFG->CG1, SYSCONF_CG1_SCI2)
#define __DDL_RCC_SCI1_CLK_HOLD()            CLEAR_BIT(SYSCFG->CG1, SYSCONF_CG1_SCI1)
#define __DDL_RCC_SCI0_CLK_HOLD()            CLEAR_BIT(SYSCFG->CG1, SYSCONF_CG1_SCI0)
#define __DDL_RCC_SPI_CLK_HOLD()             CLEAR_BIT(SYSCFG->CG1, SYSCONF_CG1_SPI)
#define __DDL_RCC_IOW_CLK_HOLD()             CLEAR_BIT(SYSCFG->CG1, SYSCONF_CG1_IOW)


/**
  * @brief  Initialize the HOSC oscillator for system clock
  * @param  void
  * @retval DDL status
  */
DDL_Status_T DDL_RCC_HoscEnable(void);

/**
  * @brief  Initialize the HIRC oscillator for system clock
  * @param  void
  * @retval DDL status
  */
DDL_Status_T DDL_RCC_HircEnable(void);

/**
  * @brief  Initialize the LIRC oscillator for system clock
  * @param  void
  * @retval DDL status
  */
DDL_Status_T DDL_RCC_LircEnable(void);

/**
  * @brief  Disable the HOSC oscillator for system clock
  * @param  void
  * @retval DDL status
  */
DDL_Status_T DDL_RCC_HoscDisable(void);

/**
  * @brief  Disable the HIRC oscillator for system clock
  * @param  void
  * @retval DDL status
  */
DDL_Status_T DDL_RCC_HircDisable(void);

/**
  * @brief  Disable the LIRC oscillator for system clock
  * @param  void
  * @retval DDL status
  */
DDL_Status_T DDL_RCC_LircDisable(void);

/**
  * @brief  Disable the PLL for system clock
  * @param  void
  * @retval DDL status
  */
DDL_Status_T DDL_RCC_PLLDisable(void);

/**
  * @brief  Configure MCLK source
  * @param  void
  * @retval DDL status
  */
DDL_Status_T  DDL_RCC_MclkSrcConfig(uint32_t mClkSource);

/**
  * @brief  Configure the elements of structure RCC_PLLInit_T to default values.
  * @param  pllInitStruct  pointer to a RCC_PLLInit_T structure that contains
  *                        the configuration information PLL.
  * @retval None.
  */
void DDL_RCC_PllStructInit(RCC_PLLInit_T  *pllInitStruct);

/**
  * @brief  Initializes the PLL by the specified parameters in the RCC_PLLInit_T.
  *
  * @param  pllInitStruct  pointer to a RCC_PLLInit_T structure that contains
  *                        the configuration information PLL.
  *
  * @retval DDL status
  */
DDL_Status_T DDL_RCC_PllConfig(RCC_PLLInit_T  *pllInitStruct);

/**
  * @brief  Configure system clock to the expected working frequency.
  * @param  rccClkInitStruct  pointer to a RCC_ClkInit_T structure that contains
  *                           the configuration information for system clock.
  * @retval DDL status
  */
DDL_Status_T DDL_RCC_ClkSrcConfig(RCC_ClkInit_T  *rccClkInitStruct);

/**
  * @brief  Get system working frequency.
  *
  * @retval system clock frequency.
  */
uint32_t DDL_RCC_GetSysClockFreq(void);

/**
  * @brief  Configure Base Timer clock.
  * @param  btimClkInit  pointer to a RCC_BTIMInit_T structure that contains
  *                      the configuration information for Base Timer.
  * @retval DDL status
  */
DDL_Status_T DDL_RCC_BTIM_PeriphCLKConfig(RCC_BTIMInit_T *btimClkInit);

/**
  * @brief  Configure Buzzer clock.
  * @param  buzzClkInit  pointer to a RCC_BUZInit_T structure that contains
  *                      the configuration information for Buzzer.
  * @retval DDL status
  */
DDL_Status_T DDL_RCC_BUZZ_PeriphCLKConfig(RCC_BUZInit_T *buzzClkInit);

/**
  * @brief  Configure Noise Filter clock.
  * @param  nfClkInit  pointer to a RCC_NFInit_T structure that contains
  *                    the configuration information for Noise Filter.
  * @retval DDL status
  */
DDL_Status_T DDL_RCC_NF_PeriphCLKConfig(RCC_NFInit_T *nfClkInit);

/**
  * @brief  Configure ADC's clock.
  * @param  adcClkInit  pointer to a RCC_ADCInit_T structure that contains
  *                     the configuration information for ADC.
  * @retval DDL status
  */
DDL_Status_T DDL_RCC_ADC_PeriphCLKConfig(RCC_ADCInit_T *adcClkInit);

/**
  * @brief  Configure WatchDog's clock.
  * @param  wdtClkInit  pointer to a RCC_WDTInit_T structure that contains
  *                     the configuration information for WDT.
  * @retval DDL status
  */
DDL_Status_T DDL_RCC_WDT_PeriphCLKConfig(RCC_WDTInit_T *wdtClkInit);


#ifdef __cplusplus
}
#endif

#endif /* __KPM32xx_DDL_RCC_H */
