/**
  ******************************************************************************
  * @file    kpm32xx_ddl_crc.h
  * @author  Kiwi Software Team
  * @brief   Header file of CRC DDL module.
  * @note
  *          V1.0.0, 2025/1/7.
  *
  * Copyright (c) 2024, Kiwi Instruments Co,. Ltd.
  *
  * Redistribution and use in source and binary forms, with or without modification,
  * are permitted provided that the following conditions are met:
  *
  *   1. Redistributions of source code must retain the above copyright notice,
  *      this list of conditions and the following disclaimer.
  *
  *   2. Redistributions in binary form must reproduce the above copyright notice,
  *      this list of conditions and the following disclaimer in the documentation
  *      and/or other materials provided with the distribution.
  *
  *   3. Neither the name of the copyright holder nor the names of its contributors
  *      may be used to endorse or promote products derived from this software without
  *      specific prior written permission.
  *
  * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
  * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
  * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
  * THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
  * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE
  * GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
  * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
  * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
  * THE POSSIBILITY OF SUCH DAMAGE.
  ******************************************************************************
  */



#ifndef __KPM32xx_DDL_CRC_H
#define __KPM32xx_DDL_CRC_H

#ifdef __cplusplus
extern "C" {
#endif

#include "kpm32xx_ddl_def.h"


/* CRC initialize structure definition */
typedef struct
{
	uint32_t                    inputInvert;   /* indicate the input data invert or not before calculating CRC               */
	uint32_t                    outputInvert;  /* indicate output data invert or not after calculating CRC                   */
}CRC_Init_T;


/* CRC output data invert type */
#define CRC_OUTPUT_INVT_WORD                             (0x00000001U)
#define CRC_OUTPUT_INVT_NONE                             (0x00000000U)

/* CRC input data invert type */
#define CRC_INPUT_INVT_NONE                               CRC_CTRL_ININVT_NONE
#define CRC_INPUT_INVT_BYTE                               CRC_CTRL_ININVT_BYTE
#define CRC_INPUT_INVT_HALF                               CRC_CTRL_ININVT_HALF
#define CRC_INPUT_INVT_WORD                               CRC_CTRL_ININVT_WORD


/**
  * @brief  Configure the elements of structure CRC_Init_T to default values.
  * @param  pCrc Pointer to a CRC_Init_T structure that contains
  *                  the configuration information for the given crc module.
  */
void DDL_CRC_StructInit(CRC_Init_T *pCrc);

/**
  * @brief  Initialize the CRC according to the specified
  *         parameters in the CRC_Init_T.
  * @param  pCrc  CRC_Init_T
  */
void DDL_CRC_Init(CRC_Init_T *pCrc);

/**
  * @brief  DeInitialize the CRC peripheral.
  * @param  hcrc  CRC handle
  */
void DDL_CRC_DeInit(void);

/**
  * @brief  Compute the CRC value of the specify data.
  * @brief  default: big endian
  * @param  pCrc          CRC_Init_T
  * @param  initValue     CRC init value.
  * @param  pBuffer       pointer to the input data buffer.
  * @param  bufferLength  input data buffer length.
  * @retval crc value
  */
uint32_t DDL_CRC_Calculate(CRC_Init_T *pCrc, uint32_t initValue, uint32_t *pBuffer, uint32_t bufferLength);

/**
  * @param  initValue  CRC init value.
  */
void DDL_CRC_Set_InitValue(uint32_t initValue);

/**
  * @brief  Compute the CRC value of the specify data.
  * @brief  default: big endian
  * @param  pCrc          CRC_Init_T
  * @param  pBuffer       pointer to the input data buffer.
  * @param  bufferLength  input data buffer length.
  * @retval crc value
  */
uint32_t DDL_CRC_Accumulate(CRC_Init_T *pCrc, uint32_t *pBuffer, uint32_t bufferLength);



#ifdef __cplusplus
}
#endif

#endif /* __KPM32xx_DDL_CRC_H */

