/**
 ******************************************************************************
 * @file    kpm32xx_ddl_basetimer.h
 * @author  Kiwi Software Team
 * @brief   Header file of Base Timer module.
 * @note
 *          V1.0.0, 2024/12/20.
 *
 * Copyright (c) 2024, Kiwi Instruments Co,. Ltd.
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 *	1. Redistributions of source code must retain the above copyright notice,
 *	   this list of conditions and the following disclaimer.
 *
 *	2. Redistributions in binary form must reproduce the above copyright notice,
 *	   this list of conditions and the following disclaimer in the documentation
 *	   and/or other materials provided with the distribution.
 *
 *	3. Neither the name of the copyright holder nor the names of its contributors
 *	   may be used to endorse or promote products derived from this software without
 *	   specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
 * THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE
 * GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 ******************************************************************************
 */

#ifndef __KPM32xx_DDL_BASETIMER_H
#define __KPM32xx_DDL_BASETIMER_H

#ifdef __cplusplus
extern "C" {
#endif

#include "kpm32xx_ddl_def.h"


/* BaseTimer clock selection */
typedef enum
{
	BTM_LIRC = SYSCONF_CSEL2_BTIM_LIRC,        /* LIRC as BaseTimer clock source */
	BTM_MCLK = SYSCONF_CSEL2_BTIM_MCLK,        /* MCLK as BaseTimer clock source */
} BTM_ClkSrc_T;

/* BaseTimer mode selection */
typedef enum
{
	MODE_16BIT,                                /* 16 Bit Mode */
	MODE_32BIT                                 /* 32 Bit Mode */
} BTM_Mode_T;

/* BaseTimer event output enable */
typedef enum
{
	EVT_DISABLE,                               /* event output disable */
	EVT_ENABLE                                 /* event output enable  */
} BTM_EVT_T;

/* BaseTimer clock division */
typedef enum
{
	DIV_16  = 0x0UL,
	DIV_32  = 0x1UL,
	DIV_64  = 0x2UL,
	DIV_256 = 0x3UL
} BTM_ClkDiv_T;

/*
 * When configured as 32Bit mode, BTM1 has no sense.
 * In this case, Base-Timer is only controlled by BTM0.
 */
typedef struct
{
	BTM_Mode_T   bitMode;          /* 32Bit or 16Bit mode         */
	BTM_EVT_T    eventOuput;       /* event Enable or Disable     */
	BTM_ClkDiv_T clkDivsion;       /* BTM x Clock Division Value  */
	uint32_t     loadValue;        /* BTM x Load Value            */
	BTM_ClkSrc_T clkSource;        /* Base-Timer Clock Source     */
} BTM_BaseInit_T;

/**
  * @brief  Make BTM peripheral active.
  * @param  None.
  * @retval None.
  */
void DDL_BTM_Instance_Active(void);

/**
  * @brief  Configure the structure of BTM_BaseInit_T to default values.
  *
  * @param  BTM_BaseInit_T Pointer to a TIMS_BaseInit_T structure that contains
  *                        the configuration information for the given BTM module.
  *
  * @retval None
  */
void DDL_BTM_StructInit(BTM_BaseInit_T *pBtmStruct);

/**
  * @brief  Initializes the BTM by the specified parameters in the BTM_BaseInit_T.
  *			
  * @param  BTMx     BTM instance.
  *			pInitBTM Pointer to a BTM_BaseInit_T structure that contains
  *                  the configuration information for the given BTM module.
  *
  * @retval NONE
  */
void DDL_BTM_Init(BTIM_Type *BTMx, BTM_BaseInit_T *pInitBTM);

/**
  * @brief  use for BTM interrupt enable.
  * @param  BTMx BTM instance.		
  * @retval None
  */
void DDL_BTM_IntEnable(BTIM_Type *BTMx);

/**
  * @brief  use for BTM interrupt disable.
  * @param  BTMx BTM instance.		
  * @retval None
  */
void DDL_BTM_IntDisable(BTIM_Type *BTMx);

/**
  * @brief  use for BTM event output enable.
  * @param  BTMx BTM instance.		
  * @retval None
  */
void DDL_BTM_EvtOutEnable(BTIM_Type *BTMx);

/**
  * @brief  use for BTM event output disable.
  * @param  BTMx BTM instance.		
  * @retval None
  */
void DDL_BTM_EvtOutDisable(BTIM_Type *BTMx);

/**
  * @brief  use for timer start.
  * @param  BTMx BTM instance.
  * @retval None
  */
void DDL_BTM_Start(BTIM_Type *BTMx);

/**
  * @brief  use for timer stop.
  * @param  BTMx  BTM instance.
  * @retval None
  */
void DDL_BTM_Stop(BTIM_Type *BTMx);

#ifdef __cplusplus
}
#endif

#endif /* __KPM32xx_DDL_BASETIMER_H */

