/**
  ******************************************************************************
  * @file    KPM32x14Txx.h
  * @author  Kiwi Software Team
  * @brief   Header file of KPM32x14Txx registers definition.
  * @note
  *          V1.0.0, 2024/12/20.
  *
  * Copyright (c) 2024, Kiwi Instruments Co,. Ltd.
  *
  * Redistribution and use in source and binary forms, with or without modification,
  * are permitted provided that the following conditions are met:
  *
  *  1. Redistributions of source code must retain the above copyright notice,
  *	 this list of conditions and the following disclaimer.
  *
  *  2. Redistributions in binary form must reproduce the above copyright notice,
  *	 this list of conditions and the following disclaimer in the documentation
  *	 and/or other materials provided with the distribution.
  *
  *  3. Neither the name of the copyright holder nor the names of its contributors
  *	 may be used to endorse or promote products derived from this software without
  *	 specific prior written permission.
  *
  * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
  * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
  * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
  * THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
  * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE
  * GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
  * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
  * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
  * THE POSSIBILITY OF SUCH DAMAGE.
  ******************************************************************************
  */


#ifndef __KPM32x14Txx_H
#define __KPM32x14Txx_H

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */


#define __CM4_REV                 0x0001U  /* Core revision r0p1                            */
#define __MPU_PRESENT             1U       /* KPM32R28Txx provides an MPU                   */
#define __VTOR_PRESENT            1U       /* Vector Table Register supported               */
#define __NVIC_PRIO_BITS          4U       /* KPM32xx uses 3 Bits for the priority Levels   */
#define __Vendor_SysTickConfig    0U       /* Set to 1 if different SysTick Config is used  */
#define __FPU_PRESENT             1U       /* FPU present                                   */


/* CPU interrupt priority */
#define __PREEMPT_INTRERRUPT_PRIORITY_0__     0  /*The highest priority*/
#define __PREEMPT_INTRERRUPT_PRIORITY_1__     1
#define __PREEMPT_INTRERRUPT_PRIORITY_2__     2
#define __PREEMPT_INTRERRUPT_PRIORITY_3__     3  /*The lowest priority*/

#define __SUB_INTRERRUPT_PRIORITY_0__         0  /*The highest priority*/
#define __SUB_INTRERRUPT_PRIORITY_1__         1
#define __SUB_INTRERRUPT_PRIORITY_2__         2
#define __SUB_INTRERRUPT_PRIORITY_3__         3  /*The lowest priority*/


typedef enum
{
	/******  Cortex-M4_FP Processor Exceptions Numbers **********************************************************/
	NonMaskableInt_IRQn         = -14,    /*  2 Non Maskable Interrupt                                          */
	HardFault_IRQn              = -13,    /*  3 Cortex-M Hard Fault Interrupt                                   */
	SVC_IRQn                    = -5,     /*  11 Cortex-M SV Call Interrupt                                     */
	PendSV_IRQn                 = -2,     /*  14 Cortex-M Pend SV Interrupt                                     */
	SysTick_IRQn                = -1,     /*  15 Cortex-M System Tick Interrupt                                 */
	/******  KPM32x14Txx specific Interrupt Numbers *************************************************************/
	WWDG_IRQn                   = 0,      /*  Window WatchDog Interrupt                                         */
	LVD_IRQn                    = 1,      /*  LVD Interrupt                                                     */
	RESERVED_2_IRQn             = 2,      /*  Reserved                                                          */
	FLASH_IRQn                  = 3,      /*  FLASH Interrupt                                                   */
	DMA_IRQn                    = 4,      /*  DMA  Interrupt                                                    */
	CAN_IRQn                    = 5,      /*  CAN Interrupt                                                     */
	RESERVED_6_IRQn             = 6,      /*  Reserved                                                          */
	SPI_IRQn                    = 7,      /*  SPI Interrupt                                                     */
	RESERVED_8_IRQn             = 8,      /*  Reserved                                                          */
	IOW_IRQn                    = 9,      /*  IOW Interrupt                                                     */
	I2C0_IRQn                   = 10,     /*  I2C 0 Interrupt                                                   */
	I2C1_IRQn                   = 11,     /*  I2C 1 Interrupt                                                   */
	SCI0_IRQn                   = 12,     /*  SCI 0 Interrupt                                                   */
	SCI1_IRQn                   = 13,     /*  SCI 1 Interrupt                                                   */
	SCI2_IRQn                   = 14,     /*  SCI 2 Interrupt                                                   */
	RESERVED_15_IRQn            = 15,     /*  Reserved                                                          */
	ACMP0_IRQn                  = 16,     /*  ACMP 0 Interrupt                                                  */
	ACMP1_IRQn                  = 17,     /*  ACMP 1 Interrupt                                                  */
	ACMP2_IRQn                  = 18,     /*  ACMP 2 Interrupt                                                  */
	ACMP3_IRQn                  = 19,     /*  ACMP 3 Interrupt                                                  */
	ADC0_IRQn                   = 20,     /*  ADC 0 Interrupt                                                   */
	ADC1_IRQn                   = 21,     /*  ADC 1 Interrupt                                                   */
	BASETIMER0_IRQn             = 22,     /*  Base timer 0 Interrupt                                            */
	BASETIMER1_IRQn             = 23,     /*  Base timer 1 Interrupt                                           */
	INTP0_IRQn                  = 24,     /*  INTP0 Interrupt                                                   */
	INTP1_IRQn                  = 25,     /*  INTP1 Interrupt                                                   */
	INTP2_IRQn                  = 26,     /*  INTP2 Interrupt                                                   */
	INTP3_IRQn                  = 27,     /*  INTP3 Interrupt                                                   */
	INTP4_IRQn                  = 28,     /*  INTP4 Interrupt                                                   */
	INTP5_IRQn                  = 29,     /*  INTP5 Interrupt                                                   */
	INTP6_IRQn                  = 30,     /*  INTP6 Interrupt                                                   */
	INTP7_IRQn                  = 31,     /*  INTP7 Interrupt                                                   */
	TIMA0_IRQn					= 32,	  /*  TIMA 0 Interrupt								                    */
	TIMA1_IRQn					= 33,	  /*  TIMA 1 Interrupt								                    */
	TIMH0_IRQn					= 34,	  /*  TIMH 0 Interrupt													*/
	TIMG0_IRQn					= 35,	  /*  TIMG 0 Interrupt											        */
	TIMG1_IRQn					= 36,	  /*  TIMG 1 Interrupt									                */
	TIMG2_IRQn					= 37,	  /*  TIMG 2 Interrupt											        */
	TIMS0_IRQn					= 38,	  /*  TIMS 0 Interrupt									                */
	TIMS1_IRQn					= 39,	  /*  TIMS 1 Interrupt                                                  */
	TIMS2_IRQn 			        = 40,	  /*  TIMS 2 Interrupt										            */
	TIMS3_IRQn 			        = 41,	  /*  TIMS 3 Interrupt													*/
	INTP8_IRQn					= 42,	  /*  INTP8 Interrupt											        */
	INTP9_IRQn					= 43,	  /*  INTP9 Interrupt										            */
	INTP10_IRQn					= 44,	  /*  INTP10 Interrupt										            */
	INTP11_IRQn					= 45,	  /*  INTP11 Interrupt										            */
	INTP12_IRQn					= 46,	  /*  INTP12 Interrupt										            */
	INTP13_IRQn					= 47,	  /*  INTP13 Interrupt									                */
	INTP14_IRQn					= 48,	  /*  INTP14 Interrupt													*/
	INTP15_IRQn					= 49,	  /*  INTP15 Interrupt													*/
	TIMH1_IRQn					= 50,	  /*  TIMH 1 Interrupt													*/
	TIMH2_IRQn					= 51,	  /*  TIMH 2 Interrupt													*/
	IP_UNDEFINED,
} IRQn_Type;


#include "core_cm4.h"               /* Cortex-M4_FP processor and core peripherals */
#include "system_kpm32xx.h"
#include <stdint.h>

typedef struct
{
	__IO uint32_t TRIG;                 /* ADC Software Trigger Register,       address offset: 0x00      */
	__IO uint32_t CTRL;                 /* ADC Control Register,                address offset: 0x04      */
	__IO uint32_t RESERVED0[2];         /* Reserved,                            address offset : 0x08 ~ 0x0C */
	__IO uint32_t PRIO;                 /* ADC convert priority register,       address offset: 0x10      */
	__IO uint32_t RESERVED1[3];         /* Reserved,                            address offset : 0x14 ~ 0x1C */
	__IO uint32_t INT;                  /* ADC Interrupt Enable Register,       address offset: 0x20      */
	__IO uint32_t STATUS;               /* ADC Interrupt Status Register,       address offset: 0x24      */
	__IO uint32_t CLEAR;                /* ADC Interrupt Status clear Register, address offset: 0x28      */
	__IO uint32_t RESERVED2;            /* Reserved,                            address offset : 0x2C     */
	__IO uint32_t TRGOVF;               /* ADC SOC trigger overflow register,   address offset: 0x30      */
	__IO uint32_t OVFCLR;               /* ADC SOC trigger overflow clear register, address offet: 0x34   */
	__IO uint32_t RESERVED3[2];         /* Reserved,                            address offset : 0x38 ~ 0x3C */
	__IO uint32_t RESULT[16];		    /* ADC Conversion Result Register,		address offset: 0x40~0x7C */
	__IO uint32_t RESERVED4[4];         /* Reserved,                            address offset : 0x80 ~ 0x8C */
	__IO uint32_t PPBRESULT[4];         /* Result after post-processing blocks, address offset: 0x90~0x9C */
	__IO uint32_t RESERVED5[4];         /* Reserved,                            address offset : 0xA0 ~ 0xAC */
	__IO uint32_t ENABLE;               /* ADC Channel Enable Register,         address offset : 0xB0      */
	__IO uint32_t OFFSETCAL;            /* Zero offset calibration,             address offset : 0xB4      */
	__IO uint32_t RESERVED6[2];         /* Reserved,                            address offset : 0xB8 ~ 0xBC */
	__IO uint32_t SOCCONF[16];          /* ADC SOC configure register, addres offset: 0xC0 ~ 0xFC */
} ADC_Type;

typedef struct
{
	__IO uint32_t CLK;                 /* SCI Clock Prescaler Register,  address offset : 0x00 */
	__IO uint32_t BDR;                 /* SCI Baudrate Register,         address offset : 0x04 */
	__IO uint32_t RESERVED0[2];        /* Reserved,                      address offset : 0x08 ~ 0x0C */
	__IO uint32_t MC;                  /* SCI Mode Control Register,     address offset : 0x10 */
	__IO uint32_t RESERVED1[3];        /* Reserved,                      address offset : 0x14 ~ 0x1C */
	__IO uint32_t MUART;               /* SCI UART Mode Register,        address offset : 0x20 */
	__IO uint32_t MSPI;                /* SCI SPI Mode Register,         address offset : 0x24 */
	__IO uint32_t MIIC;                /* SCI IIC Mode Register,         address offset : 0x28 */
	__IO uint32_t MSC;                 /* SCI Smart Card Mode Register,  address offset : 0x2C */
	__IO uint32_t MSYNC;               /* SCI Sync Mode Register,        address offset : 0x30 */
	__IO uint32_t RESERVED2[3];        /* Reserved,                      address offset : 0x34 ~ 0x3C */
	__IO uint32_t TXD;                 /* SCI TX Data Register,          address offset : 0x40*/
	__IO uint32_t RESERVED3[3];        /* Reserved,                      address offset : 0x44 ~ 0x4C */
	__IO uint32_t RXD;                 /* SCI RX Data Register,          address offset : 0x50*/
	__IO uint32_t RESERVED4[3];        /* Reserved,                      address offset : 0x54 ~ 0x5C */
	__IO uint32_t IE;                  /* SCI Interrupt Enable Register, address offset : 0x60*/
	__IO uint32_t ST;                  /* SCI Status Register,           address offset : 0x64*/
	__IO uint32_t CLR;                 /* SCI Status Clear Register,     address offset : 0x68*/
}SCI_Type;

typedef struct
{
	__IO uint32_t DT;                   /* I2C Data Register,                       address offset : 0x00 */
	__IO uint32_t SLAD;                 /* I2C Slave address Register,              address offset : 0x04 */
	__IO uint32_t DVAD;					/* I2C device address Register,             address offset : 0x08 */
	__IO uint32_t SHF;					/* I2C Slave address Register,              address offset : 0x0C */
	__IO uint32_t CTRL;                 /* I2C Control Register for each channel,   address offset : 0x10 */
	__IO uint32_t DIV;					/* I2C Clock freq division Register,        address offset : 0x14 */
	__IO uint32_t WID;                  /* I2C High and Low Level Control Register, address offset : 0x18 */
	__IO uint32_t OUTCTL;				/* I2C Out Control Register,                address offset : 0x1C */
	__IO uint32_t FIFOCTL;              /* I2C FIFO Control Register,               address offset : 0x20 */
	__IO uint32_t FIFOST;               /* I2C FIFO Status Register,                address offset : 0x24 */
	__IO uint32_t DATACNT;              /* ,                                address offset : 0x28 */
	__IO uint32_t SDADLY;               /* ,                                address offset : 0x2C */
	__IO uint32_t SR;                   /* I2C Status Register,                     address offset : 0x30 */
	__IO uint32_t RESERVED1[3];         /* Reserved,                                address offset : 0x34~0x3C */
	__IO uint32_t INTEN;				/* I2C Interrupt Control Register,          address offset : 0x40 */
	__IO uint32_t INTSR;                /* I2C Interrupt Status Register,           address offset : 0x44 */
	__IO uint32_t TMOUTCFG;
	__IO uint32_t TMOUTCNT;
} I2C_Type;

typedef struct
{
	__IO uint32_t CTRL;                 /* CRC Control Register, address offset: 0x00 */
	__IO uint32_t DT;                   /* CRC Data Register,    address offset: 0x04 */
} CRC_Type;

typedef struct
{
	__IO uint32_t CTRL;                 /* Base Timer Control Register,            address offset: 0x00 */
	__IO uint32_t LD;                   /* Base Timer 16bit Time Load Register,    address offset: 0x04 */
	__IO uint32_t CNT;                  /* Base Timer 16bit Time counter Register, address offset: 0x08 */
} BTIM_Type;

typedef struct
{
	__IO uint32_t ST;                   /* Base Timer Common Status Register, address offset: 0x00 */
} BTIM_CommType;

typedef struct
{
	__IO uint32_t INTEN;                /* IOW Interrupt Enable Register, address offset: 0x00 */
	__IO uint32_t CTRL;                 /* IOW Control Register,          address offset: 0x04 */
	__IO uint32_t EDG;                  /* IOW Edge Register,             address offset: 0x08 */
	__IO uint32_t INTSR;                /* IOW Interrupt Status Register, address offset: 0x0C */
} IOW_Type;

typedef struct
{
	__IO uint32_t CTRL;                  /* WWDG Control Register,       address offset: 0x00        */
	__IO uint32_t LD;                    /* WWDG Load Register,          address offset: 0x04        */
	__IO uint32_t WLD;                   /* WWDG Window Load Register,   address offset: 0x08        */
	__IO uint32_t ST;                    /* WWDG Status Register,        address offset: 0x0C        */
	__IO uint32_t CNT;                   /* WWDG Counter Value Register, address offset: 0x10        */
	__IO uint32_t RESERVED[3];           /* Reserved,                    address offset: 0x14 ~ 0x1C */
	__IO uint32_t KEY;                   /* WWDG Key Register,           address offset: 0x20        */
} WWDG_Type;

typedef struct
{
	__IO uint32_t CTRL;                  /* TMU Control Register,          address offset: 0x0 */
	__IO uint32_t RESERVED0[3];          /* Reserved,                      address offset: 0x4 ~ 0xC */
    __IO uint32_t AGLF;                  /* TMU Angle Float Register,      address offset: 0x10 */
	__IO uint32_t RESERVED1;             /* Reserved,                      address offset: 0x14 */
    __IO uint32_t COSF;                  /* TMU Cosin Float Register,      address offset: 0x18 */
    __IO uint32_t SINF;                  /* TMU Sine Float Register,       address offset: 0x1C */
	__IO uint32_t AGLB;                  /* TMU Angle Binary Register,     address offset: 0x20 */
	__IO uint32_t DVIT;                  /* TMU Deviation Binary Register, address offset: 0x24 */
	__IO uint32_t COSB;                  /* TMU Cosin Binary Register,     address offset: 0x28 */
	__IO uint32_t SINB;                  /* TMU Sine Binary Register,      address offset: 0x2C */
	__IO uint32_t ST;                    /* TMU Status Register,           address offset: 0x30 */
}TMU_Type;

typedef struct
{
	__IO uint32_t RESERVED0;             /* Reserved,                            address offset: 0x00        */
	__IO uint32_t SWT;                   /* EBUS Software Trigger Register       address offset: 0x04        */
	__IO uint32_t CPU;                   /* EBUS Trigger CPU Register,           address offset: 0x08        */
	__IO uint32_t RESERVED[13];           /* Reserved,                            address offset: 0x0C ~ 0x1C */
	__IO uint32_t CHN[16];                /* EBUS Channel Configuration Register, address offset: 0x20 ~ 0x3C */
} EBUS_Type;

typedef struct
{
	__IO uint32_t IOD;                   /* Port IO Detection Register,              address offset: 0x00 */
	__IO uint32_t STA;                   /* Port IO Interrupt Status Register,       address offset: 0x04 */
	__IO uint32_t CLR;                   /* Port IO Interrupt Status Clear Register, address offset: 0x08 */
	__IO uint32_t DBEN;
	__IO uint32_t SRCSEL1;
	__IO uint32_t SRCSEL2;
} IOPORT_Type;

typedef struct
{
	__IO uint32_t ADDR;                  /* FLASH Address Register, address offset: 0x00 */
	__IO uint32_t DATA;                  /* FLASH Data Register,    address offset: 0x04 */
	__IO uint32_t ST;                    /* FLASH Status Register,  address offset: 0x08 */
	__IO uint32_t SYS;                   /* FLASH System Register,  address offset: 0x0C */
	__IO uint32_t CTRL;                  /* FLASH Control Register, address offset: 0x10 */
	__IO uint32_t CMD;                   /* FLASH Command Register, address offset: 0x14 */
	__IO uint32_t KEY;                   /* FLASH Key Register,     address offset: 0x18 */
	__IO uint32_t PROT;                  /* FLASH Protect Register, address offset: 0x1C */
} FLASH_Type;

typedef struct
{
	__IO uint32_t CTRL;                  /* ICACHE Control Register, address offset: 0x00 */
	__IO uint32_t ST;                    /* ICACHE Status Register,  address offset: 0x04 */
	__IO uint32_t RESERVED0[3];          /* Reserved 0, address offset: 0x08 */
	__IO uint32_t HIT;                   /* ICACHE Hit Statistics Register, address offset: 0x14 */
	__IO uint32_t MISS;                  /* ICACHE Miss Statistics Register, address offset: 0x18 */
} ICACHE_Type;

typedef struct
{
	__IO uint32_t CTRL;                  /* DCACHE Control Register, address offset: 0x00 */
	__IO uint32_t ST;                    /* DCACHE Status Register,  address offset: 0x04 */
	__IO uint32_t RESERVED0[3];          /* Reserved 0, address offset: 0x08 */
	__IO uint32_t HIT;                   /* DCACHE Hit Statistics Register, address offset: 0x14 */
	__IO uint32_t MISS;                  /* DCACHE Miss Statistics Register, address offset: 0x18 */
} DCACHE_Type;

typedef struct
{
	__IO uint32_t RESERVED0[4];          /* Reserved,                                address offset: 0x00 ~ 0xC  */
	__IO uint32_t DT;                    /* GPIO Data Register,                      address offset: 0x10        */
	__IO uint32_t DO;                    /* GPIO Data Output Register,               address offset: 0x14        */
	__IO uint32_t DOSR;                  /* GPIO Data Output Set and Reset Register, address offset: 0x18        */
	__IO uint32_t TOGL;                  /* GPIO Data Output Toggle Register,        address offset: 0x1C        */
	__IO uint32_t OES;                   /* GPIO AS Output Enable Register,          address offset: 0x20        */
	__IO uint32_t OEC;                   /* GPIO AS Output Disable Register,         address offset: 0x24        */
	__IO uint32_t RESERVED1[5];          /* Reserved,                                address offset: 0x28 ~ 0x38 */
	__IO uint32_t EVT;                   /* GPIO AS Event Configuration Register,    address offset: 0x3C        */
} GPIO_Type;

typedef struct
{
	__IO uint32_t CTRL;                  /* DMA Control Register,                    address offset: 0x00 */
	__IO uint32_t ADDR_KEEP;             /* DMA Address Keep Register,               address offset: 0x04 */
	__IO uint32_t CH_SW_REQ;             /* DMA Control Register,                    address offset: 0x08 */
	__IO uint32_t RESERVED0;             /* Reserved,                                address offset: 0x0C */
	__IO uint32_t INT_EN;                /* DMA Interrupt Enable Register,           address offset: 0x10 */
	__IO uint32_t INT_FLAG;              /* DMA Interrupt Flag Register,             address offset: 0x14 */
	__IO uint32_t STATUS;                /* DMA Status Register,                     address offset: 0x18 */
	__IO uint32_t ERR;                   /* DMA Error Status Register,               address offset: 0x1C */
	__IO uint32_t MASK_REQ_SET;          /* DMA Channel Mask Request Set Register,   address offset: 0x20 */
	__IO uint32_t MASK_REQ_CLR;          /* DMA Channel Mask Request Clear Register, address offset: 0x24 */
	__IO uint32_t CH_EN_SET;             /* DMA Channel Enable Set Register,         address offset: 0x28 */
	__IO uint32_t CH_EN_CLR;             /* DMA Channel Enable Clear Register,       address offset: 0x2C */
	__IO uint32_t CH_PRI_SET;            /* DMA Channel Priority Set Register,       address offset: 0x30 */
	__IO uint32_t CH_PRI_CLR;            /* DMA Channel Priority Clear Register,     address offset: 0x34 */
	__IO uint32_t RESERVED1[2];          /* Reserved,                                address offset: 0x38 */
	__IO uint32_t CH_SEL;                /* DMA Channel Select Register,             address offset: 0x40 */
	__IO uint32_t RESERVED2[3];          /* Reserved,                                address offset: 0x44 */
	__IO uint32_t EBUS_CTRL;             /* DMA Control Register,                    address offset: 0x50 */
} DMA_Type;

typedef struct
{
	__IO uint32_t DR;                    /* SPI Data Register,                     address offset: 0x00  */
	__IO uint32_t RESERVED1[3];          /* Reserved,                              address offset: 0x04 ~ 0x0C */
	__IO uint32_t CFG;                   /* SPI Configuration Register,            address offset: 0x10  */
	__IO uint32_t FRT;                   /* SPI Frame Format Register,             address offset: 0x14  */
	__IO uint32_t FRW;                   /* SPI Frame Width Register,              address offset: 0x18  */
	__IO uint32_t DLY;                   /* SPI Sample Delay Register,             address offset: 0x1C  */
	__IO uint32_t BRD;                   /* SPI Baudrate Register,                 address offset: 0x20  */
	__IO uint32_t CSCTL;                 /* SPI CS Pin Control Register,           address offset: 0x24  */
	__IO uint32_t CSTIM;                 /* SPI CS Pin Timing Register,            address offset: 0x28  */
	__IO uint32_t SCK;                   /* SPI SCK Pin Control Register,          address offset: 0x2C */
	__IO uint32_t FFCTL;                 /* SPI FIFO Control Register,             address offset: 0x30  */
	__IO uint32_t FFST;                  /* SPI FIFO Status Register,              address offset: 0x34  */
	__IO uint32_t ST;                    /* SPI Status Register,                   address offset: 0x38  */
	__IO uint32_t RESERVED3;             /* Reserved,                              address offset: 0x3C */
	__IO uint32_t DMATX;                 /* SPI DMA TX Register,                   address offset: 0x40  */
	__IO uint32_t DMARX;                 /* SPI DMA RX Register,                   address offset: 0x44  */
	__IO uint32_t RESERVED4[2];          /* Reserved,                              address offset: 0x48 ~ 0x4C */
	__IO uint32_t INTR;                  /* SPI Interrupt Control Register,        address offset: 0x50  */
	__IO uint32_t INTST;                 /* SPI Interrupt Status Register,         address offset: 0x54  */
} SPI_Type;

typedef struct
{
	__IO uint32_t RESERVED0[2];          /* Reserved,                     address offset: 0x00 ~ 0x04   */
	__IO uint32_t ID;                    /* MCU chip ID,                  address offset: 0x08          */
    __IO uint32_t RESERVED1;             /* Reserved,                     address offset: 0x0C          */
	__IO uint32_t RSTS;                  /* System Reset & Status,        address offset: 0x10          */
	__IO uint32_t RESERVED2;             /* Reserved         ,            address offset: 0x14          */
	__IO uint32_t PMS;                   /* Power mode Status,            address offset: 0x18          */
	__IO uint32_t RESERVED3;             /* Reserved         ,            address offset: 0x1C          */
	__IO uint32_t NMI;                   /* NMI Status       ,            address offset: 0x20          */
	__IO uint32_t RESERVED4[15];         /* Reserved,                     address offset: 0x24 ~ 0x5C   */
	__IO uint32_t SC;                    /* System Configure Control,     address offset: 0x60          */
	__IO uint32_t RESERVED5[3];          /* Reserved,                     address offset: 0x64 ~ 0x6C   */
	__IO uint32_t PMC;                   /* Power mode Control,           address offset: 0x70          */
	__IO uint32_t RESERVED6[7];          /* Reserved,                     address offset: 0x74 ~ 0x8C   */
	__IO uint32_t RESERVED7;             /* Reserved,                     address offset: 0x90          */
	__IO uint32_t TICKCALIB;             /* System tick calibration,      address offset: 0x94          */
	__IO uint32_t RESERVED8[25];         /* Reserved,                     address offset: 0x98 ~ 0xF8   */
	__IO uint32_t KEY;                   /* Reserved,                     address offset: 0xFC          */
	__IO uint32_t PMU;                   /* Analog PMU Control,           address offset: 0x100         */
	__IO uint32_t RESERVED9[3];          /* Reserved,                     address offset: 0x104 ~ 0x10C */
	__IO uint32_t IRC[3];                /* IRC Control,                  address offset: 0x110 ~ 0X118 */
	__IO uint32_t RESERVED10[5];         /* Reserved,                     address offset: 0x11C ~ 0x12C */
	__IO uint32_t HOSC;                  /* HOSC Control,                 address offset: 0x130         */
	__IO uint32_t RESERVED11[3];          /* Reserved,                     address offset: 0x134 ~ 0x13C */
	__IO uint32_t LOSC;                  /* LOSC Control,                 address offset: 0x140         */
	__IO uint32_t RESERVED12[3];         /* Reserved,                     address offset: 0x144 ~ 0x14C */
	__IO uint32_t PLL;                   /* PLL,                          address offset: 0x150         */
	__IO uint32_t RESERVED13[3];         /* Reserved,                     address offset: 0x154 ~ 0x15C */
	__IO uint32_t TMPS;                  /* Temperature Sensor,           address offset: 0x160         */
	__IO uint32_t RESERVED14[15];        /* Reserved,                     address offset: 0x164 ~ 0x19C */
	__IO uint32_t FMNT0;                 /* Frequency Monitor 0,          address offset: 0x1A0         */
	__IO uint32_t FMNT1;                 /* Frequency Monitor 1,          address offset: 0x1A4         */
	__IO uint32_t FMNT2;                 /* Frequency Monitor 2,          address offset: 0x1A8         */
	__IO uint32_t RESERVED15;            /* Reserved,                     address offset: 0x1AC         */
	__IO uint32_t CMNT;                  /* Clock Monitor,                address offset: 0x1B0         */
	__IO uint32_t RESERVED16[19];        /* Reserved,                     address offset: 0x1B4 ~ 0x1FC */
	__IO uint32_t LVD1CTRL;              /* Voltage Detector Control,     address offset: 0x200         */
	__IO uint32_t LVD1STS;               /* Voltage Detector Status,      address offset: 0x204         */
	__IO uint32_t LVD2CTRL;              /* Voltage Detector Control,     address offset: 0x208         */
	__IO uint32_t LVD2STS;               /* Voltage Detector Status,      address offset: 0x20C         */
	__IO uint32_t RESERVED17[28];        /* Reserved,                     address offset: 0x210 ~ 0x27C */
	__IO uint32_t RC0;                   /* Reset Control 0,              address offset: 0x280         */
	__IO uint32_t RC1;                   /* Reset Control 1,              address offset: 0x284         */
	__IO uint32_t RESERVED18[2];         /* Reserved,                     address offset: 0x288 ~ 0x28C */
	__IO uint32_t CG0;                   /* Clock Gate 0,                 address offset: 0x290         */
	__IO uint32_t CG1;                   /* Clock Gate 1,                 address offset: 0x294         */
	__IO uint32_t RESERVED19[2];         /* Reserved,                     address offset: 0x298 ~ 0x29C */
	__IO uint32_t CSEL0;                 /* Clock Selection 0,            address offset: 0x2A0         */
	__IO uint32_t CSEL1;                 /* Clock Selection 1,            address offset: 0x2A4         */
	__IO uint32_t CSEL2;                 /* Clock Selection 2,            address offset: 0x2A8         */
	__IO uint32_t CSEL3;                 /* Clock Selection 3,            address offset: 0x2AC         */
	__IO uint32_t RESERVED20[4];         /* Reserved,                     address offset: 0x2B0 ~ 0x2BC */
	__IO uint32_t NF[1];                 /* Noise Filter Clock Selection, address offset: 0x2C0         */
	__IO uint32_t RESERVED21[15];        /* Reserved,                     address offset: 0x2C4 ~ 0x2FC */
	__IO uint32_t IOPDF[2];              /* IOPAD Filter 0~1,             address offset: 0x300 ~ 0x304 */
	__IO uint32_t RESERVED22[6];         /* Reserved,                     address offset: 0x308 ~ 0x31C */
	__IO uint32_t PDEN[2];               /* PAD Input Enable,             address offset: 0x320 ~ 0x324 */
	__IO uint32_t RESERVED23[6];         /* Reserved,                     address offset: 0x328 ~ 0x33C */
	__IO uint32_t PDDRV[2];              /* PAD Driving,                  address offset: 0x340 ~ 0x344 */
	__IO uint32_t RESERVED24[6];         /* Reserved,                     address offset: 0x348 ~ 0x35C */
	__IO uint32_t PDPLUP[2];             /* PAD pull-up,                  address offset: 0x360 ~ 0x364 */
	__IO uint32_t RESERVED25[6];         /* Reserved,                     address offset: 0x368 ~ 0x37C */
	__IO uint32_t PDPLDN[2];             /* PAD pull-Down,                address offset: 0x380 ~ 0x384 */
	__IO uint32_t RESERVED26[6];         /* Reserved,                     address offset: 0x388 ~ 0x39C */
	__IO uint32_t IOMDSEL[4];            /* IO mode Selection,            address offset: 0x3A0 ~ 0x3AC */
	__IO uint32_t RESERVED27[12];        /* Reserved,                     address offset: 0x3B0 ~ 0x3DC */
	__IO uint32_t ALTF[8];               /* IO alternate Function,        address offset: 0x3E0 ~ 0x3FC */
	__IO uint32_t RESERVED28[24];        /* Reserved,                     address offset: 0x400 ~ 0x45C */
	__IO uint32_t PDOD[3];               /* PAD Open Drain,               address offset: 0x460 ~ 0x464 */
	__IO uint32_t RESERVED29[21];        /* Reserved,                     address offset: 0x468 ~ 0x4BC */
	__IO uint32_t DBG;                   /* Debug,                        address offset: 0x4C0         */
	__IO uint32_t RESERVED30[15];        /* Reserved,                     address offset: 0x4C4 ~ 0x4FC */
	__IO uint32_t SRMIS;                 /* SRAM Interrupt Status,        address offset: 0x500         */
	__IO uint32_t SADDR;                 /* SRAM parity Error address,    address offset: 0x504         */
} SysConfig_Type;

typedef struct
{
	__IO uint32_t CR0;                   /* TIM Control Register0,                 		address offset: 0x00        */
	__IO uint32_t CR1;                   /* TIM Control Register1,                 		address offset: 0x04        */
	__IO uint32_t SMCR;         		 /* TIM Mode Control Register,             		address offset: 0x08        */
	__IO uint32_t DIER;                  /* TIM DMA and Interrupt Enable Register, 		address offset: 0x0C        */
	__IO uint32_t SR;                    /* TIM State Register,                    		address offset: 0x10        */
	__IO uint32_t EGR;                   /* TIM Event Generate Register,				address offset: 0x14        */
	__IO uint32_t CCMR0;                 /* TIM Capture/Compare Mode Register,     		address offset: 0x18        */
	__IO uint32_t CCMR1;                 /* TIM Capture/Compare Mode Register,     		address offset: 0x1C        */
	__IO uint32_t CCMR2;         		 /* TIM Capture/Compare Mode Register,     		address offset: 0x20        */
	__IO uint32_t CCER;         		 /* TIM Capture/Compare Mode Enable Register,    address offset: 0x24        */
	__IO uint32_t CNT;         			 /* TIM Counter Register,        				address offset: 0x28        */
	__IO uint32_t PSC;            		 /* TIM Prescaler,	                            address offset: 0x2C        */
	__IO uint32_t ARR;        			 /* TIM Auto-Reload Register, 		     		address offset: 0x30        */
	__IO uint32_t RCR;        			 /* TIM Retry Counter Register,   				address offset: 0x34        */
	__IO uint32_t CCR0;        			 /* TIM Capture/Compare Register0,           	address offset: 0x38        */
	__IO uint32_t CCR1;        			 /* TIM Capture/Compare Register1,           	address offset: 0x3C        */
	__IO uint32_t CCR2;        			 /* TIM Capture/Compare Register2,           	address offset: 0x40        */
	__IO uint32_t CCR3;        			 /* TIM Capture/Compare Register3,           	address offset: 0x44        */
	__IO uint32_t CCR4;        			 /* TIM Capture/Compare Register4,           	address offset: 0x48        */
	__IO uint32_t CCR5;        			 /* TIM Capture/Compare Register5,           	address offset: 0x4C        */
	__IO uint32_t BDTR;                  /* TIM Break And Dead-Time Register,	        address offset: 0x50        */
	__IO uint32_t RESERVED[3];			 /* RESERVED,									address offset: 0x54 -- 0x5C*/
	__IO uint32_t ECR;                 	 /* TIM Event Control Register,	        		address offset: 0x60        */
	__IO uint32_t SETKR;                 /* TIM Set Key Register,	       				address offset: 0x64        */
} TIMA_Type;

typedef struct
{
	__IO uint32_t CON;					 /* TIM Control Register,          				address offset: 0x00        */
	__IO uint32_t STAT;					 /* TIM status Register,           				address offset: 0x04        */
	__IO uint32_t IOCON;				 /* TIM IO Control Register,       				address offset: 0x08        */
	__IO uint32_t EVT;				 	 /* TIM Event Register,            				address offset: 0x0C        */
	__IO uint32_t FPCI;				 	 /* TIM FPCI Register,             				address offset: 0x10        */
	__IO uint32_t CLPCI;				 /* TIM CLPCI Register,                 		address offset: 0x14        */
	__IO uint32_t FFPCI;				 /* TIM FFPCI Register,                 		address offset: 0x18        */
	__IO uint32_t SPCI;				 	 /* TIM SPCI Register,                 			address offset: 0x1C        */
	__IO uint32_t LEB;				 	 /* TIM Leading Edge Blanking Register, 		address offset: 0x20        */
	__IO uint32_t PHASE;				 /* TIM Phase Register,            				address offset: 0x24        */
	__IO uint32_t DC;				 	 /* TIM Duty Cycle Register,        			address offset: 0x28        */
	__IO uint32_t DCA;				 	 /* TIM Duty Cycle Adjustment Register,			address offset: 0x2C        */
	__IO uint32_t PER;				 	 /* TIM Period Register,           				address offset: 0x30        */
	__IO uint32_t TRIGA;				 /* TIM TrigA,                 					address offset: 0x34        */
	__IO uint32_t TRIGB;				 /* TIM TrigB,                 					address offset: 0x38        */
	__IO uint32_t TRIGC;				 /* TIM TrigC,                 					address offset: 0x3C        */
	__IO uint32_t DT0;				 	 /* TIM Deadtime0,                 				address offset: 0x40        */
	__IO uint32_t CAP;				 	 /* TIM Capture Register,                 		address offset: 0x44        */
} TIMH_Type;

typedef struct
{
	__IO uint32_t CR0;                          /* TIM Control Register0,                 		address offset: 0x00        */
	__IO uint32_t CR1;                  		/* TIM Control Register1,                 		address offset: 0x04        */
	__IO uint32_t SMCR;         				/* TIM Mode Control Register,             		address offset: 0x08        */
	__IO uint32_t DIER;        				    /* TIM DMA and Interrupt Enable Register, 		address offset: 0x0C        */
	__IO uint32_t SR;      						/* TIM State Register,                    		address offset: 0x10        */
	__IO uint32_t EGR;       				    /* TIM Event Generate Register,					address offset: 0x14        */
	__IO uint32_t CCMR0;       					/* TIM Capture/Compare Mode Register,     		address offset: 0x18        */
	__IO uint32_t CCMR1;         				/* TIM Capture/Compare Mode Register,     		address offset: 0x1C        */
	__IO uint32_t CCER;         				/* TIM Capture/Compare Mode Enable Register,    address offset: 0x20        */
	__IO uint32_t CNT;         				  	/* TIM Counter Register,        				address offset: 0x24        */
	__IO uint32_t PSC;            				/* TIM Prescaler,	                            address offset: 0x28        */
	__IO uint32_t ARR;        				  	/* TIM Auto-Reload Register, 		     		address offset: 0x2C        */
	__IO uint32_t RESERVED0;			        /* Reserved,                            		address offset: 0x30 		*/
	__IO uint32_t CCR0;        					/* TIM Capture/Compare Register0,           	address offset: 0x34        */
	__IO uint32_t CCR1;        					/* TIM Capture/Compare Register1,           	address offset: 0x38        */
	__IO uint32_t CCR2;        					/* TIM Capture/Compare Register2,           	address offset: 0x3C        */
	__IO uint32_t CCR3;        					/* TIM Capture/Compare Register3,           	address offset: 0x40        */
	__IO uint32_t RESERVED[8];         		    /* Reserved,                            		address offset: 0x44 ~ 0x60 */
	__IO uint32_t SETKR;                 		/* TIM Break And Dead-Time Register,	        address offset: 0x64        */
} TIMG_Type;

typedef struct
{
    __IO uint32_t CR;                    /* TIM Control Register,                address offset: 0x00        */
    __IO uint32_t MCR;         		     /* TIM Mode Control Register,           address offset: 0x04        */
    __IO uint32_t CCMR;                  /* TIM Capture/Compare Mode Register,   address offset: 0x08        */
    __IO uint32_t RESERVED0[5];          /* Reserved,                            address offset: 0x0C ~ 0x1C */
    __IO uint32_t CCER;       		     /* TIM Capture/Compare Enable Register, address offset: 0x20        */
	__IO uint32_t RESERVED1[15];         /* Reserved,                            address offset: 0x24 ~ 0x5C */
	__IO uint32_t CNT;                   /* TIM Counter Register,                address offset: 0x60        */
	__IO uint32_t PSC;                   /* TIM Prescaler Register,              address offset: 0x64        */
	__IO uint32_t ARR;                   /* TIM Auto-Reload Register,            address offset: 0x68        */
	__IO uint32_t RCR;					 /* TIM repeat Counter Register,         address offset: 0x6C        */
	__IO uint32_t CCR1;                  /* TIM Capture/Compare Register 1,      address offset: 0x70        */
	__IO uint32_t CCR2;                  /* TIM Capture/Compare Register 2,      address offset: 0x74        */
	__IO uint32_t RESERVED3[6];          /* Reserved,                            address offset: 0x78 ~ 0x8C */
	__IO uint32_t BDTR;                  /* TIM Break And Dead-Time Register,    address offset: 0x90        */
	__IO uint32_t SETKR;				 /* TIM Config Key Register,             address offset: 0x94        */
	__IO uint32_t RESERVED4[6];          /* Reserved,                            address offset: 0x98 ~ 0xAC */
	__IO uint32_t EGR;                   /* TIM Event Generation Register,       address offset: 0xB0        */
	__IO uint32_t RESERVED5[3];          /* Reserved,                            address offset: 0xB4 ~ 0xBC */
	__IO uint32_t IER;                   /* TIM Interrupt Enable Register,       address offset: 0xC0        */
	__IO uint32_t SR;                    /* TIM Status Register,                 address offset: 0xC4        */
} TIMS_Type;

typedef struct
{
	__IO uint32_t TIMA0_IMCR;            /* TIMA0 Input Repetition Control Register,      address offset: 0x00      */
	__IO uint32_t RESERVED0[2];          /* Reserved,                            		  address offset: 0x4 ~ 0x8 */
	__IO uint32_t TIMG0_IMCR;            /* TIMG0 Input Repetition Control Register,      address offset: 0x0C      */
	__IO uint32_t TIMG1_IMCR;			 /* TIMG1 Input Repetition Control Register,	  address offset: 0x10		*/
	__IO uint32_t TIMG2_IMCR;			 /* TIMG2 Input Repetition Control Register,	  address offset: 0x14		*/
	__IO uint32_t TIMS_IMCR;			 /* TIMS  Input Repetition Control Register,	  address offset: 0x18		*/
	__IO uint32_t RESERVED1[9];          /* Reserved,                            		  address offset: 0x1C ~ 3C */
	__IO uint32_t TIMA_ITCR;			 /* TIMA  Internal Trigger  Control Register,	  address offset: 0x40		*/
	__IO uint32_t TIMG_ITCR;			 /* TIMG  Internal Trigger  Control Register,	  address offset: 0x44		*/
	__IO uint32_t TIMS_ITCR;			 /* TIMS  Internal Trigger  Control Register,	  address offset: 0x48		*/
	__IO uint32_t RESERVED2;         	 /* Reserved,                            		  address offset: 0x4C		*/
	__IO uint32_t TIMA0_SBCR;            /* TIMA0 System Broken Circuit Control Register, address offset: 0x50      */
	__IO uint32_t TIMA1_SBCR;			 /* TIMA1 System Broken Circuit Control Register, address offset: 0x54		*/
	__IO uint32_t TIMH_SBCR;			 /* TIMA2 System Broken Circuit Control Register, address offset: 0x58		*/
	__IO uint32_t TIMS0_SBCR;			 /* TIMS0 System Broken Circuit Control Register, address offset: 0x5C		*/
	__IO uint32_t TIMS1_SBCR;			 /* TIMS1 System Broken Circuit Control Register, address offset: 0x60		*/
	__IO uint32_t TIMS2_SBCR;			 /* TIMS2 System Broken Circuit Control Register, address offset: 0x64		*/
	__IO uint32_t TIMS3_SBCR;			 /* TIMS3 System Broken Circuit Control Register, address offset: 0x68		*/
	__IO uint32_t RESERVED3;          	 /* Reserved,                            		  address offset: 0x6C		*/
	__IO uint32_t TIMA_ETCR0;			 /* TIMA  Extern Trigger Control Register0,		  address offset: 0x70		*/
	__IO uint32_t TIMA_ETCR1;			 /* TIMA  Extern Trigger Control Register1,		  address offset: 0x74		*/
	__IO uint32_t RESERVED4[2];          /* Reserved,                            		  address offset: 0x78 ~ 7C	*/
	__IO uint32_t TIMA_OCCR;			 /* TIMA  Clear Control Register,				  address offset: 0x80		*/
	__IO uint32_t TIMG_OCCR;			 /* TIMG  Clear Control Register,				  address offset: 0x84		*/
	__IO uint32_t TIMS_OCCR;			 /* TIMS  Clear Control Register,				  address offset: 0x88	    */
} TIM_Common_Type;

typedef struct
{
	__IO uint32_t PCLKCON;            	/* PWM Control Register,					      address offset: 0x00      */
	__IO uint32_t FSCL;          		/* PWM frequency scale Register,                  address offset: 0x04		*/
	__IO uint32_t FSMINPER;            	/* PWM frequency scaling Minimum period Register, address offset: 0x08      */
	__IO uint32_t MPHASE;			 	/* Master Phase Register,	      				  address offset: 0x0c		*/
	__IO uint32_t MDC;			 		/* Master Duty Cycle Register,	      			  address offset: 0x10		*/
	__IO uint32_t MPER;			 		/* Master Period Register,	      				  address offset: 0x14		*/
	__IO uint32_t LFSR;          		/* Linear Feedback Shift Register,                address offset: 0x18		*/
	__IO uint32_t CMBTRIG;				/* CMBTRIG Register,	  						  address offset: 0x1c		*/
	__IO uint32_t LOGCONA;			 	/* LOGCONy Register,	  						  address offset: 0x20		*/
	__IO uint32_t LOGCONB;			 	/* LOGCONy Register,	  						  address offset: 0x24		*/
	__IO uint32_t RESERVED[6];          /* Reserved,                            		  address offset: 0x28 ~ 3C */
	__IO uint32_t PWMEVTA;			 	/* PWM Event Output_A Register,	  				  address offset: 0x40		*/
	__IO uint32_t PWMEVTB;			 	/* PWM Event Output_B Register,	  				  address offset: 0x44		*/
	__IO uint32_t PWMEVTC;			 	/* PWM Event Output_B Register,	  				  address offset: 0x48		*/
	__IO uint32_t RESERVED1[5];         /* Reserved,                            		  address offset: 0x4C ~ 5c	*/
	__IO uint32_t HRCON;				/* HR Mode Control Register,	  				  address offset: 0x60		*/
} TIMH_Common_Type;

/* Controller Area Network */
typedef struct
{
	__IO uint32_t ID;              /* CAN Identification Register, address offset: 0x00 */
	__IO uint32_t RESERVED0;       /* Reserved, address offset: 0x04*/
	__IO uint32_t CTRL;            /* CAN Control Register, address offset: 0x08 */
	__IO uint32_t RESERVED1;       /* Reserved, address offset: 0x0C */
	__IO uint32_t CFG;             /* CAN Configuration Register, address offset: 010 */
	__IO uint32_t IE;              /* CAN Interrupt Enable Register, address offset: 0x14 */
	__IO uint32_t BUSTIM;          /* CAN Bus Timing Register, address offset: 0x18 */
	__IO uint32_t RESERVED2;       /* Reserved, address offset: 0x1C */
	__IO uint32_t ACPTCODE0;       /* CAN Acceptance Code Register, address offset: 0x20 */
	__IO uint32_t ACPTMASK0;       /* CAN Acceptance Mask Register, address offset: 0x24 */
	__IO uint32_t ACPTCODE1;       /* CAN Acceptance Code Register, address offset: 0x28 */
	__IO uint32_t ACPTMASK1;       /* CAN Acceptance Mask Register, address offset: 0x2C */
	__IO uint32_t ERRWARNTHR;      /* CAN error warning threshold register, address offset: 0x30 */
	__IO uint32_t RESERVED4;	   /* Reserved, address offset: 0x34 */
	__IO uint32_t OUTCTRL;         /* CAN Output Control Register, address offset: 0x38 */
	__IO uint32_t RESERVED5;       /* Reserved, address offset: 0x3C */
	__IO uint32_t ST;              /* CAN Status Register, address offset: 0x40 */
	__IO uint32_t INTST;           /* CAN Interrupt Register, address offset: 0x44 */
	__IO uint32_t RXBUFST;         /* CAN Receive Buffer Status Register, address offset: 0x48 */
	__IO uint32_t RESERVED6;       /* Reserved, address offset: 0x4C */
	__IO uint32_t TXFRMINFO;       /* CAN TX Frame Information Register, address offset: 0x50 */
	__IO uint32_t TXID;            /* CAN TX Identifier Register, address offset: 0x54 */
	__IO uint32_t TXDATA0;         /* CAN TX Data Byte Part 0 Register, address offset: 0x58 */
	__IO uint32_t TXDATA1;         /* CAN TX Data Byte Part 1 Register, address offset: 0x5C */
	__IO uint32_t RXFRMINFO;       /* CAN RX Frame Information Register, address offset: 0x60 */
	__IO uint32_t RXID;            /* CAN RX Identifier Register, address offset: 0x64 */
	__IO uint32_t RXDATA0;         /* CAN RX Data Byte Part 0 Register, address offset: 0x68 */
	__IO uint32_t RXDATA1;         /* CAN RX Data Byte Part 1 Register, address offset: 0x6C */
	__IO uint32_t ARBITLOST;       /* CAN Arbitration lost bit register, address offset: 0x70 */
	__IO uint32_t ERRINFO;         /* CAN Error Information Register, address offset: 0x74 */
	__IO uint32_t ERRCNT;          /* CAN Error Counter Register, address offset: 0x78 */
} CAN_Type;

/* ACMP Unit */
typedef struct
{
	__IO uint32_t ACMP_CTRL;			   /* ACMP Control Register, address offset: 0x00 */
	__IO uint32_t ACMP_PN_SEL;	           /* ACMP Input Select Register, address offset: 0x04 */
	__IO uint32_t ACMP_FLT_CTRL;           /* ACMP Filter Control Register, address offset: 0x08 */
	__IO uint32_t ACMP_ST;       		   /* ACMP State Register, address offset: 0x0C*/
	__IO uint32_t DAC_CTRL;		  		   /* DAC  Control Register, address offset: 0x10 */
	__IO uint32_t RESERVED0;               /* RESERVED, address offset: 0x14*/
	__IO uint32_t DAC_VAL_T;		   	   /* DAC Data Register, address offset: 0x18 */
	__IO uint32_t DAC_SDAT_T;   		   /* DAC Toggle Data Register, address offset: 0x1C*/
	__IO uint32_t DAC_RAMP_DLY_R;		   /* DAC Ramp Delay Register, address offset: 0x20 */
	__IO uint32_t DAC_RAMP_STEP_R;		   /* DAC Ramp Step Register, address offset: 0x24 */
	__IO uint32_t DAC_RAMP_SLOPE_R;		   /* DAC Ramp Slope Register, address offset: 0x28 */
	__IO uint32_t RESERVED1;               /* RESERVED1, address offset: 0x2C*/
	__IO uint32_t ACMP_ASEL;			   /* ACMP Input IO Select Register, address offset: 0x30 */
} ACMP_Type;

#define FLASH_BASE                  (0x08000000UL)  /* FLASH base address */
#define SRAM_BASE                   (0x20000000UL)  /* SRAM base address */
#define SRAM_SIZE                   (0x00008000UL)  /* SRAM size */
#define PERIPH_BASE                 (0x40000000UL)  /* Peripheral base address */
#define GPIOPORT_BASE               (0x50000000UL)  /* IOPORT base address */
#define SRAM_SIZE_MAX               (0x00008000UL)  /* maximum SRAM size (up to 32 KBytes) */

/* Flash mapping */
#define FLASH_CODE_BASE              FLASH_BASE
#define FLASH_CODE_SIZE              (0x00040000UL)
#define FLASH_CODE_PAGESIZE          (0x00000200UL)
#define FLASH_CODE_MIN_SECTNUM       0
#define FLASH_CODE_MAX_SECTNUM       511
#define FLASH_CODE_TOTAL_PAGES       512

#define FLASH_BOOTROM_BASE           (0x1FFF0200UL)
#define FLASH_BOOTROM_SIZE           (0x00000C00UL)
#define FLASH_BOOTROM_PAGESIZE       (0x00000200UL)
#define FLASH_BOOTROM_MIN_SECTNUM    0
#define FLASH_BOOTROM_MAX_SECTNUM    5

#define FLASH_DATA_BASE              (0x1FFF1000UL)
#define FLASH_DATA_SIZE              (0x00001000UL)
#define FLASH_DATA_PAGESIZE          (0x00000200UL)
#define FLASH_DATA_MIN_SECTNUM       0
#define FLASH_DATA_MAX_SECTNUM       7


/* Peripheral memory map */
#define APBPERIPH_BASE        (PERIPH_BASE)


#define WWDG_BASE             (APBPERIPH_BASE + 0x00002C00UL)
#define ADC0_BASE             (APBPERIPH_BASE + 0x00032000UL)
#define ADC1_BASE             (APBPERIPH_BASE + 0x00033000UL)
#define FLASH_R_BASE          (APBPERIPH_BASE + 0x00012000UL)
#define ICACHE_BASE           (APBPERIPH_BASE + 0x00012400UL)
#define DCACHE_BASE           (APBPERIPH_BASE + 0x00012800UL)
#define DMA_BASE              (APBPERIPH_BASE + 0x00020000UL)
#define DMA_CHANNEL_BASE      (APBPERIPH_BASE + 0x00020100UL)
#define CRC_BASE              (APBPERIPH_BASE + 0x00030000UL)
#define IOW_BASE              (APBPERIPH_BASE + 0x00007400UL)
#define I2C0_BASE             (APBPERIPH_BASE + 0x00005400UL)
#define I2C1_BASE             (APBPERIPH_BASE + 0x00005800UL)
#define TIMA0_BASE            (APBPERIPH_BASE + 0x00040000UL)
#define TIMA1_BASE            (APBPERIPH_BASE + 0x00040400UL)
#define TIMH_COMMON_BASE      (APBPERIPH_BASE + 0x00040800UL)
#define TIMH0_BASE            (APBPERIPH_BASE + 0x00040900UL)
#define TIMH1_BASE            (APBPERIPH_BASE + 0x00040980UL)
#define TIMH2_BASE            (APBPERIPH_BASE + 0x00040A00UL)
#define TIMG0_BASE            (APBPERIPH_BASE + 0x00041000UL)
#define TIMG1_BASE            (APBPERIPH_BASE + 0x00041400UL)
#define TIMG2_BASE            (APBPERIPH_BASE + 0x00041800UL)
#define TIMS0_BASE            (APBPERIPH_BASE + 0x00042000UL)
#define TIMS1_BASE            (APBPERIPH_BASE + 0x00042400UL)
#define TIMS2_BASE            (APBPERIPH_BASE + 0x00042800UL)
#define TIMS3_BASE            (APBPERIPH_BASE + 0x00042C00UL)
#define TIM_COMMON_BASE       (APBPERIPH_BASE + 0x00046000UL)
#define BTIM00_BASE           (APBPERIPH_BASE + 0x00003000UL)
#define BTIM01_BASE           (APBPERIPH_BASE + 0x00003020UL)
#define BTIM0_COMM_BASE       (APBPERIPH_BASE + 0x00003040UL)
#define BTIM10_BASE           (APBPERIPH_BASE + 0x00003100UL)
#define BTIM11_BASE           (APBPERIPH_BASE + 0x00003120UL)
#define BTIM1_COMM_BASE       (APBPERIPH_BASE + 0x00003140UL)
#define TMU_BASE              (APBPERIPH_BASE + 0x00031000UL)
#define EBUS_BASE             (APBPERIPH_BASE + 0x00010800UL)
#define IOPORT_BASE           (APBPERIPH_BASE + 0x0001080CUL)
#define CAN_BASE              (APBPERIPH_BASE + 0x00001000UL)
#define ACMP0_BASE            (APBPERIPH_BASE + 0x00006000UL)
#define ACMP1_BASE            (APBPERIPH_BASE + 0x00006040UL)
#define ACMP2_BASE            (APBPERIPH_BASE + 0x00006080UL)
#define ACMP3_BASE            (APBPERIPH_BASE + 0x000060C0UL)

#define SCI0_BASE             (APBPERIPH_BASE + 0x00004400UL)
#define SCI1_BASE             (APBPERIPH_BASE + 0x00004800UL)
#define SCI2_BASE             (APBPERIPH_BASE + 0x00004C00UL)

#define SPI_BASE              (APBPERIPH_BASE + 0x00002000UL)

#define SYS_REG_BASE          (APBPERIPH_BASE + 0x00010000UL)

/*IO Port*/
#define GPIOA_BASE            (GPIOPORT_BASE + 0x00000000UL)
#define GPIOB_BASE            (GPIOPORT_BASE + 0x00001000UL)
#define GPIOC_BASE            (GPIOPORT_BASE + 0x00002000UL)
#define GPIOD_BASE            (GPIOPORT_BASE + 0x00003000UL)
#define GPIOE_BASE            (GPIOPORT_BASE + 0x00004000UL)
#define GPIOX_BASE            GPIOA_BASE


#define WWDG                  ((WWDG_Type *) WWDG_BASE)
#define ADC0                  ((ADC_Type *) ADC0_BASE)
#define ADC1                  ((ADC_Type *) ADC1_BASE)
#define FLASHCTRL             ((FLASH_Type *) FLASH_R_BASE)
#define ICACHE                ((ICACHE_Type *) ICACHE_BASE)
#define DCACHE                ((DCACHE_Type *) DCACHE_BASE)
#define DMA                   ((DMA_Type *) DMA_BASE)
#define CRC                   ((CRC_Type *) CRC_BASE)
#define IOW                   ((IOW_Type *) IOW_BASE)
#define TIMA0                 ((TIMA_Type *) TIMA0_BASE)
#define TIMA1                 ((TIMA_Type *) TIMA1_BASE)

#define TIMH0                 ((TIMH_Type *) TIMH0_BASE)
#define TIMH1                 ((TIMH_Type *) TIMH1_BASE)
#define TIMH2                 ((TIMH_Type *) TIMH2_BASE)

#define TIMG0                 ((TIMG_Type *) TIMG0_BASE)
#define TIMG1                 ((TIMG_Type *) TIMG1_BASE)
#define TIMG2                 ((TIMG_Type *) TIMG2_BASE)
#define TIMS0                 ((TIMS_Type *) TIMS0_BASE)
#define TIMS1                 ((TIMS_Type *) TIMS1_BASE)
#define TIMS2                 ((TIMS_Type *) TIMS2_BASE)
#define TIMS3                 ((TIMS_Type *) TIMS3_BASE)
#define TIM_COMMON            ((TIM_Common_Type *) TIM_COMMON_BASE)
#define TIMH_COMMON           ((TIMH_Common_Type *) TIMH_COMMON_BASE)

#define GPIOA                 ((GPIO_Type *) GPIOA_BASE)
#define GPIOB                 ((GPIO_Type *) GPIOB_BASE)
#define GPIOC                 ((GPIO_Type *) GPIOC_BASE)
#define GPIOD                 ((GPIO_Type *) GPIOD_BASE)
#define GPIOE                 ((GPIO_Type *) GPIOE_BASE)
#define BTIM00    			  ((BTIM_Type *) BTIM00_BASE)
#define BTIM01    			  ((BTIM_Type *) BTIM01_BASE)
#define BTIM0_COMM            ((BTIM_CommType *) BTIM0_COMM_BASE)
#define BTIM10    			  ((BTIM_Type *) BTIM10_BASE)
#define BTIM11    			  ((BTIM_Type *) BTIM11_BASE)
#define BTIM1_COMM            ((BTIM_CommType *) BTIM1_COMM_BASE)

#define SYSCFG                ((SysConfig_Type *) SYS_REG_BASE)
#define EBUS                  ((EBUS_Type *) EBUS_BASE)
#define TMU                   ((TMU_Type *) TMU_BASE)
#define IOPORT                ((IOPORT_Type *)IOPORT_BASE)

#define SCI0                  ((SCI_Type *)SCI0_BASE)
#define SCI1                  ((SCI_Type *)SCI1_BASE)
#define SCI2                  ((SCI_Type *)SCI2_BASE)

#define SPI                   ((SPI_Type *)SPI_BASE)


#define I2C0    			  ((I2C_Type *)  I2C0_BASE)
#define I2C1				  ((I2C_Type *)  I2C1_BASE)
#define CAN                   ((CAN_Type *) CAN_BASE)


#define ACMP0                 ((ACMP_Type *)ACMP0_BASE)
#define ACMP1                 ((ACMP_Type *)ACMP1_BASE)
#define ACMP2                 ((ACMP_Type *)ACMP2_BASE)
#define ACMP3                 ((ACMP_Type *)ACMP3_BASE)

/******************************************************************************/
/*                                                                            */
/*                    System Configuration Register                           */
/*                                                                            */
/******************************************************************************/
/****************************** System Configuration Register *************************/
#define SYSCONF_SC_PADNMIEN_Pos            (8U) /*Lockup reset enable*/
#define SYSCONF_SC_PADNMIEN_Msk            (0x1UL << SYSCONF_SC_PADNMIEN_Pos)
#define SYSCONF_SC_PADNMIEN                SYSCONF_SC_PADNMIEN_Msk

#define SYSCONF_SC_LOCKUP_Pos              (4U) /*Lockup reset enable*/
#define SYSCONF_SC_LOCKUP_Msk              (0x1UL << SYSCONF_SC_LOCKUP_Pos)
#define SYSCONF_SC_LOCKUP                  SYSCONF_SC_LOCKUP_Msk

#define SYSCONF_SC_BOOTSEQ_Pos             (0U) /*Boot Squence configuration*/
#define SYSCONF_SC_BOOTSEQ_Msk             (0x3UL << SYSCONF_SC_BOOTSEQ_Pos)
#define SYSCONF_SC_BOOTSEQ_SRAM            (0x0UL << SYSCONF_SC_BOOTSEQ_Pos)
#define SYSCONF_SC_BOOTSEQ_BOOTROM         (0x1UL << SYSCONF_SC_BOOTSEQ_Pos)
#define SYSCONF_SC_BOOTSEQ_FLASH           (0x3UL << SYSCONF_SC_BOOTSEQ_Pos)

/****************************** Reset Status Register *************************/
#define SYSCONF_RSTS_POR_Pos               (0U)
#define SYSCONF_RSTS_POR_Msk               (0x1UL << SYSCONF_RSTS_POR_Pos)
#define SYSCONF_RSTS_POR                   SYSCONF_RSTS_POR_Msk
#define SYSCONF_RSTS_WDT_Pos               (1U)
#define SYSCONF_RSTS_WDT_Msk               (0x1UL << SYSCONF_RSTS_WDT_Pos)
#define SYSCONF_RSTS_WDT                   SYSCONF_RSTS_WDT_Msk
#define SYSCONF_RSTS_LOCKUP_Pos            (2U)
#define SYSCONF_RSTS_LOCKUP_Msk            (0x1UL << SYSCONF_RSTS_LOCKUP_Pos)
#define SYSCONF_RSTS_LOCKUP                SYSCONF_RSTS_LOCKUP_Msk
#define SYSCONF_RSTS_SYSREQ_Pos            (3U) /*System reset request*/
#define SYSCONF_RSTS_SYSREQ_Msk            (0x1UL << SYSCONF_RSTS_SYSREQ_Pos)
#define SYSCONF_RSTS_SYSREQ                SYSCONF_RSTS_SYSREQ_Msk
#define SYSCONF_RSTS_LOSCLS_Pos            (6U) /*LOSC loss*/
#define SYSCONF_RSTS_LOSCLS_Msk            (0x1UL << SYSCONF_RSTS_LOSCLS_Pos)
#define SYSCONF_RSTS_LOSCLS                SYSCONF_RSTS_LOSCLS_Msk
#define SYSCONF_RSTS_HOSCLS_Pos            (7U) /*HOSC loss*/
#define SYSCONF_RSTS_HOSCLS_Msk            (0x1UL << SYSCONF_RSTS_HOSCLS_Pos)
#define SYSCONF_RSTS_HOSCLS                SYSCONF_RSTS_HOSCLS_Msk
#define SYSCONF_RSTS_PORWARNUP_Pos         (10U)
#define SYSCONF_RSTS_PORWARNUP_Msk         (0x1UL << SYSCONF_RSTS_PORWARNUP_Pos)
#define SYSCONF_RSTS_PORWARNUP             SYSCONF_RSTS_PORWARNUP_Msk
#define SYSCONF_RSTS_FLASHWAKEUP_Pos       (11U)
#define SYSCONF_RSTS_FLASHWAKEUP_Msk       (0x1UL << SYSCONF_RSTS_FLASHWAKEUP_Pos)
#define SYSCONF_RSTS_FLASHWAKEUP           SYSCONF_RSTS_FLASHWAKEUP_Msk
#define SYSCONF_RSTS_OPTREAD_Pos           (12U)
#define SYSCONF_RSTS_OPTREAD_Msk           (0x1UL << SYSCONF_RSTS_OPTREAD_Pos)
#define SYSCONF_RSTS_OPTREAD               SYSCONF_RSTS_OPTREAD_Msk
#define SYSCONF_RSTS_FRMSTP_Pos            (18U) /*Return from lower power mode*/
#define SYSCONF_RSTS_FRMSTP_Msk            (0x1UL << SYSCONF_RSTS_FRMSTP_Pos)
#define SYSCONF_RSTS_FRMSTP                SYSCONF_RSTS_FRMSTP_Msk
#define SYSCONF_RSTS_HIRCRDY_Pos           (24U) /*HIRC clock is stable*/
#define SYSCONF_RSTS_HIRCRDY_Msk           (0x1UL << SYSCONF_RSTS_HIRCRDY_Pos)
#define SYSCONF_RSTS_HIRCRDY               SYSCONF_RSTS_HIRCRDY_Msk
#define SYSCONF_RSTS_LIRCRDY_Pos           (25U) /*LIRC clock is stable*/
#define SYSCONF_RSTS_LIRCRDY_Msk           (0x1UL << SYSCONF_RSTS_LIRCRDY_Pos)
#define SYSCONF_RSTS_LIRCRDY               SYSCONF_RSTS_LIRCRDY_Msk
#define SYSCONF_RSTS_HOSCRDY_Pos           (26U) /*HOSC clock is stable*/
#define SYSCONF_RSTS_HOSCRDY_Msk           (0x1UL << SYSCONF_RSTS_HOSCRDY_Pos)
#define SYSCONF_RSTS_HOSCRDY               SYSCONF_RSTS_HOSCRDY_Msk
#define SYSCONF_RSTS_HIRC2RDY_Pos          (27U) /*HIRC2 clock is stable*/
#define SYSCONF_RSTS_HIRC2RDY_Msk          (0x1UL << SYSCONF_RSTS_HIRC2RDY_Pos)
#define SYSCONF_RSTS_HIRC2RDY              SYSCONF_RSTS_HIRC2RDY_Msk
#define SYSCONF_RSTS_PLLRDY_Pos            (28U) /*PLL clock is stable*/
#define SYSCONF_RSTS_PLLRDY_Msk            (0x1UL << SYSCONF_RSTS_PLLRDY_Pos)
#define SYSCONF_RSTS_PLLRDY                SYSCONF_RSTS_PLLRDY_Msk

/****************************** NMI Status Register *************************/
#define SYSCONF_NMI_MCLKLOSSPEND_Pos       (0U) /* M clock loss */
#define SYSCONF_NMI_MCLKLOSSPEND_Msk       (0x1UL << SYSCONF_NMI_MCLKLOSSPEND_Pos)
#define SYSCONF_NMI_MCLKLOSSPEND           SYSCONF_NMI_MCLKLOSSPEND_Msk

#define SYSCONF_NMI_PLLLOSSPEND_Pos        (1U) /* PLL clock loss */
#define SYSCONF_NMI_PLLLOSSPEND_Msk        (0x1UL << SYSCONF_NMI_PLLLOSSPEND_Pos)
#define SYSCONF_NMI_PLLLOSSPEND            SYSCONF_NMI_PLLLOSSPEND_Msk

#define SYSCONF_NMI_FRQERR_Pos             (2U) /* Frequence error */
#define SYSCONF_NMI_FRQERR_Msk             (0x1UL << SYSCONF_NMI_FRQERR_Pos)
#define SYSCONF_NMI_FRQERR                 SYSCONF_NMI_FRQERR_Msk

#define SYSCONF_NMI_SRMPRTERR_Pos          (3U) /* SRAM parity error */
#define SYSCONF_NMI_SRMPRTERR_Msk          (0x1UL << SYSCONF_NMI_SRMPRTERR_Pos)
#define SYSCONF_NMI_SRMPRTERR              SYSCONF_NMI_SRMPRTERR_Msk

#define SYSCONF_NMI_CACHEPRTERR_Pos        (4U) /* Cache parity error */
#define SYSCONF_NMI_CACHEPRTERR_Msk        (0x1UL << SYSCONF_NMI_CACHEPRTERR_Pos)
#define SYSCONF_NMI_CACHEPRTERR            SYSCONF_NMI_CACHEPRTERR_Msk

#define SYSCONF_NMI_PADNMIPEND_Pos         (5U) /* SRAM parity error */
#define SYSCONF_NMI_PADNMIPEND_Msk         (0x1UL << SYSCONF_NMI_PADNMIPEND_Pos)
#define SYSCONF_NMI_PADNMIPEND             SYSCONF_NMI_PADNMIPEND_Msk

/***************************** HOSC Control Register **************************/
#define SYSCONF_HOSC_EN_Pos                (0U) /* HOSC Enable */
#define SYSCONF_HOSC_EN_Msk                (0x1UL << SYSCONF_HOSC_EN_Pos)
#define SYSCONF_HOSC_EN                    SYSCONF_HOSC_EN_Msk

#define SYSCONF_HOSC_EXTCLKEN_Pos          (1U) /* External clock Enable */
#define SYSCONF_HOSC_EXTCLKEN_Msk          (0x1UL << SYSCONF_HOSC_EXTCLKEN_Pos)
#define SYSCONF_HOSC_EXTCLKEN              SYSCONF_HOSC_EXTCLKEN_Msk

#define SYSCONF_HOSC_GMADJUST_Pos          (4U) /* GM parameter adjust */
#define SYSCONF_HOSC_GMADJUST_Msk          (0xFUL << SYSCONF_HOSC_GMADJUST_Pos)
#define SYSCONF_HOSC_GMADJUST              SYSCONF_HOSC_GMADJUST_Msk

#define SYSCONF_HOSC_RESEN_Pos             (8U) /* Internal parallel resistance enable */
#define SYSCONF_HOSC_RESEN_Msk             (0x1UL << SYSCONF_HOSC_RESEN_Pos)
#define SYSCONF_HOSC_RESEN                 SYSCONF_HOSC_RESEN_Msk

#define SYSCONF_HOSC_RESSEL_Pos            (9U) /* Internal parallel resistance select */
#define SYSCONF_HOSC_RESSEL_Msk            (0x3UL << SYSCONF_HOSC_RESSEL_Pos)
#define SYSCONF_HOSC_RESSEL                SYSCONF_HOSC_RESSEL_Msk

#define SYSCONF_HOSC_STABLETIME_Pos        (12U) /* HOSC stable time select */
#define SYSCONF_HOSC_STABLETIME_Msk        (0x7UL << SYSCONF_HOSC_STABLETIME_Pos)
#define SYSCONF_HOSC_STABLETIME            SYSCONF_HOSC_STABLETIME_Msk

#define SYSCONF_HOSC_DIVIDE_Pos            (16U) /* HOSC frequency division select */
#define SYSCONF_HOSC_DIVIDE_Msk            (0x3UL << SYSCONF_HOSC_DIVIDE_Pos)
#define SYSCONF_HOSC_DIVIDE                SYSCONF_HOSC_DIVIDE_Msk

#define SYSCONF_HOSC_CLKOUTEN_Pos          (18U) /* HOSC clock out enable */
#define SYSCONF_HOSC_CLKOUTEN_Msk          (0x1UL << SYSCONF_HOSC_CLKOUTEN_Pos)
#define SYSCONF_HOSC_CLKOUTEN              SYSCONF_HOSC_CLKOUTEN_Msk

#define SYSCONF_HOSC_SEL_Pos               (30U) /* HOSC Selection */
#define SYSCONF_HOSC_SEL_Msk               (0x3UL << SYSCONF_HOSC_SEL_Pos)
#define SYSCONF_HOSC_SEL_HINOUT            SYSCONF_HOSC_SEL_Msk
#define SYSCONF_HOSC_SEL_NORM              (0x0UL << SYSCONF_HOSC_SEL_Pos)


/***************************** LOSC Control Register **************************/
#define SYSCONF_LOSC_EN_Pos                (0U) /* LOSC Enable */
#define SYSCONF_LOSC_EN_Msk                (0x1UL << SYSCONF_LOSC_EN_Pos)
#define SYSCONF_LOSC_EN                    SYSCONF_LOSC_EN_Msk

#define SYSCONF_LOSC_EXTCLKEN_Pos          (1U) /* External clock Enable */
#define SYSCONF_LOSC_EXTCLKEN_Msk          (0x1UL << SYSCONF_LOSC_EXTCLKEN_Pos)
#define SYSCONF_LOSC_EXTCLKEN              SYSCONF_LOSC_EXTCLKEN_Msk

#define SYSCONF_LOSC_GMADJUST_Pos          (4U) /* GM parameter adjust */
#define SYSCONF_LOSC_GMADJUST_Msk          (0xFUL << SYSCONF_LOSC_GMADJUST_Pos)
#define SYSCONF_LOSC_GMADJUST              SYSCONF_LOSC_GMADJUST_Msk

#define SYSCONF_LOSC_STABLETIME_Pos        (12U) /* LOSC stable time select */
#define SYSCONF_LOSC_STABLETIME_Msk        (0x7UL << SYSCONF_LOSC_STABLETIME_Pos)
#define SYSCONF_LOSC_STABLETIME            SYSCONF_LOSC_STABLETIME_Msk

#define SYSCONF_LOSC_CLKOUTEN_Pos          (16U) /* LOSC clock out enable */
#define SYSCONF_LOSC_CLKOUTEN_Msk          (0x1UL << SYSCONF_LOSC_CLKOUTEN_Pos)
#define SYSCONF_LOSC_CLKOUTEN              SYSCONF_LOSC_CLKOUTEN_Msk

#define SYSCONF_LOSC_SEL_Pos               (30U) /* LOSC Selection */
#define SYSCONF_LOSC_SEL_Msk               (0x3UL << SYSCONF_LOSC_SEL_Pos)
#define SYSCONF_LOSC_SEL_LINOUT            SYSCONF_LOSC_SEL_Msk
#define SYSCONF_LOSC_SEL_NORM              (0x0UL << SYSCONF_LOSC_SEL_Pos)

/*************************** LVD1 Control Register ************************/
#define SYSCONF_LVD1CTRL_EN_Pos            (0U)
#define SYSCONF_LVD1CTRL_EN_Msk            (0x1UL << SYSCONF_LVD1CTRL_EN_Pos)
#define SYSCONF_LVD1CTRL_EN                SYSCONF_LVD1CTRL_EN_Msk

#define SYSCONF_LVD1CTRL_RST_Pos           (4U)
#define SYSCONF_LVD1CTRL_RST_Msk           (0x1UL << SYSCONF_LVD1CTRL_RST_Pos)
#define SYSCONF_LVD1CTRL_RST               SYSCONF_LVD1CTRL_RST_Msk

#define SYSCONF_LVD1CTRL_SEL_Pos           (8U)
#define SYSCONF_LVD1CTRL_SEL_Msk           (0xFUL << SYSCONF_LVD1CTRL_SEL_Pos)
#define SYSCONF_LVD1CTRL_SEL               SYSCONF_LVD1CTRL_SEL_Msk

/*************************** LVD1 Status Register ************************/
#define SYSCONF_LVD1STS_FLAG_Pos           (1U)
#define SYSCONF_LVD1STS_FLAG_Msk           (0x1UL << SYSCONF_LVD1STS_FLAG_Pos)
#define SYSCONF_LVD1STS_FLAG               SYSCONF_LVD1STS_FLAG_Msk

/*************************** LVD2 Control Register ************************/
#define SYSCONF_LVD2CTRL_EN_Pos            (0U)
#define SYSCONF_LVD2CTRL_EN_Msk            (0x1UL << SYSCONF_LVD2CTRL_EN_Pos)
#define SYSCONF_LVD2CTRL_EN                SYSCONF_LVD2CTRL_EN_Msk

#define SYSCONF_LVD2CTRL_IE_Pos            (3U)
#define SYSCONF_LVD2CTRL_IE_Msk            (0x1UL << SYSCONF_LVD2CTRL_IE_Pos)
#define SYSCONF_LVD2CTRL_IE                SYSCONF_LVD2CTRL_IE_Msk

#define SYSCONF_LVD2CTRL_SEL_Pos           (8U)
#define SYSCONF_LVD2CTRL_SEL_Msk           (0xFUL << SYSCONF_LVD2CTRL_SEL_Pos)
#define SYSCONF_LVD2CTRL_SEL               SYSCONF_LVD2CTRL_SEL_Msk

#define SYSCONF_LVD2CTRL_CLKPSC_Pos        (12U)
#define SYSCONF_LVD2CTRL_CLKPSC_Msk        (0x7UL << SYSCONF_LVD2CTRL_CLKPSC_Pos)

#define SYSCONF_LVD2CTRL_THR_Pos           (16U)
#define SYSCONF_LVD2CTRL_THR_Msk           (0xFFUL << SYSCONF_LVD2CTRL_THR_Pos)

/*************************** LVD2 Status Register ************************/
#define SYSCONF_LVD2STS_FLAG_Pos           (1U)
#define SYSCONF_LVD2STS_FLAG_Msk           (0x1UL << SYSCONF_LVD2STS_FLAG_Pos)
#define SYSCONF_LVD2STS_FLAG               SYSCONF_LVD2STS_FLAG_Msk


/***************************** HIRC Control Register **************************/
#define SYSCONF_HIRC_PWRDN_Pos             (0U) /* HIRC Power-down */
#define SYSCONF_HIRC_PWRDN_Msk             (0x1UL << SYSCONF_HIRC_PWRDN_Pos)
#define SYSCONF_HIRC_PWRDN                 SYSCONF_HIRC_PWRDN_Msk
#define SYSCONF_LIRC_PWRDN_Pos             (16U) /* LIRC Power-down */
#define SYSCONF_LIRC_PWRDN_Msk             (0x1UL << SYSCONF_LIRC_PWRDN_Pos)
#define SYSCONF_LIRC_PWRDN                 SYSCONF_LIRC_PWRDN_Msk

/************************** Power mode Control Register ***********************/
#define SYSCONF_PMC_PMUEN_Pos              (0U) /*PMU enable*/
#define SYSCONF_PMC_PMUEN_Msk              (0x1UL << SYSCONF_PMC_PMUEN_Pos)
#define SYSCONF_PMC_PMUEN                  SYSCONF_PMC_PMUEN_Msk
#define SYSCONF_PMC_STP_Pos                (1U) /*Flash standby enable*/
#define SYSCONF_PMC_STP_Msk                (0x1UL << SYSCONF_PMC_STP_Pos)
#define SYSCONF_PMC_STP                    SYSCONF_PMC_STP_Msk
#define SYSCONF_PMC_FLSTBY_Pos             (4U) /*Flash standby enable*/
#define SYSCONF_PMC_FLSTBY_Msk             (0x1UL << SYSCONF_PMC_FLSTBY_Pos)
#define SYSCONF_PMC_FLSTBY                 SYSCONF_PMC_FLSTBY_Msk

/****************************** Clock Gate 0 Register **********************/
#define SYSCONF_CG0_I2C0_Pos               (0U)
#define SYSCONF_CG0_I2C0_Msk               (0x1UL << SYSCONF_CG0_I2C0_Pos)
#define SYSCONF_CG0_I2C0                   SYSCONF_CG0_I2C0_Msk
#define SYSCONF_CG0_I2C1_Pos               (1U)
#define SYSCONF_CG0_I2C1_Msk               (0x1UL << SYSCONF_CG0_I2C1_Pos)
#define SYSCONF_CG0_I2C1                   SYSCONF_CG0_I2C1_Msk
#define SYSCONF_CG0_GPIO_Pos               (4U)
#define SYSCONF_CG0_GPIO_Msk               (0x1UL << SYSCONF_CG0_GPIO_Pos)
#define SYSCONF_CG0_GPIO                   SYSCONF_CG0_GPIO_Msk
#define SYSCONF_CG0_DMA_Pos                (5U)
#define SYSCONF_CG0_DMA_Msk                (0x1UL << SYSCONF_CG0_DMA_Pos)
#define SYSCONF_CG0_DMA                    SYSCONF_CG0_DMA_Msk
#define SYSCONF_CG0_TMU_Pos                (6U)
#define SYSCONF_CG0_TMU_Msk                (0x1UL << SYSCONF_CG0_TMU_Pos)
#define SYSCONF_CG0_TMU                    SYSCONF_CG0_TMU_Msk
#define SYSCONF_CG0_EBUS_Pos               (7U)
#define SYSCONF_CG0_EBUS_Msk               (0x1UL << SYSCONF_CG0_EBUS_Pos)
#define SYSCONF_CG0_EBUS                   SYSCONF_CG0_EBUS_Msk
#define SYSCONF_CG0_ADC0_Pos               (8U)
#define SYSCONF_CG0_ADC0_Msk               (0x1UL << SYSCONF_CG0_ADC0_Pos)
#define SYSCONF_CG0_ADC0                   SYSCONF_CG0_ADC0_Msk
#define SYSCONF_CG0_ADC1_Pos               (9U)
#define SYSCONF_CG0_ADC1_Msk               (0x1UL << SYSCONF_CG0_ADC1_Pos)
#define SYSCONF_CG0_ADC1                   SYSCONF_CG0_ADC1_Msk
#define SYSCONF_CG0_BTIM_Pos         	   (12U)
#define SYSCONF_CG0_BTIM_Msk          	   (0x1UL << SYSCONF_CG0_BTIM_Pos)
#define SYSCONF_CG0_BTIM              	   SYSCONF_CG0_BTIM_Msk
#define SYSCONF_CG0_TIMA0_Pos              (16U)
#define SYSCONF_CG0_TIMA0_Msk              (0x1UL << SYSCONF_CG0_TIMA0_Pos)
#define SYSCONF_CG0_TIMA0                  SYSCONF_CG0_TIMA0_Msk
#define SYSCONF_CG0_TIMA1_Pos              (17U)
#define SYSCONF_CG0_TIMA1_Msk              (0x1UL << SYSCONF_CG0_TIMA1_Pos)
#define SYSCONF_CG0_TIMA1                  SYSCONF_CG0_TIMA1_Msk
#define SYSCONF_CG0_TIMH_Pos               (18U)
#define SYSCONF_CG0_TIMH_Msk               (0x1UL << SYSCONF_CG0_TIMH_Pos)
#define SYSCONF_CG0_TIMH                   SYSCONF_CG0_TIMH_Msk
#define SYSCONF_CG0_TIMG0_Pos              (19U)
#define SYSCONF_CG0_TIMG0_Msk              (0x1UL << SYSCONF_CG0_TIMG0_Pos)
#define SYSCONF_CG0_TIMG0                  SYSCONF_CG0_TIMG0_Msk
#define SYSCONF_CG0_TIMG1_Pos              (20U)
#define SYSCONF_CG0_TIMG1_Msk              (0x1UL << SYSCONF_CG0_TIMG1_Pos)
#define SYSCONF_CG0_TIMG1                  SYSCONF_CG0_TIMG1_Msk
#define SYSCONF_CG0_TIMG2_Pos              (21U)
#define SYSCONF_CG0_TIMG2_Msk              (0x1UL << SYSCONF_CG0_TIMG2_Pos)
#define SYSCONF_CG0_TIMG2                  SYSCONF_CG0_TIMG2_Msk
#define SYSCONF_CG0_BUZ1_Pos               (25U)
#define SYSCONF_CG0_BUZ1_Msk               (0x1UL << SYSCONF_CG0_BUZ1_Pos)
#define SYSCONF_CG0_BUZ1                   SYSCONF_CG0_BUZ1_Msk
#define SYSCONF_CG0_WDT_Pos                (31U)
#define SYSCONF_CG0_WDT_Msk                (0x1UL << SYSCONF_CG0_WDT_Pos)
#define SYSCONF_CG0_WDT                    SYSCONF_CG0_WDT_Msk

/****************************** Clock Gate 1 Register **********************/
#define SYSCONF_CG1_IOW_Pos                (1U)
#define SYSCONF_CG1_IOW_Msk                (0x1UL << SYSCONF_CG1_IOW_Pos)
#define SYSCONF_CG1_IOW                    SYSCONF_CG1_IOW_Msk
#define SYSCONF_CG1_SPI_Pos                (2U)
#define SYSCONF_CG1_SPI_Msk                (0x1UL << SYSCONF_CG1_SPI_Pos)
#define SYSCONF_CG1_SPI                    SYSCONF_CG1_SPI_Msk
#define SYSCONF_CG1_SCI0_Pos               (8U)
#define SYSCONF_CG1_SCI0_Msk               (0x1UL << SYSCONF_CG1_SCI0_Pos)
#define SYSCONF_CG1_SCI0                   SYSCONF_CG1_SCI0_Msk
#define SYSCONF_CG1_SCI1_Pos               (9U)
#define SYSCONF_CG1_SCI1_Msk               (0x1UL << SYSCONF_CG1_SCI1_Pos)
#define SYSCONF_CG1_SCI1                   SYSCONF_CG1_SCI1_Msk
#define SYSCONF_CG1_SCI2_Pos               (10U)
#define SYSCONF_CG1_SCI2_Msk               (0x1UL << SYSCONF_CG1_SCI2_Pos)
#define SYSCONF_CG1_SCI2                   SYSCONF_CG1_SCI2_Msk
#define SYSCONF_CG1_DAC_Pos                (12U)
#define SYSCONF_CG1_DAC_Msk                (0x1UL << SYSCONF_CG1_DAC_Pos)
#define SYSCONF_CG1_DAC                    SYSCONF_CG1_DAC_Msk
#define SYSCONF_CG1_ACMP_Pos               (13U)
#define SYSCONF_CG1_ACMP_Msk               (0x1UL << SYSCONF_CG1_ACMP_Pos)
#define SYSCONF_CG1_ACMP                   SYSCONF_CG1_ACMP_Msk
#define SYSCONF_CG1_CAN_Pos                (14U)
#define SYSCONF_CG1_CAN_Msk                (0x1UL << SYSCONF_CG1_CAN_Pos)
#define SYSCONF_CG1_CAN                    SYSCONF_CG1_CAN_Msk
#define SYSCONF_CG1_TIMS0_Pos              (16U)
#define SYSCONF_CG1_TIMS0_Msk              (0x1UL << SYSCONF_CG1_TIMS0_Pos)
#define SYSCONF_CG1_TIMS0                  SYSCONF_CG1_TIMS0_Msk
#define SYSCONF_CG1_TIMS1_Pos              (17U)
#define SYSCONF_CG1_TIMS1_Msk              (0x1UL << SYSCONF_CG1_TIMS1_Pos)
#define SYSCONF_CG1_TIMS1                  SYSCONF_CG1_TIMS1_Msk
#define SYSCONF_CG1_TIMS2_Pos              (18U)
#define SYSCONF_CG1_TIMS2_Msk              (0x1UL << SYSCONF_CG1_TIMS2_Pos)
#define SYSCONF_CG1_TIMS2                  SYSCONF_CG1_TIMS2_Msk
#define SYSCONF_CG1_TIMS3_Pos              (19U)
#define SYSCONF_CG1_TIMS3_Msk              (0x1UL << SYSCONF_CG1_TIMS3_Pos)
#define SYSCONF_CG1_TIMS3                  SYSCONF_CG1_TIMS3_Msk

/****************************** Reset Control 0 Register **********************/
#define SYSCONF_RC0_TIMG2_Pos              (21U)
#define SYSCONF_RC0_TIMG2_Msk              (0x1UL << SYSCONF_RC0_TIMG2_Pos)
#define SYSCONF_RC0_TIMG2                  SYSCONF_RC0_TIMG2_Msk
#define SYSCONF_RC0_TIMG1_Pos              (20U)
#define SYSCONF_RC0_TIMG1_Msk              (0x1UL << SYSCONF_RC0_TIMG1_Pos)
#define SYSCONF_RC0_TIMG1                  SYSCONF_RC0_TIMG1_Msk
#define SYSCONF_RC0_TIMG0_Pos              (19U)
#define SYSCONF_RC0_TIMG0_Msk              (0x1UL << SYSCONF_RC0_TIMG0_Pos)
#define SYSCONF_RC0_TIMG0                  SYSCONF_RC0_TIMG0_Msk
#define SYSCONF_RC0_TIMH_Pos               (18U)
#define SYSCONF_RC0_TIMH_Msk               (0x1UL << SYSCONF_RC0_TIMH_Pos)
#define SYSCONF_RC0_TIMH                   SYSCONF_RC0_TIMH_Msk
#define SYSCONF_RC0_TIMA1_Pos              (17U)
#define SYSCONF_RC0_TIMA1_Msk              (0x1UL << SYSCONF_RC0_TIMA1_Pos)
#define SYSCONF_RC0_TIMA1                  SYSCONF_RC0_TIMA1_Msk
#define SYSCONF_RC0_TIMA0_Pos              (16U)
#define SYSCONF_RC0_TIMA0_Msk              (0x1UL << SYSCONF_RC0_TIMA0_Pos)
#define SYSCONF_RC0_TIMA0                  SYSCONF_RC0_TIMA0_Msk
#define SYSCONF_RC0_BTIM_Pos         	   (12U)
#define SYSCONF_RC0_BTIM_Msk          	   (0x1UL << SYSCONF_RC0_BTIM_Pos)
#define SYSCONF_RC0_BTIM             	   SYSCONF_RC0_BTIM_Msk
#define SYSCONF_RC0_ADC1_Pos               (9U)
#define SYSCONF_RC0_ADC1_Msk               (0x1UL << SYSCONF_RC0_ADC1_Pos)
#define SYSCONF_RC0_ADC1                   SYSCONF_RC0_ADC1_Msk
#define SYSCONF_RC0_ADC0_Pos               (8U)
#define SYSCONF_RC0_ADC0_Msk               (0x1UL << SYSCONF_RC0_ADC0_Pos)
#define SYSCONF_RC0_ADC0                   SYSCONF_RC0_ADC0_Msk
#define SYSCONF_RC0_EBUS_Pos               (7U)
#define SYSCONF_RC0_EBUS_Msk               (0x1UL << SYSCONF_RC0_EBUS_Pos)
#define SYSCONF_RC0_EBUS                   SYSCONF_RC0_EBUS_Msk
#define SYSCONF_RC0_TMU_Pos                (6U)
#define SYSCONF_RC0_TMU_Msk                (0x1UL << SYSCONF_RC0_TMU_Pos)
#define SYSCONF_RC0_TMU                    SYSCONF_RC0_TMU_Msk
#define SYSCONF_RC0_DMA_Pos                (5U)
#define SYSCONF_RC0_DMA_Msk                (0x1UL << SYSCONF_RC0_DMA_Pos)
#define SYSCONF_RC0_DMA                    SYSCONF_RC0_DMA_Msk
#define SYSCONF_RC0_GPIO_Pos               (4U)
#define SYSCONF_RC0_GPIO_Msk               (0x1UL << SYSCONF_RC0_GPIO_Pos)
#define SYSCONF_RC0_GPIO                   SYSCONF_RC0_GPIO_Msk
#define SYSCONF_RC0_I2C1_Pos               (1U)
#define SYSCONF_RC0_I2C1_Msk               (0x1UL << SYSCONF_RC0_I2C1_Pos)
#define SYSCONF_RC0_I2C1                   SYSCONF_RC0_I2C1_Msk
#define SYSCONF_RC0_I2C0_Pos               (0U)
#define SYSCONF_RC0_I2C0_Msk               (0x1UL << SYSCONF_RC0_I2C0_Pos)
#define SYSCONF_RC0_I2C0                   SYSCONF_RC0_I2C0_Msk


/****************************** Reset Control 1 Register **********************/
#define SYSCONF_RC1_TIMS3_Pos              (19U)
#define SYSCONF_RC1_TIMS3_Msk              (0x1UL << SYSCONF_RC1_TIMS3_Pos)
#define SYSCONF_RC1_TIMS3                  SYSCONF_RC1_TIMS3_Msk
#define SYSCONF_RC1_TIMS2_Pos              (18U)
#define SYSCONF_RC1_TIMS2_Msk              (0x1UL << SYSCONF_RC1_TIMS2_Pos)
#define SYSCONF_RC1_TIMS2                  SYSCONF_RC1_TIMS2_Msk
#define SYSCONF_RC1_TIMS1_Pos              (17U)
#define SYSCONF_RC1_TIMS1_Msk              (0x1UL << SYSCONF_RC1_TIMS1_Pos)
#define SYSCONF_RC1_TIMS1                  SYSCONF_RC1_TIMS1_Msk
#define SYSCONF_RC1_TIMS0_Pos              (16U)
#define SYSCONF_RC1_TIMS0_Msk              (0x1UL << SYSCONF_RC1_TIMS0_Pos)
#define SYSCONF_RC1_TIMS0                  SYSCONF_RC1_TIMS0_Msk
#define SYSCONF_RC1_CAN_Pos                (14U)
#define SYSCONF_RC1_CAN_Msk                (0x1UL << SYSCONF_RC1_CAN_Pos)
#define SYSCONF_RC1_CAN                    SYSCONF_RC1_CAN_Msk
#define SYSCONF_RC1_ACMP_Pos               (13U)
#define SYSCONF_RC1_ACMP_Msk               (0x1UL << SYSCONF_RC1_ACMP_Pos)
#define SYSCONF_RC1_ACMP                   SYSCONF_RC1_ACMP_Msk
#define SYSCONF_RC1_DAC_Pos                (12U)
#define SYSCONF_RC1_DAC_Msk                (0x1UL << SYSCONF_RC1_DAC_Pos)
#define SYSCONF_RC1_DAC                    SYSCONF_RC1_DAC_Msk
#define SYSCONF_RC1_SCI2_Pos               (10U)
#define SYSCONF_RC1_SCI2_Msk               (0x1UL << SYSCONF_RC1_SCI2_Pos)
#define SYSCONF_RC1_SCI2                   SYSCONF_RC1_SCI2_Msk
#define SYSCONF_RC1_SCI1_Pos               (9U)
#define SYSCONF_RC1_SCI1_Msk               (0x1UL << SYSCONF_RC1_SCI1_Pos)
#define SYSCONF_RC1_SCI1                   SYSCONF_RC1_SCI1_Msk
#define SYSCONF_RC1_SCI0_Pos               (8U)
#define SYSCONF_RC1_SCI0_Msk               (0x1UL << SYSCONF_RC1_SCI0_Pos)
#define SYSCONF_RC1_SCI0                   SYSCONF_RC1_SCI0_Msk
#define SYSCONF_RC1_SPI_Pos                (2U)
#define SYSCONF_RC1_SPI_Msk                (0x1UL << SYSCONF_RC1_SPI_Pos)
#define SYSCONF_RC1_SPI                    SYSCONF_RC1_SPI_Msk
#define SYSCONF_RC1_IOW_Pos                (1U)
#define SYSCONF_RC1_IOW_Msk                (0x1UL << SYSCONF_RC1_IOW_Pos)
#define SYSCONF_RC1_IOW                    SYSCONF_RC1_IOW_Msk

/*************************** Clock Monitor Register ***********************/
#define SYSCONF_CMNT_MCLKEN_Pos           (0U) /*M Clock monitor Enable*/
#define SYSCONF_CMNT_MCLKEN_Msk           (0x1UL << SYSCONF_CMNT_MCLKEN_Pos)
#define SYSCONF_CMNT_MCLKEN               SYSCONF_CMNT_MCLKEN_Msk
#define SYSCONF_CMNT_MCLKLPDEN_Pos        (1U) /*M Clock Loss Pending Enable*/
#define SYSCONF_CMNT_MCLKLPDEN_Msk        (0x1UL << SYSCONF_CMNT_MCLKLPDEN_Pos)
#define SYSCONF_CMNT_MCLKLPDEN            SYSCONF_CMNT_MCLKLPDEN_Msk
#define SYSCONF_CMNT_MCLKBAKVLD_Pos       (8U) /*M Clock Backup Status */
#define SYSCONF_CMNT_MCLKBAKVLD_Msk       (0x1UL << SYSCONF_CMNT_MCLKBAKVLD_Pos)
#define SYSCONF_CMNT_MCLKBAKVLD           SYSCONF_CMNT_MCLKBAKVLD_Msk
#define SYSCONF_CMNT_PLLEN_Pos            (16U) /*PLL Clock monitor Enable*/
#define SYSCONF_CMNT_PLLEN_Msk            (0x1UL << SYSCONF_CMNT_PLLEN_Pos)
#define SYSCONF_CMNT_PLLEN                SYSCONF_CMNT_PLLEN_Msk
#define SYSCONF_CMNT_PLLLPDEN_Pos         (17U) /*PLL Clock Loss Pending Enable*/
#define SYSCONF_CMNT_PLLLPDEN_Msk         (0x1UL << SYSCONF_CMNT_PLLLPDEN_Pos)
#define SYSCONF_CMNT_PLLLPDEN             SYSCONF_CMNT_PLLLPDEN_Msk
#define SYSCONF_CMNT_PLLBAKVLD_Pos        (24U) /*PLL Clock Backup Status */
#define SYSCONF_CMNT_PLLBAKVLD_Msk        (0x1UL << SYSCONF_CMNT_PLLBAKVLD_Pos)
#define SYSCONF_CMNT_PLLBAKVLD            SYSCONF_CMNT_PLLBAKVLD_Msk

/********************** Frequency Monitor Register 0********************/
#define SYSCONF_FMNT0_CLKOBJ_Pos          (16U) /* Target Clock Selection */
#define SYSCONF_FMNT0_CLKOBJ_Msk          (0x7UL << SYSCONF_FMNT0_CLKOBJ_Pos)
#define SYSCONF_FMNT0_CLKOBJ              SYSCONF_FMNT0_CLKOBJ_Msk
#define SYSCONF_FMNT0_CLKREFDIV_Pos       (12U) /* Reference Clock Division */
#define SYSCONF_FMNT0_CLKREFDIV_Msk       (0x7UL << SYSCONF_FMNT0_CLKREFDIV_Pos)
#define SYSCONF_FMNT0_CLKREFDIV           SYSCONF_FMNT0_CLKREFDIV_Msk
#define SYSCONF_FMNT0_CLKREF_Pos          (8U) /* Reference Clock Selection */
#define SYSCONF_FMNT0_CLKREF_Msk          (0x7UL << SYSCONF_FMNT0_CLKREF_Pos)
#define SYSCONF_FMNT0_CLKREF              SYSCONF_FMNT0_CLKREF_Msk
#define SYSCONF_FMNT0_FERRPNDEN_Pos       (1U) /* Frequency Error Pending Enable */
#define SYSCONF_FMNT0_FERRPNDEN_Msk       (0x1UL << SYSCONF_FMNT0_FERRPNDEN_Pos)
#define SYSCONF_FMNT0_FERRPNDEN           SYSCONF_FMNT0_FERRPNDEN_Msk
#define SYSCONF_FMNT0_FMNTEN_Pos          (0U) /* Frequency Monitor Enable */
#define SYSCONF_FMNT0_FMNTEN_Msk          (0x1UL << SYSCONF_FMNT0_FMNTEN_Pos)
#define SYSCONF_FMNT0_FMNTEN              SYSCONF_FMNT0_FMNTEN_Msk

/********************** Frequency Monitor Register 1********************/
#define SYSCONF_FMNT1_UPP_Pos              (16U) /* Upper Threshold */
#define SYSCONF_FMNT1_UPP_Msk              (0xFFFFUL << SYSCONF_FMNT1_UPP_Pos)
#define SYSCONF_FMNT1_UPP                  SYSCONF_FMNT1_UPP_Msk
#define SYSCONF_FMNT1_LOW_Pos              (0U) /* Lower Threshold */
#define SYSCONF_FMNT1_LOW_Msk              (0xFFFFUL << SYSCONF_FMNT1_LOW_Pos)
#define SYSCONF_FMNT1_LOW                  SYSCONF_FMNT1_LOW_Msk

/********************** Frequency Monitor Register 2********************/
#define SYSCONF_FMNT2_ME_Pos               (16U) /* Measure End */
#define SYSCONF_FMNT2_ME_Msk               (0xFFUL << SYSCONF_FMNT2_ME_Pos)
#define SYSCONF_FMNT2_ME                   SYSCONF_FMNT2_ME_Msk
#define SYSCONF_FMNT2_CLKPRD_Pos           (0U) /* Measure End */
#define SYSCONF_FMNT2_CLKPRD_Msk           (0xFFFFUL << SYSCONF_FMNT2_CLKPRD_Pos)
#define SYSCONF_FMNT2_CLKPRD               SYSCONF_FMNT2_CLKPRD_Msk

/*************************** NMI Status Register ***********************/
#define SYSCONF_NMI_MCLKLPD_Pos           (0U) /* M Clock Loss pending */
#define SYSCONF_NMI_MCLKLPD_Msk           (0x1UL << SYSCONF_NMI_MCLKLPD_Pos)
#define SYSCONF_NMI_MCLKLPD               SYSCONF_NMI_MCLKLPD_Msk
#define SYSCONF_NMI_PLLLPD_Pos            (1U) /* PLL Clock Loss pending */
#define SYSCONF_NMI_PLLLPD_Msk            (0x1UL << SYSCONF_NMI_PLLLPD_Pos)
#define SYSCONF_NMI_PLLLPD                SYSCONF_NMI_PLLLPD_Msk
#define SYSCONF_NMI_FRQERR_Pos            (2U) /* Frequence Error */
#define SYSCONF_NMI_FRQERR_Msk            (0x1UL << SYSCONF_NMI_FRQERR_Pos)
#define SYSCONF_NMI_FRQERR                SYSCONF_NMI_FRQERR_Msk

/*************************** HOSC Monitor 0 Register **********************/
#define SYSCONF_HMON0_EN_Pos               (0U) /*HOSC monitor enable*/
#define SYSCONF_HMON0_EN_Msk               (0x1UL << SYSCONF_HMON0_EN_Pos)
#define SYSCONF_HMON0_EN                   SYSCONF_HMON0_EN_Msk
#define SYSCONF_HMON0_LOSSPEND_Pos         (1U) /*HOSC loss pend to NMI interrupt enable*/
#define SYSCONF_HMON0_LOSSPEND_Msk         (0x1UL << SYSCONF_HMON0_LOSSPEND_Pos)
#define SYSCONF_HMON0_LOSSPEND             SYSCONF_HMON0_LOSSPEND_Msk

/*************************** HOSC Monitor 1 Register **********************/
#define SYSCONF_HMON1_PRDLOW_Pos           (0U) /* HOSC lower-limit */
#define SYSCONF_HMON1_PRDLOW_Msk           (0xFFFUL << SYSCONF_HMON1_PRDLOW_Pos)
#define SYSCONF_HMON1_PRDLOW               SYSCONF_HMON1_PRDLOW_Msk
#define SYSCONF_HMON1_PRDHIGH_Pos          (16U) /* HOSC upper-limit */
#define SYSCONF_HMON1_PRDHIGH_Msk          (0xFFFUL << SYSCONF_HMON1_PRDHIGH_Pos)
#define SYSCONF_HMON1_PRDHIGH              SYSCONF_HMON1_PRDHIGH_Msk

/*************************** LOSC Monitor Register ************************/
#define SYSCONF_LMON_EN_Pos                (0U) /*LOSC monitor enable*/
#define SYSCONF_LMON_EN_Msk                (0x1UL << SYSCONF_LMON_EN_Pos)
#define SYSCONF_LMON_EN                    SYSCONF_LMON_EN_Msk
#define SYSCONF_LMON_LOSSPEND_Pos          (1U) /*LOSC loss pend to NMI interrupt enable*/
#define SYSCONF_LMON_LOSSPEND_Msk          (0x1UL << SYSCONF_LMON_LOSSPEND_Pos)
#define SYSCONF_LMON_LOSSPEND              SYSCONF_LMON_LOSSPEND_Msk


/****************************** pll Control Register **********************/
#define SYSCONF_PLL_EN_Pos                 (0U)  /*PLL enable*/
#define SYSCONF_PLL_EN_Msk                 (0x1UL << SYSCONF_PLL_EN_Pos)
#define SYSCONF_PLL_EN                     SYSCONF_PLL_EN_Msk

#define SYSCONF_PLL_CLKSRC_Pos             (1U)   /*PLL clock source*/
#define SYSCONF_PLL_CLKSRC_Msk             (0x3UL << SYSCONF_PLL_CLKSRC_Pos)
#define SYSCONF_PLL_CLKSRC                 SYSCONF_PLL_CLKSRC_Msk

#define SYSCONF_PLL_DIV_Pos                (4U)   /* pll prescaler */
#define SYSCONF_PLL_DIV_Msk                (0xFFUL << SYSCONF_PLL_DIV_Pos)
#define SYSCONF_PLL_DIV                    SYSCONF_PLL_DIV_Msk

#define SYSCONF_PLL_PLLN_Pos               (12U)  /* pll N */
#define SYSCONF_PLL_PLLN_Msk               (0x7FUL << SYSCONF_PLL_PLLN_Pos)
#define SYSCONF_PLL_PLLN                   SYSCONF_PLL_PLLN_Msk

#define SYSCONF_PLL_PLLM_Pos               (20U)  /* pll M */
#define SYSCONF_PLL_PLLM_Msk               (0xFUL << SYSCONF_PLL_PLLM_Pos)
#define SYSCONF_PLL_PLLM                   SYSCONF_PLL_PLLM_Msk

#define SYSCONF_PLL_KVCO_Pos               (24U)  /* VCO */
#define SYSCONF_PLL_KVCO_Msk               (0x3UL << SYSCONF_PLL_KVCO_Pos)
#define SYSCONF_PLL_KVCO                   SYSCONF_PLL_KVCO_Msk

#define SYSCONF_PLL_CPC_Pos                (26U)  /* CPC */
#define SYSCONF_PLL_CPC_Msk                (0x3UL << SYSCONF_PLL_CPC_Pos)
#define SYSCONF_PLL_CPC                    SYSCONF_PLL_CPC_Msk

#define SYSCONF_PLL_LDO_Pos                (28U)  /* LDO */
#define SYSCONF_PLL_LDO_Msk                (0x7UL << SYSCONF_PLL_LDO_Pos)
#define SYSCONF_PLL_LDO                    SYSCONF_PLL_LDO_Msk

#define SYSCONF_PLL_CLKOUTEN_Pos           (31U)  /* Clock output enable */
#define SYSCONF_PLL_CLKOUTEN_Msk           (0x7UL << SYSCONF_PLL_CLKOUTEN_Pos)
#define SYSCONF_PLL_CLKOUTEN               SYSCONF_PLL_CLKOUTEN_Msk

/************************* Clock Selection 0 Register *************************/
#define SYSCONF_CSEL0_MCLK_Pos             (0U) /* M Clock Sources */
#define SYSCONF_CSEL0_MCLK_Msk             (0x1UL << SYSCONF_CSEL0_MCLK_Pos)
#define SYSCONF_CSEL0_MCLK                 SYSCONF_CSEL0_MCLK_Msk
#define SYSCONF_CSEL0_FCLK_Pos             (2U) /* System Clock (FCLK) Sources */
#define SYSCONF_CSEL0_FCLK_Msk             (0x1UL << SYSCONF_CSEL0_FCLK_Pos)
#define SYSCONF_CSEL0_FCLK                 SYSCONF_CSEL0_FCLK_Msk
#define SYSCONF_CSEL0_FDIV_Pos             (4U) /* Division for System Clock */
#define SYSCONF_CSEL0_FDIV_Msk             (0xFUL << SYSCONF_CSEL0_FDIV_Pos)
#define SYSCONF_CSEL0_FDIV                 SYSCONF_CSEL0_FDIV_Msk
#define SYSCONF_CSEL0_MCLKST_Pos           (16U) /* To indicate M Clock source */
#define SYSCONF_CSEL0_MCLKST_Msk           (0x1UL << SYSCONF_CSEL0_MCLKST_Pos)
#define SYSCONF_CSEL0_MCLKST               SYSCONF_CSEL0_MCLKST_Msk
#define SYSCONF_CSEL0_FCLKST_Pos           (18U) /* To indicate the real system clock source */
#define SYSCONF_CSEL0_FCLKST_Msk           (0x1UL << SYSCONF_CSEL0_FCLKST_Pos)
#define SYSCONF_CSEL0_FCLKST               SYSCONF_CSEL0_FCLKST_Msk
#define SYSCONF_CSEL0_SWERR_Pos            (20U) /* Clock Switch Error. It happens when the working clock source is not the expected one. */
#define SYSCONF_CSEL0_SWERR_Msk            (0x1UL << SYSCONF_CSEL0_SWERR_Pos)
#define SYSCONF_CSEL0_SWERR                SYSCONF_CSEL0_SWERR_Msk

/************************* Clock Selection 1 Register *************************/
#define SYSCONF_CSEL1_BUZ1DIV_Pos          (8U) /* BUZ1 Division */
#define SYSCONF_CSEL1_BUZ1DIV_Msk          (0xFUL << SYSCONF_CSEL1_BUZ1DIV_Pos)
#define SYSCONF_CSEL1_BUZ1DIV              SYSCONF_CSEL1_BUZ1DIV_Msk


/************************* Clock Selection 2 Register *************************/
#define SYSCONF_CSEL2_ADC1DIV_Pos          (24U) /* Division for ADC Clock */
#define SYSCONF_CSEL2_ADC1DIV_Msk          (0x7FUL << SYSCONF_CSEL2_ADC1DIV_Pos)
#define SYSCONF_CSEL2_ADC1DIV              SYSCONF_CSEL2_ADC1DIV_Msk
#define SYSCONF_CSEL2_BTIM_Pos             (22U) /*BaseTimer clock selection*/
#define SYSCONF_CSEL2_BTIM_Msk             (0x1UL << SYSCONF_CSEL2_BTIM_Pos)
#define SYSCONF_CSEL2_BTIM_LIRC            (0x0UL << SYSCONF_CSEL2_BTIM_Pos)
#define SYSCONF_CSEL2_BTIM_MCLK            (0x1UL << SYSCONF_CSEL2_BTIM_Pos)
#define SYSCONF_CSEL2_WDT_Pos              (20U) /*WDT clock selection*/
#define SYSCONF_CSEL2_WDT_Msk              (0x1UL << SYSCONF_CSEL2_WDT_Pos)
#define SYSCONF_CSEL2_WDT_LIRC             (0x0UL << SYSCONF_CSEL2_WDT_Pos)
#define SYSCONF_CSEL2_WDT_MCLK             (0x1UL << SYSCONF_CSEL2_WDT_Pos)
#define SYSCONF_CSEL2_BUZ1_Pos             (16U) /*BUZZ1 clock selection*/
#define SYSCONF_CSEL2_BUZ1_Msk             (0x7UL << SYSCONF_CSEL2_BUZ1_Pos)
#define SYSCONF_CSEL2_BUZ1_FCLK            (0x0UL << SYSCONF_CSEL2_BUZ1_Pos)
#define SYSCONF_CSEL2_BUZ1_HIRC            (0x1UL << SYSCONF_CSEL2_BUZ1_Pos)
#define SYSCONF_CSEL2_BUZ1_LIRC            (0x2UL << SYSCONF_CSEL2_BUZ1_Pos)
#define SYSCONF_CSEL2_BUZ1_HOSC            (0x3UL << SYSCONF_CSEL2_BUZ1_Pos)
#define SYSCONF_CSEL2_BUZ1_HIRC2           (0x4UL << SYSCONF_CSEL2_BUZ1_Pos)
#define SYSCONF_CSEL2_BUZ1_PLL             (0x5UL << SYSCONF_CSEL2_BUZ1_Pos)
#define SYSCONF_CSEL2_NFCLK2_Pos           (14U) /* NF Clock 2*/
#define SYSCONF_CSEL2_NFCLK2_Msk           (0x3UL << SYSCONF_CSEL2_NFCLK2_Pos)
#define SYSCONF_CSEL2_NFCLK2               SYSCONF_CSEL2_NFCLK2_Msk
#define SYSCONF_CSEL2_NFCLK1_Pos           (11U) /* NF Clock 1*/
#define SYSCONF_CSEL2_NFCLK1_Msk           (0x7UL << SYSCONF_CSEL2_NFCLK1_Pos)
#define SYSCONF_CSEL2_NFCLK1               SYSCONF_CSEL2_NFCLK1_Msk
#define SYSCONF_CSEL2_NFCLK0_Pos           (9U) /* NF Clock 0*/
#define SYSCONF_CSEL2_NFCLK0_Msk           (0x3UL << SYSCONF_CSEL2_NFCLK0_Pos)
#define SYSCONF_CSEL2_NFCLK0               SYSCONF_CSEL2_NFCLK0_Msk
#define SYSCONF_CSEL2_NF_Pos               (8U) /* NF Clock */
#define SYSCONF_CSEL2_NF_Msk               (0x1UL << SYSCONF_CSEL2_NF_Pos)
#define SYSCONF_CSEL2_NF                   SYSCONF_CSEL2_NF_Msk
#define SYSCONF_CSEL2_NF_HIRC              (0x0UL << SYSCONF_CSEL2_NF_Pos)
#define SYSCONF_CSEL2_NF_FCLK              (0x1UL << SYSCONF_CSEL2_NF_Pos)
#define SYSCONF_CSEL2_ADC0DIV_Pos          (0U) /* Division for ADC Clock */
#define SYSCONF_CSEL2_ADC0DIV_Msk          (0x7FUL << SYSCONF_CSEL2_ADC0DIV_Pos)
#define SYSCONF_CSEL2_ADC0DIV              SYSCONF_CSEL2_ADC0DIV_Msk


/************************* Clock Selection 3 Register *************************/


/************************* IO mode Configuration 3 Register *************************/
/* SYSCONF_IOMDSEL_PORTX_0_Pos, in which x means A, B, C ,D for KPM32xx Serials MCU */
#define SYSCONF_IOMDSEL_PORTX_0_Pos        (0U)  /*Port X.0*/
#define SYSCONF_IOMDSEL_PORTX_0_Msk        (0x3UL << SYSCONF_IOMDSEL_PORTX_0_Pos)
#define SYSCONF_IOMDSEL_PORTX_1_Pos        (2U)  /*Port X.1*/
#define SYSCONF_IOMDSEL_PORTX_1_Msk        (0x3UL << SYSCONF_IOMDSEL_PORTX_1_Pos)
#define SYSCONF_IOMDSEL_PORTX_2_Pos        (4U)  /*Port X.2*/
#define SYSCONF_IOMDSEL_PORTX_2_Msk        (0x3UL << SYSCONF_IOMDSEL_PORTX_2_Pos)
#define SYSCONF_IOMDSEL_PORTX_3_Pos        (6U)  /*Port X.3*/
#define SYSCONF_IOMDSEL_PORTX_3_Msk        (0x3UL << SYSCONF_IOMDSEL_PORTX_3_Pos)
#define SYSCONF_IOMDSEL_PORTX_4_Pos        (8U)  /*Port X.4*/
#define SYSCONF_IOMDSEL_PORTX_4_Msk        (0x3UL << SYSCONF_IOMDSEL_PORTX_4_Pos)
#define SYSCONF_IOMDSEL_PORTX_5_Pos        (10U)  /*Port X.5*/
#define SYSCONF_IOMDSEL_PORTX_5_Msk        (0x3UL << SYSCONF_IOMDSEL_PORTX_5_Pos)
#define SYSCONF_IOMDSEL_PORTX_6_Pos        (12U)  /*Port X.6*/
#define SYSCONF_IOMDSEL_PORTX_6_Msk        (0x3UL << SYSCONF_IOMDSEL_PORTX_6_Pos)
#define SYSCONF_IOMDSEL_PORTX_7_Pos        (14U)  /*Port X.7*/
#define SYSCONF_IOMDSEL_PORTX_7_Msk        (0x3UL << SYSCONF_IOMDSEL_PORTX_7_Pos)
#define SYSCONF_IOMDSEL_PORTX_8_Pos        (16U)  /*Port X.8*/
#define SYSCONF_IOMDSEL_PORTX_8_Msk        (0x3UL << SYSCONF_IOMDSEL_PORTX_0_Pos)
#define SYSCONF_IOMDSEL_PORTX_9_Pos        (18U)  /*Port X.9*/
#define SYSCONF_IOMDSEL_PORTX_9_Msk        (0x3UL << SYSCONF_IOMDSEL_PORTX_1_Pos)
#define SYSCONF_IOMDSEL_PORTX_10_Pos       (20U)  /*Port X.10*/
#define SYSCONF_IOMDSEL_PORTX_10_Msk       (0x3UL << SYSCONF_IOMDSEL_PORTX_2_Pos)
#define SYSCONF_IOMDSEL_PORTX_11_Pos       (22U)  /*Port X.11*/
#define SYSCONF_IOMDSEL_PORTX_11_Msk       (0x3UL << SYSCONF_IOMDSEL_PORTX_3_Pos)
#define SYSCONF_IOMDSEL_PORTX_12_Pos       (24U)  /*Port X.12*/
#define SYSCONF_IOMDSEL_PORTX_12_Msk       (0x3UL << SYSCONF_IOMDSEL_PORTX_4_Pos)
#define SYSCONF_IOMDSEL_PORTX_13_Pos       (26U)  /*Port X.13*/
#define SYSCONF_IOMDSEL_PORTX_13_Msk       (0x3UL << SYSCONF_IOMDSEL_PORTX_5_Pos)
#define SYSCONF_IOMDSEL_PORTX_14_Pos       (28U)  /*Port X.14*/
#define SYSCONF_IOMDSEL_PORTX_14_Msk       (0x3UL << SYSCONF_IOMDSEL_PORTX_6_Pos)
#define SYSCONF_IOMDSEL_PORTX_15_Pos       (30U)  /*Port X.15*/
#define SYSCONF_IOMDSEL_PORTX_15_Msk       (0x3UL << SYSCONF_IOMDSEL_PORTX_7_Pos)

#define SYSCONF_IOMDSEL_PIN_GPIO           (0x00UL)
#define SYSCONF_IOMDSEL_PIN_ALTF           (0x01UL)
#define SYSCONF_IOMDSEL_PIN_ANAL           (0x02UL)
#define SYSCONF_IOMDSEL_PIN_MASK           (0x03UL)

/***************** alternate Function Selection Register **********************/
/* SYSCONF_AF_PORTX_0_Pos, in which x means A, B, C ,D for KPM32xx Serials MCU */
#define SYSCONF_AF_PORTX_0_Pos             (0U)  /*Port X.0*/
#define SYSCONF_AF_PORTX_0_Msk             (0xFUL << SYSCONF_AF_PORTX_0_Pos)
#define SYSCONF_AF_PORTX_0                 SYSCONF_AF_PORTX_0_Msk
#define SYSCONF_AF_PORTX_1_Pos             (4U)  /*Port X.1*/
#define SYSCONF_AF_PORTX_1_Msk             (0xFUL << SYSCONF_AF_PORTX_1_Pos)
#define SYSCONF_AF_PORTX_1                 SYSCONF_AF_PORTX_1_Msk
#define SYSCONF_AF_PORTX_2_Pos             (8U)  /*Port X.2*/
#define SYSCONF_AF_PORTX_2_Msk             (0xFUL << SYSCONF_AF_PORTX_2_Pos)
#define SYSCONF_AF_PORTX_2                 SYSCONF_AF_PORTX_2_Msk
#define SYSCONF_AF_PORTX_3_Pos             (12U) /*Port X.3*/
#define SYSCONF_AF_PORTX_3_Msk             (0xFUL << SYSCONF_AF_PORTX_3_Pos)
#define SYSCONF_AF_PORTX_3                 SYSCONF_AF_PORTX_3_Msk
#define SYSCONF_AF_PORTX_4_Pos             (16U) /*Port X.4*/
#define SYSCONF_AF_PORTX_4_Msk             (0xFUL << SYSCONF_AF_PORTX_4_Pos)
#define SYSCONF_AF_PORTX_4                 SYSCONF_AF_PORTX_4_Msk
#define SYSCONF_AF_PORTX_5_Pos             (20U) /*Port X.5*/
#define SYSCONF_AF_PORTX_5_Msk             (0xFUL << SYSCONF_AF_PORTX_5_Pos)
#define SYSCONF_AF_PORTX_5                 SYSCONF_AF_PORTX_5_Msk
#define SYSCONF_AF_PORTX_6_Pos             (24U) /*Port X.6*/
#define SYSCONF_AF_PORTX_6_Msk             (0xFUL << SYSCONF_AF_PORTX_6_Pos)
#define SYSCONF_AF_PORTX_6                 SYSCONF_AF_PORTX_6_Msk
#define SYSCONF_AF_PORTX_7_Pos             (28U) /*Port X.7*/
#define SYSCONF_AF_PORTX_7_Msk             (0xFUL << SYSCONF_AF_PORTX_7_Pos)
#define SYSCONF_AF_PORTX_7                 SYSCONF_AF_PORTX_7_Msk
#define SYSCONF_AF_PORTX_8_Pos             (0U) /*Port X.8*/
#define SYSCONF_AF_PORTX_8_Msk             (0xFUL << SYSCONF_AF_PORTX_8_Pos)
#define SYSCONF_AF_PORTX_8                 SYSCONF_AF_PORTX_8_Msk
#define SYSCONF_AF_PORTX_9_Pos             (4U) /*Port X.9*/
#define SYSCONF_AF_PORTX_9_Msk             (0xFUL << SYSCONF_AF_PORTX_9_Pos)
#define SYSCONF_AF_PORTX_9                 SYSCONF_AF_PORTX_9_Msk
#define SYSCONF_AF_PORTX_10_Pos            (8U) /*Port X.10*/
#define SYSCONF_AF_PORTX_10_Msk            (0xFUL << SYSCONF_AF_PORTX_10_Msk)
#define SYSCONF_AF_PORTX_10                SYSCONF_AF_PORTX_10_Msk
#define SYSCONF_AF_PORTX_11_Pos            (12U) /*Port X.11*/
#define SYSCONF_AF_PORTX_11_Msk            (0xFUL << SYSCONF_AF_PORTX_11_Pos)
#define SYSCONF_AF_PORTX_11                SYSCONF_AF_PORTX_11_Msk
#define SYSCONF_AF_PORTX_12_Pos            (16U) /*Port X.12*/
#define SYSCONF_AF_PORTX_12_Msk            (0xFUL << SYSCONF_AF_PORTX_12_Pos)
#define SYSCONF_AF_PORTX_12                SYSCONF_AF_PORTX_12_Msk
#define SYSCONF_AF_PORTX_13_Pos            (20U) /*Port X.13*/
#define SYSCONF_AF_PORTX_13_Msk            (0xFUL << SYSCONF_AF_PORTX_13_Pos)
#define SYSCONF_AF_PORTX_13                SYSCONF_AF_PORTX_13_Msk
#define SYSCONF_AF_PORTX_14_Pos            (24U) /*Port X.14*/
#define SYSCONF_AF_PORTX_14_Msk            (0xFUL << SYSCONF_AF_PORTX_14_Pos)
#define SYSCONF_AF_PORTX_14                SYSCONF_AF_PORTX_14_Msk
#define SYSCONF_AF_PORTX_15_Pos            (28U) /*Port X.15*/
#define SYSCONF_AF_PORTX_15_Msk            (0xFUL << SYSCONF_AF_PORTX_15_Pos)
#define SYSCONF_AF_PORTX_15                SYSCONF_AF_PORTX_15_Msk
#define SYSCONF_AF_MASK                    (0x0FU)


/******************************************************************************/
/*                                                                            */
/*                                Hardware TMU                                */
/*                                                                            */
/******************************************************************************/

/******************************* TMU Control register *************************/
#define TMU_CTRL_EN_Pos               (0U) /* TMU Enable */
#define TMU_CTRL_EN_Msk               (0x1UL << TMU_CTRL_EN_Pos)
#define TMU_CTRL_EN                   TMU_CTRL_EN_Msk

/******************************* TMU Status register *************************/
#define TMU_ST_BUSY_Pos               (0U) /* Busy Status */
#define TMU_ST_BUSY_Msk               (0x1UL << TMU_ST_BUSY_Pos)
#define TMU_ST_BUSY                   TMU_ST_BUSY_Msk


/*****************************************************************************/
/*                                                                           */
/*                                   SCI                                     */
/*                                                                           */
/*****************************************************************************/
/*********************** SCI Clock Prescaler register ************************/
#define SCI_CLK_PRESCALER_Pos         (0U)   /* SCI Clock prescaler */
#define SCI_CLK_PRESCALER_Msk         (0xFUL << SCI_CLK_PRESCALER_Pos)
#define SCI_CLK_PRESCALER             SCI_CLK_PRESCALER_Msk

/************************** SCI Baudrate register ***************************/
#define SCI_BDR_DIV2_Pos              (16U)   /* Baudrate division 2*/
#define SCI_BDR_DIV2_Msk              (0x1FFUL << SCI_BDR_DIV2_Pos)
#define SCI_BDR_DIV2                  SCI_BDR_DIV2_Msk
#define SCI_BDR_DIV1_Pos              (0U)    /* Baudrate division 1*/
#define SCI_BDR_DIV1_Msk              (0x1FUL << SCI_BDR_DIV1_Pos)
#define SCI_BDR_DIV1                  SCI_BDR_DIV1_Msk

/************************** SCI Mode Control register ***********************/
#define SCI_MC_CLKFILT_Pos            (25U)   /* Clock Filter Enable*/
#define SCI_MC_CLKFILT_Msk            (0x1UL << SCI_MC_CLKFILT_Pos)
#define SCI_MC_CLKFILT                SCI_MC_CLKFILT_Msk
#define SCI_MC_DATFILT_Pos            (24U)   /* Data Filter Enable*/
#define SCI_MC_DATFILT_Msk            (0x1UL << SCI_MC_DATFILT_Pos)
#define SCI_MC_DATFILT                SCI_MC_DATFILT_Msk
#define SCI_MC_SCKOUT_Pos             (18U)   /* SCK Output */
#define SCI_MC_SCKOUT_Msk             (0x3UL << SCI_MC_SCKOUT_Pos)
#define SCI_MC_SCKOUT                 SCI_MC_SCKOUT_Msk
#define SCI_MC_DATOUT_Pos             (16U)   /* Data Output*/
#define SCI_MC_DATOUT_Msk             (0x3UL << SCI_MC_DATOUT_Pos)
#define SCI_MC_DATOUT                 SCI_MC_DATOUT_Msk
#define SCI_MC_DATOUT_HWC             (0x0UL << SCI_MC_DATOUT_Pos)
#define SCI_MC_DATOUT_LOW             (0x2UL << SCI_MC_DATOUT_Pos)
#define SCI_MC_DATOUT_HIGH            (0x3UL << SCI_MC_DATOUT_Pos)
#define SCI_MC_DATINV_Pos             (13U)   /* Data Inverse*/
#define SCI_MC_DATINV_Msk             (0x1UL << SCI_MC_DATINV_Pos)
#define SCI_MC_DATINV                 SCI_MC_DATINV_Msk
#define SCI_MC_DATDIR_Pos             (12U)   /* Data Direction*/
#define SCI_MC_DATDIR_Msk             (0x1UL << SCI_MC_DATDIR_Pos)
#define SCI_MC_DATDIR                 SCI_MC_DATDIR_Msk
#define SCI_MC_RXEN_Pos               (9U)   /* Data RX Enable*/
#define SCI_MC_RXEN_Msk               (0x1UL << SCI_MC_RXEN_Pos)
#define SCI_MC_RXEN                   SCI_MC_RXEN_Msk
#define SCI_MC_TXEN_Pos               (8U)   /* Data TX Enable*/
#define SCI_MC_TXEN_Msk               (0x1UL << SCI_MC_TXEN_Pos)
#define SCI_MC_TXEN                   SCI_MC_TXEN_Msk
#define SCI_MC_DMAEN_Pos              (4U)   /* DMA Enable*/
#define SCI_MC_DMAEN_Msk              (0x1UL << SCI_MC_DMAEN_Pos)
#define SCI_MC_DMAEN                  SCI_MC_DMAEN_Msk
#define SCI_MC_MODSEL_Pos             (0U)   /* DMA Mode Selection*/
#define SCI_MC_MODSEL_Msk             (0x7UL << SCI_MC_MODSEL_Pos)
#define SCI_MC_MODSEL                 SCI_MC_MODSEL_Msk

/************************ SCI Uart Mode Control register *******************/
#define SCI_MUART_MPBIT_Pos           (10U)   /* Multi-processor Bit */
#define SCI_MUART_MPBIT_Msk           (0x1UL << SCI_MUART_MPBIT_Pos)
#define SCI_MUART_MPBIT               SCI_MUART_MPBIT_Msk
#define SCI_MUART_MPDETEN_Pos         (9U)   /* Multi-processor Address Detection */
#define SCI_MUART_MPDETEN_Msk         (0x1UL << SCI_MUART_MPDETEN_Pos)
#define SCI_MUART_MPDETEN             SCI_MUART_MPDETEN_Msk
#define SCI_MUART_MPEN_Pos            (8U)   /* Multi-processor Enable */
#define SCI_MUART_MPEN_Msk            (0x1UL << SCI_MUART_MPEN_Pos)
#define SCI_MUART_MPEN                SCI_MUART_MPEN_Msk
#define SCI_MUART_RTSEN_Pos           (6U)   /* RTS Enable */
#define SCI_MUART_RTSEN_Msk           (0x1UL << SCI_MUART_RTSEN_Pos)
#define SCI_MUART_RTSEN               SCI_MUART_RTSEN_Msk
#define SCI_MUART_CTSEN_Pos           (5U)   /* CTS Enable */
#define SCI_MUART_CTSEN_Msk           (0x1UL << SCI_MUART_CTSEN_Pos)
#define SCI_MUART_CTSEN               SCI_MUART_CTSEN_Msk
#define SCI_MUART_STPLEN_Pos          (4U)   /* Stop Bit Length */
#define SCI_MUART_STPLEN_Msk          (0x1UL << SCI_MUART_STPLEN_Pos)
#define SCI_MUART_STPLEN              SCI_MUART_STPLEN_Msk
#define SCI_MUART_PRTSEL_Pos          (2U)   /* Parity Selection */
#define SCI_MUART_PRTSEL_Msk          (0x3UL << SCI_MUART_PRTSEL_Pos)
#define SCI_MUART_PRTSEL              SCI_MUART_PRTSEL_Msk
#define SCI_MUART_DATLEN_Pos          (0U)   /* Data Bit Length */
#define SCI_MUART_DATLEN_Msk          (0x3UL << SCI_MUART_DATLEN_Pos)
#define SCI_MUART_DATLEN              SCI_MUART_DATLEN_Msk


/************************ SCI SPI Mode Control register *******************/
#define SCI_MSPI_CSDEACT_Pos          (8U)   /* SPI CS Deactive */
#define SCI_MSPI_CSDEACT_Msk          (0x1UL << SCI_MSPI_CSDEACT_Pos)
#define SCI_MSPI_CSDEACT              SCI_MSPI_CSDEACT_Msk
#define SCI_MSPI_CSEN_Pos             (3U)   /* SPI CS Enable */
#define SCI_MSPI_CSEN_Msk             (0x1UL << SCI_MSPI_CSEN_Pos)
#define SCI_MSPI_CSEN                 SCI_MSPI_CSEN_Msk
#define SCI_MSPI_MODSEL_Pos           (2U)   /* SPI Master or Slave Selection */
#define SCI_MSPI_MODSEL_Msk           (0x1UL << SCI_MSPI_MODSEL_Pos)
#define SCI_MSPI_MODSEL               SCI_MSPI_MODSEL_Msk
#define SCI_MSPI_CLKPHS_Pos           (1U)   /* SPI Clock Phase */
#define SCI_MSPI_CLKPHS_Msk           (0x1UL << SCI_MSPI_CLKPHS_Pos)
#define SCI_MSPI_CLKPHS               SCI_MSPI_CLKPHS_Msk
#define SCI_MSPI_DATPHS_Pos           (0U)   /* SPI Data Phase */
#define SCI_MSPI_DATPHS_Msk           (0x1UL << SCI_MSPI_DATPHS_Pos)
#define SCI_MSPI_DATPHS               SCI_MSPI_DATPHS_Msk


/************************* SCI I2C Mode Control register *********************/
#define SCI_MI2C_STP_Pos              (14U)   /* I2C Stop Signal triggered */
#define SCI_MI2C_STP_Msk              (0x1UL << SCI_MI2C_STP_Pos)
#define SCI_MI2C_STP                  SCI_MI2C_STP_Msk
#define SCI_MI2C_RSTT_Pos             (13U)   /* I2C Restart Signal triggered */
#define SCI_MI2C_RSTT_Msk             (0x1UL << SCI_MI2C_RSTT_Pos)
#define SCI_MI2C_RSTT                 SCI_MI2C_RSTT_Msk
#define SCI_MI2C_STT_Pos              (12U)   /* I2C Start Signal triggered */
#define SCI_MI2C_STT_Msk              (0x1UL << SCI_MI2C_STT_Pos)
#define SCI_MI2C_STT                  SCI_MI2C_STT_Msk
#define SCI_MI2C_FSTMOD_Pos           (6U)   /* I2C Fast Mode, usually for 1Mbps Communication */
#define SCI_MI2C_FSTMOD_Msk           (0x1UL << SCI_MI2C_FSTMOD_Pos)
#define SCI_MI2C_FSTMOD               SCI_MI2C_FSTMOD_Msk
#define SCI_MI2C_HOLD_Pos             (8U)   /* I2C SDA Hold Time */
#define SCI_MI2C_HOLD_Msk             (0x3UL << SCI_MI2C_HOLD_Pos)
#define SCI_MI2C_HOLD                 SCI_MI2C_HOLD_Msk
#define SCI_MI2C_CLKSYNC_Pos          (5U)   /* I2C Clock Sync Enable */
#define SCI_MI2C_CLKSYNC_Msk          (0x1UL << SCI_MI2C_CLKSYNC_Pos)
#define SCI_MI2C_CLKSYNC              SCI_MI2C_CLKSYNC_Msk
#define SCI_MI2C_ACK_Pos              (4U)   /* Send I2C ACK and NACK */
#define SCI_MI2C_ACK_Msk              (0x1UL << SCI_MI2C_ACK_Pos)
#define SCI_MI2C_ACK                  SCI_MI2C_ACK_Msk

/************************* SCI Sync Mode Control register *********************/
#define SCI_MSYNC_CLKTRG_Pos         (16U)   /* Sync Interface Clock Triger */
#define SCI_MSYNC_CLKTRG_Msk         (0x1UL << SCI_MSYNC_CLKTRG_Pos)
#define SCI_MSYNC_CLKTRG             SCI_MSYNC_CLKTRG_Msk
#define SCI_MSYNC_RTS_Pos            (6U)   /* Sync Interface RTS Enable */
#define SCI_MSYNC_RTS_Msk            (0x1UL << SCI_MSYNC_RTS_Pos)
#define SCI_MSYNC_RTS                SCI_MSYNC_RTS_Msk
#define SCI_MSYNC_CTS_Pos            (5U)   /* Sync Interface CTS Enable */
#define SCI_MSYNC_CTS_Msk            (0x1UL << SCI_MSYNC_CTS_Pos)
#define SCI_MSYNC_CTS                SCI_MSYNC_CTS_Msk
#define SCI_MSYNC_MODSEL_Pos         (2U)   /* Sync Interface Master or Slave Selection */
#define SCI_MSYNC_MODSEL_Msk         (0x1UL << SCI_MSYNC_MODSEL_Pos)
#define SCI_MSYNC_MODSEL             SCI_MSYNC_MODSEL_Msk

/************************* SCI Sync Mode Control register *********************/
#define SCI_MSC_PRTSEL_Pos           (2U)   /* Parity Selection */
#define SCI_MSC_PRTSEL_Msk           (0x3UL << SCI_MSC_PRTSEL_Pos)
#define SCI_MSC_PRTSEL               SCI_MSC_PRTSEL_Msk
#define SCI_MSC_BLKEN_Pos            (0U)   /* Smart Card Block Enable */
#define SCI_MSC_BLKEN_Msk            (0x1UL << SCI_MSC_BLKEN_Pos)
#define SCI_MSC_BLKEN                SCI_MSC_BLKEN_Msk

/***************************** SCI Status register ***************************/
#define SCI_ST_RXGOING_Pos           (14U)   /* SCI RX going */
#define SCI_ST_RXGOING_Msk           (0x1UL << SCI_ST_RXGOING_Pos)
#define SCI_ST_RXGOING               SCI_ST_RXGOING_Msk
#define SCI_ST_TXGOING_Pos           (13U)   /* SCI TX going */
#define SCI_ST_TXGOING_Msk           (0x1UL << SCI_ST_TXGOING_Pos)
#define SCI_ST_TXGOING               SCI_ST_TXGOING_Msk
#define SCI_ST_NACK_Pos              (12U)   /* SCI IIC NACK */
#define SCI_ST_NACK_Msk              (0x1UL << SCI_ST_NACK_Pos)
#define SCI_ST_NACK                  SCI_ST_NACK_Msk
#define SCI_ST_FNS_Pos               (10U)   /* SCI IIC Restart and Finish complete */
#define SCI_ST_FNS_Msk               (0x1UL << SCI_ST_FNS_Pos)
#define SCI_ST_FNS                   SCI_ST_FNS_Msk
#define SCI_ST_SMC_Pos               (9U)   /* SCI Smart Card Error */
#define SCI_ST_SMC_Msk               (0x1UL << SCI_ST_SMC_Pos)
#define SCI_ST_SMC                   SCI_ST_SMC_Msk
#define SCI_ST_MPDET_Pos             (8U)   /* SCI Multi-processor Address Detection */
#define SCI_ST_MPDET_Msk             (0x1UL << SCI_ST_MPDET_Pos)
#define SCI_ST_MPDET                 SCI_ST_MPDET_Msk
#define SCI_ST_FRMERR_Pos            (7U)   /* SCI Frame Error */
#define SCI_ST_FRMERR_Msk            (0x1UL << SCI_ST_FRMERR_Pos)
#define SCI_ST_FRMERR                SCI_ST_FRMERR_Msk
#define SCI_ST_PRTERR_Pos            (6U)   /* SCI Parity Error */
#define SCI_ST_PRTERR_Msk            (0x1UL << SCI_ST_PRTERR_Pos)
#define SCI_ST_PRTERR                SCI_ST_PRTERR_Msk
#define SCI_ST_UDRERR_Pos            (5U)   /* SCI Underflow Error */
#define SCI_ST_UDRERR_Msk            (0x1UL << SCI_ST_UDRERR_Pos)
#define SCI_ST_UDRERR                SCI_ST_UDRERR_Msk
#define SCI_ST_OVRERR_Pos            (4U)   /* SCI Overrun Error */
#define SCI_ST_OVRERR_Msk            (0x1UL << SCI_ST_OVRERR_Pos)
#define SCI_ST_OVRERR                SCI_ST_OVRERR_Msk
#define SCI_ST_RXEND_Pos             (3U)   /* SCI RX End */
#define SCI_ST_RXEND_Msk             (0x1UL << SCI_ST_RXEND_Pos)
#define SCI_ST_RXEND                 SCI_ST_RXEND_Msk
#define SCI_ST_TXEND_Pos             (2U)   /* SCI TX End */
#define SCI_ST_TXEND_Msk             (0x1UL << SCI_ST_TXEND_Pos)
#define SCI_ST_TXEND                 SCI_ST_TXEND_Msk
#define SCI_ST_RXBUFFUL_Pos          (1U)   /* SCI RX Buffer Full */
#define SCI_ST_RXBUFFUL_Msk          (0x1UL << SCI_ST_RXBUFFUL_Pos)
#define SCI_ST_RXBUFFUL              SCI_ST_RXBUFFUL_Msk
#define SCI_ST_TXBUFEMPT_Pos         (0U)   /* SCI TX Buffer Empty */
#define SCI_ST_TXBUFEMPT_Msk         (0x1UL << SCI_ST_TXBUFEMPT_Pos)
#define SCI_ST_TXBUFEMPT             SCI_ST_TXBUFEMPT_Msk

/********************* SCI Status Interrupt Enable register ************************/
#define SCI_IE_FNS_Pos              (10U)   /* SCI IIC Restart and Finish complete */
#define SCI_IE_FNS_Msk              (0x1UL << SCI_IE_FNS_Pos)
#define SCI_IE_FNS                  SCI_IE_FNS_Msk
#define SCI_IE_SMC_Pos              (9U)   /* SCI Smart Card Error */
#define SCI_IE_SMC_Msk              (0x1UL << SCI_IE_SMC_Pos)
#define SCI_IE_SMC                  SCI_IE_SMC_Msk
#define SCI_IE_MPDET_Pos            (8U)   /* SCI Multi-processor Address Detection */
#define SCI_IE_MPDET_Msk            (0x1UL << SCI_IE_MPDET_Pos)
#define SCI_IE_MPDET                SCI_IE_MPDET_Msk
#define SCI_IE_FRMERR_Pos           (7U)   /* SCI Frame Error */
#define SCI_IE_FRMERR_Msk           (0x1UL << SCI_IE_FRMERR_Pos)
#define SCI_IE_FRMERR               SCI_IE_FRMERR_Msk
#define SCI_IE_PRTERR_Pos           (6U)   /* SCI Parity Error */
#define SCI_IE_PRTERR_Msk           (0x1UL << SCI_IE_PRTERR_Pos)
#define SCI_IE_PRTERR               SCI_IE_PRTERR_Msk
#define SCI_IE_UDRERR_Pos           (5U)   /* SCI Overrun Error */
#define SCI_IE_UDRERR_Msk           (0x1UL << SCI_IE_UDRERR_Pos)
#define SCI_IE_UDRERR               SCI_IE_UDRERR_Msk
#define SCI_IE_OVRERR_Pos           (4U)   /* SCI Overrun Error */
#define SCI_IE_OVRERR_Msk           (0x1UL << SCI_IE_OVRERR_Pos)
#define SCI_IE_OVRERR               SCI_IE_OVRERR_Msk
#define SCI_IE_RXEND_Pos            (3U)   /* SCI RX End */
#define SCI_IE_RXEND_Msk            (0x1UL << SCI_IE_RXEND_Pos)
#define SCI_IE_RXEND                SCI_IE_RXEND_Msk
#define SCI_IE_TXEND_Pos            (2U)   /* SCI TX End */
#define SCI_IE_TXEND_Msk            (0x1UL << SCI_IE_TXEND_Pos)
#define SCI_IE_TXEND                SCI_IE_TXEND_Msk
#define SCI_IE_RXBUFFUL_Pos         (1U)   /* SCI RX Buffer Full */
#define SCI_IE_RXBUFFUL_Msk         (0x1UL << SCI_IE_RXBUFFUL_Pos)
#define SCI_IE_RXBUFFUL             SCI_IE_RXBUFFUL_Msk
#define SCI_IE_TXBUFEMPT_Pos        (0U)   /* SCI TX Buffer Empty */
#define SCI_IE_TXBUFEMPT_Msk        (0x1UL << SCI_IE_TXBUFEMPT_Pos)
#define SCI_IE_TXBUFEMPT            SCI_IE_TXBUFEMPT_Msk

/*************************** SCI Status Clear register ***************************/
#define SCI_CLR_FNS_Pos              (10U)   /* SCI IIC Restart and Finish complete */
#define SCI_CLR_FNS_Msk              (0x1UL << SCI_CLR_FNS_Pos)
#define SCI_CLR_FNS                  SCI_CLR_FNS_Msk
#define SCI_CLR_SMC_Pos              (9U)   /* SCI Smart Card Error */
#define SCI_CLR_SMC_Msk              (0x1UL << SCI_CLR_SMC_Pos)
#define SCI_CLR_SMC                  SCI_CLR_SMC_Msk
#define SCI_CLR_MPDET_Pos            (8U)   /* SCI Multi-processor Address Detection */
#define SCI_CLR_MPDET_Msk            (0x1UL << SCI_CLR_MPDET_Pos)
#define SCI_CLR_MPDET                SCI_CLR_MPDET_Msk
#define SCI_CLR_FRMERR_Pos           (7U)   /* SCI Frame Error */
#define SCI_CLR_FRMERR_Msk           (0x1UL << SCI_CLR_FRMERR_Pos)
#define SCI_CLR_FRMERR               SCI_CLR_FRMERR_Msk
#define SCI_CLR_PRTERR_Pos           (6U)   /* SCI Parity Error */
#define SCI_CLR_PRTERR_Msk           (0x1UL << SCI_CLR_PRTERR_Pos)
#define SCI_CLR_PRTERR               SCI_CLR_PRTERR_Msk
#define SCI_CLR_UDRERR_Pos           (5U)   /* SCI Overrun Error */
#define SCI_CLR_UDRERR_Msk           (0x1UL << SCI_CLR_UDRERR_Pos)
#define SCI_CLR_UDRERR               SCI_CLR_UDRERR_Msk
#define SCI_CLR_OVRERR_Pos           (4U)   /* SCI Overrun Error */
#define SCI_CLR_OVRERR_Msk           (0x1UL << SCI_CLR_OVRERR_Pos)
#define SCI_CLR_OVRERR               SCI_CLR_OVRERR_Msk
#define SCI_CLR_RXEND_Pos            (3U)   /* SCI RX End */
#define SCI_CLR_RXEND_Msk            (0x1UL << SCI_CLR_RXEND_Pos)
#define SCI_CLR_RXEND                SCI_CLR_RXEND_Msk
#define SCI_CLR_TXEND_Pos            (2U)   /* SCI TX End */
#define SCI_CLR_TXEND_Msk            (0x1UL << SCI_CLR_TXEND_Pos)
#define SCI_CLR_TXEND                SCI_CLR_TXEND_Msk
#define SCI_CLR_RXBUFFUL_Pos         (1U)   /* SCI RX Buffer Full */
#define SCI_CLR_RXBUFFUL_Msk         (0x1UL << SCI_CLR_RXBUFFUL_Pos)
#define SCI_CLR_RXBUFFUL             SCI_CLR_RXBUFFUL_Msk
#define SCI_CLR_TXBUFEMPT_Pos        (0U)   /* SCI TX Buffer Empty */
#define SCI_CLR_TXBUFEMPT_Msk        (0x1UL << SCI_CLR_TXBUFEMPT_Pos)
#define SCI_CLR_TXBUFEMPT            SCI_CLR_TXBUFEMPT_Msk


/*****************************************************************************/
/*                                                                           */
/*                             Full Feature I2C                              */
/*                                                                           */
/*****************************************************************************/
/*****************  Bit definition for I2C CTRL register  ********************/

#define I2C_CTRL_DATALEN_Pos           (24U)   /* I2C Clock Presaler */
#define I2C_CTRL_DATALEN_Msk           (0x3UL << I2C_CTRL_DATALEN_Pos)
#define I2C_CTRL_DATALEN               I2C_CTRL_DATALEN_Msk
#define I2C_CTRL_DMAEN_Pos             (16U)   /* DMA Enable */
#define I2C_CTRL_DMAEN_Msk             (0x1UL << I2C_CTRL_DMAEN_Pos)
#define I2C_CTRL_DMAEN                 I2C_CTRL_DMAEN_Msk
#define I2C_CTRL_ADDRMASK_Pos		   (13U)
#define I2C_CTRL_ADDRMASK_Msk		   (0x1UL << I2C_CTRL_ADDRMASK_Pos)
#define I2C_CTRL_ADDRMASK			   I2C_CTRL_ADDRMASK_Msk
#define I2C_CTRL_DATADIR_Pos		   (12U)
#define I2C_CTRL_DATADIR_Msk		   (0x1UL << I2C_CTRL_DATADIR_Pos)
#define I2C_CTRL_DATADIR			   I2C_CTRL_DATADIR_Msk
#define I2C_CTRL_FLTEN_Pos			   (11U)
#define I2C_CTRL_FLTEN_Msk			   (0x1UL << I2C_CTRL_FLTEN_Pos)
#define I2C_CTRL_FLTEN 				   I2C_CTRL_FLTEN_Msk
#define I2C_CTRL_ACKEN_Pos             (9U) /* ACK Response Enable */
#define I2C_CTRL_ACKEN_Msk             (0x1UL << I2C_CTRL_ACKEN_Pos)
#define I2C_CTRL_ACKEN                 I2C_CTRL_ACKEN_Msk
#define I2C_CTRL_WTIM_Pos              (8U)  /* Raise Interrupt on 8th or 9th Clock */
#define I2C_CTRL_WTIM_Msk              (0x1UL << I2C_CTRL_WTIM_Pos)
#define I2C_CTRL_WTIM                  I2C_CTRL_WTIM_Msk
#define I2C_CTRL_WREL_Pos              (7U) /* SCL-Level-Low Wait Cancel*/
#define I2C_CTRL_WREL_Msk              (0x1UL << I2C_CTRL_WREL_Pos)
#define I2C_CTRL_WREL                  I2C_CTRL_WREL_Msk
#define I2C_CTRL_LREL_Pos              (6U) /* Communication Cancel */
#define I2C_CTRL_LREL_Msk              (0x1UL << I2C_CTRL_LREL_Pos)
#define I2C_CTRL_LREL                  I2C_CTRL_LREL_Msk
#define I2C_CTRL_SPT_Pos               (5U)  /* Generate Stop Signal */
#define I2C_CTRL_SPT_Msk               (0x1UL << I2C_CTRL_SPT_Pos)
#define I2C_CTRL_SPT                   I2C_CTRL_SPT_Msk
#define I2C_CTRL_STT_Pos               (4U) /* Generate Start Signal */
#define I2C_CTRL_STT_Msk               (0x1UL << I2C_CTRL_STT_Pos)
#define I2C_CTRL_STT                   I2C_CTRL_STT_Msk
#define I2C_CTRL_ADMOD_Pos             (2U) /* Generate Start Signal */
#define I2C_CTRL_ADMOD_Msk             (0x1UL << I2C_CTRL_ADMOD_Pos)
#define I2C_CTRL_ADMOD                 I2C_CTRL_ADMOD_Msk
#define I2C_CTRL_SMBUS_Pos			   (1U)
#define I2C_CTRL_SMBUS_Msk			   (0x1UL << I2C_CTRL_SMBUS_Pos)
#define I2C_CTRL_SMBUS    			   I2C_CTRL_SMBUS_Msk
#define I2C_CTRL_I2CEN_Pos             (0U) /* I2C Module Enable */
#define I2C_CTRL_I2CEN_Msk             (0x1UL << I2C_CTRL_I2CEN_Pos)
#define I2C_CTRL_I2CEN                 I2C_CTRL_I2CEN_Msk

/****************  Bit definition for I2C WIDTH register  ********************/
#define I2C_WID_LL_Pos                 (0U) /* Low Level Width */
#define I2C_WID_LL_Msk                 (0xFUL << I2C_WID_LL_Pos)
#define I2C_WID_LL                     I2C_WID_LL_Msk
#define I2C_WID_HL_Pos                 (16U) /* High Level Width */
#define I2C_WID_HL_Msk                 (0xFUL << I2C_WID_HL_Pos)
#define I2C_WID_HL                     I2C_WID_HL_Msk

/****************  Bit definition for I2C OUTCTL register  ********************/
#define I2C_OUTCTL_SDAOUT_Pos                 (5U)
#define I2C_OUTCTL_SDAOUT_Msk                 (0x1UL << I2C_OUTCTL_SDAOUT_Pos)
#define I2C_OUTCTL_SDAOUT                     I2C_OUTCTL_SDAOUT_Msk
#define I2C_OUTCTL_SCLOUT_Pos                 (4U)
#define I2C_OUTCTL_SCLOUT_Msk                 (0x1UL << I2C_OUTCTL_SCLOUT_Pos)
#define I2C_OUTCTL_SCLOUT                     I2C_OUTCTL_SCLOUT_Msk
#define I2C_OUTCTL_SDACTL_Pos                 (1U)
#define I2C_OUTCTL_SDACTL_Msk                 (0x1UL << I2C_OUTCTL_SDACTL_Pos)
#define I2C_OUTCTL_SDACTL                     I2C_OUTCTL_SDACTL_Msk
#define I2C_OUTCTL_SCLCTL_Pos                 (0U)
#define I2C_OUTCTL_SCLCTL_Msk                 (0x1UL << I2C_OUTCTL_SCLCTL_Pos)
#define I2C_OUTCTL_SCLCTL                     I2C_OUTCTL_SCLCTL_Msk

/************  Bit definition for I2C FIFO CTRL register  *************/
#define I2C_FIFOCTL_FIFOCLR_Pos		   (8U)
#define I2C_FIFOCTL_FIFOCLR_Msk		   (0x1UL << I2C_FIFOCTL_FIFOCLR_Pos)
#define I2C_FIFOCTL_FIFOCLR			   I2C_FIFOCTL_FIFOCLR_Msk
/************  Bit definition for I2C FIFO CTRL register  *************/

/************  Bit definition for I2C FIFO STATE register  *************/
#define I2C_FIFOST_FIFOTHR_Pos		       (10U)
#define I2C_FIFOST_FIFOTHR_Msk		       (0x1UL << I2C_FIFOST_FIFOTHR_Pos)
#define I2C_FIFOST_FIFOTHR			       I2C_FIFOST_FIFOTHR_Msk
#define I2C_FIFOST_FIFOFULL_Pos		       (9U)
#define I2C_FIFOST_FIFOFULL_Msk		       (0x1UL << I2C_FIFOST_FIFOFULL_Pos)
#define I2C_FIFOST_FIFOFULL			       I2C_FIFOST_FIFOFULL_Msk
#define I2C_FIFOST_FIFOEMPTY_Pos		   (8U)
#define I2C_FIFOST_FIFOEMPTY_Msk		   (0x1UL << I2C_FIFOEMPTY_FIFOFULL_Pos)
#define I2C_FIFOST_FIFOEMPTY			   I2C_FIFOST_FIFOEMPTY_Msk
#define I2C_FIFOST_FIFONUM_Pos		       (9U)
#define I2C_FIFOST_FIFONUM_Msk		       (0x1UL << I2C_FIFOST_FIFONUM_Pos)
#define I2C_FIFOST_FIFONUM			       I2C_FIFOST_FIFONUM_Msk
/************  Bit definition for I2C FIFO CTRL register  *************/

/****************  Bit definition for I2C Status register  *******************/
#define I2C_SR_DMAERR_Pos              (17U) /* NACK May Occurr During DMA Transcation */
#define I2C_SR_DMAERR_Msk              (0x1UL << I2C_SR_DMAERR_Pos)
#define I2C_SR_DMAERR                  I2C_SR_DMAERR_Msk
#define I2C_SR_CMPLT_Pos               (16U) /* Data Transaction Complete Flag */
#define I2C_SR_CMPLT_Msk               (0x1UL << I2C_SR_CMPLT_Pos)
#define I2C_SR_CMPLT                   I2C_SR_CMPLT_Msk
#define I2C_SR_ALD_Pos                 (11U) /* Arbitration Failure Status */
#define I2C_SR_ALD_Msk                 (0x1UL << I2C_SR_ALD_Pos)
#define I2C_SR_ALD                     I2C_SR_ALD_Msk
#define I2C_SR_WAIT_Pos                (10U) /* Arbitration Failure Status */
#define I2C_SR_WAIT_Msk                (0x1UL << I2C_SR_WAIT_Pos)
#define I2C_SR_WAIT                    I2C_SR_WAIT_Msk
#define I2C_SR_SDALVL_Pos              (9U) /* SDA Level Status */
#define I2C_SR_SDALVL_Msk              (0x1UL << I2C_SR_SDALVL_Pos)
#define I2C_SR_SDALVL                  I2C_SR_SDALVL_Msk
#define I2C_SR_SCLLVL_Pos              (8U) /* SCL Level Status */
#define I2C_SR_SCLLVL_Msk              (0x1UL << I2C_SR_SCLLVL_Pos)
#define I2C_SR_SCLLVL                  I2C_SR_SCLLVL_Msk
#define I2C_SR_STP_Pos                 (7U) /* Stop Signal Status */
#define I2C_SR_STP_Msk                 (0x1UL << I2C_SR_STP_Pos)
#define I2C_SR_STP                     I2C_SR_STP_Msk
#define I2C_SR_STT_Pos                 (6U) /* Start Signal Status */
#define I2C_SR_STT_Msk                 (0x1UL << I2C_SR_STT_Pos)
#define I2C_SR_STT                     I2C_SR_STT_Msk
#define I2C_SR_ACK_Pos                 (5U) /* ACK Status */
#define I2C_SR_ACK_Msk                 (0x1UL << I2C_SR_ACK_Pos)
#define I2C_SR_ACK                     I2C_SR_ACK_Msk
#define I2C_SR_TRC_Pos                 (4U) /* Transmission or Reception Status */
#define I2C_SR_TRC_Msk                 (0x1UL << I2C_SR_TRC_Pos)
#define I2C_SR_TRC                     I2C_SR_TRC_Msk
#define I2C_SR_COL_Pos                 (3U) /* address Match Status for Slave mode */
#define I2C_SR_COL_Msk                 (0x1UL << I2C_SR_COL_Pos)
#define I2C_SR_COL                     I2C_SR_COL_Msk
#define I2C_SR_EXC_Pos                 (2U) /* Extended Code Match Status for Slave mode */
#define I2C_SR_EXC_Msk                 (0x1UL << I2C_SR_EXC_Pos)
#define I2C_SR_EXC                     I2C_SR_EXC_Msk
#define I2C_SR_MSTS_Pos                (1U) /* Master Communication Status */
#define I2C_SR_MSTS_Msk                (0x1UL << I2C_SR_MSTS_Pos)
#define I2C_SR_MSTS                    I2C_SR_MSTS_Msk
#define I2C_SR_BUSY_Pos                (0U) /* Busy Status */
#define I2C_SR_BUSY_Msk                (0x1UL << I2C_SR_BUSY_Pos)
#define I2C_SR_BUSY                    I2C_SR_BUSY_Msk

/************  Bit definition for I2C Interrupt Control register  *************/
#define I2C_INTEN_FIFOTHRE_Pos 			   (10U) /*  */
#define I2C_INTEN_FIFOTHRE_Msk 			   (0x1UL << I2C_INTEN_FIFOTHRE_Pos)
#define I2C_INTEN_FIFOTHRE                 I2C_INTEN_FIFOTHRE_Msk
#define I2C_INTEN_FIFOFULL_Pos             (9U) /*  */
#define I2C_INTEN_FIFOFULL_Msk             (0x1UL << I2C_INTEN_FIFOFULL_Pos)
#define I2C_INTEN_FIFOFULL                 I2C_INTEN_FIFOFULL_Msk
#define I2C_INTEN_FIFOEMPTY_Pos            (8U)/*   */
#define I2C_INTEN_FIFOEMPTY_Msk            (0x1UL << I2C_INTEN_FIFOEMPTY_Pos)
#define I2C_INTEN_FIFOEMPTY                I2C_INTEN_FIFOEMPTY_Msk
#define I2C_INTEN_TCNTEN_Pos               (6U)/*   */
#define I2C_INTEN_TCNTEN_Msk               (0x1UL << I2C_INTEN_TCNTEN_Pos)
#define I2C_INTEN_TCNTEN                   I2C_INTEN_TCNTEN_Msk
#define I2C_INTEN_TIMOEN_Pos               (5U)/*   */
#define I2C_INTEN_TIMOEN_Msk               (0x1UL << I2C_INTEN_TIMOEN_Pos)
#define I2C_INTEN_TIMOEN                   I2C_INTEN_TIMOEN_Msk
#define I2C_INTEN_LALD_Pos                 (4U)/* LALD  */
#define I2C_INTEN_LALD_Msk                 (0x1UL << I2C_INTEN_LALD_Pos)
#define I2C_INTEN_LALD                     I2C_INTEN_LALD_Msk
#define I2C_INTEN_NACK_Pos                 (3U)/* NACK  */
#define I2C_INTEN_NACK_Msk                 (0x1UL << I2C_INTEN_NACK_Pos)
#define I2C_INTEN_NACK                     I2C_INTEN_NACK_Msk
#define I2C_INTEN_STOPINT_Pos			   (2u)
#define I2C_INTEN_STOPINT_Msk              (0x1UL << I2C_INTEN_STOPINT_Pos)
#define I2C_INTEN_STOPINT				   I2C_INTEN_STOPINT_Msk
#define I2C_INTEN_WTIMEN_Pos			   (1U)
#define I2C_INTEN_WTIMEN_Msk			   (0x1UL << I2C_INTEN_WTIMEN_Pos)
#define I2C_INTEN_WTIMEN				   I2C_INTEN_WTIMEN_Msk
#define I2C_INTEN_ADDRINT_Pos			   (0U)
#define I2C_INTEN_ADDRINT_Msk			   (0x1UL << I2C_INTEN_ADDRINT_Pos)
#define I2C_INTEN_ADDRINT				   I2C_INTEN_ADDRINT_Msk

/************  Bit definition for I2C Interrupt Status register  *************/
#define I2C_INTSR_FIFOTHRE_Pos		   (10U)
#define I2C_INTSR_FIFOTHRE_Msk		   (0x1UL << I2C_INTSR_FIFOTHRE_Pos)
#define I2C_INTSR_FIFOTHRE			   I2C_INTSR_FIFOTHRE_Msk
#define I2C_INTSR_FIFOFULL_Pos		   (9U)
#define I2C_INTSR_FIFOFULL_Msk		   (0x1UL << I2C_INTSR_FIFOFULL_Pos)
#define I2C_INTSR_FIFOFULL			   I2C_INTSR_FIFOFULL_Msk
#define I2C_INTSR_FIFOEMPTY_Pos		   (8U)
#define I2C_INTSR_FIFOEMPTY_Msk		   (0x1UL << I2C_INTSR_FIFOEMPTY_Pos)
#define I2C_INTSR_FIFOEMPTY			   I2C_INTSR_FIFOEMPTY_Msk
#define I2C_INTSR_TCNTDONE_Pos		   (6U)
#define I2C_INTSR_TCNTDONE_Msk		   (0x1UL << I2C_INTSR_TCNTDONE_Pos)
#define I2C_INTSR_TCNTDONE			   I2C_INTSR_TCNTDONE_Msk
#define I2C_INTSR_TIMOINT_Pos		   (5U)
#define I2C_INTSR_TIMOINT_Msk		   (0x1UL << I2C_INTSR_TIMOINT_Pos)
#define I2C_INTSR_TIMOINT			   I2C_INTSR_TIMOINT_Msk
#define I2C_INTSR_LALDINT_Pos		   (4U)
#define I2C_INTSR_LALDINT_Msk		   (0x1UL << I2C_INTSR_LALDINT_Pos)
#define I2C_INTSR_LALDINT			   I2C_INTSR_LALDINT_Msk
#define I2C_INTSR_NACK_Pos			   (3U)
#define I2C_INTSR_NACK_Msk			   (0x1UL << I2C_INTSR_NACK_Pos)
#define I2C_INTSR_NACK 				   I2C_INTSR_NACK_Msk
#define I2C_INTSR_STP_Pos              (2U) /* Stop Signal Interrupt Status */
#define I2C_INTSR_STP_Msk              (0x1UL << I2C_INTSR_STP_Pos)
#define I2C_INTSR_STP                  I2C_INTSR_STP_Msk
#define I2C_INTSR_DATA_Pos             (1U) /* Data Interrupt Status */
#define I2C_INTSR_DATA_Msk             (0x1UL << I2C_INTSR_DATA_Pos)
#define I2C_INTSR_DATA                 I2C_INTSR_DATA_Msk
#define I2C_INTSR_ADDR_Pos             (0U) /* address Match Status */
#define I2C_INTSR_ADDR_Msk             (0x1UL << I2C_INTSR_ADDR_Pos)
#define I2C_INTSR_ADDR                 I2C_INTSR_ADDR_Msk
/************  Bit definition for I2C TMOUTCFG register  *************/
#define I2C_TMOUTCFG_LTMOEN_Pos		   (2U)
#define I2C_TMOUTCFG_LTMOEN_Msk		   (0x1UL << I2C_TMOUTCFG_LTMOEN_Pos)
#define I2C_TMOUTCFG_LTMOEN			   I2C_TMOUTCFG_LTMOEN_Msk
#define I2C_TMOUTCFG_HTMOEN_Pos		   (1U)
#define I2C_TMOUTCFG_HTMOEN_Msk		   (0x1UL << I2C_TMOUTCFG_HTMOEN_Pos)
#define I2C_TMOUTCFG_HTMOEN			   I2C_TMOUTCFG_HTMOEN_Msk
#define I2C_TMOUTCFG_TMOEN_Pos		   (2U)
#define I2C_TMOUTCFG_TMOEN_Msk		   (0x1UL << I2C_TMOUTCFG_TMOEN_Pos)
#define I2C_TMOUTCFG_TMOEN			   I2C_TMOUTCFG_TMOEN_Msk

/*****************************************************************************/
/*                                                                           */
/*                                   Base-Timer                              */
/*                                                                           */
/*****************************************************************************/
/*****************  Bit definition for BTIM_CTRL register  *******************/
#define BTIM_CTRL_MOD_Pos               (31U)   /* 32Bit or 16Bit Timer */
#define BTIM_CTRL_MOD_Msk               (0x1UL << BTIM_CTRL_MOD_Pos)
#define BTIM_CTRL_MOD                   BTIM_CTRL_MOD_Msk

#define BTIM_CTRL_LDBSY_Pos             (16U)   /* Timer Load Busy Status */
#define BTIM_CTRL_LDBSY_Msk             (0x1UL << BTIM_CTRL_LDBSY_Pos)
#define BTIM_CTRL_LDBSY                 BTIM_CTRL_LDBSY_Msk
#define BTIM_CTRL_IE_Pos                (7U)    /* Timer Interrupt Enable */
#define BTIM_CTRL_IE_Msk                (0x1UL << BTIM_CTRL_IE_Pos)
#define BTIM_CTRL_IE                    BTIM_CTRL_IE_Msk
#define BTIM_CTRL_EVTOE_Pos             (6U)   /* Timer Event Output Enable */
#define BTIM_CTRL_EVTOE_Msk             (0x1UL << BTIM_CTRL_EVTOE_Pos)
#define BTIM_CTRL_EVTOE                 BTIM_CTRL_EVTOE_Msk
#define BTIM_CTRL_DIV_Pos               (4U)   /* Timer Clock Division */
#define BTIM_CTRL_DIV_Msk               (0x3UL << BTIM_CTRL_DIV_Pos)
#define BTIM_CTRL_DIV                   BTIM_CTRL_DIV_Msk
#define BTIM_CTRL_EN_Pos                (0U)   /* Timer Module Enable */
#define BTIM_CTRL_EN_Msk                (0x1UL << BTIM_CTRL_EN_Pos)
#define BTIM_CTRL_EN                    BTIM_CTRL_EN_Msk

/********************  Bit definition for BTIM_ST register  *******************/
#define BTIM_ST_TM0INT_Pos              (0U)   /* Timer Interrupt Status */
#define BTIM_ST_TM0INT_Msk              (0x1UL << BTIM_ST_TM0INT_Pos)
#define BTIM_ST_TM0INT                  BTIM_ST_TM0INT_Msk
#define BTIM_ST_TM1INT_Pos              (16U)   /* Timer Interrupt Status */
#define BTIM_ST_TM1INT_Msk              (0x1UL << BTIM_ST_TM1INT_Pos)
#define BTIM_ST_TM1INT                  BTIM_ST_TM1INT_Msk


/******************************************************************************/
/*                                                                            */
/*                      Analog to Digital Converter (ADC)                     */
/*                                                                            */
/******************************************************************************/

/********************  Bit definition for ADC_TRIG register  *******************/
#define ADC_TRIG_BURST_Pos                 (16U)
#define ADC_TRIG_BURST_Msk                 (0x1UL << ADC_TRIG_BURST_Pos)
#define ADC_TRIG_BURST                     ADC_TRIG_BURST_Msk

#define ADC_TRIG_STOP_Pos                  (17U)
#define ADC_TRIG_STOP_Msk                  (0x1UL << ADC_TRIG_STOP_Pos)
#define ADC_TRIG_STOP                      ADC_TRIG_STOP_Msk

/********************  Bit definition for ADC_CTRL register  *******************/
#define ADC_CTRL_ENABLE_Pos                (0U)
#define ADC_CTRL_ENABLE_Msk                (0x1UL << ADC_CTRL_ENABLE_Pos)
#define ADC_CTRL_ENABLE                    ADC_CTRL_ENABLE_Msk

#define ADC_CTRL_MODE_Pos                  (2U)
#define ADC_CTRL_MODE_Msk                  (0x3UL << ADC_CTRL_MODE_Pos)
#define ADC_CTRL_MODE_STOP                 (0x0UL << ADC_CTRL_MODE_Pos)
#define ADC_CTRL_MODE_SINGLE               (0x1UL << ADC_CTRL_MODE_Pos)

#define ADC_CTRL_RESOLUTION_Pos            (4U)
#define ADC_CTRL_RESOLUTION_Msk            (0x3UL << ADC_CTRL_RESOLUTION_Pos)

#define ADC_CTRL_DMA_Pos                   (12U)
#define ADC_CTRL_DMA_Msk                   (0x1UL << ADC_CTRL_DMA_Pos)
#define ADC_CTRL_DMA                       ADC_CTRL_DMA_Msk

#define ADC_CTRL_BURSTEN_Pos               (16U)
#define ADC_CTRL_BURSTEN_Msk               (0x1UL << ADC_CTRL_BURSTEN_Pos)
#define ADC_CTRL_BURSTEN                   ADC_CTRL_BURSTEN_Msk

#define ADC_CTRL_BURSTHWTRIGEN_Pos         (20U)
#define ADC_CTRL_BURSTHWTRIGEN_Msk         (0x1UL << ADC_CTRL_BURSTHWTRIGEN_Pos)
#define ADC_CTRL_BURSTHWTRIGEN             ADC_CTRL_BURSTHWTRIGEN_Msk

#define ADC_CTRL_BURSTHWTRIGSEL_Pos        (24U)
#define ADC_CTRL_BURSTHWTRIGSEL_Msk        (0xFUL << ADC_CTRL_BURSTHWTRIGSEL_Pos)

#define ADC_CTRL_BURSTLEN_Pos              (28U)
#define ADC_CTRL_BURSTLEN_Msk              (0xFUL << ADC_CTRL_BURSTLEN_Pos)

/********************  Bit definition for ADC_PRIO register  ********************/
#define ADC_PRIO_HIGHSEL_Pos               (0U)
#define ADC_PRIO_HIGHSEL_Msk               (0x1FUL << ADC_PRIO_HIGHSEL_Pos)

#define ADC_PRIO_RRP_Pos                   (8U)
#define ADC_PRIO_RRP_Msk                   (0x1FUL << ADC_PRIO_RRP_Pos)

/********************  Bit definition for ADC_INT register  *****************/
#define ADC_INT_SOC0INTEN_Pos              (0U)
#define ADC_INT_SOC0INTEN_Msk              (0x1UL << ADC_INT_SOC0INTEN_Pos)
#define ADC_INT_SOC0INTEN                  ADC_INT_SOC0INTEN_Msk

#define ADC_INT_SOC1INTEN_Pos              (1U)
#define ADC_INT_SOC1INTEN_Msk              (0x1UL << ADC_INT_SOC1INTEN_Pos)
#define ADC_INT_SOC1INTEN                  ADC_INT_SOC1INTEN_Msk

#define ADC_INT_SOC2INTEN_Pos              (2U)
#define ADC_INT_SOC2INTEN_Msk              (0x1UL << ADC_INT_SOC2INTEN_Pos)
#define ADC_INT_SOC2INTEN                  ADC_INT_SOC2INTEN_Msk

#define ADC_INT_SOC3INTEN_Pos              (3U)
#define ADC_INT_SOC3INTEN_Msk              (0x1UL << ADC_INT_SOC3INTEN_Pos)
#define ADC_INT_SOC3INTEN                  ADC_INT_SOC3INTEN_Msk

#define ADC_INT_SOC4INTEN_Pos              (4U)
#define ADC_INT_SOC4INTEN_Msk              (0x1UL << ADC_INT_SOC4INTEN_Pos)
#define ADC_INT_SOC4INTEN                  ADC_INT_SOC4INTEN_Msk

#define ADC_INT_SOC5INTEN_Pos              (5U)
#define ADC_INT_SOC5INTEN_Msk              (0x1UL << ADC_INT_SOC5INTEN_Pos)
#define ADC_INT_SOC5INTEN                  ADC_INT_SOC5INTEN_Msk

#define ADC_INT_SOC6INTEN_Pos              (6U)
#define ADC_INT_SOC6INTEN_Msk              (0x1UL << ADC_INT_SOC6INTEN_Pos)
#define ADC_INT_SOC6INTEN                  ADC_INT_SOC6INTEN_Msk

#define ADC_INT_SOC7INTEN_Pos              (7U)
#define ADC_INT_SOC7INTEN_Msk              (0x1UL << ADC_INT_SOC7INTEN_Pos)
#define ADC_INT_SOC7INTEN                  ADC_INT_SOC7INTEN_Msk

#define ADC_INT_SOC8INTEN_Pos              (8U)
#define ADC_INT_SOC8INTEN_Msk              (0x1UL << ADC_INT_SOC8INTEN_Pos)
#define ADC_INT_SOC8INTEN                  ADC_INT_SOC8INTEN_Msk

#define ADC_INT_SOC9INTEN_Pos              (9U)
#define ADC_INT_SOC9INTEN_Msk              (0x1UL << ADC_INT_SOC9INTEN_Pos)
#define ADC_INT_SOC9INTEN                  ADC_INT_SOC9INTEN_Msk

#define ADC_INT_SOC10INTEN_Pos             (10U)
#define ADC_INT_SOC10INTEN_Msk             (0x1UL << ADC_INT_SOC10INTEN_Pos)
#define ADC_INT_SOC10INTEN                 ADC_INT_SOC10INTEN_Msk

#define ADC_INT_SOC11INTEN_Pos             (11U)
#define ADC_INT_SOC11INTEN_Msk             (0x1UL << ADC_INT_SOC11INTEN_Pos)
#define ADC_INT_SOC11INTEN                 ADC_INT_SOC11INTEN_Msk

#define ADC_INT_SOC12INTEN_Pos             (12U)
#define ADC_INT_SOC12INTEN_Msk             (0x1UL << ADC_INT_SOC12INTEN_Pos)
#define ADC_INT_SOC12INTEN                 ADC_INT_SOC12INTEN_Msk

#define ADC_INT_SOC13INTEN_Pos             (13U)
#define ADC_INT_SOC13INTEN_Msk             (0x1UL << ADC_INT_SOC13INTEN_Pos)
#define ADC_INT_SOC13INTEN                 ADC_INT_SOC13INTEN_Msk

#define ADC_INT_SOC14INTEN_Pos             (14U)
#define ADC_INT_SOC14INTEN_Msk             (0x1UL << ADC_INT_SOC14INTEN_Pos)
#define ADC_INT_SOC14INTEN                 ADC_INT_SOC14INTEN_Msk

#define ADC_INT_SOC15INTEN_Pos             (15U)
#define ADC_INT_SOC15INTEN_Msk             (0x1UL << ADC_INT_SOC15INTEN_Pos)
#define ADC_INT_SOC15INTEN                 ADC_INT_SOC15INTEN_Msk

#define ADC_INT_PPB0COMPHINTEN_Pos         (16U)
#define ADC_INT_PPB0COMPHINTEN_Msk         (0x1UL << ADC_INT_PPB0COMPHINTEN_Pos)
#define ADC_INT_PPB0COMPHINTEN             ADC_INT_PPB0COMPHINTEN_Msk

#define ADC_INT_PPB0COMPLINTEN_Pos         (17U)
#define ADC_INT_PPB0COMPLINTEN_Msk         (0x1UL << ADC_INT_PPB0COMPLINTEN_Pos)
#define ADC_INT_PPB0COMPLINTEN             ADC_INT_PPB0COMPLINTEN_Msk

#define ADC_INT_PPB0ZEROINTEN_Pos          (18U)
#define ADC_INT_PPB0ZEROINTEN_Msk          (0x1UL << ADC_INT_PPB0ZEROINTEN_Pos)
#define ADC_INT_PPB0ZEROINTEN              ADC_INT_PPB0ZEROINTEN_Msk

#define ADC_INT_PPB1COMPHINTEN_Pos         (19U)
#define ADC_INT_PPB1COMPHINTEN_Msk         (0x1UL << ADC_INT_PPB1COMPHINTEN_Pos)
#define ADC_INT_PPB1COMPHINTEN             ADC_INT_PPB1COMPHINTEN_Msk

#define ADC_INT_PPB1COMPLINTEN_Pos         (20U)
#define ADC_INT_PPB1COMPLINTEN_Msk         (0x1UL << ADC_INT_PPB1COMPLINTEN_Pos)
#define ADC_INT_PPB1COMPLINTEN             ADC_INT_PPB1COMPLINTEN_Msk

#define ADC_INT_PPB1ZEROINTEN_Pos          (21U)
#define ADC_INT_PPB1ZEROINTEN_Msk          (0x1UL << ADC_INT_PPB1ZEROINTEN_Pos)
#define ADC_INT_PPB1ZEROINTEN              ADC_INT_PPB1ZEROINTEN_Msk

#define ADC_INT_PPB2COMPHINTEN_Pos         (22U)
#define ADC_INT_PPB2COMPHINTEN_Msk         (0x1UL << ADC_INT_PPB2COMPHINTEN_Pos)
#define ADC_INT_PPB2COMPHINTEN             ADC_INT_PPB2COMPHINTEN_Msk

#define ADC_INT_PPB2COMPLINTEN_Pos         (23U)
#define ADC_INT_PPB2COMPLINTEN_Msk         (0x1UL << ADC_INT_PPB2COMPLINTEN_Pos)
#define ADC_INT_PPB2COMPLINTEN             ADC_INT_PPB2COMPLINTEN_Msk

#define ADC_INT_PPB2ZEROINTEN_Pos          (24U)
#define ADC_INT_PPB2ZEROINTEN_Msk          (0x1UL << ADC_INT_PPB2ZEROINTEN_Pos)
#define ADC_INT_PPB2ZEROINTEN              ADC_INT_PPB2ZEROINTEN_Msk

#define ADC_INT_PPB3COMPHINTEN_Pos         (25U)
#define ADC_INT_PPB3COMPHINTEN_Msk         (0x1UL << ADC_INT_PPB3COMPHINTEN_Pos)
#define ADC_INT_PPB3COMPHINTEN             ADC_INT_PPB3COMPHINTEN_Msk

#define ADC_INT_PPB3COMPLINTEN_Pos         (26U)
#define ADC_INT_PPB3COMPLINTEN_Msk         (0x1UL << ADC_INT_PPB3COMPLINTEN_Pos)
#define ADC_INT_PPB3COMPLINTEN             ADC_INT_PPB3COMPLINTEN_Msk

#define ADC_INT_PPB3ZEROINTEN_Pos          (27U)
#define ADC_INT_PPB3ZEROINTEN_Msk          (0x1UL << ADC_INT_PPB3ZEROINTEN_Pos)
#define ADC_INT_PPB3ZEROINTEN              ADC_INT_PPB3ZEROINTEN_Msk

#define ADC_INT_SOCANYINTEN_Pos            (28U)
#define ADC_INT_SOCANYINTEN_Msk            (0x1UL << ADC_INT_SOCANYINTEN_Pos)
#define ADC_INT_SOCANYINTEN                ADC_INT_SOCANYINTEN_Msk

#define ADC_INT_SOCOVFINTEN_Pos            (29U)
#define ADC_INT_SOCOVFINTEN_Msk            (0x1UL << ADC_INT_SOCOVFINTEN_Pos)
#define ADC_INT_SOCOVFINTEN                ADC_INT_SOCOVFINTEN_Msk

/********************  Bit definition for ADC_STATUS register  *****************/
#define ADC_STATUS_SOC0INT_Pos              (0U)
#define ADC_STATUS_SOC0INT_Msk              (0x1UL << ADC_STATUS_SOC0INT_Pos)
#define ADC_STATUS_SOC0INT                  ADC_STATUS_SOC0INT_Msk

#define ADC_STATUS_SOC1INT_Pos              (1U)
#define ADC_STATUS_SOC1INT_Msk              (0x1UL << ADC_STATUS_SOC1INT_Pos)
#define ADC_STATUS_SOC1INT                  ADC_STATUS_SOC1INT_Msk

#define ADC_STATUS_SOC2INT_Pos              (2U)
#define ADC_STATUS_SOC2INT_Msk              (0x1UL << ADC_STATUS_SOC2INT_Pos)
#define ADC_STATUS_SOC2INT                  ADC_STATUS_SOC2INT_Msk

#define ADC_STATUS_SOC3INT_Pos              (3U)
#define ADC_STATUS_SOC3INT_Msk              (0x1UL << ADC_STATUS_SOC3INT_Pos)
#define ADC_STATUS_SOC3INT                  ADC_STATUS_SOC3INT_Msk

#define ADC_STATUS_SOC4INT_Pos              (4U)
#define ADC_STATUS_SOC4INT_Msk              (0x1UL << ADC_STATUS_SOC4INT_Pos)
#define ADC_STATUS_SOC4INT                  ADC_STATUS_SOC4INT_Msk

#define ADC_STATUS_SOC5INT_Pos              (5U)
#define ADC_STATUS_SOC5INT_Msk              (0x1UL << ADC_STATUS_SOC5INT_Pos)
#define ADC_STATUS_SOC5INT                  ADC_STATUS_SOC5INT_Msk

#define ADC_STATUS_SOC6INT_Pos              (6U)
#define ADC_STATUS_SOC6INT_Msk              (0x1UL << ADC_STATUS_SOC6INT_Pos)
#define ADC_STATUS_SOC6INT                  ADC_STATUS_SOC6INT_Msk

#define ADC_STATUS_SOC7INT_Pos              (7U)
#define ADC_STATUS_SOC7INT_Msk              (0x1UL << ADC_STATUS_SOC7INT_Pos)
#define ADC_STATUS_SOC7INT                  ADC_STATUS_SOC7INT_Msk

#define ADC_STATUS_SOC8INT_Pos              (8U)
#define ADC_STATUS_SOC8INT_Msk              (0x1UL << ADC_STATUS_SOC8INT_Pos)
#define ADC_STATUS_SOC8INT                  ADC_STATUS_SOC8INT_Msk

#define ADC_STATUS_SOC9INT_Pos              (9U)
#define ADC_STATUS_SOC9INT_Msk              (0x1UL << ADC_STATUS_SOC9INT_Pos)
#define ADC_STATUS_SOC9INT                  ADC_STATUS_SOC9INT_Msk

#define ADC_STATUS_SOC10INT_Pos             (10U)
#define ADC_STATUS_SOC10INT_Msk             (0x1UL << ADC_STATUS_SOC10INT_Pos)
#define ADC_STATUS_SOC10INT                 ADC_STATUS_SOC10INT_Msk

#define ADC_STATUS_SOC11INT_Pos             (11U)
#define ADC_STATUS_SOC11INT_Msk             (0x1UL << ADC_STATUS_SOC11INT_Pos)
#define ADC_STATUS_SOC11INT                 ADC_STATUS_SOC11INT_Msk

#define ADC_STATUS_SOC12INT_Pos             (12U)
#define ADC_STATUS_SOC12INT_Msk             (0x1UL << ADC_STATUS_SOC12INT_Pos)
#define ADC_STATUS_SOC12INT                 ADC_STATUS_SOC12INT_Msk

#define ADC_STATUS_SOC13INT_Pos             (13U)
#define ADC_STATUS_SOC13INT_Msk             (0x1UL << ADC_STATUS_SOC13INT_Pos)
#define ADC_STATUS_SOC13INT                 ADC_STATUS_SOC13INT_Msk

#define ADC_STATUS_SOC14INT_Pos             (14U)
#define ADC_STATUS_SOC14INT_Msk             (0x1UL << ADC_STATUS_SOC14INT_Pos)
#define ADC_STATUS_SOC14INT                 ADC_STATUS_SOC14INT_Msk

#define ADC_STATUS_SOC15INT_Pos             (15U)
#define ADC_STATUS_SOC15INT_Msk             (0x1UL << ADC_STATUS_SOC15INT_Pos)
#define ADC_STATUS_SOC15INT                 ADC_STATUS_SOC15INT_Msk

#define ADC_STATUS_PPB0COMPHINT_Pos         (16U)
#define ADC_STATUS_PPB0COMPHINT_Msk         (0x1UL << ADC_STATUS_PPB0COMPHINT_Pos)
#define ADC_STATUS_PPB0COMPHINT             ADC_STATUS_PPB0COMPHINT_Msk

#define ADC_STATUS_PPB0COMPLINT_Pos         (17U)
#define ADC_STATUS_PPB0COMPLINT_Msk         (0x1UL << ADC_STATUS_PPB0COMPLINT_Pos)
#define ADC_STATUS_PPB0COMPLINT             ADC_STATUS_PPB0COMPLINT_Msk

#define ADC_STATUS_PPB0ZEROINT_Pos          (18U)
#define ADC_STATUS_PPB0ZEROINT_Msk          (0x1UL << ADC_STATUS_PPB0ZEROINT_Pos)
#define ADC_STATUS_PPB0ZEROINT              ADC_STATUS_PPB0ZEROINT_Msk

#define ADC_STATUS_PPB1COMPHINT_Pos         (19U)
#define ADC_STATUS_PPB1COMPHINT_Msk         (0x1UL << ADC_STATUS_PPB1COMPHINT_Pos)
#define ADC_STATUS_PPB1COMPHINT             ADC_STATUS_PPB1COMPHINT_Msk

#define ADC_STATUS_PPB1COMPLINT_Pos         (20U)
#define ADC_STATUS_PPB1COMPLINT_Msk         (0x1UL << ADC_STATUS_PPB1COMPLINT_Pos)
#define ADC_STATUS_PPB1COMPLINT             ADC_STATUS_PPB1COMPLINT_Msk

#define ADC_STATUS_PPB1ZEROINT_Pos          (21U)
#define ADC_STATUS_PPB1ZEROINT_Msk          (0x1UL << ADC_STATUS_PPB1ZEROINT_Pos)
#define ADC_STATUS_PPB1ZEROINT              ADC_STATUS_PPB1ZEROINT_Msk

#define ADC_STATUS_PPB2COMPHINT_Pos         (22U)
#define ADC_STATUS_PPB2COMPHINT_Msk         (0x1UL << ADC_STATUS_PPB2COMPHINT_Pos)
#define ADC_STATUS_PPB2COMPHINT             ADC_STATUS_PPB2COMPHINT_Msk

#define ADC_STATUS_PPB2COMPLINT_Pos         (23U)
#define ADC_STATUS_PPB2COMPLINT_Msk         (0x1UL << ADC_STATUS_PPB2COMPLINT_Pos)
#define ADC_STATUS_PPB2COMPLINT             ADC_STATUS_PPB2COMPLINT_Msk

#define ADC_STATUS_PPB2ZEROINT_Pos          (24U)
#define ADC_STATUS_PPB2ZEROINT_Msk          (0x1UL << ADC_STATUS_PPB2ZEROINT_Pos)
#define ADC_STATUS_PPB2ZEROINT              ADC_STATUS_PPB2ZEROINT_Msk

#define ADC_STATUS_PPB3COMPHINT_Pos         (25U)
#define ADC_STATUS_PPB3COMPHINT_Msk         (0x1UL << ADC_STATUS_PPB3COMPHINT_Pos)
#define ADC_STATUS_PPB3COMPHINT             ADC_STATUS_PPB3COMPHINT_Msk

#define ADC_STATUS_PPB3COMPLINT_Pos         (26U)
#define ADC_STATUS_PPB3COMPLINT_Msk         (0x1UL << ADC_STATUS_PPB3COMPLINT_Pos)
#define ADC_STATUS_PPB3COMPLINT             ADC_STATUS_PPB3COMPLINT_Msk

#define ADC_STATUS_PPB3ZEROINT_Pos          (27U)
#define ADC_STATUS_PPB3ZEROINT_Msk          (0x1UL << ADC_STATUS_PPB3ZEROINT_Pos)
#define ADC_STATUS_PPB3ZEROINT              ADC_STATUS_PPB3ZEROINT_Msk

#define ADC_STATUS_SOCANYINT_Pos            (28U)
#define ADC_STATUS_SOCANYINT_Msk            (0x1UL << ADC_STATUS_SOCANYINT_Pos)
#define ADC_STATUS_SOCANYINT                ADC_STATUS_SOCANYINT_Msk

#define ADC_STATUS_SOCOVFINT_Pos            (29U)
#define ADC_STATUS_SOCOVFINT_Msk            (0x1UL << ADC_STATUS_SOCOVFINT_Pos)
#define ADC_STATUS_SOCOVFINT                ADC_STATUS_SOCOVFINT_Msk

#define ADC_STATUS_BUSY_Pos                 (31U)
#define ADC_STATUS_BUSY_Msk                 (0x1UL << ADC_STATUS_BUSY_Pos)
#define ADC_STATUS_BUSY                     ADC_STATUS_BUSY_Msk

/********************  Bit definition for ADC_CLEAR register  *****************/
#define ADC_CLEAR_SOC0INT_Pos              (0U)
#define ADC_CLEAR_SOC0INT_Msk              (0x1UL << ADC_CLEAR_SOC0INT_Pos)
#define ADC_CLEAR_SOC0INT                  ADC_CLEAR_SOC0INT_Msk

#define ADC_CLEAR_SOC1INT_Pos              (1U)
#define ADC_CLEAR_SOC1INT_Msk              (0x1UL << ADC_CLEAR_SOC1INT_Pos)
#define ADC_CLEAR_SOC1INT                  ADC_CLEAR_SOC1INT_Msk

#define ADC_CLEAR_SOC2INT_Pos              (2U)
#define ADC_CLEAR_SOC2INT_Msk              (0x1UL << ADC_CLEAR_SOC2INT_Pos)
#define ADC_CLEAR_SOC2INT                  ADC_CLEAR_SOC2INT_Msk

#define ADC_CLEAR_SOC3INT_Pos              (3U)
#define ADC_CLEAR_SOC3INT_Msk              (0x1UL << ADC_CLEAR_SOC3INT_Pos)
#define ADC_CLEAR_SOC3INT                  ADC_CLEAR_SOC3INT_Msk

#define ADC_CLEAR_SOC4INT_Pos              (4U)
#define ADC_CLEAR_SOC4INT_Msk              (0x1UL << ADC_CLEAR_SOC4INT_Pos)
#define ADC_CLEAR_SOC4INT                  ADC_CLEAR_SOC4INT_Msk

#define ADC_CLEAR_SOC5INT_Pos              (5U)
#define ADC_CLEAR_SOC5INT_Msk              (0x1UL << ADC_CLEAR_SOC5INT_Pos)
#define ADC_CLEAR_SOC5INT                  ADC_CLEAR_SOC5INT_Msk

#define ADC_CLEAR_SOC6INT_Pos              (6U)
#define ADC_CLEAR_SOC6INT_Msk              (0x1UL << ADC_CLEAR_SOC6INT_Pos)
#define ADC_CLEAR_SOC6INT                  ADC_CLEAR_SOC6INT_Msk

#define ADC_CLEAR_SOC7INT_Pos              (7U)
#define ADC_CLEAR_SOC7INT_Msk              (0x1UL << ADC_CLEAR_SOC7INT_Pos)
#define ADC_CLEAR_SOC7INT                  ADC_CLEAR_SOC7INT_Msk

#define ADC_CLEAR_SOC8INT_Pos              (8U)
#define ADC_CLEAR_SOC8INT_Msk              (0x1UL << ADC_CLEAR_SOC8INT_Pos)
#define ADC_CLEAR_SOC8INT                  ADC_CLEAR_SOC8INT_Msk

#define ADC_CLEAR_SOC9INT_Pos              (9U)
#define ADC_CLEAR_SOC9INT_Msk              (0x1UL << ADC_CLEAR_SOC9INT_Pos)
#define ADC_CLEAR_SOC9INT                  ADC_CLEAR_SOC9INT_Msk

#define ADC_CLEAR_SOC10INT_Pos             (10U)
#define ADC_CLEAR_SOC10INT_Msk             (0x1UL << ADC_CLEAR_SOC10INT_Pos)
#define ADC_CLEAR_SOC10INT                 ADC_CLEAR_SOC10INT_Msk

#define ADC_CLEAR_SOC11INT_Pos             (11U)
#define ADC_CLEAR_SOC11INT_Msk             (0x1UL << ADC_CLEAR_SOC11INT_Pos)
#define ADC_CLEAR_SOC11INT                 ADC_CLEAR_SOC11INT_Msk

#define ADC_CLEAR_SOC12INT_Pos             (12U)
#define ADC_CLEAR_SOC12INT_Msk             (0x1UL << ADC_CLEAR_SOC12INT_Pos)
#define ADC_CLEAR_SOC12INT                 ADC_CLEAR_SOC12INT_Msk

#define ADC_CLEAR_SOC13INT_Pos             (13U)
#define ADC_CLEAR_SOC13INT_Msk             (0x1UL << ADC_CLEAR_SOC13INT_Pos)
#define ADC_CLEAR_SOC13INT                 ADC_CLEAR_SOC13INT_Msk

#define ADC_CLEAR_SOC14INT_Pos             (14U)
#define ADC_CLEAR_SOC14INT_Msk             (0x1UL << ADC_CLEAR_SOC14INT_Pos)
#define ADC_CLEAR_SOC14INT                 ADC_CLEAR_SOC14INT_Msk

#define ADC_CLEAR_SOC15INT_Pos             (15U)
#define ADC_CLEAR_SOC15INT_Msk             (0x1UL << ADC_CLEAR_SOC15INT_Pos)
#define ADC_CLEAR_SOC15INT                 ADC_CLEAR_SOC15INT_Msk

#define ADC_CLEAR_PPB0COMPHINT_Pos         (16U)
#define ADC_CLEAR_PPB0COMPHINT_Msk         (0x1UL << ADC_CLEAR_PPB0COMPHINT_Pos)
#define ADC_CLEAR_PPB0COMPHINT             ADC_CLEAR_PPB0COMPHINT_Msk

#define ADC_CLEAR_PPB0COMPLINT_Pos         (17U)
#define ADC_CLEAR_PPB0COMPLINT_Msk         (0x1UL << ADC_CLEAR_PPB0COMPLINT_Pos)
#define ADC_CLEAR_PPB0COMPLINT             ADC_CLEAR_PPB0COMPLINT_Msk

#define ADC_CLEAR_PPB0ZEROINT_Pos          (18U)
#define ADC_CLEAR_PPB0ZEROINT_Msk          (0x1UL << ADC_CLEAR_PPB0ZEROINT_Pos)
#define ADC_CLEAR_PPB0ZEROINT              ADC_CLEAR_PPB0ZEROINT_Msk

#define ADC_CLEAR_PPB1COMPHINT_Pos         (19U)
#define ADC_CLEAR_PPB1COMPHINT_Msk         (0x1UL << ADC_CLEAR_PPB1COMPHINT_Pos)
#define ADC_CLEAR_PPB1COMPHINT             ADC_CLEAR_PPB1COMPHINT_Msk

#define ADC_CLEAR_PPB1COMPLINT_Pos         (20U)
#define ADC_CLEAR_PPB1COMPLINT_Msk         (0x1UL << ADC_CLEAR_PPB1COMPLINT_Pos)
#define ADC_CLEAR_PPB1COMPLINT             ADC_STATUS_PPB1COMPLINT_Msk

#define ADC_CLEAR_PPB1ZEROINT_Pos          (21U)
#define ADC_CLEAR_PPB1ZEROINT_Msk          (0x1UL << ADC_CLEAR_PPB1ZEROINT_Pos)
#define ADC_CLEAR_PPB1ZEROINT              ADC_CLEAR_PPB1ZEROINT_Msk

#define ADC_CLEAR_PPB2COMPHINT_Pos         (22U)
#define ADC_CLEAR_PPB2COMPHINT_Msk         (0x1UL << ADC_CLEAR_PPB2COMPHINT_Pos)
#define ADC_CLEAR_PPB2COMPHINT             ADC_CLEAR_PPB2COMPHINT_Msk

#define ADC_CLEAR_PPB2COMPLINT_Pos         (23U)
#define ADC_CLEAR_PPB2COMPLINT_Msk         (0x1UL << ADC_CLEAR_PPB2COMPLINT_Pos)
#define ADC_CLEAR_PPB2COMPLINT             ADC_CLEAR_PPB2COMPLINT_Msk

#define ADC_CLEAR_PPB2ZEROINT_Pos          (24U)
#define ADC_CLEAR_PPB2ZEROINT_Msk          (0x1UL << ADC_CLEAR_PPB2ZEROINT_Pos)
#define ADC_CLEAR_PPB2ZEROINT              ADC_CLEAR_PPB2ZEROINT_Msk

#define ADC_CLEAR_PPB3COMPHINT_Pos         (25U)
#define ADC_CLEAR_PPB3COMPHINT_Msk         (0x1UL << ADC_CLEAR_PPB3COMPHINT_Pos)
#define ADC_CLEAR_PPB3COMPHINT             ADC_CLEAR_PPB3COMPHINT_Msk

#define ADC_CLEAR_PPB3COMPLINT_Pos         (26U)
#define ADC_CLEAR_PPB3COMPLINT_Msk         (0x1UL << ADC_CLEAR_PPB3COMPLINT_Pos)
#define ADC_CLEAR_PPB3COMPLINT             ADC_CLEAR_PPB3COMPLINT_Msk

#define ADC_CLEAR_PPB3ZEROINT_Pos          (27U)
#define ADC_CLEAR_PPB3ZEROINT_Msk          (0x1UL << ADC_CLEAR_PPB3ZEROINT_Pos)
#define ADC_CLEAR_PPB3ZEROINT              ADC_CLEAR_PPB3ZEROINT_Msk

#define ADC_CLEAR_SOCANYINT_Pos            (28U)
#define ADC_CLEAR_SOCANYINT_Msk            (0x1UL << ADC_CLEAR_SOCANYINT_Pos)
#define ADC_CLEAR_SOCANYINT                ADC_CLEAR_SOCANYINT_Msk

#define ADC_CLEAR_SOCOVFINT_Pos            (29U)
#define ADC_CLEAR_SOCOVFINT_Msk            (0x1UL << ADC_CLEAR_SOCOVFINT_Pos)
#define ADC_CLEAR_SOCOVFINT                ADC_CLEAR_SOCOVFINT_Msk

#define ADC_CLEAR_BUSY_Pos                 (31U)
#define ADC_CLEAR_BUSY_Msk                 (0x1UL << ADC_CLEAR_BUSY_Pos)
#define ADC_CLEAR_BUSY                     ADC_CLEAR_BUSY_Msk

/********************  Bit definition for ADC_TRGOVF register  *****************/
#define ADC_TRGOVF_SOC0OVF_Pos              (0U)
#define ADC_TRGOVF_SOC0OVF_Msk              (0x1UL << ADC_TRGOVF_SOC0OVF_Pos)
#define ADC_TRGOVF_SOC0OVF                  ADC_TRGOVF_SOC0OVF_Msk

#define ADC_TRGOVF_SOC1OVF_Pos              (1U)
#define ADC_TRGOVF_SOC1OVF_Msk              (0x1UL << ADC_TRGOVF_SOC1OVF_Pos)
#define ADC_TRGOVF_SOC1OVF                  ADC_TRGOVF_SOC1OVF_Msk

#define ADC_TRGOVF_SOC2OVF_Pos              (2U)
#define ADC_TRGOVF_SOC2OVF_Msk              (0x1UL << ADC_TRGOVF_SOC2OVF_Pos)
#define ADC_TRGOVF_SOC2OVF                  ADC_TRGOVF_SOC2OVF_Msk

#define ADC_TRGOVF_SOC3OVF_Pos              (3U)
#define ADC_TRGOVF_SOC3OVF_Msk              (0x1UL << ADC_TRGOVF_SOC3OVF_Pos)
#define ADC_TRGOVF_SOC3OVF                  ADC_TRGOVF_SOC3OVF_Msk

#define ADC_TRGOVF_SOC4OVF_Pos              (4U)
#define ADC_TRGOVF_SOC4OVF_Msk              (0x1UL << ADC_TRGOVF_SOC4OVF_Pos)
#define ADC_TRGOVF_SOC4OVF                  ADC_TRGOVF_SOC4OVF_Msk

#define ADC_TRGOVF_SOC5OVF_Pos              (5U)
#define ADC_TRGOVF_SOC5OVF_Msk              (0x1UL << ADC_TRGOVF_SOC5OVF_Pos)
#define ADC_TRGOVF_SOC5OVF                  ADC_TRGOVF_SOC5OVF_Msk

#define ADC_TRGOVF_SOC6OVF_Pos              (6U)
#define ADC_TRGOVF_SOC6OVF_Msk              (0x1UL << ADC_TRGOVF_SOC6OVF_Pos)
#define ADC_TRGOVF_SOC6OVF                  ADC_TRGOVF_SOC6OVF_Msk

#define ADC_TRGOVF_SOC7OVF_Pos              (7U)
#define ADC_TRGOVF_SOC7OVF_Msk              (0x1UL << ADC_TRGOVF_SOC7OVF_Pos)
#define ADC_TRGOVF_SOC7OVF                  ADC_TRGOVF_SOC7OVF_Msk

#define ADC_TRGOVF_SOC8OVF_Pos              (8U)
#define ADC_TRGOVF_SOC8OVF_Msk              (0x1UL << ADC_TRGOVF_SOC8OVF_Pos)
#define ADC_TRGOVF_SOC8OVF                  ADC_TRGOVF_SOC8OVF_Msk

#define ADC_TRGOVF_SOC9OVF_Pos              (9U)
#define ADC_TRGOVF_SOC9OVF_Msk              (0x1UL << ADC_TRGOVF_SOC9OVF_Pos)
#define ADC_TRGOVF_SOC9OVF                  ADC_TRGOVF_SOC9OVF_Msk

#define ADC_TRGOVF_SOC10OVF_Pos             (10U)
#define ADC_TRGOVF_SOC10OVF_Msk             (0x1UL << ADC_TRGOVF_SOC10OVF_Pos)
#define ADC_TRGOVF_SOC10OVF                 ADC_TRGOVF_SOC10OVF_Msk

#define ADC_TRGOVF_SOC11OVF_Pos             (11U)
#define ADC_TRGOVF_SOC11OVF_Msk             (0x1UL << ADC_TRGOVF_SOC11OVF_Pos)
#define ADC_TRGOVF_SOC11OVF                 ADC_TRGOVF_SOC11OVF_Msk

#define ADC_TRGOVF_SOC12OVF_Pos             (12U)
#define ADC_TRGOVF_SOC12OVF_Msk             (0x1UL << ADC_TRGOVF_SOC12OVF_Pos)
#define ADC_TRGOVF_SOC12OVF                 ADC_TRGOVF_SOC12OVF_Msk

#define ADC_TRGOVF_SOC13OVF_Pos             (13U)
#define ADC_TRGOVF_SOC13OVF_Msk             (0x1UL << ADC_TRGOVF_SOC13OVF_Pos)
#define ADC_TRGOVF_SOC13OVF                 ADC_TRGOVF_SOC13OVF_Msk

#define ADC_TRGOVF_SOC14OVF_Pos             (14U)
#define ADC_TRGOVF_SOC14OVF_Msk             (0x1UL << ADC_TRGOVF_SOC14OVF_Pos)
#define ADC_TRGOVF_SOC14OVF                 ADC_TRGOVF_SOC14OVF_Msk

#define ADC_TRGOVF_SOC15OVF_Pos             (15U)
#define ADC_TRGOVF_SOC15OVF_Msk             (0x1UL << ADC_TRGOVF_SOC15OVF_Pos)
#define ADC_TRGOVF_SOC15OVF                 ADC_TRGOVF_SOC15OVF_Msk

/********************  Bit definition for ADC_OVFCLR register  *****************/
#define ADC_OVFCLR_SOC0OVF_Pos              (0U)
#define ADC_OVFCLR_SOC0OVF_Msk              (0x1UL << ADC_OVFCLR_SOC0OVF_Pos)
#define ADC_OVFCLR_SOC0OVF                  ADC_OVFCLR_SOC0OVF_Msk

#define ADC_OVFCLR_SOC1OVF_Pos              (1U)
#define ADC_OVFCLR_SOC1OVF_Msk              (0x1UL << ADC_OVFCLR_SOC1OVF_Pos)
#define ADC_OVFCLR_SOC1OVF                  ADC_OVFCLR_SOC1OVF_Msk

#define ADC_OVFCLR_SOC2OVF_Pos              (2U)
#define ADC_OVFCLR_SOC2OVF_Msk              (0x1UL << ADC_TRGOVF_SOC2OVF_Pos)
#define ADC_OVFCLR_SOC2OVF                  ADC_TRGOVF_SOC2OVF_Msk

#define ADC_OVFCLR_SOC3OVF_Pos              (3U)
#define ADC_OVFCLR_SOC3OVF_Msk              (0x1UL << ADC_OVFCLR_SOC3OVF_Pos)
#define ADC_OVFCLR_SOC3OVF                  ADC_OVFCLR_SOC3OVF_Msk

#define ADC_OVFCLR_SOC4OVF_Pos              (4U)
#define ADC_OVFCLR_SOC4OVF_Msk              (0x1UL << ADC_OVFCLR_SOC4OVF_Pos)
#define ADC_OVFCLR_SOC4OVF                  ADC_OVFCLR_SOC4OVF_Msk

#define ADC_OVFCLR_SOC5OVF_Pos              (5U)
#define ADC_OVFCLR_SOC5OVF_Msk              (0x1UL << ADC_OVFCLR_SOC5OVF_Pos)
#define ADC_OVFCLR_SOC5OVF                  ADC_OVFCLR_SOC5OVF_Msk

#define ADC_OVFCLR_SOC6OVF_Pos              (6U)
#define ADC_OVFCLR_SOC6OVF_Msk              (0x1UL << ADC_OVFCLR_SOC6OVF_Pos)
#define ADC_OVFCLR_SOC6OVF                  ADC_OVFCLR_SOC6OVF_Msk

#define ADC_OVFCLR_SOC7OVF_Pos              (7U)
#define ADC_OVFCLR_SOC7OVF_Msk              (0x1UL << ADC_OVFCLR_SOC7OVF_Pos)
#define ADC_OVFCLR_SOC7OVF                  ADC_OVFCLR_SOC7OVF_Msk

#define ADC_OVFCLR_SOC8OVF_Pos              (8U)
#define ADC_OVFCLR_SOC8OVF_Msk              (0x1UL << ADC_OVFCLR_SOC8OVF_Pos)
#define ADC_OVFCLR_SOC8OVF                  ADC_OVFCLR_SOC8OVF_Msk

#define ADC_OVFCLR_SOC9OVF_Pos              (9U)
#define ADC_OVFCLR_SOC9OVF_Msk              (0x1UL << ADC_OVFCLR_SOC9OVF_Pos)
#define ADC_OVFCLR_SOC9OVF                  ADC_OVFCLR_SOC9OVF_Msk

#define ADC_OVFCLR_SOC10OVF_Pos             (10U)
#define ADC_OVFCLR_SOC10OVF_Msk             (0x1UL << ADC_OVFCLR_SOC10OVF_Pos)
#define ADC_OVFCLR_SOC10OVF                 ADC_OVFCLR_SOC10OVF_Msk

#define ADC_OVFCLR_SOC11OVF_Pos             (11U)
#define ADC_OVFCLR_SOC11OVF_Msk             (0x1UL << ADC_OVFCLR_SOC11OVF_Pos)
#define ADC_OVFCLR_SOC11OVF                 ADC_OVFCLR_SOC11OVF_Msk

#define ADC_OVFCLR_SOC12OVF_Pos             (12U)
#define ADC_OVFCLR_SOC12OVF_Msk             (0x1UL << ADC_OVFCLR_SOC12OVF_Pos)
#define ADC_OVFCLR_SOC12OVF                 ADC_OVFCLR_SOC12OVF_Msk

#define ADC_OVFCLR_SOC13OVF_Pos             (13U)
#define ADC_OVFCLR_SOC13OVF_Msk             (0x1UL << ADC_OVFCLR_SOC13OVF_Pos)
#define ADC_OVFCLR_SOC13OVF                 ADC_OVFCLR_SOC13OVF_Msk

#define ADC_OVFCLR_SOC14OVF_Pos             (14U)
#define ADC_OVFCLR_SOC14OVF_Msk             (0x1UL << ADC_OVFCLR_SOC14OVF_Pos)
#define ADC_OVFCLR_SOC14OVF                 ADC_OVFCLR_SOC14OVF_Msk

#define ADC_OVFCLR_SOC15OVF_Pos             (15U)
#define ADC_OVFCLR_SOC15OVF_Msk             (0x1UL << ADC_OVFCLR_SOC15OVF_Pos)
#define ADC_OVFCLR_SOC15OVF                 ADC_OVFCLR_SOC15OVF_Msk

/********************  Bit definition for ADC_SOCCONF register  *****************/
#define ADC_SOCCONF_HWTRIGEN_Pos            (0U)
#define ADC_SOCCONF_HWTRIGEN_Msk            (0x1UL << ADC_SOCCONF_HWTRIGEN_Pos)
#define ADC_SOCCONF_HWTRIGEN                ADC_SOCCONF_HWTRIGEN_Msk

#define ADC_SOCCONF_HWTRIGSEL_Pos           (4U)
#define ADC_SOCCONF_HWTRIGSEL_Msk           (0xFUL << ADC_SOCCONF_HWTRIGSEL_Pos)
#define ADC_SOCCONF_HWTRIGSEL               ADC_SOCCONF_HWTRIGSEL_Msk

#define ADC_SOCCONF_SAMPLETIME_Pos          (12U)
#define ADC_SOCCONF_SAMPLETIME_Msk          (0xFUL << ADC_SOCCONF_SAMPLETIME_Pos)
#define ADC_SOCCONF_SAMPLETIME              ADC_SOCCONF_SAMPLETIME_Msk

#define ADC_SOCCONF_CHNSEL_Pos              (20U)
#define ADC_SOCCONF_CHNSEL_Msk              (0x1FUL << ADC_SOCCONF_CHNSEL_Pos)
#define ADC_SOCCONF_CHNSEL                  ADC_SOCCONF_CHNSEL_Msk

/********************  Bit definition for ADC_PPBCONF register  *****************/
#define ADC_PPBCONF_ENABLE_Pos              (0U)
#define ADC_PPBCONF_ENABLE_Msk              (0x1UL << ADC_PPBCONF_ENABLE_Pos)
#define ADC_PPBCONF_ENABLE                  ADC_PPBCONF_ENABLE_Msk

#define ADC_PPBCONF_SOCSEL_Pos              (4U)
#define ADC_PPBCONF_SOCSEL_Msk              (0xFUL << ADC_PPBCONF_SOCSEL_Pos)
#define ADC_PPBCONF_SOCSEL                  ADC_PPBCONF_SOCSEL_Msk

#define ADC_PPBCONF_CALSEL_Pos              (8U)
#define ADC_PPBCONF_CALSEL_Msk              (0x1UL << ADC_PPBCONF_CALSEL_Pos)
#define ADC_PPBCONF_CALSEL                  ADC_PPBCONF_CALSEL_Msk

/********************  Bit definition for ADC_PPBOFFCAL register  *****************/
#define ADC_PPBOFFCAL_VALUE_Pos             (0U)
#define ADC_PPBOFFCAL_VALUE_Msk             (0x3FFUL << ADC_PPBOFFCAL_VALUE_Pos)

/********************  Bit definition for ADC_PPBOFFREF register  *****************/
#define ADC_PPBOFFREF_VALUE_Pos             (0U)
#define ADC_PPBOFFREF_VALUE_Msk             (0xFFFUL << ADC_PPBOFFCAL_VALUE_Pos)

/********************  Bit definition for ADC_PPBCOMPH register  *****************/
#define ADC_PPBCOMPH_VALUE_Pos              (0U)
#define ADC_PPBCOMPH_VALUE_Msk              (0x1FFFUL << ADC_PPBCOMPH_VALUE_Pos)

/********************  Bit definition for ADC_PPBCOMPL register  *****************/
#define ADC_PPBCOMPL_VALUE_Pos              (0U)
#define ADC_PPBCOMPL_VALUE_Msk              (0x1FFFUL << ADC_PPBCOMPL_VALUE_Pos)

/********************  Bit definition for ADC_OFFSETCAL register  *****************/
#define ADC_OFFSETCAL_VALUE_Pos             (0U)
#define ADC_OFFSETCAL_VALUE_Msk             (0x7FUL << ADC_OFFSETCAL_VALUE_Pos)


/******************************************************************************/
/*                                     TIMA                                   */
/******************************************************************************/
/**************  TIM Control Register0  ***************************************/
#define TIMA_CR0_CNT_ST_Pos                     (10)
#define TIMA_CR0_CNT_ST_Msk                     (0x1UL << TIMA_CR0_CNT_ST_Pos)
#define TIMA_CR0_CNT_ST                         TIMA_CR0_CNT_ST_Msk
#define TIMA_CR0_CDK_Pos                   		(8U)
#define TIMA_CR0_CDK_Msk                  		(0x3UL << TIMA_CR0_CDK_Pos)
#define TIMA_CR0_CDK							TIMA_CR0_CDK_Msk
#define TIMA_CR0_ARPE_Pos                  		(7U)
#define TIMA_CR0_ARPE_Msk                  		(0x1UL << TIMA_CR0_ARPE_Pos)
#define TIMA_CR0_ARPE                   		TIMA_CR0_ARPE_Msk
#define TIMA_CR0_CMS_Pos                   		(5U)
#define TIMA_CR0_CMS_Msk                  		(0x3UL << TIMA_CR0_CMS_Pos)
#define TIMA_CR0_CMS                            TIMA_CR0_CMS_Msk
#define TIMA_CR0_DIR_Pos                   		(4U)
#define TIMA_CR0_DIR_Msk                  		(0x1UL << TIMA_CR0_DIR_Pos)
#define TIMA_CR0_DIR	                   		TIMA_CR0_DIR_Msk
#define TIMA_CR0_OPM_Pos                   		(3U)
#define TIMA_CR0_OPM_Msk                  		(0x1UL << TIMA_CR0_OPM_Pos)
#define TIMA_CR0_OPM							TIMA_CR0_OPM_Msk
#define TIMA_CR0_URS_Pos                   		(2U)
#define TIMA_CR0_URS_Msk                   		(0x1UL << TIMA_CR0_URS_Pos)
#define TIMA_CR0_URS					  		TIMA_CR0_URS_Msk
#define TIMA_CR0_UDIS_Pos				  		(1U)
#define TIMA_CR0_UDIS_Msk                 		(0x1UL << TIMA_CR0_UDIS_Pos)
#define TIMA_CR0_UDIS							TIMA_CR0_UDIS_Msk
#define TIMA_CR0_CEN_Pos				      	(0U)
#define TIMA_CR0_CEN_Msk					  	(0x1UL << TIMA_CR0_CEN_Pos)
#define TIMA_CR0_CEN				      		TIMA_CR0_CEN_Msk
/***************  TIM Control Register1   ****************************/
#define TIMA_CR1_MMS1_Pos                       (24U)
#define TIMA_CR1_MMS1_Msk                 		(0xFUL << TIMA_CR1_MMS1_Pos)
#define TIMA_CR1_MMS1                           TIMA_CR1_MMS1_Msk
#define TIMA_CR1_HIZ3_Pos                       (22U)
#define TIMA_CR1_HIZ3_Msk                 		(0x1UL << TIMA_CR1_HIZ3_Pos)
#define TIMA_CR1_HIZ3                           TIMA_CR1_HIZ3_Msk
#define TIMA_CR1_HIZ2N_Pos                      (21U)
#define TIMA_CR1_HIZ2N_Msk                 		(0x1UL << TIMA_CR1_HIZ2N_Pos)
#define TIMA_CR1_HIZ2N                          TIMA_CR1_HIZ2N_Msk
#define TIMA_CR1_HIZ2_Pos                       (20U)
#define TIMA_CR1_HIZ2_Msk                 		(0x1UL << TIMA_CR1_HIZ2_Pos)
#define TIMA_CR1_HIZ2                           TIMA_CR1_HIZ2_Msk
#define TIMA_CR1_HIZ1N_Pos                      (19U)
#define TIMA_CR1_HIZ1N_Msk                 		(0x1UL << TIMA_CR1_HIZ1N_Pos)
#define TIMA_CR1_HIZ1N                          TIMA_CR1_HIZ1N_Msk
#define TIMA_CR1_HIZ1_Pos                       (18U)
#define TIMA_CR1_HIZ1_Msk                 		(0x1UL << TIMA_CR1_HIZ1_Pos)
#define TIMA_CR1_HIZ1                           TIMA_CR1_HIZ1_Msk
#define TIMA_CR1_HIZ0N_Pos                      (17U)
#define TIMA_CR1_HIZ0N_Msk                 		(0x1UL << TIMA_CR1_HIZ0N_Pos)
#define TIMA_CR1_HIZ0N                          TIMA_CR1_HIZ0N_Msk
#define TIMA_CR1_HIZ0_Pos                       (16U)
#define TIMA_CR1_HIZ0_Msk                 		(0x1UL << TIMA_CR1_HIZ0_Pos)
#define TIMA_CR1_HIZ0                           TIMA_CR1_HIZ0_Msk
#define TIMA_CR1_OIS3_Pos				  		(14U)
#define TIMA_CR1_OIS3_Msk				  		(0x1UL << TIMA_CR1_OIS3_Pos)
#define TIMA_CR1_OIS3              	  			TIMA_CR1_OIS3_Msk
#define TIMA_CR1_OIS2N_Pos				  		(13U)
#define TIMA_CR1_OIS2N_Msk				  		(0x1UL << TIMA_CR1_OIS2N_Pos)
#define TIMA_CR1_OIS2N              	  		TIMA_CR1_OIS2N_Msk
#define TIMA_CR1_OIS2_Pos				  		(12U)
#define TIMA_CR1_OIS2_Msk				  		(0x1UL << TIMA_CR1_OIS2_Pos)
#define TIMA_CR1_OIS2              	  			TIMA_CR1_OIS2_Msk
#define TIMA_CR1_OIS1N_Pos				  		(11U)
#define TIMA_CR1_OIS1N_Msk				  		(0x1UL << TIMA_CR1_OIS1N_Pos)
#define TIMA_CR1_OIS1N              	  		TIMA_CR1_OIS1N_Msk
#define TIMA_CR1_OIS1_Pos				  		(10U)
#define TIMA_CR1_OIS1_Msk				  		(0x1UL << TIMA_CR1_OIS1_Pos)
#define TIMA_CR1_OIS1              	  			TIMA_CR1_OIS1_Msk
#define TIMA_CR1_OIS0N_Pos				  		(9U)
#define TIMA_CR1_OIS0N_Msk				  		(0x1UL << TIMA_CR1_OIS0N_Pos)
#define TIMA_CR1_OIS0N              	  		TIMA_CR1_OIS0N_Msk
#define TIMA_CR1_OIS0_Pos				  		(8U)
#define TIMA_CR1_OIS0_Msk				  		(0x1UL << TIMA_CR1_OIS0_Pos)
#define TIMA_CR1_OIS0              	  			TIMA_CR1_OIS0_Msk
#define TIMA_CR1_TI0S_Pos				  		(7U)
#define TIMA_CR1_TI0S_Msk				  		(0x1UL << TIMA_CR1_TI0S_Pos)
#define TIMA_CR1_TI0S           	  				TIMA_CR1_TI0S_Msk
#define TIMA_CR1_MMS_Pos					  	(4U)
#define TIMA_CR1_MMS_Msk				  		(0x7UL << TIMA_CR1_MMS_Pos)
#define TIMA_CR1_MMS                   			TIMA_CR1_MMS_Msk
#define TIMA_CR1_CCDS_Pos				  		(3U)
#define TIMA_CR1_CCDS_Msk				  		(0x1UL << TIMA_CR1_CCDS_Pos)
#define TIMA_CR1_CCDS                   		TIMA_CR1_CCDS_Msk
#define TIMA_CR1_CCUS_Pos				  		(2U)
#define TIMA_CR1_CCUS_Msk				  		(0x1UL << TIMA_CR1_CCUS_Pos)
#define TIMA_CR1_CCUS                   		TIMA_CR1_CCUS_Msk
#define TIMA_CR1_CCPC_Pos				  		(0U)
#define TIMA_CR1_CCPC_Msk				  		(0x1UL << TIMA_CR1_CCPC_Pos)
#define TIMA_CR1_CCPC                   		TIMA_CR1_CCPC_Msk
/************  TIM Mode Control Register  **************************/
#define TIMA_SMCR_QEM_Pos                       (24U)
#define TIMA_SMCR_QEM_Msk                       (0x7UL << TIMA_SMCR_QEM_Pos)
#define TIMA_SMCR_QEM                           TIMA_SMCR_QEM_Msk
#define TIMA_SMCR_OCCS_Pos                      (20U)
#define TIMA_SMCR_OCCS_Msk                      (0x1UL << TIMA_SMCR_OCCS_Pos)
#define TIMA_SMCR_OCCS                          TIMA_SMCR_OCCS_Msk
#define TIMA_SMCR_ETEN_Pos                      (16U)
#define TIMA_SMCR_ETEN_Msk 						(0x1UL << TIMA_SMCR_ETEN_Pos)
#define TIMA_SMCR_ETEN							TIMA_SMCR_ETEN_Msk
#define TIMA_SMCR_ETP_Pos				   		(15U)
#define TIMA_SMCR_ETP_Msk				   		(0x1UL << TIMA_SMCR_ETP_Pos)
#define TIMA_SMCR_ETP                    		TIMA_SMCR_ETP_Msk
#define TIMA_SMCR_ECE_Pos				   		(14U)
#define TIMA_SMCR_ECE_Msk				   		(0x1UL << TIMA_SMCR_ECE_Pos)
#define TIMA_SMCR_ECE                    		TIMA_SMCR_ECE_Msk
#define TIMA_SMCR_ETPS_Pos				   		(12U)
#define TIMA_SMCR_ETPS_Msk				   		(0x3UL << TIMA_SMCR_ETPS_Pos)
#define TIMA_SMCR_ETPS                  		TIMA_SMCR_ETPS_Msk
#define TIMA_SMCR_ETF_Pos				   		(8U)
#define TIMA_SMCR_ETF_Msk                 		(0xfUL << TIMA_SMCR_ETF_Pos)
#define TIMA_SMCR_ETF            	   			TIMA_SMCR_ETF_Msk
#define TIMA_SMCR_MSM_Pos				   		(7U)
#define TIMA_SMCR_MSM_Msk				   		(0x1UL << TIMA_SMCR_MSM_Pos)
#define TIMA_SMCR_MSM                   		TIMA_SMCR_MSM_Msk
#define TIMA_SMCR_TS_Pos					   	(4U)
#define TIMA_SMCR_TS_Msk				   		(0x7UL << TIMA_SMCR_TS_Pos)
#define TIMA_SMCR_TS					  	 	TIMA_SMCR_TS_Msk
#define TIMA_SMCR_SMS_Pos                   	(0U)
#define TIMA_SMCR_SMS_Msk				   		(0x7UL << TIMA_SMCR_SMS_Pos)
#define TIMA_SMCR_SMS                   		TIMA_SMCR_SMS_Msk
/********  TIM DMA and Interrupt Enable Register   **********/
#define TIMA_DIER_CC5IE_Pos                     (25U)
#define TIMA_DIER_CC5IE_Msk                     (0x1UL << TIMA_DIER_CC5IE_Pos)
#define TIMA_DIER_CC5IE                         TIMA_DIER_CC5IE_Msk
#define TIMA_DIER_CC4IE_Pos                     (24U)
#define TIMA_DIER_CC4IE_Msk                     (0x1UL << TIMA_DIER_CC4IE_Pos)
#define TIMA_DIER_CC4IE                         TIMA_DIER_CC4IE_Msk
#define TIMA_DIER_AC3IE_Pos                     (19U)
#define TIMA_DIER_AC3IE_Msk                     (0x1UL << TIMA_DIER_AC3IE_Pos)
#define TIMA_DIER_AC3IE                         TIMA_DIER_AC3IE_Msk
#define TIMA_DIER_AC2IE_Pos                     (18U)
#define TIMA_DIER_AC2IE_Msk                     (0x1UL << TIMA_DIER_AC2IE_Pos)
#define TIMA_DIER_AC2IE                         TIMA_DIER_AC2IE_Msk
#define TIMA_DIER_AC1IE_Pos                     (17U)
#define TIMA_DIER_AC1IE_Msk                     (0x1UL << TIMA_DIER_AC1IE_Pos)
#define TIMA_DIER_AC1IE                         TIMA_DIER_AC1IE_Msk
#define TIMA_DIER_AC0IE_Pos                     (16U)
#define TIMA_DIER_AC0IE_Msk                     (0x1UL << TIMA_DIER_AC0IE_Pos)
#define TIMA_DIER_AC0IE                         TIMA_DIER_AC0IE_Msk
#define TIMA_DIER_TDE_Pos						(14U)
#define TIMA_DIER_TDE_Msk						(0x1UL << TIMA_DIER_TDE_Pos)
#define TIMA_DIER_TDE 							TIMA_DIER_TDE_Msk
#define TIMA_DIER_COMDE_Pos						(13U)
#define TIMA_DIER_COMDE_Msk						(0x1UL << TIMA_DIER_COMDE_Pos)
#define TIMA_DIER_COMDE 						TIMA_DIER_COMDE_Msk
#define TIMA_DIER_CC3DE_Pos						(12U)
#define TIMA_DIER_CC3DE_Msk						(0x1UL << TIMA_DIER_CC3DE_Pos)
#define TIMA_DIER_CC3DE 						TIMA_DIER_CC3DE_Msk
#define TIMA_DIER_CC2DE_Pos						(11U)
#define TIMA_DIER_CC2DE_Msk						(0x1UL << TIMA_DIER_CC2DE_Pos)
#define TIMA_DIER_CC2DE 						TIMA_DIER_CC2DE_Msk
#define TIMA_DIER_CC1DE_Pos						(10U)
#define TIMA_DIER_CC1DE_Msk						(0x1UL << TIMA_DIER_CC1DE_Pos)
#define TIMA_DIER_CC1DE 						TIMA_DIER_CC1DE_Msk
#define TIMA_DIER_CC0DE_Pos						(9U)
#define TIMA_DIER_CC0DE_Msk						(0x1UL << TIMA_DIER_CC0DE_Pos)
#define TIMA_DIER_CC0DE 						TIMA_DIER_CC0DE_Msk
#define TIMA_DIER_UDE_Pos						(8U)
#define TIMA_DIER_UDE_Msk						(0x1UL << TIMA_DIER_UDE_Pos)
#define TIMA_DIER_UDE	 						TIMA_DIER_UDE_Msk
#define TIMA_DIER_BIE_Pos						(7U)
#define TIMA_DIER_BIE_Msk						(0x1UL << TIMA_DIER_BIE_Pos)
#define TIMA_DIER_BIE	 						TIMA_DIER_BIE_Msk
#define TIMA_DIER_TIE_Pos						(6U)
#define TIMA_DIER_TIE_Msk						(0x1UL << TIMA_DIER_TIE_Pos)
#define TIMA_DIER_TIE	 						TIMA_DIER_TIE_Msk
#define TIMA_DIER_COMIE_Pos						(5U)
#define TIMA_DIER_COMIE_Msk						(0x1UL << TIMA_DIER_COMIE_Pos)
#define TIMA_DIER_COMIE	 						TIMA_DIER_COMIE_Msk
#define TIMA_DIER_CC3IE_Pos						(4U)
#define TIMA_DIER_CC3IE_Msk						(0x1UL << TIMA_DIER_CC3IE_Pos)
#define TIMA_DIER_CC3IE 						TIMA_DIER_CC3IE_Msk
#define TIMA_DIER_CC2IE_Pos						(3U)
#define TIMA_DIER_CC2IE_Msk						(0x1UL << TIMA_DIER_CC2IE_Pos)
#define TIMA_DIER_CC2IE 						TIMA_DIER_CC2IE_Msk
#define TIMA_DIER_CC1IE_Pos						(2U)
#define TIMA_DIER_CC1IE_Msk						(0x1UL << TIMA_DIER_CC1IE_Pos)
#define TIMA_DIER_CC1IE 						TIMA_DIER_CC1IE_Msk
#define TIMA_DIER_CC0IE_Pos						(1U)
#define TIMA_DIER_CC0IE_Msk						(0x1UL << TIMA_DIER_CC0IE_Pos)
#define TIMA_DIER_CC0IE 						TIMA_DIER_CC0IE_Msk
#define TIMA_DIER_UIE_Pos						(0U)
#define TIMA_DIER_UIE_Msk						(0x1UL << TIMA_DIER_UIE_Pos)
#define TIMA_DIER_UIE	 						TIMA_DIER_UIE_Msk
/*******  TIM State Register    ****************************/
#define TIMA_SR_EVENT1F_Pos                     (27U)
#define TIMA_SR_EVENT1F_Msk                     (0x1UL << TIMA_SR_EVENT1F_Pos)
#define TIMA_SR_EVENT1F                         TIMA_SR_EVENT1F_Msk
#define TIMA_SR_EVENT0F_Pos                     (26U)
#define TIMA_SR_EVENT0F_Msk                     (0x1UL << TIMA_SR_EVENT0F_Pos)
#define TIMA_SR_EVENT0F                         TIMA_SR_EVENT0F_Msk
#define TIMA_SR_CC5IF_Pos                    	(25U)
#define TIMA_SR_CC5IF_Msk                     	(0x1UL << TIMA_SR_CC5IF_Pos)
#define TIMA_SR_CC5IF   						TIMA_SR_CC5IF_Msk
#define TIMA_SR_CC4IF_Pos                    	(24U)
#define TIMA_SR_CC4IF_Msk                     	(0x1UL << TIMA_SR_CC4IF_Pos)
#define TIMA_SR_CC4IF   						TIMA_SR_CC4IF_Msk
#define TIMA_SR_AC3IF_Pos                    	(19U)
#define TIMA_SR_AC3IF_Msk                     	(0x1UL << TIMA_SR_AC3IF_Pos)
#define TIMA_SR_AC3IF   						TIMA_SR_AC3IF_Msk
#define TIMA_SR_AC2IF_Pos                    	(18U)
#define TIMA_SR_AC2IF_Msk                     	(0x1UL << TIMA_SR_AC2IF_Pos)
#define TIMA_SR_AC2IF   						TIMA_SR_AC2IF_Msk
#define TIMA_SR_AC1IF_Pos                    	(17U)
#define TIMA_SR_AC1IF_Msk                     	(0x1UL << TIMA_SR_AC1IF_Pos)
#define TIMA_SR_AC1IF   						TIMA_SR_AC1IF_Msk
#define TIMA_SR_AC0IF_Pos                    	(16U)
#define TIMA_SR_AC0IF_Msk                     	(0x1UL << TIMA_SR_AC0IF_Pos)
#define TIMA_SR_AC0IF   						TIMA_SR_AC0IF_Msk
#define TIMA_SR_CC3OF_Pos                    	(12U)
#define TIMA_SR_CC3OF_Msk						(0x1UL << TIMA_SR_CC3OF_Pos)
#define TIMA_SR_CC3OF							TIMA_SR_CC3OF_Msk
#define TIMA_SR_CC2OF_Pos                    	(11U)
#define TIMA_SR_CC2OF_Msk						(0x1UL << TIMA_SR_CC2OF_Pos)
#define TIMA_SR_CC2OF							TIMA_SR_CC2OF_Msk
#define TIMA_SR_CC1OF_Pos                    	(10U)
#define TIMA_SR_CC1OF_Msk						(0x1UL << TIMA_SR_CC1OF_Pos)
#define TIMA_SR_CC1OF							TIMA_SR_CC1OF_Msk
#define TIMA_SR_CC0OF_Pos                    	(9U)
#define TIMA_SR_CC0OF_Msk						(0x1UL << TIMA_SR_CC0OF_Pos)
#define TIMA_SR_CC0OF							TIMA_SR_CC0OF_Msk
#define TIMA_SR_BIF_Pos                    		(7U)
#define TIMA_SR_BIF_Msk							(0x1UL << TIMA_SR_BIF_Pos)
#define TIMA_SR_BIF								TIMA_SR_BIF_Msk
#define TIMA_SR_TIF_Pos                    		(6U)
#define TIMA_SR_TIF_Msk							(0x1UL << TIMA_SR_TIF_Pos)
#define TIMA_SR_TIF								TIMA_SR_TIF_Msk
#define TIMA_SR_COMIF_Pos                    	(5U)
#define TIMA_SR_COMIF_Msk						(0x1UL << TIMA_SR_COMIF_Pos)
#define TIMA_SR_COMIF							TIMA_SR_COMIF_Msk
#define TIMA_SR_CC3IF_Pos                    	(4U)
#define TIMA_SR_CC3IF_Msk						(0x1UL << TIMA_SR_CC3IF_Pos)
#define TIMA_SR_CC3IF							TIMA_SR_CC3IF_Msk
#define TIMA_SR_CC2IF_Pos                    	(3U)
#define TIMA_SR_CC2IF_Msk						(0x1UL << TIMA_SR_CC2IF_Pos)
#define TIMA_SR_CC2IF							TIMA_SR_CC2IF_Msk
#define TIMA_SR_CC1IF_Pos                    	(2U)
#define TIMA_SR_CC1IF_Msk						(0x1UL << TIMA_SR_CC1IF_Pos)
#define TIMA_SR_CC1IF							TIMA_SR_CC1IF_Msk
#define TIMA_SR_CC0IF_Pos                    	(1U)
#define TIMA_SR_CC0IF_Msk						(0x1UL << TIMA_SR_CC0IF_Pos)
#define TIMA_SR_CC0IF							TIMA_SR_CC0IF_Msk
#define TIMA_SR_UIF_Pos                    		(0U)
#define TIMA_SR_UIF_Msk							(0x1UL << TIMA_SR_UIF_Pos)
#define TIMA_SR_UIF								TIMA_SR_UIF_Msk
/**********  TIM Event Generate Register  ************************/
#define TIMA_EGR_BG_Pos                      	(7U)
#define TIMA_EGR_BG_Msk							(0x1UL << TIMA_EGR_BG_Pos)
#define TIMA_EGR_BG	 							TIMA_EGR_BG_Msk
#define TIMA_EGR_TG_Pos                      	(6U)
#define TIMA_EGR_TG_Msk							(0x1UL << TIMA_EGR_TG_Pos)
#define TIMA_EGR_TG 							TIMA_EGR_TG_Msk
#define TIMA_EGR_COMG_Pos                    	(5U)
#define TIMA_EGR_COMG_Msk						(0x1UL << TIMA_EGR_COMG_Pos)
#define TIMA_EGR_COMG	 						TIMA_EGR_COMG_Msk
#define TIMA_EGR_CC3G_Pos                    	(4U)
#define TIMA_EGR_CC3G_Msk						(0x1UL << TIMA_EGR_CC3G_Pos)
#define TIMA_EGR_CC3G	 						TIMA_EGR_CC3G_Msk
#define TIMA_EGR_CC2G_Pos                    	(3U)
#define TIMA_EGR_CC2G_Msk						(0x1UL << TIMA_EGR_CC2G_Pos)
#define TIMA_EGR_CC2G 							TIMA_EGR_CC2G_Msk
#define TIMA_EGR_CC1G_Pos                    	(2U)
#define TIMA_EGR_CC1G_Msk						(0x1UL << TIMA_EGR_CC1G_Pos)
#define TIMA_EGR_CC1G	 						TIMA_EGR_CC1G_Msk
#define TIMA_EGR_CC0G_Pos                    	(1U)
#define TIMA_EGR_CC0G_Msk						(0x1UL << TIMA_EGR_CC0G_Pos)
#define TIMA_EGR_CC0G	 						TIMA_EGR_CC0G_Msk
#define TIMA_EGR_UG_Pos                    		(0U)
#define TIMA_EGR_UG_Msk							(0x1UL << TIMA_EGR_UG_Pos)
#define TIMA_EGR_UG	 							TIMA_EGR_UG_Msk
/*********************  TIM Capture/Compare Mode Register0(OUTPUT)  *****************/
#define TIMA_CCMR0_OC1CE_Pos					(24U)
#define TIMA_CCMR0_OC1CE_Msk					(0x1UL << TIMA_CCMR0_OC1CE_Pos)
#define TIMA_CCMR0_OC1CE 						TIMA_CCMR0_OC1CE_Msk
#define TIMA_CCMR0_OC1M_Pos						(20U)
#define TIMA_CCMR0_OC1M_Msk						(0xFUL << TIMA_CCMR0_OC1M_Pos)
#define TIMA_CCMR0_OC1M							TIMA_CCMR0_OC1M_Msk
#define TIMA_CCMR0_OC1PE_Pos                 	(19U)
#define TIMA_CCMR0_OC1PE_Msk					(0x1UL << TIMA_CCMR0_OC1PE_Pos)
#define TIMA_CCMR0_OC1PE 						TIMA_CCMR0_OC1PE_Msk
#define TIMA_CCMR0_OC1FE_Pos                 	(18U)
#define TIMA_CCMR0_OC1FE_Msk					(0x1UL << TIMA_CCMR0_OC1FE_Pos)
#define TIMA_CCMR0_OC1FE 						TIMA_CCMR0_OC1FE_Msk
#define TIMA_CCMR0_CC1S_Pos                  	(16U)
#define TIMA_CCMR0_CC1S_Msk    	            	(0x3UL << TIMA_CCMR0_CC1S_Pos)
#define TIMA_CCMR0_CC1S							TIMA_CCMR0_CC1S_Msk
#define TIMA_CCMR0_OC0CE_Pos					(8U)
#define TIMA_CCMR0_OC0CE_Msk					(0x1UL << TIMA_CCMR0_OC0CE_Pos)
#define TIMA_CCMR0_OC0CE						TIMA_CCMR0_OC0CE_Msk
#define TIMA_CCMR0_OC0M_Pos						(4U)
#define TIMA_CCMR0_OC0M_Msk						(0xFUL << TIMA_CCMR0_OC0M_Pos)
#define TIMA_CCMR0_OC0M							TIMA_CCMR0_OC0M_Msk
#define TIMA_CCMR0_OC0PE_Pos               	 	(3U)
#define TIMA_CCMR0_OC0PE_Msk					(0x1UL << TIMA_CCMR0_OC0PE_Pos)
#define TIMA_CCMR0_OC0PE 						TIMA_CCMR0_OC0PE_Msk
#define TIMA_CCMR0_OC0FE_Pos                 	(2U)
#define TIMA_CCMR0_OC0FE_Msk					(0x1UL << TIMA_CCMR0_OC0FE_Pos)
#define TIMA_CCMR0_OC0FE 						TIMA_CCMR0_OC0FE_Msk
#define TIMA_CCMR0_CC0S_Pos                  	(0U)
#define TIMA_CCMR0_CC0S_Msk 	               	(0x3UL << TIMA_CCMR0_CC0S_Pos)
#define TIMA_CCMR0_CC0S		 					TIMA_CCMR0_CC0S_Msk
/**************  TIM Capture/Compare Mode Register0(INPUT)  ***************/
#define TIMA_CCMR0_IC1F_Pos                  	(20U)
#define TIMA_CCMR0_IC1F_Msk                		(0xFUL << TIMA_CCMR0_IC1F_Pos)
#define TIMA_CCMR0_IC1F							TIMA_CCMR0_IC1F_Msk
#define TIMA_CCMR0_IC1PSC_Pos                	(18U)
#define TIMA_CCMR0_IC1PSC_Msk              		(0x3UL << TIMA_CCMR0_IC1PSC_Pos)
#define TIMA_CCMR0_IC1PSC               		TIMA_CCMR0_IC1PSC_Msk
#define TIMA_CCMR0_IC0F_Pos                  	(4U)
#define TIMA_CCMR0_IC0F_Msk                		(0xFUL << TIMA_CCMR0_IC0F_Pos)
#define TIMA_CCMR0_IC0F 						TIMA_CCMR0_IC0F_Msk
#define TIMA_CCMR0_IC0PSC_Pos                	(2U)
#define TIMA_CCMR0_IC0PSC_Msk              		(0x3UL << TIMA_CCMR0_IC0PSC_Pos)
#define TIMA_CCMR0_IC0PSC               		TIMA_CCMR0_IC0PSC_Msk

/**************** TIM Capture/Compare Mode Register1(OUTPUT) ********************/
#define TIMA_CCMR1_OC3CE_Pos					(24U)
#define TIMA_CCMR1_OC3CE_Msk					(0x1UL << TIMA_CCMR1_OC3CE_Pos)
#define TIMA_CCMR1_OC3CE 						TIMA_CCMR1_OC3CE_Msk
#define TIMA_CCMR1_OC3M_Pos						(20U)
#define TIMA_CCMR1_OC3M_Msk						(0xFUL << TIMA_CCMR1_OC3M_Pos)
#define TIMA_CCMR1_OC3M							TIMA_CCMR1_OC3M_Msk
#define TIMA_CCMR1_OC3PE_Pos                 	(19U)
#define TIMA_CCMR1_OC3PE_Msk					(0x1UL << TIMA_CCMR1_OC3PE_Pos)
#define TIMA_CCMR1_OC3PE 						TIMA_CCMR1_OC3PE_Msk
#define TIMA_CCMR1_OC3FE_Pos                 	(18U)
#define TIMA_CCMR1_OC3FE_Msk					(0x1UL << TIMA_CCMR1_OC3FE_Pos)
#define TIMA_CCMR1_OC3FE	 					TIMA_CCMR1_OC3FE_Msk
#define TIMA_CCMR1_CC3S_Pos                  	(16U)
#define TIMA_CCMR1_CC3S_Msk          	      	(0x3UL << TIMA_CCMR1_CC3S_Pos)
#define TIMA_CCMR1_CC3S							TIMA_CCMR1_CC3S_Msk
#define TIMA_CCMR1_OC2CE_Pos					(8U)
#define TIMA_CCMR1_OC2CE_Msk					(0x1UL << TIMA_CCMR1_OC2CE_Pos)
#define TIMA_CCMR1_OC2CE						TIMA_CCMR1_OC2CE_Msk
#define TIMA_CCMR1_OC2M_Pos						(4U)
#define TIMA_CCMR1_OC2M_Msk						(0xFUL << TIMA_CCMR1_OC2M_Pos)
#define TIMA_CCMR1_OC2M							TIMA_CCMR1_OC2M_Msk
#define TIMA_CCMR1_OC2PE_Pos                 	(3U)
#define TIMA_CCMR1_OC2PE_Msk					(0x1UL << TIMA_CCMR1_OC2PE_Pos)
#define TIMA_CCMR1_OC2PE 						TIMA_CCMR1_OC2PE_Msk
#define TIMA_CCMR1_OC2FE_Pos                 	(2U)
#define TIMA_CCMR1_OC2FE_Msk					(0x1UL << TIMA_CCMR1_OC2FE_Pos)
#define TIMA_CCMR1_OC2FE 						TIMA_CCMR1_OC2FE_Msk
#define TIMA_CCMR1_CC2S_Pos                  	(0U)
#define TIMA_CCMR1_CC2S_Msk                		(0x3UL << TIMA_CCMR1_CC2S_Pos)
#define TIMA_CCMR1_CC2S 						TIMA_CCMR1_CC2S_Msk
/****************  TIM Capture/Compare Mode Register1(INPUT)  ***************/
#define TIMA_CCMR1_IC3F_Pos                  	(20U)
#define TIMA_CCMR1_IC3F_Msk                		(0xFUL << TIMA_CCMR1_IC3F_Pos)
#define TIMA_CCMR1_IC3F 						TIMA_CCMR1_IC3F_Msk
#define TIMA_CCMR1_IC3PSC_Pos                	(18U)
#define TIMA_CCMR1_IC3PSC_Msk              		(0x3UL << TIMA_CCMR1_IC3PSC_Pos)
#define TIMA_CCMR1_IC3PSC               		TIMA_CCMR1_IC3PSC_Msk
#define TIMA_CCMR1_IC2F_Pos                  	(4U)
#define TIMA_CCMR1_IC2F_Msk                		(0xFUL << TIMA_CCMR1_IC2F_Pos)
#define TIMA_CCMR1_IC2F                 		TIMA_CCMR1_IC2F_Msk
#define TIMA_CCMR1_IC2PSC_Pos                	(2U)
#define TIMA_CCMR1_IC2PSC_Msk              		(0x3UL << TIMA_CCMR1_IC2PSC_Pos)
#define TIMA_CCMR1_IC2PSC             			TIMA_CCMR1_IC2PSC_Msk
/**************** TIM Capture/Compare Mode Register2(OUTPUT) ********************/
#define TIMA_CCMR2_OC5CE_Pos					(24U)
#define TIMA_CCMR2_OC5CE_Msk					(0x1UL << TIMA_CCMR2_OC5CE_Pos)
#define TIMA_CCMR2_OC5CE 						TIMA_CCMR2_OC5CE_Msk
#define TIMA_CCMR2_OC5M_Pos						(20U)
#define TIMA_CCMR2_OC5M_Msk						(0xFUL << TIMA_CCMR2_OC5M_Pos)
#define TIMA_CCMR2_OC5M							TIMA_CCMR2_OC5M_Msk
#define TIMA_CCMR2_OC5PE_Pos                 	(19U)
#define TIMA_CCMR2_OC5PE_Msk					(0x1UL << TIMA_CCMR2_OC5PE_Pos)
#define TIMA_CCMR2_OC5PE 						TIMA_CCMR2_OC5PE_Msk
#define TIMA_CCMR2_OC5FE_Pos                 	(18U)
#define TIMA_CCMR2_OC5FE_Msk					(0x1UL << TIMA_CCMR2_OC5FE_Pos)
#define TIMA_CCMR2_OC5FE	 					TIMA_CCMR2_OC5FE_Msk
#define TIMA_CCMR2_CC5S_Pos                  	(16U)
#define TIMA_CCMR2_CC5S_Msk          	      	(0x3UL << TIMA_CCMR2_CC5S_Pos)
#define TIMA_CCMR2_CC5S							TIMA_CCMR2_CC5S_Msk
#define TIMA_CCMR2_OC4CE_Pos					(8U)
#define TIMA_CCMR2_OC4CE_Msk					(0x1UL << TIMA_CCMR2_OC4CE_Pos)
#define TIMA_CCMR2_OC4CE						TIMA_CCMR2_OC4CE_Msk
#define TIMA_CCMR2_OC4M_Pos						(4U)
#define TIMA_CCMR2_OC4M_Msk						(0xFUL << TIMA_CCMR2_OC4M_Pos	)
#define TIMA_CCMR2_OC4M							TIMA_CCMR2_OC4M_Msk
#define TIMA_CCMR2_OC4PE_Pos                 	(3U)
#define TIMA_CCMR2_OC4PE_Msk					(0x1UL << TIMA_CCMR2_OC4PE_Pos)
#define TIMA_CCMR2_OC4PE 						TIMA_CCMR2_OC4PE_Msk
#define TIMA_CCMR2_OC4FE_Pos                 	(2U)
#define TIMA_CCMR2_OC4FE_Msk					(0x1UL << TIMA_CCMR2_OC4FE_Pos)
#define TIMA_CCMR2_OC4FE 						TIMA_CCMR2_OC4FE_Msk
#define TIMA_CCMR2_CC4S_Pos                  	(0U)
#define TIMA_CCMR2_CC4S_Msk                		(0x3UL << TIMA_CCMR2_CC4S_Pos)
#define TIMA_CCMR2_CC4S 						TIMA_CCMR2_CC4S_Msk
/****************  TIM Capture/Compare Mode Register2(INPUT)  ***************/
#define TIMA_CCMR2_IC5F_Pos                  	(20U)
#define TIMA_CCMR2_IC5F_Msk                		(0xFUL << TIMA_CCMR2_IC5F_Pos)
#define TIMA_CCMR2_IC5F 						TIMA_CCMR2_IC5F_Msk
#define TIMA_CCMR2_IC5PSC_Pos                	(18U)
#define TIMA_CCMR2_IC5PSC_Msk              		(0x3UL << TIMA_CCMR2_IC5PSC_Pos)
#define TIMA_CCMR2_IC5PSC	               		TIMA_CCMR2_IC5PSC_Msk
#define TIMA_CCMR2_IC4F_Pos                  	(4U)
#define TIMA_CCMR2_IC4F_Msk                		(0xFUL << TIMA_CCMR2_IC4F_Pos)
#define TIMA_CCMR2_IC4F                 		TIMA_CCMR2_IC4F_Msk
#define TIMA_CCMR2_IC4PSC_Pos                	(2U)
#define TIMA_CCMR2_IC4PSC_Msk              		(0x3UL << TIMA_CCMR2_IC4PSC_Pos)
#define TIMA_CCMR2_IC4PSC             			TIMA_CCMR2_IC4PSC_Msk
/********  TIM Capture/Compare Mode Enable Register   *********/
#define TIMA_CCER_CC3NP_Pos                  	(15U)
#define TIMA_CCER_CC3NP_Msk                  	(0x1UL << TIMA_CCER_CC3NP_Pos)
#define TIMA_CCER_CC3NP   	                	TIMA_CCER_CC3NP_Msk
#define TIMA_CCER_CC3P_Pos                   	(13U)
#define TIMA_CCER_CC3P_Msk                   	(0x1UL << TIMA_CCER_CC3P_Pos)
#define TIMA_CCER_CC3P   	                 	TIMA_CCER_CC3P_Msk
#define TIMA_CCER_CC3E_Pos                   	(12U)
#define TIMA_CCER_CC3E_Msk                   	(0x1UL << TIMA_CCER_CC3E_Pos)
#define TIMA_CCER_CC3E                   		TIMA_CCER_CC3E_Msk
#define TIMA_CCER_CC2NP_Pos                  	(11U)
#define TIMA_CCER_CC2NP_Msk                  	(0x1UL << TIMA_CCER_CC2NP_Pos)
#define TIMA_CCER_CC2NP	                 		TIMA_CCER_CC2NP_Msk
#define TIMA_CCER_CC2NE_Pos                  	(10U)
#define TIMA_CCER_CC2NE_Msk                  	(0x1UL << TIMA_CCER_CC2NE_Pos)
#define TIMA_CCER_CC2NE   	                	TIMA_CCER_CC2NE_Msk
#define TIMA_CCER_CC2P_Pos                   	(9U)
#define TIMA_CCER_CC2P_Msk                  	(0x1UL << TIMA_CCER_CC2P_Pos)
#define TIMA_CCER_CC2P   	                 	TIMA_CCER_CC2P_Msk
#define TIMA_CCER_CC2E_Pos                   	(8U)
#define TIMA_CCER_CC2E_Msk                   	(0x1UL << TIMA_CCER_CC2E_Pos)
#define TIMA_CCER_CC2E	                    	TIMA_CCER_CC2E_Msk
#define TIMA_CCER_CC1NP_Pos                  	(7U)
#define TIMA_CCER_CC1NP_Msk                  	(0x1UL << TIMA_CCER_CC1NP_Pos)
#define TIMA_CCER_CC1NP	  	                 	TIMA_CCER_CC1NP_Msk
#define TIMA_CCER_CC1NE_Pos                  	(6U)
#define TIMA_CCER_CC1NE_Msk                  	(0x1UL << TIMA_CCER_CC1NE_Pos)
#define TIMA_CCER_CC1NE		                   	TIMA_CCER_CC1NE_Msk
#define TIMA_CCER_CC1P_Pos                   	(5U)
#define TIMA_CCER_CC1P_Msk                   	(0x1UL << TIMA_CCER_CC1P_Pos)
#define TIMA_CCER_CC1P							TIMA_CCER_CC1P_Msk
#define TIMA_CCER_CC1E_Pos                 		(4U)
#define TIMA_CCER_CC1E_Msk                   	(0x1UL << TIMA_CCER_CC1E_Pos)
#define TIMA_CCER_CC1E	                 		TIMA_CCER_CC1E_Msk
#define TIMA_CCER_CC0NP_Pos                  	(3U)
#define TIMA_CCER_CC0NP_Msk                  	(0x1UL << TIMA_CCER_CC0NP_Pos)
#define TIMA_CCER_CC0NP	    	               	TIMA_CCER_CC0NP_Msk
#define TIMA_CCER_CC0NE_Pos                  	(2U)
#define TIMA_CCER_CC0NE_Msk                  	(0x1UL << TIMA_CCER_CC0NE_Pos)
#define TIMA_CCER_CC0NE		                   	TIMA_CCER_CC0NE_Msk
#define TIMA_CCER_CC0P_Pos                   	(1U)
#define TIMA_CCER_CC0P_Msk                   	(0x1UL << TIMA_CCER_CC0P_Pos)
#define TIMA_CCER_CC0P     		               	TIMA_CCER_CC0P_Msk
#define TIMA_CCER_CC0E_Pos                   	(0U)
#define TIMA_CCER_CC0E_Msk                   	(0x1UL << TIMA_CCER_CC0E_Pos)
#define TIMA_CCER_CC0E	                    	TIMA_CCER_CC0E_Msk

/***************  TIM Break And Dead-Time Register   *****************/
#define TIMA_BDTR_DTGP_Pos                      (24U)
#define TIMA_BDTR_DTGP_Msk                      (0xFFUL << TIMA_BDTR_DTGP_Pos )
#define TIMA_BDTR_DTGP							TIMA_BDTR_DTGP_Msk
#define TIMA_BDTR_DTGM_Pos                      (17U)
#define TIMA_BDTR_DTGM_Msk                      (0x1UL << TIMA_BDTR_DTGM_Pos )
#define TIMA_BDTR_DTGM							TIMA_BDTR_DTGM_Msk
#define TIMA_BDTR_DTGPE_Pos                     (16U)
#define TIMA_BDTR_DTGPE_Msk                     (0x1UL << TIMA_BDTR_DTGPE_Pos )
#define TIMA_BDTR_DTGPE							TIMA_BDTR_DTGPE_Msk
#define TIMA_BDTR_MOE_Pos                    	(15U)
#define TIMA_BDTR_MOE_Msk                    	(0x1UL << TIMA_BDTR_MOE_Pos)
#define TIMA_BDTR_MOE                     		TIMA_BDTR_MOE_Msk
#define TIMA_BDTR_AOE_Pos                    	(14U)
#define TIMA_BDTR_AOE_Msk                    	(0x1UL << TIMA_BDTR_AOE_Pos)
#define TIMA_BDTR_AOE                     		TIMA_BDTR_AOE_Msk
#define TIMA_BDTR_BKE_Pos                    	(12U)
#define TIMA_BDTR_BKE_Msk                    	(0x1UL << TIMA_BDTR_BKE_Pos)
#define TIMA_BDTR_BKE                     		TIMA_BDTR_BKE_Msk
#define TIMA_BDTR_LOCK_Pos                   	(8U)
#define TIMA_BDTR_LOCK_Msk                 		(0x3UL << TIMA_BDTR_LOCK_Pos)
#define TIMA_BDTR_LOCK                 			TIMA_BDTR_LOCK_Msk
/***********  TIM Event Control Register   *******/
#define TIMA_ECR_EVNT1PSC_Pos                   (12U)
#define TIMA_ECR_EVNT1PSC_Msk					(0x7UL << TIMA_ECR_EVNT1PSC_Pos)
#define TIMA_ECR_EVNT1PSC  						TIMA_ECR_EVNT1PSC_Msk
#define TIMA_ECR_EVNT1CNTCLR_Pos                (10U)
#define TIMA_ECR_EVNT1CNTCLR_Msk				(0x1UL << TIMA_ECR_EVNT1CNTCLR_Pos)
#define TIMA_ECR_EVNT1CNTCLR   					TIMA_ECR_EVNT1CNTCLR_Msk
#define TIMA_ECR_EVNT1SEL_Pos                   (8U)
#define TIMA_ECR_EVNT1SEL_Msk					(0x3UL << TIMA_ECR_EVNT1SEL_Pos)
#define TIMA_ECR_EVNT1SEL   					TIMA_ECR_EVNT1SEL_Msk
#define TIMA_ECR_EVNT0PSC_Pos                   (4U)
#define TIMA_ECR_EVNT0PSC_Msk					(0x7UL << TIMA_ECR_EVNT0PSC_Pos)
#define TIMA_ECR_EVNT0PSC  						TIMA_ECR_EVNT0PSC_Msk
#define TIMA_ECR_EVNT0CNTCLR_Pos                (2U)
#define TIMA_ECR_EVNT0CNTCLR_Msk				(0x1UL << TIMA_ECR_EVNT0CNTCLR_Pos)
#define TIMA_ECR_EVNT0CNTCLR   					TIMA_ECR_EVNT0CNTCLR_Msk
#define TIMA_ECR_EVNT0SEL_Pos                   (0U)
#define TIMA_ECR_EVNT0SEL_Msk					(0x3UL << TIMA_ECR_EVNT0SEL_Pos)
#define TIMA_ECR_EVNT0SEL   					TIMA_ECR_EVNT0SEL_Msk

/******************************************************************************/
/*                                     TIMH                                   */
/******************************************************************************/
/**************  TIMH Clock Control Register  *********************************/
#define TIMH_PCLKCON_HRWORK_Pos                 (31U)
#define TIMH_PCLKCON_HRWORK_Msk                 (0x1UL << TIMH_PCLKCON_HRWORK_Pos)
#define TIMH_PCLKCON_HRWORK                     TIMH_PCLKCON_HRWORK_Msk

#define TIMH_PCLKCON_HRIST_Pos                  (16U)
#define TIMH_PCLKCON_HRIST_Msk                  (0xFF << TIMH_PCLKCON_HRIST_Pos)
#define TIMH_PCLKCON_HRIST						TIMH_PCLKCON_HRIST_Msk

#define TIMH_PCLKCON_HRRDY_Pos                  (15U)
#define TIMH_PCLKCON_HRRDY_Msk                  (0x1UL << TIMH_PCLKCON_HRRDY_Pos)
#define TIMH_PCLKCON_HRRDY                   	TIMH_PCLKCON_HRRDY_Msk

#define TIMH_PCLKCON_HRERR_Pos                  (14U)
#define TIMH_PCLKCON_HRERR_Msk                  (0x1UL << TIMH_PCLKCON_HRERR_Pos)
#define TIMH_PCLKCON_HRERR                      TIMH_PCLKCON_HRERR_Msk

#define TIMH_PCLKCON_LOCK_Pos                   (8U)
#define TIMH_PCLKCON_LOCK_Msk                  	(0x1UL << TIMH_PCLKCON_LOCK_Pos)
#define TIMH_PCLKCON_LOCK	                   	TIMH_PCLKCON_LOCK_Msk

#define TIMH_PCLKCON_DIVSEL_Pos                 (0U)
#define TIMH_PCLKCON_DIVSEL_Msk                 (0x3UL << TIMH_PCLKCON_DIVSEL_Pos)
#define TIMH_PCLKCON_DIVSEL						TIMH_PCLKCON_DIVSEL_Msk

/**************  TIMH CMBTRIG Register        *********************************/
#define TIMH_CMBTRIG_CTB2EN_Pos                 (18U)
#define TIMH_CMBTRIG_CTB2EN_Msk                 (0x1UL << TIMH_CMBTRIG_CTB2EN_Pos)
#define TIMH_CMBTRIG_CTB2EN                     TIMH_CMBTRIG_CTB2EN_Msk

#define TIMH_CMBTRIG_CTB1EN_Pos                 (17U)
#define TIMH_CMBTRIG_CTB1EN_Msk                 (0x1 << TIMH_CMBTRIG_CTB1EN_Pos)
#define TIMH_CMBTRIG_CTB1EN						TIMH_CMBTRIG_CTB1EN_Msk

#define TIMH_CMBTRIG_CTB0EN_Pos                 (16U)
#define TIMH_CMBTRIG_CTB0EN_Msk                 (0x1UL << TIMH_CMBTRIG_CTB0EN_Pos)
#define TIMH_CMBTRIG_CTB0EN                   	TIMH_CMBTRIG_CTB0EN_Msk

#define TIMH_CMBTRIG_CTA2EN_Pos                 (2U)
#define TIMH_CMBTRIG_CTA2EN_Msk                 (0x1UL << TIMH_CMBTRIG_CTA2EN_Pos)
#define TIMH_CMBTRIG_CTA2EN                     TIMH_CMBTRIG_CTA2EN_Msk

#define TIMH_CMBTRIG_CTA1EN_Pos                 (1U)
#define TIMH_CMBTRIG_CTA1EN_Msk                 (0x1UL << TIMH_CMBTRIG_CTA1EN_Pos)
#define TIMH_CMBTRIG_CTA1EN	                   	TIMH_CMBTRIG_CTA1EN_Msk

#define TIMH_CMBTRIG_CTA0EN_Pos                 (0U)
#define TIMH_CMBTRIG_CTA0EN_Msk                 (0x1UL << TIMH_CMBTRIG_CTA0EN_Pos)
#define TIMH_CMBTRIG_CTA0EN						TIMH_CMBTRIG_CTA0EN_Msk

/**************  TIMH LOGCONy Register        *********************************/
#define TIMH_LOGCONy_PWMS1y_Pos                 (12U)
#define TIMH_LOGCONy_PWMS1y_Msk                 (0xFUL << TIMH_LOGCONy_PWMS1y_Pos)
#define TIMH_LOGCONy_PWMS1y                     TIMH_LOGCONy_PWMS1y_Msk

#define TIMH_LOGCONy_PWMS0y_Pos                 (8U)
#define TIMH_LOGCONy_PWMS0y_Msk                 (0xFUL << TIMH_LOGCONy_PWMS0y_Pos)
#define TIMH_LOGCONy_PWMS0y						TIMH_LOGCONy_PWMS0y_Msk

#define TIMH_LOGCONy_S1yPOL_Pos                 (7U)
#define TIMH_LOGCONy_S1yPOL_Msk                 (0x1UL << TIMH_LOGCONy_S1yPOL_Pos)
#define TIMH_LOGCONy_S1yPOL                   	TIMH_LOGCONy_S1yPOL_Msk

#define TIMH_LOGCONy_S0yPOL_Pos                 (6U)
#define TIMH_LOGCONy_S0yPOL_Msk                 (0x1UL << TIMH_LOGCONy_S0yPOL_Pos)
#define TIMH_LOGCONy_S0yPOL                     TIMH_LOGCONy_S0yPOL_Msk

#define TIMH_LOGCONy_PWMLFy_Pos                 (4U)
#define TIMH_LOGCONy_PWMLFy_Msk                 (0x3UL << TIMH_LOGCONy_PWMLFy_Pos)
#define TIMH_LOGCONy_PWMLFy	                   	TIMH_LOGCONy_PWMLFy_Msk

#define TIMH_LOGCONy_PWMLFyD_Pos                (0U)
#define TIMH_LOGCONy_PWMLFyD_Msk                (0x3UL << TIMH_LOGCONy_PWMLFyD_Pos)
#define TIMH_LOGCONy_PWMLFyD					TIMH_LOGCONy_PWMLFyD_Msk

/**************  TIMH PWMEVTy Register        *********************************/
#define TIMH_PWMEVTy_EVTySTRD_Pos               (13U)
#define TIMH_PWMEVTy_EVTySTRD_Msk               (0x1UL << TIMH_PWMEVTy_EVTySTRD_Pos)
#define TIMH_PWMEVTy_EVTySTRD					TIMH_PWMEVTy_EVTySTRD_Msk

#define TIMH_PWMEVTy_EVTySEL_Pos                (4U)
#define TIMH_PWMEVTy_EVTySEL_Msk                (0xFUL << TIMH_PWMEVTy_EVTySEL_Pos)
#define TIMH_PWMEVTy_EVTySEL					TIMH_PWMEVTy_EVTySEL_Msk

#define TIMH_PWMEVTy_EVTyPGS_Pos                (0U)
#define TIMH_PWMEVTy_EVTyPGS_Msk                (0x7UL << TIMH_PWMEVTy_EVTyPGS_Pos)
#define TIMH_PWMEVTy_EVTyPGS					TIMH_PWMEVTy_EVTyPGS_Msk

/**************  TIMH PGxCON Register         *********************************/
#define TIMH_PGxCON_MDCSEL_Pos                  (31U)    /*  */
#define TIMH_PGxCON_MDCSEL_Msk                  (0x1UL << TIMH_PGxCON_MDCSEL_Pos)
#define TIMH_PGxCON_MDCSEL                      TIMH_PGxCON_MDCSEL_Msk

#define TIMH_PGxCON_MPERSEL_Pos                 (30U)    /*  */
#define TIMH_PGxCON_MPERSEL_Msk                 (0x1UL << TIMH_PGxCON_MPERSEL_Pos)
#define TIMH_PGxCON_MPERSEL                     TIMH_PGxCON_MPERSEL_Msk

#define TIMH_PGxCON_MPHSEL_Pos                  (29U)    /*  */
#define TIMH_PGxCON_MPHSEL_Msk                  (0x1UL << TIMH_PGxCON_MPHSEL_Pos)
#define TIMH_PGxCON_MPHSEL                      TIMH_PGxCON_MPHSEL_Msk

#define TIMH_PGxCON_MSTEN_Pos                   (27U)    /*  */
#define TIMH_PGxCON_MSTEN_Msk                   (0x1UL << TIMH_PGxCON_MSTEN_Pos)
#define TIMH_PGxCON_MSTEN                       TIMH_PGxCON_MSTEN_Msk

#define TIMH_PGxCON_UPDMOD_Pos                  (24U)    /*  */
#define TIMH_PGxCON_UPDMOD_Msk                  (0x7UL << TIMH_PGxCON_UPDMOD_Pos)
#define TIMH_PGxCON_UPDMOD                      TIMH_PGxCON_UPDMOD_Msk

#define TIMH_PGxCON_TRGMOD_Pos                  (22U)    /*  */
#define TIMH_PGxCON_TRGMOD_Msk                  (0x1UL << TIMH_PGxCON_TRGMOD_Pos)
#define TIMH_PGxCON_TRGMOD                      TIMH_PGxCON_TRGMOD_Msk

#define TIMH_PGxCON_SOCS_Pos                    (16U)    /*  */
#define TIMH_PGxCON_SOCS_Msk                    (0xFUL << TIMH_PGxCON_SOCS_Pos)
#define TIMH_PGxCON_SOCS                        TIMH_PGxCON_SOCS_Msk

#define TIMH_PGxCON_ON_Pos                      (15U)    /* PGx enable */
#define TIMH_PGxCON_ON_Msk                      (0x1UL << TIMH_PGxCON_ON_Pos)
#define TIMH_PGxCON_ON                          TIMH_PGxCON_ON_Msk

#define TIMH_PGxCON_TRGCNT_Pos                  (8U)    /* PGx[TRGCNT] */
#define TIMH_PGxCON_TRGCNT_Msk                  (0x7UL << TIMH_PGxCON_TRGCNT_Pos)
#define TIMH_PGxCON_TRGCNT                      TIMH_PGxCON_TRGCNT_Msk

#define TIMH_PGxCON_HREN_Pos                    (7U)    /* PGx hr_mode select */
#define TIMH_PGxCON_HREN_Msk                    (0x1UL << TIMH_PGxCON_HREN_Pos)
#define TIMH_PGxCON_HREN                        TIMH_PGxCON_HREN_Msk

#define TIMH_PGxCON_CLKSEL_Pos                  (3U)    /* PGx clock select */
#define TIMH_PGxCON_CLKSEL_Msk                  (0x3UL << TIMH_PGxCON_CLKSEL_Pos)
#define TIMH_PGxCON_CLKSEL                      TIMH_PGxCON_CLKSEL_Msk

#define TIMH_PGxCON_MODSEL_Pos                  (0U)    /* PGx running mode */
#define TIMH_PGxCON_MODSEL_Msk                  (0x7UL << TIMH_PGxCON_MODSEL_Pos)
#define TIMH_PGxCON_MODSEL                      TIMH_PGxCON_MODSEL_Msk

/**************  TIMH PGxSTAT Register        ************************************************************/
#define TIMH_PGxSTAT_ADTRG2_Pos                  (28U)    /*  */
#define TIMH_PGxSTAT_ADTRG2_Msk                  (0x1UL << TIMH_PGxSTAT_ADTRG2_Pos)
#define TIMH_PGxSTAT_ADTRG2                      TIMH_PGxSTAT_ADTRG2_Msk

#define TIMH_PGxSTAT_ADTRG1_Pos                  (27U)    /*  */
#define TIMH_PGxSTAT_ADTRG1_Msk                  (0x1UL << TIMH_PGxSTAT_ADTRG1_Pos)
#define TIMH_PGxSTAT_ADTRG1                      TIMH_PGxSTAT_ADTRG1_Msk

#define TIMH_PGxSTAT_TRIGC_Pos                   (26U)    /*  */
#define TIMH_PGxSTAT_TRIGC_Msk                   (0x1UL << TIMH_PGxSTAT_TRIGC_Pos)
#define TIMH_PGxSTAT_TRIGC                       TIMH_PGxSTAT_TRIGC_Msk

#define TIMH_PGxSTAT_TRIGB_Pos                   (25U)    /*  */
#define TIMH_PGxSTAT_TRIGB_Msk                   (0x1UL << TIMH_PGxSTAT_TRIGB_Pos)
#define TIMH_PGxSTAT_TRIGB                       TIMH_PGxSTAT_TRIGB_Msk

#define TIMH_PGxSTAT_TRIGA_Pos                   (24U)    /*  */
#define TIMH_PGxSTAT_TRIGA_Msk                   (0x1UL << TIMH_PGxSTAT_TRIGA_Pos)
#define TIMH_PGxSTAT_TRIGA                       TIMH_PGxSTAT_TRIGA_Msk

#define TIMH_PGxSTAT_TOGERR_Pos                  (17U)    /*  */
#define TIMH_PGxSTAT_TOGERR_Msk                  (0x1UL << TIMH_PGxSTAT_TOGERR_Pos)
#define TIMH_PGxSTAT_TOGERR                      TIMH_PGxSTAT_TOGERR_Msk

#define TIMH_PGxSTAT_IEVT_Pos                    (16U)    /*  */
#define TIMH_PGxSTAT_IEVT_Msk                    (0x1UL << TIMH_PGxSTAT_IEVT_Pos)
#define TIMH_PGxSTAT_IEVT                        TIMH_PGxSTAT_IEVT_Msk

#define TIMH_PGxSTAT_SEVT_Pos                    (15U)    /*  */
#define TIMH_PGxSTAT_SEVT_Msk                    (0x1UL << TIMH_PGxSTAT_SEVT_Pos)
#define TIMH_PGxSTAT_SEVT                        TIMH_PGxSTAT_SEVT_Msk

#define TIMH_PGxSTAT_FLTEVT_Pos                  (14U)    /*  */
#define TIMH_PGxSTAT_FLTEVT_Msk                  (0x1UL << TIMH_PGxSTAT_FLTEVT_Pos)
#define TIMH_PGxSTAT_FLTEVT                      TIMH_PGxSTAT_FLTEVT_Msk

#define TIMH_PGxSTAT_CLEVT_Pos                   (13U)    /*  */
#define TIMH_PGxSTAT_CLEVT_Msk                   (0x1UL << TIMH_PGxSTAT_CLEVT_Pos)
#define TIMH_PGxSTAT_CLEVT                       TIMH_PGxSTAT_CLEVT_Msk

#define TIMH_PGxSTAT_FFEVT_Pos                   (12U)    /*  */
#define TIMH_PGxSTAT_FFEVT_Msk                   (0x1UL << TIMH_PGxSTAT_FFEVT_Pos)
#define TIMH_PGxSTAT_FFEVT                       TIMH_PGxSTAT_FFEVT_Msk

#define TIMH_PGxSTAT_SACT_Pos                    (11U)    /*  */
#define TIMH_PGxSTAT_SACT_Msk                    (0x1UL << TIMH_PGxSTAT_SACT_Pos)
#define TIMH_PGxSTAT_SACT                        TIMH_PGxSTAT_SACT_Msk

#define TIMH_PGxSTAT_FLTACT_Pos                  (10U)    /*  */
#define TIMH_PGxSTAT_FLTACT_Msk                  (0x1UL << TIMH_PGxSTAT_FLTACT_Pos)
#define TIMH_PGxSTAT_FLTACT                      TIMH_PGxSTAT_FLTACT_Msk

#define TIMH_PGxSTAT_CLACT_Pos                   (9U)    /*  */
#define TIMH_PGxSTAT_CLACT_Msk                   (0x1UL << TIMH_PGxSTAT_CLACT_Pos)
#define TIMH_PGxSTAT_CLACT                       TIMH_PGxSTAT_CLACT_Msk

#define TIMH_PGxSTAT_FFACT_Pos                   (8U)    /*  */
#define TIMH_PGxSTAT_FFACT_Msk                   (0x1UL << TIMH_PGxSTAT_FFACT_Pos)
#define TIMH_PGxSTAT_FFACT                       TIMH_PGxSTAT_FFACT_Msk

#define TIMH_PGxSTAT_TRSET_Pos                   (7U)    /*  */
#define TIMH_PGxSTAT_TRSET_Msk                   (0x1UL << TIMH_PGxSTAT_TRSET_Pos)
#define TIMH_PGxSTAT_TRSET                       TIMH_PGxSTAT_TRSET_Msk

#define TIMH_PGxSTAT_TRCLR_Pos                   (6U)    /*  */
#define TIMH_PGxSTAT_TRCLR_Msk                   (0x1UL << TIMH_PGxSTAT_TRCLR_Pos)
#define TIMH_PGxSTAT_TRCLR                       TIMH_PGxSTAT_TRCLR_Msk

#define TIMH_PGxSTAT_CAP_Pos                     (5U)    /*  */
#define TIMH_PGxSTAT_CAP_Msk                     (0x1UL << TIMH_PGxSTAT_CAP_Pos)
#define TIMH_PGxSTAT_CAP                         TIMH_PGxSTAT_CAP_Msk

#define TIMH_PGxSTAT_UPDATE_Pos                  (4U)    /*  */
#define TIMH_PGxSTAT_UPDATE_Msk                  (0x1UL << TIMH_PGxSTAT_UPDATE_Pos)
#define TIMH_PGxSTAT_UPDATE                      TIMH_PGxSTAT_UPDATE_Msk

#define TIMH_PGxSTAT_UPDREQ_Pos                  (3U)    /*  */
#define TIMH_PGxSTAT_UPDREQ_Msk                  (0x1UL << TIMH_PGxSTAT_UPDREQ_Pos)
#define TIMH_PGxSTAT_UPDREQ                      TIMH_PGxSTAT_UPDREQ_Msk

#define TIMH_PGxSTAT_STEER_Pos                   (2U)    /*  */
#define TIMH_PGxSTAT_STEER_Msk                   (0x1UL << TIMH_PGxSTAT_STEER_Pos)
#define TIMH_PGxSTAT_STEER                       TIMH_PGxSTAT_STEER_Msk

#define TIMH_PGxSTAT_CAHALF_Pos                  (1U)    /*  */
#define TIMH_PGxSTAT_CAHALF_Msk                  (0x1UL << TIMH_PGxSTAT_CAHALF_Pos)
#define TIMH_PGxSTAT_CAHALF                      TIMH_PGxSTAT_CAHALF_Msk

#define TIMH_PGxSTAT_TRIG_Pos                    (0U)    /*  */
#define TIMH_PGxSTAT_TRIG_Msk                    (0x1UL << TIMH_PGxSTAT_TRIG_Pos)
#define TIMH_PGxSTAT_TRIG                        TIMH_PGxSTAT_TRIG_Msk

/**************  TIMH PGxIOCON Register       *********************************/
#define TIMH_PGxIOCON_CAPSRC_Pos					(28U)    /*  */
#define TIMH_PGxIOCON_CAPSRC_Msk                (0x7UL << TIMH_PGxIOCON_CAPSRC_Pos)
#define TIMH_PGxIOCON_CAPSRC                      TIMH_PGxIOCON_CAPSRC_Msk

#define TIMH_PGxIOCON_DTCMPSEL_Pos                (24U)    /*  */
#define TIMH_PGxIOCON_DTCMPSEL_Msk                (0x1UL << TIMH_PGxIOCON_DTCMPSEL_Pos)
#define TIMH_PGxIOCON_DTCMPSEL                    TIMH_PGxIOCON_DTCMPSEL_Msk

#define TIMH_PGxIOCON_IDAT_Pos					  (22U)
#define TIMH_PGxIOCON_IDAT_Msk                    (0x3 << TIMH_PGxIOCON_IDAT_Pos)
#define TIMH_PGxIOCON_IDAT                        TIMH_PGxIOCON_IDAT_Msk

#define TIMH_PGxIOCON_PMOD_Pos                    (20U)    /*  */
#define TIMH_PGxIOCON_PMOD_Msk                    (0x3UL << TIMH_PGxIOCON_PMOD_Pos)
#define TIMH_PGxIOCON_PMOD                        TIMH_PGxIOCON_PMOD_Msk

#define TIMH_PGxIOCON_PENH_Pos                  (19U)    /*  */
#define TIMH_PGxIOCON_PENH_Msk                  (0x1UL << TIMH_PGxIOCON_PENH_Pos)
#define TIMH_PGxIOCON_PENH                      TIMH_PGxIOCON_PENH_Msk

#define TIMH_PGxIOCON_PENL_Pos                  (18U)    /*  */
#define TIMH_PGxIOCON_PENL_Msk                  (0x1UL << TIMH_PGxIOCON_PENL_Pos)
#define TIMH_PGxIOCON_PENL                      TIMH_PGxIOCON_PENL_Msk

#define TIMH_PGxIOCON_POLH_Pos                    (17U)    /*  */
#define TIMH_PGxIOCON_POLH_Msk                    (0x1UL << TIMH_PGxIOCON_POLH_Pos)
#define TIMH_PGxIOCON_POLH                        TIMH_PGxIOCON_POLH_Msk

#define TIMH_PGxIOCON_POLL_Pos                    (16U)    /*  */
#define TIMH_PGxIOCON_POLL_Msk                    (0x1UL << TIMH_PGxIOCON_POLL_Pos)
#define TIMH_PGxIOCON_POLL                        TIMH_PGxIOCON_POLL_Msk

#define TIMH_PGxIOCON_CLMOD_Pos                   (15U)    /*  */
#define TIMH_PGxIOCON_CLMOD_Msk                   (0x1UL << TIMH_PGxIOCON_CLMOD_Pos)
#define TIMH_PGxIOCON_CLMOD                       TIMH_PGxIOCON_CLMOD_Msk

#define TIMH_PGxIOCON_SWAP_Pos                    (14U)    /*  */
#define TIMH_PGxIOCON_SWAP_Msk                    (0x1UL << TIMH_PGxIOCON_SWAP_Pos)
#define TIMH_PGxIOCON_SWAP                        TIMH_PGxIOCON_SWAP_Msk

#define TIMH_PGxIOCON_OVRENH_Pos                  (13U)    /*  */
#define TIMH_PGxIOCON_OVRENH_Msk                  (0x1UL << TIMH_PGxIOCON_OVRENH_Pos)
#define TIMH_PGxIOCON_OVRENH                      TIMH_PGxIOCON_OVRENH_Msk

#define TIMH_PGxIOCON_OVRENL_Pos                  (12U)    /*  */
#define TIMH_PGxIOCON_OVRENL_Msk                  (0x1UL << TIMH_PGxIOCON_OVRENL_Pos)
#define TIMH_PGxIOCON_OVRENL                      TIMH_PGxIOCON_OVRENL_Msk

#define TIMH_PGxIOCON_OVRDAT_Pos                  (10U)    /*  */
#define TIMH_PGxIOCON_OVRDAT_Msk                  (0x3UL << TIMH_PGxIOCON_OVRDAT_Pos)
#define TIMH_PGxIOCON_OVRDAT                      TIMH_PGxIOCON_OVRDAT_Msk

#define TIMH_PGxIOCON_OSYNC_Pos                   (8U)    /*  */
#define TIMH_PGxIOCON_OSYNC_Msk                   (0x3UL << TIMH_PGxIOCON_OSYNC_Pos)
#define TIMH_PGxIOCON_OSYNC                       TIMH_PGxIOCON_OSYNC_Msk

#define TIMH_PGxIOCON_FLTDAT_Pos                  (6U)    /*  */
#define TIMH_PGxIOCON_FLTDAT_Msk                  (0x3UL << TIMH_PGxIOCON_FLTDAT_Pos)
#define TIMH_PGxIOCON_FLTDAT                      TIMH_PGxIOCON_FLTDAT_Msk

#define TIMH_PGxIOCON_CLDAT_Pos                   (4U)    /*  */
#define TIMH_PGxIOCON_CLDAT_Msk                   (0x3UL << TIMH_PGxIOCON_CLDAT_Pos)
#define TIMH_PGxIOCON_CLDAT                       TIMH_PGxIOCON_CLDAT_Msk

#define TIMH_PGxIOCON_FFDAT_Pos                   (2U)    /*  */
#define TIMH_PGxIOCON_FFDAT_Msk                   (0x3UL << TIMH_PGxIOCON_FFDAT_Pos)
#define TIMH_PGxIOCON_FFDAT                       TIMH_PGxIOCON_FFDAT_Msk

#define TIMH_PGxIOCON_DBDAT_Pos                   (0U)    /*  */
#define TIMH_PGxIOCON_DBDAT_Msk                   (0x3UL << TIMH_PGxIOCON_DBDAT_Pos)
#define TIMH_PGxIOCON_DBDAT                       TIMH_PGxIOCON_DBDAT_Msk

/**************  TIMH PGxEVT Register           ************************************************************/
#define TIMH_PGxEVT_FLTIEN_Pos                  (31U)    /*  */
#define TIMH_PGxEVT_FLTIEN_Msk                  (0x1UL << TIMH_PGxEVT_FLTIEN_Pos)
#define TIMH_PGxEVT_FLTIEN                      TIMH_PGxEVT_FLTIEN_Msk

#define TIMH_PGxEVT_CLIEN_Pos                   (30U)    /*  */
#define TIMH_PGxEVT_CLIEN_Msk                   (0x1UL << TIMH_PGxEVT_CLIEN_Pos)
#define TIMH_PGxEVT_CLIEN                       TIMH_PGxEVT_CLIEN_Msk

#define TIMH_PGxEVT_FFIEN_Pos                   (29U)    /*  */
#define TIMH_PGxEVT_FFIEN_Msk                   (0x1UL << TIMH_PGxEVT_FFIEN_Pos)
#define TIMH_PGxEVT_FFIEN                       TIMH_PGxEVT_FFIEN_Msk

#define TIMH_PGxEVT_SIEN_Pos                    (28U)    /*  */
#define TIMH_PGxEVT_SIEN_Msk                    (0x1UL << TIMH_PGxEVT_SIEN_Pos)
#define TIMH_PGxEVT_SIEN                        TIMH_PGxEVT_SIEN_Msk

#define TIMH_PGxEVT_IEVTSEL_Pos                 (24U)    /*  */
#define TIMH_PGxEVT_IEVTSEL_Msk                 (0x7UL << TIMH_PGxEVT_IEVTSEL_Pos)
#define TIMH_PGxEVT_IEVTSEL                     TIMH_PGxEVT_IEVTSEL_Msk

#define TIMH_PGxEVT_ADTR2EN3_Pos                (23U)    /*  */
#define TIMH_PGxEVT_ADTR2EN3_Msk                (0x1UL << TIMH_PGxEVT_ADTR2EN3_Pos)
#define TIMH_PGxEVT_ADTR2EN3                    TIMH_PGxEVT_ADTR2EN3_Msk

#define TIMH_PGxEVT_ADTR2EN2_Pos                (22U)    /*  */
#define TIMH_PGxEVT_ADTR2EN2_Msk                (0x1UL << TIMH_PGxEVT_ADTR2EN2_Pos)
#define TIMH_PGxEVT_ADTR2EN2                    TIMH_PGxEVT_ADTR2EN2_Msk

#define TIMH_PGxEVT_ADTR2EN1_Pos                (21U)    /*  */
#define TIMH_PGxEVT_ADTR2EN1_Msk                (0x1UL << TIMH_PGxEVT_ADTR2EN1_Pos)
#define TIMH_PGxEVT_ADTR2EN1                    TIMH_PGxEVT_ADTR2EN1_Msk

#define TIMH_PGxEVT_ADTR1OFS_Pos                (16U)    /*  */
#define TIMH_PGxEVT_ADTR1OFS_Msk                (0x1FUL << TIMH_PGxEVT_ADTR1OFS_Pos)
#define TIMH_PGxEVT_ADTR1OFS                    TIMH_PGxEVT_ADTR1OFS_Msk

#define TIMH_PGxEVT_ADTR1PS_Pos                 (11U)    /*  */
#define TIMH_PGxEVT_ADTR1PS_Msk                 (0x1FUL << TIMH_PGxEVT_ADTR1PS_Pos)
#define TIMH_PGxEVT_ADTR1PS                     TIMH_PGxEVT_ADTR1PS_Msk

#define TIMH_PGxEVT_ADTR1EN3_Pos                (10U)    /*  */
#define TIMH_PGxEVT_ADTR1EN3_Msk                (0x1UL << TIMH_PGxEVT_ADTR1EN3_Pos)
#define TIMH_PGxEVT_ADTR1EN3                    TIMH_PGxEVT_ADTR1EN3_Msk

#define TIMH_PGxEVT_ADTR1EN2_Pos                (19U)    /*  */
#define TIMH_PGxEVT_ADTR1EN2_Msk                (0x1UL << TIMH_PGxEVT_ADTR1EN2_Pos)
#define TIMH_PGxEVT_ADTR1EN2                    TIMH_PGxEVT_ADTR1EN2_Msk

#define TIMH_PGxEVT_ADTR1EN1_Pos                (8U)    /*  */
#define TIMH_PGxEVT_ADTR1EN1_Msk                (0x1UL << TIMH_PGxEVT_ADTR1EN1_Pos)
#define TIMH_PGxEVT_ADTR1EN1                    TIMH_PGxEVT_ADTR1EN1_Msk

#define TIMH_PGxEVT_UPDTRG_Pos                  (3U)    /*  */
#define TIMH_PGxEVT_UPDTRG_Msk                  (0x3UL << TIMH_PGxEVT_UPDTRG_Pos)
#define TIMH_PGxEVT_UPDTRG                      TIMH_PGxEVT_UPDTRG_Msk

#define TIMH_PGxEVT_PGTRGSEL_Pos                (0U)    /*  */
#define TIMH_PGxEVT_PGTRGSEL_Msk                (0x3UL << TIMH_PGxEVT_PGTRGSEL_Pos)
#define TIMH_PGxEVT_PGTRGSEL                    TIMH_PGxEVT_PGTRGSEL_Msk

/**************  TIMH PGxyPCI(x=[0:2], y=[F,CL,FF,S]) Register  ********************************/
#define TIMH_PGxyPCI_BPEN_Pos                   (31U)    /*  */
#define TIMH_PGxyPCI_BPEN_Msk                   (0x1UL << TIMH_PGxyPCI_BPEN_Pos)
#define TIMH_PGxyPCI_BPEN                       TIMH_PGxyPCI_BPEN_Msk

#define TIMH_PGxyPCI_BPSEL_Pos                	(28U)    /*  */
#define TIMH_PGxyPCI_BPSEL_Msk                	(0x7UL << TIMH_PGxyPCI_BPSEL_Pos)
#define TIMH_PGxyPCI_BPSEL                     	TIMH_PGxyPCI_BPSEL_Msk

#define TIMH_PGxyPCI_ACP_Pos                	(24U)    /*  */
#define TIMH_PGxyPCI_ACP_Msk                	(0x7UL << TIMH_PGxyPCI_ACP_Pos)
#define TIMH_PGxyPCI_ACP                    	TIMH_PGxyPCI_ACP_Msk

#define TIMH_PGxyPCI_SWPCI_Pos                  (23U)    /*  */
#define TIMH_PGxyPCI_SWPCI_Msk                  (0x1UL << TIMH_PGxyPCI_SWPCI_Pos)
#define TIMH_PGxyPCI_SWPCI                      TIMH_PGxyPCI_SWPCI_Msk

#define TIMH_PGxyPCI_SWPCIM_Pos                	(21U)    /*  */
#define TIMH_PGxyPCI_SWPCIM_Msk                	(0x3UL << TIMH_PGxyPCI_SWPCIM_Pos)
#define TIMH_PGxyPCI_SWPCIM                     TIMH_PGxyPCI_SWPCIM_Msk

#define TIMH_PGxyPCI_LATMOD_Pos                	(20U)    /*  */
#define TIMH_PGxyPCI_LATMOD_Msk                	(0x1UL << TIMH_PGxyPCI_LATMOD_Pos)
#define TIMH_PGxyPCI_LATMOD                    	TIMH_PGxyPCI_LATMOD_Msk

#define TIMH_PGxyPCI_TQPS_Pos                	(19U)    /*  */
#define TIMH_PGxyPCI_TQPS_Msk                	(0x1UL << TIMH_PGxyPCI_TQPS_Pos)
#define TIMH_PGxyPCI_TQPS                    	TIMH_PGxyPCI_TQPS_Msk

#define TIMH_PGxyPCI_TQSS_Pos                	(16U)    /*  */
#define TIMH_PGxyPCI_TQSS_Msk                	(0x3UL << TIMH_PGxyPCI_TQSS_Pos)
#define TIMH_PGxyPCI_TQSS                     	TIMH_PGxyPCI_TQSS_Msk

#define TIMH_PGxyPCI_TSYNCDIS_Pos               (15U)    /*  */
#define TIMH_PGxyPCI_TSYNCDIS_Msk               (0x1UL << TIMH_PGxyPCI_TSYNCDIS_Pos)
#define TIMH_PGxyPCI_TSYNCDIS                   TIMH_PGxyPCI_TSYNCDIS_Msk

#define TIMH_PGxyPCI_TERM_Pos                	(12U)    /*  */
#define TIMH_PGxyPCI_TERM_Msk                	(0x7UL << TIMH_PGxyPCI_TERM_Pos)
#define TIMH_PGxyPCI_TERM                    	TIMH_PGxyPCI_TERM_Msk

#define TIMH_PGxyPCI_AQPS_Pos                	(11U)    /*  */
#define TIMH_PGxyPCI_AQPS_Msk                	(0x1UL << TIMH_PGxyPCI_AQPS_Pos)
#define TIMH_PGxyPCI_AQPS                    	TIMH_PGxyPCI_AQPS_Msk

#define TIMH_PGxyPCI_AQSS_Pos                	(8U)    /* Acceptance Qualifier Source Selection */
#define TIMH_PGxyPCI_AQSS_Msk                	(0x7UL << TIMH_PGxyPCI_AQSS_Pos)
#define TIMH_PGxyPCI_AQSS                    	TIMH_PGxyPCI_AQSS_Msk

#define TIMH_PGxyPCI_SWTERM_Pos                	(7U)    /* PCI sofeware stop */
#define TIMH_PGxyPCI_SWTERM_Msk                	(0x1UL << TIMH_PGxyPCI_SWTERM_Pos)
#define TIMH_PGxyPCI_SWTERM                    	TIMH_PGxyPCI_SWTERM_Msk

#define TIMH_PGxyPCI_PSYNC_Pos                	(6U)    /*  */
#define TIMH_PGxyPCI_PSYNC_Msk                	(0x1UL << TIMH_PGxyPCI_PSYNC_Pos)
#define TIMH_PGxyPCI_PSYNC                    	TIMH_PGxyPCI_PSYNC_Msk

#define TIMH_PGxyPCI_PPS_Pos                	(5U)    /*  */
#define TIMH_PGxyPCI_PPS_Msk                	(0x1UL << TIMH_PGxyPCI_PPS_Pos)
#define TIMH_PGxyPCI_PPS                    	TIMH_PGxyPCI_PPS_Msk

#define TIMH_PGxyPCI_PSS_Pos                 	(0U)    /*  */
#define TIMH_PGxyPCI_PSS_Msk                 	(0x1FUL << TIMH_PGxyPCI_PSS_Pos)
#define TIMH_PGxyPCI_PSS                     	TIMH_PGxyPCI_PSS_Msk

/**************  TIMH PGxLEB Register         *********************************/
#define TIMH_PGxLEB_PWMPCI_Pos                	(24U)    /*  */
#define TIMH_PGxLEB_PWMPCI_Msk                	(0x7UL << TIMH_PGxLEB_PWMPCI_Pos)
#define TIMH_PGxLEB_PWMPCI                    	TIMH_PGxLEB_PWMPCI_Msk

#define TIMH_PGxLEB_PHR_Pos                		(19U)    /*  */
#define TIMH_PGxLEB_PHR_Msk                		(0x1UL << TIMH_PGxLEB_PHR_Pos)
#define TIMH_PGxLEB_PHR                    		TIMH_PGxLEB_PHR_Msk

#define TIMH_PGxLEB_PHF_Pos                		(18U)    /*  */
#define TIMH_PGxLEB_PHF_Msk                		(0x1UL << TIMH_PGxLEB_PHF_Pos)
#define TIMH_PGxLEB_PHF                    		TIMH_PGxLEB_PHF_Msk

#define TIMH_PGxLEB_PLR_Pos                		(17U)    /*  */
#define TIMH_PGxLEB_PLR_Msk                		(0x1UL << TIMH_PGxLEB_PLR_Pos)
#define TIMH_PGxLEB_PLR                    		TIMH_PGxLEB_PLR_Msk

#define TIMH_PGxLEB_PLF_Pos                		(16U)    /*  */
#define TIMH_PGxLEB_PLF_Msk                		(0x1UL << TIMH_PGxLEB_PHR_Pos)
#define TIMH_PGxLEB_PLF                    		TIMH_PGxLEB_PHR_Msk

#define TIMH_PGxLEB_LEBP_Pos                	(0U)    /* Leading-Edge Blanking Period */
#define TIMH_PGxLEB_LEBP_Msk                	(0xFFFFUL << TIMH_PGxLEB_LEBP_Pos)
#define TIMH_PGxLEB_LEBP                    	TIMH_PGxLEB_LEBP_Msk

/**************  TIMH PGxTRIGA                *********************************/
#define TIMH_PGxTRIGA_TRIGASEL_Pos                (31U)    /* */
#define TIMH_PGxTRIGA_TRIGASEL_Msk                (0x1UL << TIMH_PGxTRIGA_TRIGASEL_Pos)
#define TIMH_PGxTRIGA_TRIGASEL                    TIMH_PGxTRIGA_TRIGASEL_Msk

/**************  TIMH PGxTRIGB                *********************************/
#define TIMH_PGxTRIGB_TRIGBSEL_Pos                (31U)    /* */
#define TIMH_PGxTRIGB_TRIGBSEL_Msk                (0x1UL << TIMH_PGxTRIGB_TRIGBSEL_Pos)
#define TIMH_PGxTRIGB_TRIGBSEL                    TIMH_PGxTRIGB_TRIGBSEL_Msk

/**************  TIMH PGxTRIGC Register       *********************************/
#define TIMH_PGxTRIGC_TRIGCSEL_Pos                (31U)    /* */
#define TIMH_PGxTRIGC_TRIGCSEL_Msk                (0x1UL << TIMH_PGxTRIGC_TRIGCSEL_Pos)
#define TIMH_PGxTRIGC_TRIGCSEL                    TIMH_PGxTRIGC_TRIGCSEL_Msk


/******************************************************************************/
/*                                     TIMG                                   */
/******************************************************************************/
/***********************  TIM Control Register0  ***********************************/
#define TIMG_CR0_CNT_ST_Pos                     (10U)
#define TIMG_CR0_CNT_ST_Msk						(0x1UL << TIMG_CR0_CNT_ST_Pos)
#define TIMG_CR0_CNT_ST							TIMG_CR0_CNT_ST_Msk
#define TIMG_CR0_CDK_Pos                   		(8U)
#define TIMG_CR0_CDK_Msk                        (0x3UL << TIMG_CR0_CDK_Pos)
#define TIMG_CR0_CDK                   		 	TIMG_CR0_CDK_Msk
#define TIMG_CR0_ARPE_Pos                  		(7U)
#define TIMG_CR0_ARPE_Msk                  		(0x1UL << TIMG_CR0_ARPE_Pos)
#define TIMG_CR0_ARPE                   		 TIMG_CR0_ARPE_Msk
#define TIMG_CR0_CMS_Pos                   		(5U)
#define TIMG_CR0_CMS_Msk                  		(0x3UL << TIMG_CR0_CMS_Pos)
#define TIMG_CR0_CMS                   			TIMG_CR0_CMS_Msk
#define TIMG_CR0_DIR_Pos                   		(4U)
#define TIMG_CR0_DIR_Msk                  		(0x1UL << TIMG_CR0_DIR_Pos)
#define TIMG_CR0_DIR                  		 	TIMG_CR0_DIR_Msk
#define TIMG_CR0_OPM_Pos                   		(3U)
#define TIMG_CR0_OPM_Msk                  		(0x1UL << TIMG_CR0_OPM_Pos)
#define TIMG_CR0_OPM                   		 	TIMG_CR0_OPM_Msk
#define TIMG_CR0_URS_Pos                   		(2U)
#define TIMG_CR0_URS_Msk                   		(0x1UL << TIMG_CR0_URS_Pos)
#define TIMG_CR0_URS					  		TIMG_CR0_URS_Msk
#define TIMG_CR0_UDIS_Pos				  		(1U)
#define TIMG_CR0_UDIS_Msk                 		(0x1UL << TIMG_CR0_UDIS_Pos)
#define TIMG_CR0_UDIS				  			TIMG_CR0_UDIS_Msk
#define TIMG_CR0_CEN_Pos				      	(0U)
#define TIMG_CR0_CEN_Msk					  	(0x1UL << TIMG_CR0_CEN_Pos)
#define TIMG_CR0_CEN				      		TIMG_CR0_CEN_Msk
/***********************  TIM Control Register1  ***********************************/
#define TIMG_CR1_TI0S_Pos				  		(7U)
#define TIMG_CR1_TI0S_Msk				  		(0x1UL << TIMG_CR1_TI0S_Pos)
#define TIMG_CR1_TI0S	              	  		TIMG_CR1_TI0S_Msk
#define TIMG_CR1_MMS_Pos					  	(4U)
#define TIMG_CR1_MMS_Msk				  		(0x7UL << TIMG_CR1_MMS_Pos)
#define TIMG_CR1_MMS                   		 	TIMG_CR1_MMS_Msk
#define TIMG_CR1_CCDS_Pos				  		(3U)
#define TIMG_CR1_CCDS_Msk				  		(0x1UL << TIMG_CR1_CCDS_Pos)
#define TIMG_CR1_CCDS                   		TIMG_CR1_CCDS_Msk
/***********************  TIM Mode Control Register  ***************************/
#define TIMG_SMCR_QEES_Pos						(28U)
#define TIMG_SMCR_QEES_Msk                      (0x03UL << TIMG_SMCR_QEES_Pos)
#define TIMG_SMCR_QEES                          TIMG_SMCR_QEES_Msk
#define TIMG_SMCR_QEM_Pos						(24U)
#define TIMG_SMCR_QEM_Msk                       (0x07UL << TIMG_SMCR_QEM_Pos)
#define TIMG_SMCR_QEM                           TIMG_SMCR_QEM_Msk
#define TIMG_SMCR_OCCS_Pos						(20U)
#define TIMG_SMCR_OCCS_Msk                      (0x01UL << TIMG_SMCR_OCCS_Pos)
#define TIMG_SMCR_OCCS                          TIMG_SMCR_OCCS_Msk
#define TIMG_SMCR_ETEN_Pos                      (16U)
#define TIMG_SMCR_ETEN_Msk						(0x01UL << TIMG_SMCR_ETEN_Pos)
#define TIMG_SMCR_ETEN							TIMG_SMCR_ETEN_Msk
#define TIMG_SMCR_ETP_Pos				   		(15U)
#define TIMG_SMCR_ETP_Msk				   		(0x1UL << TIMG_SMCR_ETP_Pos)
#define TIMG_SMCR_ETP	                   		TIMG_SMCR_ETP_Msk
#define TIMG_SMCR_ECE_Pos				   		(14U)
#define TIMG_SMCR_ECE_Msk				   		(0x1UL << TIMG_SMCR_ECE_Pos)
#define TIMG_SMCR_ECE                    		TIMG_SMCR_ECE_Msk
#define TIMG_SMCR_ETPS_Pos				   		(12U)
#define TIMG_SMCR_ETPS_Msk				   		(0x3UL << TIMG_SMCR_ETPS_Pos)
#define TIMG_SMCR_ETPS                  		TIMG_SMCR_ETPS_Msk
#define TIMG_SMCR_ETF_Pos				   		(8U)
#define TIMG_SMCR_ETF_Msk                  		(0xFUL << TIMG_SMCR_ETF_Pos)
#define TIMG_SMCR_ETF            	   			TIMG_SMCR_ETF_Msk
#define TIMG_SMCR_MSM_Pos				   		(7U)
#define TIMG_SMCR_MSM_Msk				   		(0x1UL << TIMG_SMCR_MSM_Pos)
#define TIMG_SMCR_MSM	                    	TIMG_SMCR_MSM_Msk
#define TIMG_SMCR_TS_Pos					   	(4U)
#define TIMG_SMCR_TS_Msk				   		(0x7UL << TIMG_SMCR_TS_Pos)
#define TIMG_SMCR_TS					   		TIMG_SMCR_TS_Msk
#define TIMG_SMCR_SMS_Pos                   	(0U)
#define TIMG_SMCR_SMS_Msk				   		(0x7UL << TIMG_SMCR_SMS_Pos)
#define TIMG_SMCR_SMS                   		TIMG_SMCR_SMS_Msk
/*****************  TIM DMA and Interrupt Enable Register  *************************/
#define TIMG_DIER_TDE_Pos						(14U)
#define TIMG_DIER_TDE_Msk						(0x1UL << TIMG_DIER_TDE_Pos)
#define TIMG_DIER_TDE	 						TIMG_DIER_TDE_Msk
#define TIMG_DIER_CC3DE_Pos						(12U)
#define TIMG_DIER_CC3DE_Msk						(0x1UL << TIMG_DIER_CC3DE_Pos)
#define TIMG_DIER_CC3DE 						TIMG_DIER_CC3DE_Msk
#define TIMG_DIER_CC2DE_Pos						(11U)
#define TIMG_DIER_CC2DE_Msk						(0x1UL << TIMG_DIER_CC2DE_Pos)
#define TIMG_DIER_CC2DE 						TIMG_DIER_CC2DE_Msk
#define TIMG_DIER_CC1DE_Pos						(10U)
#define TIMG_DIER_CC1DE_Msk						(0x1UL << TIMG_DIER_CC1DE_Pos)
#define TIMG_DIER_CC1DE 						TIMG_DIER_CC1DE_Msk
#define TIMG_DIER_CC0DE_Pos						(9U)
#define TIMG_DIER_CC0DE_Msk						(0x1UL << TIMG_DIER_CC0DE_Pos)
#define TIMG_DIER_CC0DE 						TIMG_DIER_CC0DE_Msk
#define TIMG_DIER_UDE_Pos						(8U)
#define TIMG_DIER_UDE_Msk						(0x1UL << TIMG_DIER_UDE_Pos)
#define TIMG_DIER_UDE	 						TIMG_DIER_UDE_Msk
#define TIMG_DIER_TIE_Pos						(6U)
#define TIMG_DIER_TIE_Msk						(0x1UL << TIMG_DIER_TIE_Pos)
#define TIMG_DIER_TIE	 						TIMG_DIER_TIE_Msk
#define TIMG_DIER_CC3IE_Pos						(4U)
#define TIMG_DIER_CC3IE_Msk						(0x1UL << TIMG_DIER_CC3IE_Pos)
#define TIMG_DIER_CC3IE 						TIMG_DIER_CC3IE_Msk
#define TIMG_DIER_CC2IE_Pos						(3U)
#define TIMG_DIER_CC2IE_Msk						(0x1UL << TIMG_DIER_CC2IE_Pos)
#define TIMG_DIER_CC2IE 						TIMG_DIER_CC2IE_Msk
#define TIMG_DIER_CC1IE_Pos						(2U)
#define TIMG_DIER_CC1IE_Msk						(0x1UL << TIMG_DIER_CC1IE_Pos)
#define TIMG_DIER_CC1IE 						TIMG_DIER_CC1IE_Msk
#define TIMG_DIER_CC0IE_Pos						(1U)
#define TIMG_DIER_CC0IE_Msk						(0x1UL << TIMG_DIER_CC0IE_Pos)
#define TIMG_DIER_CC0IE 						TIMG_DIER_CC0IE_Msk
#define TIMG_DIER_UIE_Pos						(0U)
#define TIMG_DIER_UIE_Msk						(0x1UL << TIMG_DIER_UIE_Pos)
#define TIMG_DIER_UIE 						TIMG_DIER_UIE_Msk
/**************************  TIM State Register  ****************************/
#define TIMG_SR_CC3OF_Pos                    	(12U)
#define TIMG_SR_CC3OF_Msk						(0x1UL << TIMG_SR_CC3OF_Pos)
#define TIMG_SR_CC3OF							TIMG_SR_CC3OF_Msk
#define TIMG_SR_CC2OF_Pos                    	(11U)
#define TIMG_SR_CC2OF_Msk						(0x1UL << TIMG_SR_CC2OF_Pos)
#define TIMG_SR_CC2OF							TIMG_SR_CC2OF_Msk
#define TIMG_SR_CC1OF_Pos                    	(10U)
#define TIMG_SR_CC1OF_Msk						(0x1UL << TIMG_SR_CC1OF_Pos)
#define TIMG_SR_CC1OF							TIMG_SR_CC1OF_Msk
#define TIMG_SR_CC0OF_Pos                    	(9U)
#define TIMG_SR_CC0OF_Msk						(0x1UL << TIMG_SR_CC0OF_Pos)
#define TIMG_SR_CC0OF							TIMG_SR_CC0OF_Msk
#define TIMG_SR_TIF_Pos                    		(6U)
#define TIMG_SR_TIF_Msk							(0x1UL << TIMG_SR_TIF_Pos)
#define TIMG_SR_TIF								TIMG_SR_TIF_Msk
#define TIMG_SR_CC3IF_Pos                    	(4U)
#define TIMG_SR_CC3IF_Msk						(0x1UL << TIMG_SR_CC3IF_Pos)
#define TIMG_SR_CC3IF							TIMG_SR_CC3IF_Msk
#define TIMG_SR_CC2IF_Pos                    	(3U)
#define TIMG_SR_CC2IF_Msk						(0x1UL << TIMG_SR_CC2IF_Pos)
#define TIMG_SR_CC2IF							TIMG_SR_CC2IF_Msk
#define TIMG_SR_CC1IF_Pos                    	(2U)
#define TIMG_SR_CC1IF_Msk						(0x1UL << TIMG_SR_CC1IF_Pos)
#define TIMG_SR_CC1IF							TIMG_SR_CC1IF_Msk
#define TIMG_SR_CC0IF_Pos                    	(1U)
#define TIMG_SR_CC0IF_Msk						(0x1UL << TIMG_SR_CC0IF_Pos)
#define TIMG_SR_CC0IF							TIMG_SR_CC0IF_Msk
#define TIMG_SR_UIF_Pos                    		(0U)
#define TIMG_SR_UIF_Msk							(0x1UL << TIMG_SR_UIF_Pos)
#define TIMG_SR_UIF								TIMG_SR_UIF_Msk
/***********************  TIM Event Generate Register  ***********************/
#define TIMG_EGR_TG_Pos                      	(6U)
#define TIMG_EGR_TG_Msk							(0x1UL << TIMG_EGR_TG_Pos)
#define TIMG_EGR_TG 							TIMG_EGR_TG_Msk
#define TIMG_EGR_CC3G_Pos                    	(4U)
#define TIMG_EGR_CC3G_Msk						(0x1UL << TIMG_EGR_CC3G_Pos)
#define TIMG_EGR_CC3G 							TIMG_EGR_CC3G_Msk
#define TIMG_EGR_CC2G_Pos                    	(3U)
#define TIMG_EGR_CC2G_Msk						(0x1UL << TIMG_EGR_CC2G_Pos)
#define TIMG_EGR_CC2G 							TIMG_EGR_CC2G_Msk
#define TIMG_EGR_CC1G_Pos                    	(2U)
#define TIMG_EGR_CC1G_Msk						(0x1UL << TIMG_EGR_CC1G_Pos)
#define TIMG_EGR_CC1G 							TIMG_EGR_CC1G_Msk
#define TIMG_EGR_CC0G_Pos                    	(1U)
#define TIMG_EGR_CC0G_Msk						(0x1UL << TIMG_EGR_CC0G_Pos)
#define TIMG_EGR_CC0G 							TIMG_EGR_CC0G_Msk
#define TIMG_EGR_UG_Pos                    		(0U)
#define TIMG_EGR_UG_Msk							(0x1UL << TIMG_EGR_UG_Pos)
#define TIMG_EGR_UG 							TIMG_EGR_UG_Msk
/***********************  TIM Capture/Compare Mode Register0(OUTPUT)   *****************/
#define TIMG_CCMR0_OC1CE_Pos					(24U)
#define TIMG_CCMR0_OC1CE_Msk					(0x1UL << TIMG_CCMR0_OC1CE_Pos)
#define TIMG_CCMR0_OC1CE 						TIMG_CCMR0_OC1CE_Msk
#define TIMG_CCMR0_OC1M_Pos						(20U)
#define TIMG_CCMR0_OC1M_Msk						(0xFUL << TIMG_CCMR0_OC1M_Pos)
#define TIMG_CCMR0_OC1M							TIMG_CCMR0_OC1M_Msk
#define TIMG_CCMR0_OC1PE_Pos                 	(19U)
#define TIMG_CCMR0_OC1PE_Msk					(0x1UL << TIMG_CCMR0_OC1PE_Pos)
#define TIMG_CCMR0_OC1PE 						TIMG_CCMR0_OC1PE_Msk
#define TIMG_CCMR0_OC1FE_Pos                 	(18U)
#define TIMG_CCMR0_OC1FE_Msk					(0x1UL << TIMG_CCMR0_OC1FE_Pos)
#define TIMG_CCMR0_OC1FE 						TIMG_CCMR0_OC1FE_Msk
#define TIMG_CCMR0_CC1S_Pos                  	(16U)
#define TIMG_CCMR0_CC1S_Msk                		(0x3UL << TIMG_CCMR0_CC1S_Pos)
#define TIMG_CCMR0_CC1S							TIMG_CCMR0_CC1S_Msk
#define TIMG_CCMR0_OC0CE_Pos					(8U)
#define TIMG_CCMR0_OC0CE_Msk					(0x1UL << TIMG_CCMR0_OC0CE_Pos)
#define TIMG_CCMR0_OC0CE						TIMG_CCMR0_OC0CE_Msk
#define TIMG_CCMR0_OC0M_Pos						(4U)
#define TIMG_CCMR0_OC0M_Msk						(0xFUL << TIMG_CCMR0_OC0M_Pos)
#define TIMG_CCMR0_OC0M							TIMG_CCMR0_OC0M_Msk
#define TIMG_CCMR0_OC0PE_Pos                 	(3U)
#define TIMG_CCMR0_OC0PE_Msk					(0x1UL << TIMG_CCMR0_OC0PE_Pos)
#define TIMG_CCMR0_OC0PE 						TIMG_CCMR0_OC0PE_Msk
#define TIMG_CCMR0_OC0FE_Pos                 	(2U)
#define TIMG_CCMR0_OC0FE_Msk					(0x1UL << TIMG_CCMR0_OC0FE_Pos)
#define TIMG_CCMR0_OC0FE 						TIMG_CCMR0_OC0FE_Msk
#define TIMG_CCMR0_CC0S_Pos                  	(0U)
#define TIMG_CCMR0_CC0S_Msk                		(0x3UL << TIMG_CCMR0_CC0S_Pos)
#define TIMG_CCMR0_CC0S 						TIMG_CCMR0_CC0S_Msk
/***********************  TIM Capture/Compare Mode Register0(INPUT)   *****************/
#define TIMG_CCMR0_IC1F_Pos                  	(20U)
#define TIMG_CCMR0_IC1F_Msk                		(0xFUL << TIMG_CCMR0_IC1F_Pos)
#define TIMG_CCMR0_IC1F 						TIMG_CCMR0_IC1F_Msk
#define TIMG_CCMR0_IC1PSC_Pos                	(18U)
#define TIMG_CCMR0_IC1PSC_Msk              		(0x3UL << TIMG_CCMR0_IC1PSC_Pos)
#define TIMG_CCMR0_IC1PSC               		TIMG_CCMR0_IC1PSC_Msk
#define TIMG_CCMR0_IC0F_Pos                  	(4U)
#define TIMG_CCMR0_IC0F_Msk                		(0xFUL << TIMG_CCMR0_IC0F_Pos)
#define TIMG_CCMR0_IC0F 						TIMG_CCMR0_IC0F_Msk
#define TIMG_CCMR0_IC0PSC_Pos                	(2U)
#define TIMG_CCMR0_IC0PSC_Msk              		(0x3UL << TIMG_CCMR0_IC0PSC_Pos)
#define TIMG_CCMR0_IC0PSC               		TIMG_CCMR0_IC0PSC_Msk

/***********************  TIM Capture/Compare Mode Register1(OUTPUT)   *****************/
#define TIMG_CCMR1_OC3CE_Pos					(24U)
#define TIMG_CCMR1_OC3CE_Msk					(0x1UL << TIMG_CCMR1_OC3CE_Pos)
#define TIMG_CCMR1_OC3CE 						TIMG_CCMR1_OC3CE_Msk
#define TIMG_CCMR1_OC3M_Pos						(20U)
#define TIMG_CCMR1_OC3M_Msk						(0xFUL << TIMG_CCMR1_OC3M_Pos)
#define TIMG_CCMR1_OC3M							TIMG_CCMR1_OC3M_Msk
#define TIMG_CCMR1_OC3PE_Pos                 	(19U)
#define TIMG_CCMR1_OC3PE_Msk					(0x1UL << TIMG_CCMR1_OC3PE_Pos)
#define TIMG_CCMR1_OC3PE 						TIMG_CCMR1_OC3PE_Msk
#define TIMG_CCMR1_OC3FE_Pos                 	(18U)
#define TIMG_CCMR1_OC3FE_Msk					(0x1UL << TIMG_CCMR1_OC3FE_Pos)
#define TIMG_CCMR1_OC3FE 						TIMG_CCMR1_OC3FE_Msk
#define TIMG_CCMR1_CC3S_Pos                  	(16U)
#define TIMG_CCMR1_CC3S_Msk                		(0x3UL << TIMG_CCMR1_CC3S_Pos)
#define TIMG_CCMR1_CC3S							TIMG_CCMR1_CC3S_Msk
#define TIMG_CCMR1_OC2CE_Pos					(8U)
#define TIMG_CCMR1_OC2CE_Msk					(0x1UL << TIMG_CCMR1_OC2CE_Pos)
#define TIMG_CCMR1_OC2CE						TIMG_CCMR1_OC2CE_Msk
#define TIMG_CCMR1_OC2M_Pos						(4U)
#define TIMG_CCMR1_OC2M_Msk						(0xFUL << TIMG_CCMR1_OC2M_Pos)
#define TIMG_CCMR1_OC2M							TIMG_CCMR1_OC2M_Msk
#define TIMG_CCMR1_OC2PE_Pos                 	(3U)
#define TIMG_CCMR1_OC2PE_Msk					(0x1UL << TIMG_CCMR1_OC2PE_Pos)
#define TIMG_CCMR1_OC2PE 						TIMG_CCMR1_OC2PE_Msk
#define TIMG_CCMR1_OC2FE_Pos                 	(2U)
#define TIMG_CCMR1_OC2FE_Msk					(0x1UL << TIMG_CCMR1_OC2FE_Pos)
#define TIMG_CCMR1_OC2FE 						TIMG_CCMR1_OC2FE_Msk
#define TIMG_CCMR1_CC2S_Pos                  	(0U)
#define TIMG_CCMR1_CC2S_Msk                		(0x3UL << TIMG_CCMR1_CC2S_Pos)
#define TIMG_CCMR1_CC2S 						TIMG_CCMR1_CC2S_Msk
/***********************  TIM Capture/Compare Mode Register1(INPUT)   *****************/
#define TIMG_CCMR1_IC3F_Pos                  	(20U)
#define TIMG_CCMR1_IC3F_Msk                		(0xFUL << TIMG_CCMR1_IC3F_Pos)
#define TIMG_CCMR1_IC3F 						TIMG_CCMR1_IC3F_Msk
#define TIMG_CCMR1_IC3PSC_Pos                	(18U)
#define TIMG_CCMR1_IC3PSC_Msk 	             	(0x3UL << TIMG_CCMR1_IC3PSC_Pos)
#define TIMG_CCMR1_IC3PSC               		TIMG_CCMR1_IC3PSC_Msk
#define TIMG_CCMR1_IC2F_Pos                  	(4U)
#define TIMG_CCMR1_IC2F_Msk                		(0xFUL << TIMG_CCMR1_IC2F_Pos)
#define TIMG_CCMR1_IC2F                 		TIMG_CCMR1_IC2F_Msk
#define TIMG_CCMR1_IC2PSC_Pos                	(2U)
#define TIMG_CCMR1_IC2PSC_Msk              		(0x3UL << TIMG_CCMR1_IC2PSC_Pos)
#define TIMG_CCMR1_IC2PSC               		TIMG_CCMR1_IC2PSC_Msk
/*********************   TIM Capture/Compare Mode Enable Register   *****************/
#define TIMG_CCER_CC3NP_Pos                  	(15U)
#define TIMG_CCER_CC3NP_Msk                  	(0x1UL << TIMG_CCER_CC3NP_Pos)
#define TIMG_CCER_CC3NP                   		TIMG_CCER_CC3NP_Msk
#define TIMG_CCER_CC3P_Pos                   	(13U)
#define TIMG_CCER_CC3P_Msk                   	(0x1UL << TIMG_CCER_CC3P_Pos)
#define TIMG_CCER_CC3P                    		TIMG_CCER_CC3P_Msk
#define TIMG_CCER_CC3E_Pos                   	(12U)
#define TIMG_CCER_CC3E_Msk                   	(0x1UL << TIMG_CCER_CC3E_Pos)
#define TIMG_CCER_CC3E                   		TIMG_CCER_CC3E_Msk
#define TIMG_CCER_CC2NP_Pos                  	(11U)
#define TIMG_CCER_CC2NP_Msk                  	(0x1UL << TIMG_CCER_CC2NP_Pos)
#define TIMG_CCER_CC2NP                  		TIMG_CCER_CC2NP_Msk
#define TIMG_CCER_CC2P_Pos                   	(9U)
#define TIMG_CCER_CC2P_Msk                  	(0x1UL << TIMG_CCER_CC2P_Pos)
#define TIMG_CCER_CC2P                    		TIMG_CCER_CC2P_Msk
#define TIMG_CCER_CC2E_Pos                   	(8U)
#define TIMG_CCER_CC2E_Msk                   	(0x1UL << TIMG_CCER_CC2E_Pos)
#define TIMG_CCER_CC2E	                    	TIMG_CCER_CC2E_Msk
#define TIMG_CCER_CC1NP_Pos                  	(7U)
#define TIMG_CCER_CC1NP_Msk                  	(0x1UL << TIMG_CCER_CC1NP_Pos)
#define TIMG_CCER_CC1NP		                   	TIMG_CCER_CC1NP_Msk
#define TIMG_CCER_CC1P_Pos                   	(5U)
#define TIMG_CCER_CC1P_Msk                   	(0x1UL << TIMG_CCER_CC1P_Pos)
#define TIMG_CCER_CC1P   	                 	TIMG_CCER_CC1P_Msk
#define TIMG_CCER_CC1E_Pos                 		(4U)
#define TIMG_CCER_CC1E_Msk                   	(0x1UL << TIMG_CCER_CC1E_Pos)
#define TIMG_CCER_CC1E                   		TIMG_CCER_CC1E_Msk
#define TIMG_CCER_CC0NP_Pos                  	(3U)
#define TIMG_CCER_CC0NP_Msk                  	(0x1UL << TIMG_CCER_CC0NP_Pos)
#define TIMG_CCER_CC0NP  	                 	TIMG_CCER_CC0NP_Msk
#define TIMG_CCER_CC0P_Pos                   	(1U)
#define TIMG_CCER_CC0P_Msk                   	(0x1UL << TIMG_CCER_CC0P_Pos)
#define TIMG_CCER_CC0P	                    	TIMG_CCER_CC0P_Msk
#define TIMG_CCER_CC0E_Pos                   	(0U)
#define TIMG_CCER_CC0E_Msk                   	(0x1UL << TIMG_CCER_CC0E_Pos)
#define TIMG_CCER_CC0E	                    	TIMG_CCER_CC0E_Msk
/******************************************************************************/
/*                                     TIMS                                   */
/******************************************************************************/
/******************  Bit definition for TIM Control register  *****************/
#define TIMS_CR_CCE_Pos                     (26U)  /* collaborative commutation enable */
#define TIMS_CR_CCE_Msk                     (0x1UL << TIMS_CR_CCE_Pos)
#define TIMS_CR_CCE                         TIMS_CR_CCE_Msk
#define TIMS_CR_TCE_Pos                     (25U)  /* trigger commutation enable */
#define TIMS_CR_TCE_Msk                     (0x1UL << TIMS_CR_TCE_Pos)
#define TIMS_CR_TCE                         TIMS_CR_TCE_Msk
#define TIMS_CR_CCPC_Pos                    (24U)  /* Capture/compare preloaded control */
#define TIMS_CR_CCPC_Msk                    (0x1UL << TIMS_CR_CCPC_Pos)
#define TIMS_CR_CCPC                        TIMS_CR_CCPC_Msk
#define TIMS_CR_HIZ2_Pos                    (22U)  /* high resistance output enable */
#define TIMS_CR_HIZ2_Msk                    (0x1UL << TIMS_CR_HIZ2_Pos)
#define TIMS_CR_HIZ2                        TIMS_CR_HIZ2_Msk
#define TIMS_CR_HIZ1N_Pos                   (21U)  /* high resistance output enable */
#define TIMS_CR_HIZ1N_Msk                   (0x1UL << TIMS_CR_HIZ1N_Pos)
#define TIMS_CR_HIZ1N                        TIMS_CR_HIZ1N_Msk
#define TIMS_CR_HIZ1_Pos                    (20U)  /* high resistance output enable */
#define TIMS_CR_HIZ1_Msk                    (0x1UL << TIMS_CR_HIZ1_Pos)
#define TIMS_CR_HIZ1                        TIMS_CR_HIZ1_Msk
#define TIMS_CR_OIS2_Pos                    (18U)  /* OC2 output */
#define TIMS_CR_OIS2_Msk                    (0x1UL << TIMS_CR_OIS2_Pos)
#define TIMS_CR_OIS2                        TIMS_CR_OIS2_Msk
#define TIMS_CR_OIS1N_Pos                   (17U)  /* OC1N output */
#define TIMS_CR_OIS1N_Msk                   (0x1UL << TIMS_CR_OIS1N_Pos)
#define TIMS_CR_OIS1N                       TIMS_CR_OIS1N_Msk
#define TIMS_CR_OIS1_Pos                    (16U)   /* OC1 output */
#define TIMS_CR_OIS1_Msk                    (0x1UL << TIMS_CR_OIS1_Pos)
#define TIMS_CR_OIS1                        TIMS_CR_OIS1_Msk
#define TIMS_CR_CEC_Pos                     (12U)   /* counetr collaborative enable control*/
#define TIMS_CR_CEC_Msk                     (0x1UL << TIMS_CR_CEC_Pos)
#define TIMS_CR_CEC                         TIMS_CR_CEC_Msk
#define TIMS_CR_CKD_Pos                     (8U)   /* clock div */
#define TIMS_CR_CKD_Msk                     (0x3UL << TIMS_CR_CKD_Pos)
#define TIMS_CR_CKD   					   TIMS_CR_CKD_Msk
#define TIMS_CR_ARPE_Pos                    (7U)    /*auto_reload*/
#define TIMS_CR_ARPE_Msk                    (0x1UL << TIMS_CR_ARPE_Pos)
#define TIMS_CR_ARPE                        TIMS_CR_ARPE_Msk
#define TIMS_CR_CMS_Pos                     (5U)   /* center mode register */
#define TIMS_CR_CMS_Msk                     (0x3UL << TIMS_CR_CMS_Pos)
#define TIMS_CR_CMS							TIMS_CR_CMS_Msk
#define TIMS_CR_DIR_Pos                     (4U)   /* direction register */
#define TIMS_CR_DIR_Msk                     (0x1UL << TIMS_CR_DIR_Pos)
#define TIMS_CR_DIR							TIMS_CR_DIR_Msk
#define TIMS_CR_OPM_Pos                     (3U)    /*one_pulse ctrl*/
#define TIMS_CR_OPM_Msk                     (0x1UL << TIMS_CR_OPM_Pos)
#define TIMS_CR_OPM                         TIMS_CR_OPM_Msk
#define TIMS_CR_URS_Pos                     (2U)    /*one_pulse ctrl*/
#define TIMS_CR_URS_Msk                     (0x1UL << TIMS_CR_URS_Pos)
#define TIMS_CR_URS                         TIMS_CR_URS_Msk
#define TIMS_CR_UDIS_Pos                    (1U)    /*forbid update*/
#define TIMS_CR_UDIS_Msk                    (0x1UL << TIMS_CR_UDIS_Pos)
#define TIMS_CR_UDIS                        TIMS_CR_UDIS_Msk
#define TIMS_CR_CEN_Pos                     (0U)    /*counter enable control*/
#define TIMS_CR_CEN_Msk                     (0x1UL << TIMS_CR_CEN_Pos)
#define TIMS_CR_CEN                         TIMS_CR_CEN_Msk

/***********  Bit definition for TIM Mode Control Register  *************/
#define TIMS_MCR_TS_Pos                     (20U)   /*triger sign select*/
#define TIMS_MCR_TS_Msk                     (0x7UL << TIMS_MCR_TS_Pos)
#define TIMS_MCR_TS                         TIMS_MCR_TS_Msk
#define TIMS_MCR_SMS_Pos                    (16U)    /*Slave mode Select*/
#define TIMS_MCR_SMS_Msk                    (0x7UL << TIMS_MCR_SMS_Pos)
#define TIMS_MCR_SMS                        TIMS_MCR_SMS_Msk
#define TIMS_MCR_MSM_Pos                    (8U)    /*Master/Slave mode*/
#define TIMS_MCR_MSM_Msk                    (0x1UL << TIMS_MCR_MSM_Pos)
#define TIMS_MCR_MSM                        TIMS_MCR_MSM_Msk
#define TIMS_MCR_MMS_Pos                    (0U)    /*Master mode selection*/
#define TIMS_MCR_MMS_Msk                    (0x7UL << TIMS_MCR_MMS_Pos)
#define TIMS_MCR_MMS                        TIMS_MCR_MMS_Msk

/********  Bit definition for TIM Capture/Compare Mode Register (Compare) *********/
#define TIMS_CCMR_OC2M_Pos                  (12U)    /*output compare 2 mode*/
#define TIMS_CCMR_OC2M_Msk                  (0x7UL << TIMS_CCMR_OC2M_Pos)
#define TIMS_CCMR_OC2M     					TIMS_CCMR_OC2M_Msk
#define TIMS_CCMR_OC2PE_Pos                 (11U)
#define TIMS_CCMR_OC2PE_Msk                 (0x0UL << TIMS_CCMR_OC2PE_Pos)
#define TIMS_CCMR_OC2PE                     TIMS_CCMR_OC2PE_Msk
#define TIMS_CCMR_CC2S_Pos                  (8U)     /*capture/compare channel select*/
#define TIMS_CCMR_CC2S_Msk                  (0x3UL << TIMS_CCMR_CC2S_Pos)
#define TIMS_CCMR_CC2S 						TIMS_CCMR_CC2S_Msk
#define TIMS_CCMR_OC1CE_Pos                 (7U)     /*OCREF CLR*/
#define TIMS_CCMR_OC1CE_Msk                 (0x1UL << TIMS_CCMR_OC1CE_Pos)
#define TIMS_CCMR_OC1CE 					TIMS_CCMR_OC1CE_Msk
#define TIMS_CCMR_OC1M_Pos                  (4U)    /*output compare 1 mode*/
#define TIMS_CCMR_OC1M_Msk                  (0x7UL << TIMS_CCMR_OC1M_Pos)
#define TIMS_CCMR_OC1M                      TIMS_CCMR_OC1M_Msk
#define TIMS_CCMR_OC1PE_Pos                 (3U)
#define TIMS_CCMR_OC1PE_Msk                 (0x1UL << TIMS_CCMR_OC1PE_Pos)
#define TIMS_CCMR_OC1PE                     TIMS_CCMR_OC1PE_Msk
#define TIMS_CCMR_CC1S_Pos                  (0U)      /* capture/compare channel select */
#define TIMS_CCMR_CC1S_Msk                  (0x3UL << TIMS_CCMR_CC1S_Pos)
#define TIMS_CCMR_CC1S                      TIMS_CCMR_CC1S_Msk

/********  Bit definition for TIM Capture/Compare Mode Register (Capture) *********/
#define TIMS_CCMR_IC2FC_Pos                 (15U)
#define TIMS_CCMR_IC2FC_Msk                 (0x1UL << TIMS_CCMR_IC2FC_Pos)
#define TIMS_CCMR_IC2FC                     TIMS_CCMR_IC2FC_Msk
#define TIMS_CCMR_IC2F_Pos                  (12U)
#define TIMS_CCMR_IC2F_Msk                  (0x7UL << TIMS_CCMR_IC2F_Pos)
#define TIMS_CCMR_IC2F                      TIMS_CCMR_IC2F_Msk
#define TIMS_CCMR_IC2PSC_Pos                (10U)
#define TIMS_CCMR_IC2PSC_Msk                (0x3UL << TIMS_CCMR_IC2PSC_Pos)
#define TIMS_CCMR_IC2PSC					TIMS_CCMR_IC2PSC_Msk
#define TIMS_CCMR_IC1FC_Pos                 (7U)
#define TIMS_CCMR_IC1FC_Msk                 (0x1UL << TIMS_CCMR_IC1FC_Pos)
#define TIMS_CCMR_IC1FC                     TIMS_CCMR_IC1FC_Msk
#define TIMS_CCMR_IC1F_Pos                  (4U)
#define TIMS_CCMR_IC1F_Msk                  (0x7UL << TIMS_CCMR_IC1F_Pos)
#define TIMS_CCMR_IC1F                      TIMS_CCMR_IC1F_Msk
#define TIMS_CCMR_IC1PSC_Pos                (2U)
#define TIMS_CCMR_IC1PSC_Msk                (0x3UL << TIMS_CCMR_IC1PSC_Pos)
#define TIMS_CCMR_IC1PSC					TIMS_CCMR_IC1PSC_Msk

/********  Bit definition for TIM Capture/Compare Enable Register  *********/
#define TIMS_CCER_CC2NP_Pos                 (7U)
#define TIMS_CCER_CC2NP_Msk                 (0x1UL << TIMS_CCER_CC2NP_Pos)
#define TIMS_CCER_CC2NP                     TIMS_CCER_CC2NP_Msk
#define TIMS_CCER_CC2P_Pos                  (5U)
#define TIMS_CCER_CC2P_Msk                  (0x1UL << TIMS_CCER_CC2P_Pos)
#define TIMS_CCER_CC2P                      TIMS_CCER_CC2P_Msk
#define TIMS_CCER_CC2E_Pos                  (4U)
#define TIMS_CCER_CC2E_Msk                  (0x1UL << TIMS_CCER_CC2E_Pos)
#define TIMS_CCER_CC2E                      TIMS_CCER_CC2E_Msk
#define TIMS_CCER_CC1NP_Pos                 (3U)
#define TIMS_CCER_CC1NP_Msk                 (0x1UL << TIMS_CCER_CC1NP_Pos)
#define TIMS_CCER_CC1NP                     TIMS_CCER_CC1NP_Msk
#define TIMS_CCER_CC1NE_Pos                 (2U)
#define TIMS_CCER_CC1NE_Msk                 (0x1UL << TIMS_CCER_CC1NE_Pos)
#define TIMS_CCER_CC1NE                     TIMS_CCER_CC1NE_Msk
#define TIMS_CCER_CC1P_Pos                  (1U)
#define TIMS_CCER_CC1P_Msk                  (0x1UL << TIMS_CCER_CC1P_Pos)
#define TIMS_CCER_CC1P                      TIMS_CCER_CC1P_Msk
#define TIMS_CCER_CC1E_Pos                  (0U)
#define TIMS_CCER_CC1E_Msk                  (0x1UL << TIMS_CCER_CC1E_Pos)
#define TIMS_CCER_CC1E                      TIMS_CCER_CC1E_Msk

/********  Bit definition for TIM Break And Dead-Time Register  *********/
#define TIMS_BDTR_MOE_Pos                   (15U)
#define TIMS_BDTR_MOE_Msk                   (0x1UL << TIMS_BDTR_MOE_Pos)
#define TIMS_BDTR_MOE                       TIMS_BDTR_MOE_Msk
#define TIMS_BDTR_AOE_Pos                   (14U)
#define TIMS_BDTR_AOE_Msk                   (0x1UL << TIMS_BDTR_AOE_Pos)
#define TIMS_BDTR_AOE                       TIMS_BDTR_AOE_Msk
#define TIMS_BDTR_BKE_Pos                   (12U)
#define TIMS_BDTR_BKE_Msk                   (0x1UL << TIMS_BDTR_BKE_Pos)
#define TIMS_BDTR_BKE                       TIMS_BDTR_BKE_Msk
#define TIMS_BDTR_LOCK_Pos                  (8U)
#define TIMS_BDTR_LOCK_Msk                  (0x3UL << TIMS_BDTR_LOCK_Pos)
#define TIMS_BDTR_LOCK                      TIMS_BDTR_LOCK_Msk

/********  Bit definition for TIM Event Generation Register  *********/
#define TIMS_EGR_BG_Pos                     (7U)
#define TIMS_EGR_BG_Msk                     (0x1UL << TIMS_EGR_BG_Pos)
#define TIMS_EGR_BG                         TIMS_EGR_BG_Msk
#define TIMS_EGR_COMG_Pos                   (5U)
#define TIMS_EGR_COMG_Msk                   (0x1UL << TIMS_EGR_COMG_Pos)
#define TIMS_EGR_COMG                       TIMS_EGR_COMG_Msk
#define TIMS_EGR_CC2G_Pos                   (2U)
#define TIMS_EGR_CC2G_Msk                   (0x1UL << TIMS_EGR_CC2G_Pos)
#define TIMS_EGR_CC2G                       TIMS_EGR_CC2G_Msk
#define TIMS_EGR_CC1G_Pos                   (1U)
#define TIMS_EGR_CC1G_Msk                   (0x1UL << TIMS_EGR_CC1G_Pos)
#define TIMS_EGR_CC1G                       TIMS_EGR_CC1G_Msk
#define TIMS_EGR_UG_Pos                     (0U)
#define TIMS_EGR_UG_Msk                     (0x1UL << TIMS_EGR_UG_Pos)
#define TIMS_EGR_UG                         TIMS_EGR_UG_Msk

/********  Bit definition for TIM Interrupt Enable Register  *********/
#define TIMS_IER_BIE_Pos                    (7U)
#define TIMS_IER_BIE_Msk                    (0x1UL << TIMS_IER_BIE_Pos)
#define TIMS_IER_BIE                        TIMS_IER_BIE_Msk
#define TIMS_IER_TIE_Pos                    (6U)
#define TIMS_IER_TIE_Msk                    (0x1UL << TIMS_IER_TIE_Pos)
#define TIMS_IER_TIE                        TIMS_IER_TIE_Msk
#define TIMS_IER_COMIE_Pos                  (5U)
#define TIMS_IER_COMIE_Msk                  (0x1UL << TIMS_IER_COMIE_Pos)
#define TIMS_IER_COMIE                      TIMS_IER_COMIE_Msk
#define TIMS_IER_CC2IE_Pos                  (2U)
#define TIMS_IER_CC2IE_Msk                  (0x1UL << TIMS_IER_CC2IE_Pos)
#define TIMS_IER_CC2IE                      TIMS_IER_CC2IE_Msk
#define TIMS_IER_CC1IE_Pos                  (1U)
#define TIMS_IER_CC1IE_Msk                  (0x1UL << TIMS_IER_CC1IE_Pos)
#define TIMS_IER_CC1IE                      TIMS_IER_CC1IE_Msk
#define TIMS_IER_UIE_Pos                    (0U)
#define TIMS_IER_UIE_Msk                    (0x1UL << TIMS_IER_UIE_Pos)
#define TIMS_IER_UIE                        TIMS_IER_UIE_Msk

/*************  Bit definition for TIM Status Register  *************/
#define TIMS_SR_CC2OF_Pos                   (10U)
#define TIMS_SR_CC2OF_Msk                   (0x1UL << TIMS_SR_CC2OF_Pos)
#define TIMS_SR_CC2OF                       TIMS_SR_CC2OF_Msk
#define TIMS_SR_CC1OF_Pos                   (9U)
#define TIMS_SR_CC1OF_Msk                   (0x1UL << TIMS_SR_CC1OF_Pos)
#define TIMS_SR_CC1OF                       TIMS_SR_CC1OF_Msk
#define TIMS_SR_BIF_Pos                     (7U)
#define TIMS_SR_BIF_Msk                     (0x1UL << TIMS_SR_BIF_Pos)
#define TIMS_SR_BIF                         TIMS_SR_BIF_Msk
#define TIMS_SR_TIF_Pos                     (6U)
#define TIMS_SR_TIF_Msk                     (0x1UL << TIMS_SR_TIF_Pos)
#define TIMS_SR_TIF                         TIMS_SR_TIF_Msk
#define TIMS_SR_COMIF_Pos                   (5U)
#define TIMS_SR_COMIF_Msk                   (0x1UL << TIMS_SR_COMIF_Pos)
#define TIMS_SR_COMIF                       TIMS_SR_COMIF_Msk
#define TIMS_SR_CC2IF_Pos                   (2U)
#define TIMS_SR_CC2IF_Msk                   (0x1UL << TIMS_SR_CC2IF_Pos)
#define TIMS_SR_CC2IF                       TIMS_SR_CC2IF_Msk
#define TIMS_SR_CC1IF_Pos                   (1U)
#define TIMS_SR_CC1IF_Msk                   (0x1UL << TIMS_SR_CC1IF_Pos)
#define TIMS_SR_CC1IF                       TIMS_SR_CC1IF_Msk
#define TIMS_SR_UIF_Pos                     (0U)
#define TIMS_SR_UIF_Msk                     (0x1UL << TIMS_SR_UIF_Pos)
#define TIMS_SR_UIF                         TIMS_SR_UIF_Msk

/********* TIMA0 Input Repetition Control Register   *************************/
#define TIMA0_IMCR_TI1_EBUS_SEL_Pos         (12U)
#define TIMA0_IMCR_TI1_EBUS_SEL_Msk         (0xFUL << TIMA0_IMCR_TI1_EBUS_SEL_Pos)
#define TIMA0_IMCR_TI1_EBUS_SEL             TIMA0_IMCR_TI1_EBUS_SEL_Msk
#define TIMA0_IMCR_TI1_SEL_Pos         		(8U)
#define TIMA0_IMCR_TI1_SEL_Msk         		(0x1UL << TIMA0_IMCR_TI1_SEL_Pos)
#define TIMA0_IMCR_TI1_SEL             		TIMA0_IMCR_TI1_SEL_Msk
#define TIMA0_IMCR_TI0_EBUS_SEL_Pos         (4U)
#define TIMA0_IMCR_TI0_EBUS_SEL_Msk         (0xFUL << TIMA0_IMCR_TI0_EBUS_SEL_Pos)
#define TIMA0_IMCR_TI0_EBUS_SEL             TIMA0_IMCR_TI0_EBUS_SEL_Msk
#define TIMA0_IMCR_TI0_SEL_Pos         		(0U)
#define TIMA0_IMCR_TI0_SEL_Msk         		(0x1UL << TIMA0_IMCR_TI0_SEL_Pos)
#define TIMA0_IMCR_TI0_SEL             		TIMA0_IMCR_TI0_SEL_Msk
/******** TIMG0 Input Repetition Control Register	*************************/
#define TIMG0_IMCR_TI2_EBUS_SEL_Pos         (20U)
#define TIMG0_IMCR_TI2_EBUS_SEL_Msk         (0xFUL << TIMG0_IMCR_TI2_EBUS_SEL_Pos)
#define TIMG0_IMCR_TI2_EBUS_SEL             TIMG0_IMCR_TI2_EBUS_SEL_Msk
#define TIMG0_IMCR_TI2_SEL_Pos         		(16U)
#define TIMG0_IMCR_TI2_SEL_Msk         		(0x1UL << TIMG0_IMCR_TI2_SEL_Pos)
#define TIMG0_IMCR_TI2_SEL             		TIMG0_IMCR_TI2_SEL_Msk
#define TIMG0_IMCR_TI1_EBUS_SEL_Pos         (12U)
#define TIMG0_IMCR_TI1_EBUS_SEL_Msk         (0xFUL << TIMG0_IMCR_TI1_EBUS_SEL_Pos)
#define TIMG0_IMCR_TI1_EBUS_SEL             TIMG0_IMCR_TI1_EBUS_SEL_Msk
#define TIMG0_IMCR_TI1_SEL_Pos         		(8U)
#define TIMG0_IMCR_TI1_SEL_Msk         		(0x1UL << TIMG0_IMCR_TI1_SEL_Pos)
#define TIMG0_IMCR_TI1_SEL             		TIMG0_IMCR_TI1_SEL_Msk
#define TIMG0_IMCR_TI0_EBUS_SEL_Pos         (4U)
#define TIMG0_IMCR_TI0_EBUS_SEL_Msk         (0xFUL << TIMG0_IMCR_TI0_EBUS_SEL_Pos)
#define TIMG0_IMCR_TI0_EBUS_SEL             TIMG0_IMCR_TI0_EBUS_SEL_Msk
#define TIMG0_IMCR_TI0_SEL_Pos         		(0U)
#define TIMG0_IMCR_TI0_SEL_Msk         		(0x1UL << TIMG0_IMCR_TI0_SEL_Pos)
#define TIMG0_IMCR_TI0_SEL             		TIMG0_IMCR_TI0_SEL_Msk
/********* TIMG1 Input Repetition Control Register	************************/
#define TIMG1_IMCR_TI2_EBUS_SEL_Pos         (20U)
#define TIMG1_IMCR_TI2_EBUS_SEL_Msk         (0xFUL << TIMG1_IMCR_TI2_EBUS_SEL_Pos)
#define TIMG1_IMCR_TI2_EBUS_SEL             TIMG1_IMCR_TI2_EBUS_SEL_Msk
#define TIMG1_IMCR_TI2_SEL_Pos         		(16U)
#define TIMG1_IMCR_TI2_SEL_Msk         		(0x1UL << TIMG1_IMCR_TI2_SEL_Pos)
#define TIMG1_IMCR_TI2_SEL             		TIMG1_IMCR_TI2_SEL_Msk
#define TIMG1_IMCR_TI1_EBUS_SEL_Pos         (12U)
#define TIMG1_IMCR_TI1_EBUS_SEL_Msk         (0xFUL << TIMG1_IMCR_TI1_EBUS_SEL_Pos)
#define TIMG1_IMCR_TI1_EBUS_SEL             TIMG1_IMCR_TI1_EBUS_SEL_Msk
#define TIMG1_IMCR_TI1_SEL_Pos         		(8U)
#define TIMG1_IMCR_TI1_SEL_Msk         		(0x1UL << TIMG1_IMCR_TI1_SEL_Pos)
#define TIMG1_IMCR_TI1_SEL             		TIMG1_IMCR_TI1_SEL_Msk
#define TIMG1_IMCR_TI0_EBUS_SEL_Pos         (4U)
#define TIMG1_IMCR_TI0_EBUS_SEL_Msk         (0xFUL << TIMG1_IMCR_TI0_EBUS_SEL_Pos)
#define TIMG1_IMCR_TI0_EBUS_SEL             TIMG1_IMCR_TI0_EBUS_SEL_Msk
#define TIMG1_IMCR_TI0_SEL_Pos         		(0U)
#define TIMG1_IMCR_TI0_SEL_Msk         		(0x1UL << TIMG1_IMCR_TI0_SEL_Pos)
#define TIMG1_IMCR_TI0_SEL             		TIMG1_IMCR_TI0_SEL_Msk
/********* TIMG2 Input Repetition Control Register	************************/
#define TIMG2_IMCR_TI2_EBUS_SEL_Pos         (20U)
#define TIMG2_IMCR_TI2_EBUS_SEL_Msk         (0xFUL << TIMG2_IMCR_TI2_EBUS_SEL_Pos)
#define TIMG2_IMCR_TI2_EBUS_SEL             TIMG2_IMCR_TI2_EBUS_SEL_Msk
#define TIMG2_IMCR_TI2_SEL_Pos         		(16U)
#define TIMG2_IMCR_TI2_SEL_Msk         		(0x1UL << TIMG2_IMCR_TI2_SEL_Pos)
#define TIMG2_IMCR_TI2_SEL             		TIMG2_IMCR_TI2_SEL_Msk
#define TIMG2_IMCR_TI1_EBUS_SEL_Pos         (12U)
#define TIMG2_IMCR_TI1_EBUS_SEL_Msk         (0xFUL << TIMG2_IMCR_TI1_EBUS_SEL_Pos)
#define TIMG2_IMCR_TI1_EBUS_SEL             TIMG2_IMCR_TI1_EBUS_SEL_Msk
#define TIMG2_IMCR_TI1_SEL_Pos         		(8U)
#define TIMG2_IMCR_TI1_SEL_Msk         		(0x1UL << TIMG2_IMCR_TI1_SEL_Pos)
#define TIMG2_IMCR_TI1_SEL             		TIMG2_IMCR_TI1_SEL_Msk
#define TIMG2_IMCR_TI0_EBUS_SEL_Pos         (4U)
#define TIMG2_IMCR_TI0_EBUS_SEL_Msk         (0xFUL << TIMG2_IMCR_TI0_EBUS_SEL_Pos)
#define TIMG2_IMCR_TI0_EBUS_SEL             TIMG2_IMCR_TI0_EBUS_SEL_Msk
#define TIMG2_IMCR_TI0_SEL_Pos         		(0U)
#define TIMG2_IMCR_TI0_SEL_Msk         		(0x1UL << TIMG2_IMCR_TI0_SEL_Pos)
#define TIMG2_IMCR_TI0_SEL             		TIMG2_IMCR_TI0_SEL_Msk
/********* TIMS  Input Repetition Control Register		*********************/
#define TIMS3_IMCR_TI0_EBUS_SEL_Pos         (28U)
#define TIMS3_IMCE_TI0_EBUS_SEL_Msk         (0xFUL << TIMS3_IMCR_TI0_EBUS_SEL_Pos)
#define TIMS3_IMCE_TI0_EBUS_SEL				TIMS3_IMCE_TI0_EBUS_SEL_Msk
#define TIMS3_IMCR_TI0_SEL_Pos              (24U)
#define TIMS3_IMCR_TI0_SEL_Msk              (0x1UL << TIMS3_IMCR_TI0_SEL_Pos)
#define TIMS3_IMCR_TI0_SEL					TIMS3_IMCR_TI0_SEL_Msk
#define TIMS2_IMCR_TI0_EBUS_SEL_Pos         (20U)
#define TIMS2_IMCE_TI0_EBUS_SEL_Msk         (0xFUL << TIMS2_IMCR_TI0_EBUS_SEL_Pos)
#define TIMS2_IMCE_TI0_EBUS_SEL				TIMS2_IMCE_TI0_EBUS_SEL_Msk
#define TIMS2_IMCR_TI0_SEL_Pos              (16U)
#define TIMS2_IMCR_TI0_SEL_Msk              (0x1UL << TIMS2_IMCR_TI0_SEL_Pos)
#define TIMS2_IMCR_TI0_SEL					TIMS2_IMCR_TI0_SEL_Msk
#define TIMS1_IMCR_TI0_EBUS_SEL_Pos         (12U)
#define TIMS1_IMCE_TI0_EBUS_SEL_Msk         (0xFUL << TIMS1_IMCR_TI0_EBUS_SEL_Pos)
#define TIMS1_IMCE_TI0_EBUS_SEL				TIMS1_IMCE_TI0_EBUS_SEL_Msk
#define TIMS1_IMCR_TI0_SEL_Pos              (8U)
#define TIMS1_IMCR_TI0_SEL_Msk              (0x1UL << TIMS1_IMCR_TI0_SEL_Pos)
#define TIMS1_IMCR_TI0_SEL					TIMS1_IMCR_TI0_SEL_Msk
#define TIMS0_IMCR_TI0_EBUS_SEL_Pos         (4U)
#define TIMS0_IMCE_TI0_EBUS_SEL_Msk         (0xFUL << TIMS0_IMCR_TI0_EBUS_SEL_Pos)
#define TIMS0_IMCE_TI0_EBUS_SEL				TIMS0_IMCE_TI0_EBUS_SEL_Msk
#define TIMS0_IMCR_TI0_SEL_Pos              (0U)
#define TIMS0_IMCR_TI0_SEL_Msk              (0x1UL << TIMS0_IMCR_TI0_SEL_Pos)
#define TIMS0_IMCR_TI0_SEL					TIMS0_IMCR_TI0_SEL_Msk
/*************  TIMA  Internal Trigger  Control Register    *******************/
#define TIMA1_ITCR_ITR_SEL_Pos              (4U)
#define TIMA1_ITCR_ITR_SEL_Msk              (0xFUL << TIMA1_ITCR_ITR_SEL_Pos)
#define TIMA1_ITCR_ITR_SEL                  TIMA1_ITCR_ITR_SEL_Msk
#define TIMA0_ITCR_ITR_SEL_Pos              (0U)
#define TIMA0_ITCR_ITR_SEL_Msk              (0xFUL << TIMA0_ITCR_ITR_SEL_Pos)
#define TIMA0_ITCR_ITR_SEL                  TIMA0_ITCR_ITR_SEL_Msk
/*************  TIMG  Internal Trigger  Control Register    *******************/
#define TIMG2_ITCR_ITR_SEL_Pos              (8U)
#define TIMG2_ITCR_ITR_SEL_Msk              (0xFUL << TIMG2_ITCR_ITR_SEL_Pos)
#define TIMG2_ITCR_ITR_SEL                  TIMG2_ITCR_ITR_SEL_Msk
#define TIMG1_ITCR_ITR_SEL_Pos              (4U)
#define TIMG1_ITCR_ITR_SEL_Msk              (0xFUL << TIMG1_ITCR_ITR_SEL_Pos)
#define TIMG1_ITCR_ITR_SEL                  TIMG1_ITCR_ITR_SEL_Msk
#define TIMG0_ITCR_ITR_SEL_Pos              (0U)
#define TIMG0_ITCR_ITR_SEL_Msk              (0xFUL << TIMG0_ITCR_ITR_SEL_Pos)
#define TIMG0_ITCR_ITR_SEL                  TIMG0_ITCR_ITR_SEL_Msk
/*************  TIMS  Internal Trigger  Control Register    *******************/
#define TIMS3_ITCR_ITR_SEL_Pos              (12U)
#define TIMS3_ITCR_ITR_SEL_Msk              (0xFUL << TIMS3_ITCR_ITR_SEL_Pos)
#define TIMS3_ITCR_ITR_SEL                  TIMS3_ITCR_ITR_SEL_Msk
#define TIMS2_ITCR_ITR_SEL_Pos              (8U)
#define TIMS2_ITCR_ITR_SEL_Msk              (0xFUL << TIMS2_ITCR_ITR_SEL_Pos)
#define TIMS2_ITCR_ITR_SEL                  TIMS2_ITCR_ITR_SEL_Msk
#define TIMS1_ITCR_ITR_SEL_Pos              (4U)
#define TIMS1_ITCR_ITR_SEL_Msk              (0xFUL << TIMS1_ITCR_ITR_SEL_Pos)
#define TIMS1_ITCR_ITR_SEL                  TIMS1_ITCR_ITR_SEL_Msk
#define TIMS0_ITCR_ITR_SEL_Pos              (0U)
#define TIMS0_ITCR_ITR_SEL_Msk              (0xFUL << TIMS0_ITCR_ITR_SEL_Pos)
#define TIMS0_ITCR_ITR_SEL                  TIMS0_ITCR_ITR_SEL_Msk
/************* TIMA0 System Broken Circuit Control Register      ****************/
#define TIMA0_SBCR_EBK_EN_Pos               (16U)
#define TIMA0_SBCR_EBK_EN_Msk               (0xFFFFUL << TIMA0_SBCR_EBK_EN_Pos)
#define TIMA0_SBCR_EBK_EN                   TIMA0_SBCR_EBK_EN_Msk
#define TIMA0_SBCR_BRK_POL_Pos              (5U)
#define TIMA0_SBCR_BRK_POL_Msk              (0x1UL << TIMA0_SBCR_BRK_POL_Pos)
#define TIMA0_SBCR_BRK_POL                  TIMA0_SBCR_BRK_POL_Msk
#define TIMA0_SBCR_PBK_EN_Pos               (4U)
#define TIMA0_SBCR_PBK_EN_Msk               (1UL << TIMA0_SBCR_PBK_EN_Pos)
#define TIMA0_SBCR_PBK_EN                   TIMA0_SBCR_PBK_EN_Msk
#define TIMA0_SBCR_SBK3_EN_Pos              (3U)
#define TIMA0_SBCR_SBK3_EN_Msk              (1UL << TIMA0_SBCR_SBK3_EN_Pos)
#define TIMA0_SBCR_SBK3_EN                  TIMA0_SBCR_SBK3_EN_Msk
#define TIMA0_SBCR_SBK2_EN_Pos              (2U)
#define TIMA0_SBCR_SBK2_EN_Msk              (1UL << TIMA0_SBCR_SBK2_EN_Pos)
#define TIMA0_SBCR_SBK2_EN                  TIMA0_SBCR_SBK2_EN_Msk
#define TIMA0_SBCR_SBK1_EN_Pos              (1U)
#define TIMA0_SBCR_SBK1_EN_Msk              (1UL << TIMA0_SBCR_SBK1_EN_Pos)
#define TIMA0_SBCR_SBK1_EN                  TIMA0_SBCR_SBK1_EN_Msk
#define TIMA0_SBCR_SBK0_EN_Pos              (0U)
#define TIMA0_SBCR_SBK0_EN_Msk              (1UL << TIMA0_SBCR_SBK0_EN_Pos)
#define TIMA0_SBCR_SBK0_EN                  TIMA0_SBCR_SBK0_EN_Msk
/************* TIMA1 System Broken Circuit Control Register      ****************/
#define TIMA1_SBCR_EBK_EN_Pos               (16U)
#define TIMA1_SBCR_EBK_EN_Msk               (0xFFFFUL << TIMA1_SBCR_EBK_EN_Pos)
#define TIMA1_SBCR_EBK_EN                   TIMA1_SBCR_EBK_EN_Msk
#define TIMA1_SBCR_BRK_POL_Pos              (5U)
#define TIMA1_SBCR_BRK_POL_Msk              (0x1UL << TIMA1_SBCR_BRK_POL_Pos)
#define TIMA1_SBCR_BRK_POL                  TIMA1_SBCR_BRK_POL_Msk
#define TIMA1_SBCR_PBK_EN_Pos               (4U)
#define TIMA1_SBCR_PBK_EN_Msk               (1UL << TIMA1_SBCR_PBK_EN_Pos)
#define TIMA1_SBCR_PBK_EN                   TIMA1_SBCR_PBK_EN_Msk
#define TIMA1_SBCR_SBK3_EN_Pos              (3U)
#define TIMA1_SBCR_SBK3_EN_Msk              (1UL << TIMA1_SBCR_SBK3_EN_Pos)
#define TIMA1_SBCR_SBK3_EN                  TIMA1_SBCR_SBK3_EN_Msk
#define TIMA1_SBCR_SBK2_EN_Pos              (2U)
#define TIMA1_SBCR_SBK2_EN_Msk              (1UL << TIMA1_SBCR_SBK2_EN_Pos)
#define TIMA1_SBCR_SBK2_EN                  TIMA1_SBCR_SBK2_EN_Msk
#define TIMA1_SBCR_SBK1_EN_Pos              (1U)
#define TIMA1_SBCR_SBK1_EN_Msk              (1UL << TIMA1_SBCR_SBK1_EN_Pos)
#define TIMA1_SBCR_SBK1_EN                  TIMA1_SBCR_SBK1_EN_Msk
#define TIMA1_SBCR_SBK0_EN_Pos              (0U)
#define TIMA1_SBCR_SBK0_EN_Msk              (1UL << TIMA1_SBCR_SBK0_EN_Pos)
#define TIMA1_SBCR_SBK0_EN                  TIMA1_SBCR_SBK0_EN_Msk
/************* TIMA2 System Broken Circuit Control Register      ****************/
#define TIMH_SBCR_SBK2_EN_Pos               (2U)
#define TIMH_SBCR_SBK2_EN_Msk               (1UL << TIMH_SBCR_SBK2_EN_Pos)
#define TIMH_SBCR_SBK2_EN                   TIMH_SBCR_SBK2_EN_Msk
#define TIMH_SBCR_SBK1_EN_Pos               (1U)
#define TIMH_SBCR_SBK1_EN_Msk               (1UL << TIMH_SBCR_SBK1_EN_Pos)
#define TIMH_SBCR_SBK1_EN                   TIMH_SBCR_SBK1_EN_Msk
#define TIMH_SBCR_SBK0_EN_Pos               (0U)
#define TIMH_SBCR_SBK0_EN_Msk               (1UL << TIMH_SBCR_SBK0_EN_Pos)
#define TIMH_SBCR_SBK0_EN                   TIMH_SBCR_SBK0_EN_Msk
/************* TIMS0 System Broken Circuit Control Register      ****************/
#define TIMS0_SBCR_EBK_EN_Pos               (16U)
#define TIMS0_SBCR_EBK_EN_Msk               (0xFFFFUL << TIMS0_SBCR_EBK_EN_Pos)
#define TIMS0_SBCR_EBK_EN                   TIMS0_SBCR_EBK_EN_Msk
#define TIMS0_SBCR_BRK_POL_Pos              (5U)
#define TIMS0_SBCR_BRK_POL_Msk              (0x1UL << TIMS0_SBCR_BRK_POL_Pos)
#define TIMS0_SBCR_BRK_POL                  TIMS0_SBCR_BRK_POL_Msk
#define TIMS0_SBCR_PBK_EN_Pos               (4U)
#define TIMS0_SBCR_PBK_EN_Msk               (1UL << TIMS0_SBCR_PBK_EN_Pos)
#define TIMS0_SBCR_PBK_EN                   TIMS0_SBCR_PBK_EN_Msk
#define TIMS0_SBCR_SBK3_EN_Pos              (3U)
#define TIMS0_SBCR_SBK3_EN_Msk              (1UL << TIMS0_SBCR_SBK3_EN_Pos)
#define TIMS0_SBCR_SBK3_EN                  TIMS0_SBCR_SBK3_EN_Msk
#define TIMS0_SBCR_SBK2_EN_Pos              (2U)
#define TIMS0_SBCR_SBK2_EN_Msk              (1UL << TIMS0_SBCR_SBK2_EN_Pos)
#define TIMS0_SBCR_SBK2_EN                  TIMS0_SBCR_SBK2_EN_Msk
#define TIMS0_SBCR_SBK1_EN_Pos              (1U)
#define TIMS0_SBCR_SBK1_EN_Msk              (1UL << TIMS0_SBCR_SBK1_EN_Pos)
#define TIMS0_SBCR_SBK1_EN                  TIMS0_SBCR_SBK1_EN_Msk
#define TIMS0_SBCR_SBK0_EN_Pos              (0U)
#define TIMS0_SBCR_SBK0_EN_Msk              (1UL << TIMS0_SBCR_SBK0_EN_Pos)
#define TIMS0_SBCR_SBK0_EN                  TIMS0_SBCR_SBK0_EN_Msk
/************* TIMS1 System Broken Circuit Control Register      ****************/
#define TIMS1_SBCR_EBK_EN_Pos               (16U)
#define TIMS1_SBCR_EBK_EN_Msk               (0xFFFFUL << TIMS1_SBCR_EBK_EN_Pos)
#define TIMS1_SBCR_EBK_EN                   TIMS1_SBCR_EBK_EN_Msk
#define TIMS1_SBCR_BRK_POL_Pos              (5U)
#define TIMS1_SBCR_BRK_POL_Msk              (0x1UL << TIMS1_SBCR_BRK_POL_Pos)
#define TIMS1_SBCR_BRK_POL                  TIMS1_SBCR_BRK_POL_Msk
#define TIMS1_SBCR_PBK_EN_Pos               (4U)
#define TIMS1_SBCR_PBK_EN_Msk               (1UL << TIMS1_SBCR_PBK_EN_Pos)
#define TIMS1_SBCR_PBK_EN                   TIMS1_SBCR_PBK_EN_Msk
#define TIMS1_SBCR_SBK3_EN_Pos              (3U)
#define TIMS1_SBCR_SBK3_EN_Msk              (1UL << TIMS1_SBCR_SBK3_EN_Pos)
#define TIMS1_SBCR_SBK3_EN                  TIMS1_SBCR_SBK3_EN_Msk
#define TIMS1_SBCR_SBK2_EN_Pos              (2U)
#define TIMS1_SBCR_SBK2_EN_Msk              (1UL << TIMS1_SBCR_SBK2_EN_Pos)
#define TIMS1_SBCR_SBK2_EN                  TIMS1_SBCR_SBK2_EN_Msk
#define TIMS1_SBCR_SBK1_EN_Pos              (1U)
#define TIMS1_SBCR_SBK1_EN_Msk              (1UL << TIMS1_SBCR_SBK1_EN_Pos)
#define TIMS1_SBCR_SBK1_EN                  TIMS1_SBCR_SBK1_EN_Msk
#define TIMS1_SBCR_SBK0_EN_Pos              (0U)
#define TIMS1_SBCR_SBK0_EN_Msk              (1UL << TIMS1_SBCR_SBK0_EN_Pos)
#define TIMS1_SBCR_SBK0_EN                  TIMS1_SBCR_SBK0_EN_Msk
/************* TIMS2 System Broken Circuit Control Register      ****************/
#define TIMS2_SBCR_EBK_EN_Pos               (16U)
#define TIMS2_SBCR_EBK_EN_Msk               (0xFFFFUL << TIMS2_SBCR_EBK_EN_Pos)
#define TIMS2_SBCR_EBK_EN                   TIMS2_SBCR_EBK_EN_Msk
#define TIMS2_SBCR_BRK_POL_Pos              (5U)
#define TIMS2_SBCR_BRK_POL_Msk              (0x1UL << TIMS2_SBCR_BRK_POL_Pos)
#define TIMS2_SBCR_BRK_POL                  TIMS2_SBCR_BRK_POL_Msk
#define TIMS2_SBCR_PBK_EN_Pos               (4U)
#define TIMS2_SBCR_PBK_EN_Msk               (1UL << TIMS2_SBCR_PBK_EN_Pos)
#define TIMS2_SBCR_PBK_EN                   TIMS2_SBCR_PBK_EN_Msk
#define TIMS2_SBCR_SBK3_EN_Pos              (3U)
#define TIMS2_SBCR_SBK3_EN_Msk              (1UL << TIMS2_SBCR_SBK3_EN_Pos)
#define TIMS2_SBCR_SBK3_EN                  TIMS2_SBCR_SBK3_EN_Msk
#define TIMS2_SBCR_SBK2_EN_Pos              (2U)
#define TIMS2_SBCR_SBK2_EN_Msk              (1UL << TIMS2_SBCR_SBK2_EN_Pos)
#define TIMS2_SBCR_SBK2_EN                  TIMS2_SBCR_SBK2_EN_Msk
#define TIMS2_SBCR_SBK1_EN_Pos              (1U)
#define TIMS2_SBCR_SBK1_EN_Msk              (1UL << TIMS2_SBCR_SBK1_EN_Pos)
#define TIMS2_SBCR_SBK1_EN                  TIMS2_SBCR_SBK1_EN_Msk
#define TIMS2_SBCR_SBK0_EN_Pos              (0U)
#define TIMS2_SBCR_SBK0_EN_Msk              (1UL << TIMS2_SBCR_SBK0_EN_Pos)
#define TIMS2_SBCR_SBK0_EN                  TIMS2_SBCR_SBK0_EN_Msk
/************* TIMS3 System Broken Circuit Control Register      ****************/
#define TIMS3_SBCR_EBK_EN_Pos               (16U)
#define TIMS3_SBCR_EBK_EN_Msk               (0xFFFFUL << TIMS3_SBCR_EBK_EN_Pos)
#define TIMS3_SBCR_EBK_EN                   TIMS3_SBCR_EBK_EN_Msk
#define TIMS3_SBCR_BRK_POL_Pos              (5U)
#define TIMS3_SBCR_BRK_POL_Msk              (0x1UL << TIMS3_SBCR_BRK_POL_Pos)
#define TIMS3_SBCR_BRK_POL                  TIMS3_SBCR_BRK_POL_Msk
#define TIMS3_SBCR_PBK_EN_Pos               (4U)
#define TIMS3_SBCR_PBK_EN_Msk               (1UL << TIMS3_SBCR_PBK_EN_Pos)
#define TIMS3_SBCR_PBK_EN                   TIMS3_SBCR_PBK_EN_Msk
#define TIMS3_SBCR_SBK3_EN_Pos              (2U)
#define TIMS3_SBCR_SBK3_EN_Msk              (1UL << TIMS3_SBCR_SBK3_EN_Pos)
#define TIMS3_SBCR_SBK3_EN                  TIMS3_SBCR_SBK3_EN_Msk
#define TIMS3_SBCR_SBK2_EN_Pos              (2U)
#define TIMS3_SBCR_SBK2_EN_Msk              (1UL << TIMS3_SBCR_SBK2_EN_Pos)
#define TIMS3_SBCR_SBK2_EN                  TIMS3_SBCR_SBK2_EN_Msk
#define TIMS3_SBCR_SBK1_EN_Pos              (1U)
#define TIMS3_SBCR_SBK1_EN_Msk              (1UL << TIMS3_SBCR_SBK1_EN_Pos)
#define TIMS3_SBCR_SBK1_EN                  TIMS3_SBCR_SBK1_EN_Msk
#define TIMS3_SBCR_SBK0_EN_Pos              (0U)
#define TIMS3_SBCR_SBK0_EN_Msk              (1UL << TIMS3_SBCR_SBK0_EN_Pos)
#define TIMS3_SBCR_SBK0_EN                  TIMS3_SBCR_SBK0_EN_Msk
/************* TIMA  Extern Trigger Control Register		*******************/
#define TIMA1_ETCR0_EBUS_ETR_EN_Pos         (16U)
#define TIMA1_ETCR0_EBUS_ETR_EN_Msk         (0xFFFFUL << TIMA1_ETCR0_EBUS_ETR_EN_Pos)
#define TIMA1_ETCR0_EBUS_ETR_EN             TIMA1_ETCR0_EBUS_ETR_EN_Msk
#define TIMA0_ETCR0_EBUS_ETR_EN_Pos         (0U)
#define TIMA0_ETCR0_EBUS_ETR_EN_Msk         (0xFFFFUL << TIMA0_ETCR0_EBUS_ETR_EN_Pos)
#define TIMA0_ETCR0_EBUS_ETR_EN             TIMA0_ETCR0_EBUS_ETR_EN_Msk
/************* TIMA  Clear Control Register             ***********************/
#define TIMA1_OCREF_CLR_EN_Pos              (4U)
#define TIMA1_OCREF_CLR_EN_Msk              (0xFUL << TIMA1_OCREF_CLR_EN_Pos)
#define TIMA1_OCREF_CLR_EN                  TIMA1_OCREF_CLR_EN_Msk
#define TIMA0_OCREF_CLR_EN_Pos              (0U)
#define TIMA0_OCREF_CLR_EN_Msk              (0xFUL <<TIMA0_OCREF_CLR_EN_Pos)
#define TIMA0_OCREF_CLR_EN                  TIMA0_OCREF_CLR_EN_Msk
/************* TIMG  Clear Control Register             ***********************/
#define TIMG2_OCREF_CLR_EN_Pos              (8U)
#define TIMG2_OCREF_CLR_EN_Msk              (0xFUL << TIMG2_OCREF_CLR_EN_Pos)
#define TIMG2_OCREF_CLR_EN                  TIMG2_OCREF_CLR_EN_Msk
#define TIMG1_OCREF_CLR_EN_Pos              (4U)
#define TIMG1_OCREF_CLR_EN_Msk              (0xFUL << TIMG1_OCREF_CLR_EN_Pos)
#define TIMG1_OCREF_CLR_EN                  TIMG1_OCREF_CLR_EN_Msk
#define TIMG0_OCREF_CLR_EN_Pos              (0U)
#define TIMG0_OCREF_CLR_EN_Msk              (0xFUL <<TIMG0_OCREF_CLR_EN_Pos)
#define TIMG0_OCREF_CLR_EN                  TIMG0_OCREF_CLR_EN_Msk
/************* TIMS  Clear Control Register             ***********************/
#define TIMS3_OCREF_CLR_EN_Pos              (12U)
#define TIMS3_OCREF_CLR_EN_Msk              (0xFUL << TIMS3_OCREF_CLR_EN_Pos)
#define TIMS3_OCREF_CLR_EN                  TIMS3_OCREF_CLR_EN_Msk
#define TIMS2_OCREF_CLR_EN_Pos              (8U)
#define TIMS2_OCREF_CLR_EN_Msk              (0xFUL << TIMS2_OCREF_CLR_EN_Pos)
#define TIMS2_OCREF_CLR_EN                  TIMS2_OCREF_CLR_EN_Msk
#define TIMS1_OCREF_CLR_EN_Pos              (4U)
#define TIMS1_OCREF_CLR_EN_Msk              (0xFUL << TIMS1_OCREF_CLR_EN_Pos)
#define TIMS1_OCREF_CLR_EN                  TIMS1_OCREF_CLR_EN_Msk
#define TIMS0_OCREF_CLR_EN_Pos              (0U)
#define TIMS0_OCREF_CLR_EN_Msk              (0xFUL <<TIMS0_OCREF_CLR_EN_Pos)
#define TIMS0_OCREF_CLR_EN                  TIMS0_OCREF_CLR_EN_Msk


/******************************************************************************/
/*                                                                            */
/*                           DMA Controller (DMA)                             */
/*                                                                            */
/******************************************************************************/
/*******************  Bit definition for DMA_CTRL register  ********************/
#define DMA_CTRL_ENABLE_Pos                  (0U)
#define DMA_CTRL_ENABLE_Mask                 (0x1UL << DMA_CTRL_ENABLE_Pos)
#define DMA_CTRL_ENABLE                      DMA_CTRL_ENABLE_Mask

/*******************  Bit definition for DMA_ERR_STATUS register  *******************/
#define DMA_STATUS_ERROR_Pos                  (0U)
#define DMA_STATUS_ERROR_Mask                 (0x1UL << DMA_STATUS_ERROR_Pos)
#define DMA_STATUS_ERROR                      DMA_STATUS_ERROR_Mask

/*******************  Bit definition for DMA_CHx_CTRL register  ********************/
#define DMA_CHCTRL_CFGHOLDEN_Pos             (24U)
#define DMA_CHCTRL_CFGHOLDEN_Mask            (0x1UL << DMA_CHCTRL_CFGHOLDEN_Pos)
#define DMA_CHCTRL_CFGHOLDEN                 DMA_CHCTRL_CFGHOLDEN_Mask

#define DMA_CHCTRL_DATALENGTH_Pos             (8U)
#define DMA_CHCTRL_DATALENGTH_Mask            (0xFFFFUL << DMA_CHCTRL_DATALENGTH_Pos)

#define DMA_CHCTRL_DATASIZE_Pos               (6U)
#define DMA_CHCTRL_DATASIZE_Mask              (0x3UL << DMA_CHCTRL_DATASIZE_Pos)
#define DMA_CHCTRL_DATASIZE_BYTE              (0x0UL << DMA_CHCTRL_DATASIZE_Pos)
#define DMA_CHCTRL_DATASIZE_HALFWORD          (0x1UL << DMA_CHCTRL_DATASIZE_Pos)
#define DMA_CHCTRL_DATASIZE_WORD              (0x2UL << DMA_CHCTRL_DATASIZE_Pos)

#define DMA_CHCTRL_INTERVAL_Pos               (4U)
#define DMA_CHCTRL_INTERVAL_Mask              (0x3UL << DMA_CHCTRL_INTERVAL_Pos)
#define DMA_CHCTRL_INTERVAL_ONE               (0x0UL << DMA_CHCTRL_INTERVAL_Pos)
#define DMA_CHCTRL_INTERVAL_TWO               (0x1UL << DMA_CHCTRL_INTERVAL_Pos)
#define DMA_CHCTRL_INTERVAL_FOUR              (0x2UL << DMA_CHCTRL_INTERVAL_Pos)
#define DMA_CHCTRL_INTERVAL_EIGHT             (0x3UL << DMA_CHCTRL_INTERVAL_Pos)

#define DMA_CHCTRL_TRANSMITTYPE_Pos           (0U)
#define DMA_CHCTRL_TRANSMITTYPE_Mask          (0x3UL << DMA_CHCTRL_TRANSMITTYPE_Pos)
#define DMA_CHCTRL_TRANSMITTYPE_INVAL         (0x0UL << DMA_CHCTRL_TRANSMITTYPE_Pos)
#define DMA_CHCTRL_TRANSMITTYPE_NORMAL        (0x1UL << DMA_CHCTRL_TRANSMITTYPE_Pos)
#define DMA_CHCTRL_TRANSMITTYPE_AUTO          (0x2UL << DMA_CHCTRL_TRANSMITTYPE_Pos)


/******************************************************************************/
/*                                                                            */
/*                                    FLASH                                   */
/*                                                                            */
/******************************************************************************/
/*******************  Bit definition for FLASH_ST register  ********************/
#define FLASH_ST_ADDRERR_Pos                 (28U)
#define FLASH_ST_ADDRERR_Mask                (0x1UL << FLASH_ST_ADDRERR_Pos)
#define FLASH_ST_ADDRERR                     FLASH_ST_ADDRERR_Mask

#define FLASH_ST_PRVEFYERR_Pos               (24U)
#define FLASH_ST_PRVERYERR_Mask              (0x1UL << FLASH_ST_PRVEFYERR_Pos)
#define FLASH_ST_PRVERYERR                   FLASH_ST_PRVERYERR_Mask

#define FLASH_ST_ERASEERR_Pos                (20U)
#define FLASH_ST_ERASEERR_Mask               (0x1UL << FLASH_ST_ERASEERR_Pos)
#define FLASH_ST_ERASEERR                    FLASH_ST_ERASEERR_Mask

#define FLASH_ST_PROGERR_Pos                 (16U)
#define FLASH_ST_PROGERR_Mask                (0x1UL << FLASH_ST_PROGERR_Pos)
#define FLASH_ST_PROGERR                     FLASH_ST_PROGERR_Mask

#define FLASH_ST_MASSFINISH_Pos              (12U)
#define FLASH_ST_MASSFINISH_Mask             (0x1UL << FLASH_ST_MASSFINISH_Pos)
#define FLASH_ST_MASSFINISH                  FLASH_ST_MASSFINISH_Mask

#define FLASH_ST_ERASEFINISH_Pos             (8U)
#define FLASH_ST_ERASEFINISH_Mask            (0x1UL << FLASH_ST_ERASEFINISH_Pos)
#define FLASH_ST_ERASEFINISH                 FLASH_ST_ERASEFINISH_Mask

#define FLASH_ST_PROGFINISH_Pos              (4U)
#define FLASH_ST_PROGFINISH_Mask             (0x1UL << FLASH_ST_PROGFINISH_Pos)
#define FLASH_ST_PROGFINISH                  FLASH_ST_PROGFINISH_Mask

#define FLASH_ST_BUSY_Pos                    (0U)
#define FLASH_ST_BUSY_Mask                   (0x1UL << FLASH_ST_BUSY_Pos)
#define FLASH_ST_BUSY                        FLASH_ST_BUSY_Mask

/*******************  Bit definition for FLASH_SYS register  ********************/
#define FLASH_SYS_BOOTSEQ_Pos                (12U)
#define FLASH_SYS_BOOTSEQ_Mask               (0x3UL << FLASH_SYS_BOOTSEQ_Pos)

#define FLASH_SYS_SCRAMBLE_Pos               (9U)
#define FLASH_SYS_SCRAMBLE_Mask              (0x1UL << FLASH_SYS_SCRAMBLE_Pos)
#define FLASH_SYS_SCRAMBLE                   FLASH_SYS_SCRAMBLE_Mask

#define FLASH_SYS_DFLASHCTRL_Pos             (8U)
#define FLASH_SYS_DFLASHCTRL_Mask            (0x1UL << FLASH_SYS_DFLASHCTRL_Pos)

#define FLASH_SYS_FLASHSIZE_Pos              (6U)
#define FLASH_SYS_FLASHSIZE_Mask             (0x3UL << FLASH_SYS_FLASHSIZE_Pos)
#define FLASH_SYS_FLASHSIZE                   FLASH_SYS_FLASHSIZE_Mask

#define FLASH_SYS_SRAMSIZE_Pos               (4U)
#define FLASH_SYS_SRAMSIZE_Mask              (0x3UL << FLASH_SYS_SRAMSIZE_Pos)
#define FLASH_SYS_SRAMSIZE                    FLASH_SYS_SRAMSIZE_Mask

#define FLASH_SYS_OPTPROT_Pos                (3U)
#define FLASH_SYS_OPTPROT_Mask               (0x1UL << FLASH_SYS_OPTPROT_Pos)
#define FLASH_SYS_OPTPROT                    FLASH_SYS_OPTPROT_Mask

#define FLASH_SYS_BOOTPROT_Pos               (2U)
#define FLASH_SYS_BOOTPROT_Mask              (0x1UL << FLASH_SYS_BOOTPROT_Pos)
#define FLASH_SYS_BOOTPROT                   FLASH_SYS_BOOTPROT_Mask

#define FLASH_SYS_TRIMEN_Pos                 (1U)
#define FLASH_SYS_TRIMEN_Mask                (0x1UL << FLASH_SYS_TRIMEN_Pos)
#define FLASH_SYS_TRIMEN                     FLASH_SYS_TRIMEN_Mask

#define FLASH_SYS_SAFEMOD_Pos                (0U)
#define FLASH_SYS_SAFEMOD_Mask               (0x1UL << FLASH_SYS_SAFEMOD_Pos)
#define FLASH_SYS_SAFEMOD                    FLASH_SYS_SAFEMOD_Mask

/*******************  Bit definition for FLASH_CTRL register  ********************/
#define FLASH_CTRL_ERRINTEN_Pos              (12U)
#define FLASH_CTRL_ERRINTEN_Mask             (0x1UL << FLASH_CTRL_ERRINTEN_Pos)
#define FLASH_CTRL_ERRINTEN                  FLASH_CTRL_ERRINTEN_Mask

#define FLASH_CTRL_FINISHINTEN_Pos           (8U)
#define FLASH_CTRL_FINISHINTEN_Mask          (0x1UL << FLASH_CTRL_FINISHINTEN_Pos)
#define FLASH_CTRL_FINISHINTEN               FLASH_CTRL_FINISHINTEN_Mask

#define FLASH_CTRL_PROGMODE_Pos              (4U)
#define FLASH_CTRL_PROGMODE_Mask             (0x3UL << FLASH_CTRL_PROGMODE_Pos)
#define FLASH_CTRL_PGOGMODE_WORD             (0x0UL << FLASH_CTRL_PROGMODE_Pos)
#define FLASH_CTRL_PGOGMODE_HALFWORD         (0x1UL << FLASH_CTRL_PROGMODE_Pos)
#define FLASH_CTRL_PGOGMODE_BYTE             (0x2UL << FLASH_CTRL_PROGMODE_Pos)

#define FLASH_CTRL_LATENCY_Pos               (0U)
#define FLASH_CTRL_LATENCY_Mask              (0x7UL << FLASH_CTRL_LATENCY_Pos)
#define FLASH_CTRL_LATENCY_1                 (0x1UL << FLASH_CTRL_LATENCY_Pos)
#define FLASH_CTRL_LATENCY_2                 (0x2UL << FLASH_CTRL_LATENCY_Pos)
#define FLASH_CTRL_LATENCY_3                 (0x3UL << FLASH_CTRL_LATENCY_Pos)
#define FLASH_CTRL_LATENCY_4                 (0x4UL << FLASH_CTRL_LATENCY_Pos)
#define FLASH_CTRL_LATENCY_5                 (0x5UL << FLASH_CTRL_LATENCY_Pos)
#define FLASH_CTRL_LATENCY_6                 (0x6UL << FLASH_CTRL_LATENCY_Pos)
#define FLASH_CTRL_LATENCY_7                 (0x7UL << FLASH_CTRL_LATENCY_Pos)

/*******************  Bit definition for FLASH_CMD register  ********************/
#define FLASH_CMD_LOCKEN_Pos                   (12U)
#define FLASH_CMD_LOCKEN_Mask                  (0x1UL << FLASH_CMD_LOCKEN_Pos)
#define FLASH_CMD_LOCKEN                        FLASH_CMD_LOCKEN_Mask

#define FLASH_CMD_OP_Pos                       (4U)
#define FLASH_CMD_OP_Mask                      (0x3UL << FLASH_CMD_OP_Pos)
#define FLASH_CMD_OP_PROG                      (0x0UL << FLASH_CMD_OP_Pos)
#define FLASH_CMD_OP_PAGE_ERASE                (0x1UL << FLASH_CMD_OP_Pos)
#define FLASH_CMD_OP_MASS_ERASE_CODE           (0x2UL << FLASH_CMD_OP_Pos)
#define FLASH_CMD_OP_MASS_ERASE_DATA           (0x3UL << FLASH_CMD_OP_Pos)

#define FLASH_CMD_START_Pos                    (0U)
#define FLASH_CMD_START_Mask                   (0x1UL << FLASH_CMD_START_Pos)
#define FLASH_CMD_START                        FLASH_CMD_START_Mask

/******************************************************************************/
/*                                                                            */
/*                                    ICACHE                                   */
/*                                                                            */
/******************************************************************************/
/*******************  Bit definition for ICACHE_CTRL register  ********************/
#define ICACHE_CTRL_STATIS_Pos                 (6U)
#define ICACHE_CTRL_STATIS_Mask                (0x1UL << ICACHE_CTRL_STATIS_Pos)
#define ICACHE_CTRL_STATIS                     ICACHE_CTRL_STATIS_Mask

#define ICACHE_CTRL_PREFETCH_Pos               (5U)
#define ICACHE_CTRL_PREFETCH_Mask              (0x1UL << ICACHE_CTRL_PREFETCH_Pos)
#define ICACHE_CTRL_PREFETCH                   ICACHE_CTRL_PREFETCH_Mask

#define ICACHE_CTRL_ENABLE_Pos                 (0U)
#define ICACHE_CTRL_ENABLE_Mask                (0x1UL << ICACHE_CTRL_ENABLE_Pos)
#define ICACHE_CTRL_ENABLE                     ICACHE_CTRL_ENABLE_Mask

/*******************  Bit definition for ICACHE_CTRL register  ********************/
#define ICACHE_ST_CACHESTATUS_Pos              (0U)
#define ICACHE_ST_CACHESTATUS_Mask             (0x3UL << ICACHE_ST_CACHESTATUS_Pos)
#define ICACHE_ST_CACHESTATUS_CLOSE            (0x0UL << ICACHE_ST_CACHESTATUS_Pos)
#define ICACHE_ST_CACHESTATUS_ENABLING         (0x1UL << ICACHE_ST_CACHESTATUS_Pos)
#define ICACHE_ST_CACHESTATUS_ENABLED          (0x2UL << ICACHE_ST_CACHESTATUS_Pos)
#define ICACHE_ST_CACHESTATUS_CLOSING          (0x3UL << ICACHE_ST_CACHESTATUS_Pos)


/******************************************************************************/
/*                                                                            */
/*                                    DCACHE                                   */
/*                                                                            */
/******************************************************************************/
/*******************  Bit definition for DCACHE_CTRL register  ********************/
#define DCACHE_CTRL_STATIS_Pos                 (6U)
#define DCACHE_CTRL_STATIS_Mask                (0x1UL << DCACHE_CTRL_STATIS_Pos)
#define DCACHE_CTRL_STATIS                     DCACHE_CTRL_STATIS_Mask

#define DCACHE_CTRL_PREFETCH_Pos               (5U)
#define DCACHE_CTRL_PREFETCH_Mask              (0x1UL << DCACHE_CTRL_PREFETCH_Pos)
#define DCACHE_CTRL_PREFETCH                   DCACHE_CTRL_PREFETCH_Mask

#define DCACHE_CTRL_ENABLE_Pos                 (0U)
#define DCACHE_CTRL_ENABLE_Mask                (0x1UL << DCACHE_CTRL_ENABLE_Pos)
#define DCACHE_CTRL_ENABLE                     DCACHE_CTRL_ENABLE_Mask

/*******************  Bit definition forDCACHE_CTRL register  ********************/
#define DCACHE_ST_CACHESTATUS_Pos              (0U)
#define DCACHE_ST_CACHESTATUS_Mask             (0x3UL << DCACHE_ST_CACHESTATUS_Pos)
#define DCACHE_ST_CACHESTATUS_CLOSE            (0x0UL << DCACHE_ST_CACHESTATUS_Pos)
#define DCACHE_ST_CACHESTATUS_ENABLING         (0x1UL << DCACHE_ST_CACHESTATUS_Pos)
#define DCACHE_ST_CACHESTATUS_ENABLED          (0x2UL << DCACHE_ST_CACHESTATUS_Pos)
#define DCACHE_ST_CACHESTATUS_CLOSING          (0x3UL << DCACHE_ST_CACHESTATUS_Pos)


/******************************************************************************/
/*                                                                            */
/*                                   CRC                                      */
/*                                                                            */
/******************************************************************************/
/**************************** CRC CTRL Register ****************************/
#define CRC_CTRL_EN_Pos           (0U)   /* CRC Enable */
#define CRC_CTRL_EN_Msk           (0x1UL << CRC_CTRL_EN_Pos)
#define CRC_CTRL_EN               CRC_CTRL_EN_Msk

#define CRC_CTRL_ININVT_Pos       (4U)   /* CRC Input Invert */
#define CRC_CTRL_ININVT_Msk       (0x3UL << CRC_CTRL_ININVT_Pos)
#define CRC_CTRL_ININVT           CRC_CTRL_ININVT_Msk
#define CRC_CTRL_ININVT_NONE      (0x00UL << CRC_CTRL_ININVT_Pos)
#define CRC_CTRL_ININVT_BYTE      (0x01UL << CRC_CTRL_ININVT_Pos)
#define CRC_CTRL_ININVT_HALF      (0x02UL << CRC_CTRL_ININVT_Pos)
#define CRC_CTRL_ININVT_WORD      (0x03UL << CRC_CTRL_ININVT_Pos)


#define CRC_CTRL_OUTINVT_Pos      (6U)   /* CRC Output Invert */
#define CRC_CTRL_OUTINVT_Msk      (0x1UL << CRC_CTRL_OUTINVT_Pos)
#define CRC_CTRL_OUTINVT          CRC_CTRL_OUTINVT_Msk


/******************************************************************************/
/*                                                                            */
/*                                      EBUS                                  */
/*                                                                            */
/******************************************************************************/

/********** EBUS Software Trigger(CPU Trigger to Peripheral) *****************/
#define EBUS_SWT_CHN7LVL_Pos      (23U)   /* channel 7 Level Trigger */
#define EBUS_SWT_CHN7LVL_Msk      (0x1UL << EBUS_SWT_CHN7LVL_Pos)
#define EBUS_SWT_CHN7LVL          EBUS_SWT_CHN7LVL_Msk
#define EBUS_SWT_CHN6LVL_Pos      (22U)   /* channel 6 Level Trigger */
#define EBUS_SWT_CHN6LVL_Msk      (0x1UL << EBUS_SWT_CHN6LVL_Pos)
#define EBUS_SWT_CHN6LVL          EBUS_SWT_CHN6LVL_Msk
#define EBUS_SWT_CHN5LVL_Pos      (21U)   /* channel 5 Level Trigger */
#define EBUS_SWT_CHN5LVL_Msk      (0x1UL << EBUS_SWT_CHN5LVL_Pos)
#define EBUS_SWT_CHN5LVL          EBUS_SWT_CHN5LVL_Msk
#define EBUS_SWT_CHN4LVL_Pos      (20U)   /* channel 4 Level Trigger */
#define EBUS_SWT_CHN4LVL_Msk      (0x1UL << EBUS_SWT_CHN4LVL_Pos)
#define EBUS_SWT_CHN4LVL          EBUS_SWT_CHN4LVL_Msk
#define EBUS_SWT_CHN3LVL_Pos      (19U)   /* channel 3 Level Trigger */
#define EBUS_SWT_CHN3LVL_Msk      (0x1UL << EBUS_SWT_CHN3LVL_Pos)
#define EBUS_SWT_CHN3LVL          EBUS_SWT_CHN3LVL_Msk
#define EBUS_SWT_CHN2LVL_Pos      (18U)   /* channel 2 Level Trigger */
#define EBUS_SWT_CHN2LVL_Msk      (0x1UL << EBUS_SWT_CHN2LVL_Pos)
#define EBUS_SWT_CHN2LVL          EBUS_SWT_CHN2LVL_Msk
#define EBUS_SWT_CHN1LVL_Pos      (17U)   /* channel 1 Level Trigger */
#define EBUS_SWT_CHN1LVL_Msk      (0x1UL << EBUS_SWT_CHN1LVL_Pos)
#define EBUS_SWT_CHN1LVL          EBUS_SWT_CHN1LVL_Msk
#define EBUS_SWT_CHN0LVL_Pos      (16U)   /* channel 0 Level Trigger */
#define EBUS_SWT_CHN0LVL_Msk      (0x1UL << EBUS_SWT_CHN0LVL_Pos)
#define EBUS_SWT_CHN0LVL          EBUS_SWT_CHN0LVL_Msk
#define EBUS_SWT_CHN7PUL_Pos      (7U)   /* channel 7 Pulse Trigger */
#define EBUS_SWT_CHN7PUL_Msk      (0x1UL << EBUS_SWT_CHN7PUL_Pos)
#define EBUS_SWT_CHN7PUL          EBUS_SWT_CHN7PUL_Msk
#define EBUS_SWT_CHN6PUL_Pos      (6U)   /* channel 6 Pulse Trigger */
#define EBUS_SWT_CHN6PUL_Msk      (0x1UL << EBUS_SWT_CHN6PUL_Pos)
#define EBUS_SWT_CHN6PUL          EBUS_SWT_CHN6PUL_Msk
#define EBUS_SWT_CHN5PUL_Pos      (5U)   /* channel 5 Pulse Trigger */
#define EBUS_SWT_CHN5PUL_Msk      (0x1UL << EBUS_SWT_CHN5PUL_Pos)
#define EBUS_SWT_CHN5PUL          EBUS_SWT_CHN5PUL_Msk
#define EBUS_SWT_CHN4PUL_Pos      (4U)   /* channel 4 Pulse Trigger */
#define EBUS_SWT_CHN4PUL_Msk      (0x1UL << EBUS_SWT_CHN4PUL_Pos)
#define EBUS_SWT_CHN4PUL          EBUS_SWT_CHN4PUL_Msk
#define EBUS_SWT_CHN3PUL_Pos      (3U)   /* channel 3 Pulse Trigger */
#define EBUS_SWT_CHN3PUL_Msk      (0x1UL << EBUS_SWT_CHN3PUL_Pos)
#define EBUS_SWT_CHN3PUL          EBUS_SWT_CHN3PUL_Msk
#define EBUS_SWT_CHN2PUL_Pos      (2U)   /* channel 2 Pulse Trigger */
#define EBUS_SWT_CHN2PUL_Msk      (0x1UL << EBUS_SWT_CHN2PUL_Pos)
#define EBUS_SWT_CHN2PUL          EBUS_SWT_CHN2PUL_Msk
#define EBUS_SWT_CHN1PUL_Pos      (1U)   /* channel 1 Pulse Trigger */
#define EBUS_SWT_CHN1PUL_Msk      (0x1UL << EBUS_SWT_CHN1PUL_Pos)
#define EBUS_SWT_CHN1PUL          EBUS_SWT_CHN1PUL_Msk
#define EBUS_SWT_CHN0PUL_Pos      (0U)   /* channel 0 Pulse Trigger */
#define EBUS_SWT_CHN0PUL_Msk      (0x1UL << EBUS_SWT_CHN0PUL_Pos)
#define EBUS_SWT_CHN0PUL          EBUS_SWT_CHN0PUL_Msk

/********** EBUS Trigger CPU(Peripheral Trigger to CPU) *****************/
#define EBUS_CPU_CHNSEL_Pos       (4U)   /* channel Selection */
#define EBUS_CPU_CHNSEL_Msk       (0x7UL << EBUS_CPU_CHNSEL_Pos)
#define EBUS_CPU_CHNSEL           EBUS_CPU_CHNSEL_Msk
#define EBUS_CPU_EN_Pos           (0U)   /* Enable */
#define EBUS_CPU_EN_Msk           (0x7UL << EBUS_CPU_EN_Pos)
#define EBUS_CPU_EN               EBUS_CPU_EN_Msk

/*********************** EBUS channel Configuration **********************/
#define EBUS_CHN_SRCSEL_Pos       (7U)  /* EBUS channel source selection */
#define EBUS_CHN_SRCSEL_Msk       (0x1FUL << EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_SRCSEL           EBUS_CHN_SRCSEL_Msk
#define EBUS_CHN_OE_Pos           (6U)  /* EBUS output event to pad enable */
#define EBUS_CHN_OE_Msk           (0x1FUL << EBUS_CHN_OE_Pos)
#define EBUS_CHN_OE               EBUS_CHN_OE_Msk
#define EBUS_CHN_INVEN_Pos        (5U)  /* EBUS output inversion enable */
#define EBUS_CHN_INVEN_Msk        (0x1FUL << EBUS_CHN_INVEN_Pos)
#define EBUS_CHN_INVEN            EBUS_CHN_INVEN_Msk
#define EBUS_CHN_OREN_Pos         (4U)  /* EBUS output ORen with previous channel */
#define EBUS_CHN_OREN_Msk         (0x1FUL << EBUS_CHN_OREN_Pos)
#define EBUS_CHN_OREN             EBUS_CHN_OREN_Msk
#define EBUS_CHN_ANDEN_Pos        (3U) /* EBUS output ANDEN with next channel */
#define EBUS_CHN_ANDEN_Msk        (0x1FUL << EBUS_CHN_ANDEN_Pos)
#define EBUS_CHN_ANDEN            EBUS_CHN_ANDEN_Msk
#define EBUS_CHN_DETSRC_Pos       (1U) /* EBUS detection sources */
#define EBUS_CHN_DETSRC_Msk       (0x3UL << EBUS_CHN_DETSRC_Pos)
#define EBUS_CHN_DETSRC           EBUS_CHN_DETSRC_Msk
#define EBUS_CHN_ASYNC_Pos        (0U) /* EBUS asynchronization enable */
#define EBUS_CHN_ASYNC_Msk        (0x1UL << EBUS_CHN_ASYNC_Pos)
#define EBUS_CHN_ASYNC            EBUS_CHN_ASYNC_Msk



/******************************************************************************/
/*                                                                            */
/*                                     GPIO                                   */
/*                                                                            */
/******************************************************************************/

/**************************** GPIO event Register  ****************************/
#define GPIO_EVT_PIN_Pos         (28U) /*GPIO Event Pin Selection*/
#define GPIO_EVT_PIN_Msk         (0xFUL << GPIO_EVT_PIN_Pos)
#define GPIO_EVT_PIN             GPIO_EVT_PIN_Msk
#define GPIO_EVT_CHN_Pos         (16U) /*GPIO Event Channel*/
#define GPIO_EVT_CHN_Msk         (0xFUL << GPIO_EVT_CHN_Pos)
#define GPIO_EVT_CHN             GPIO_EVT_CHN_Msk
#define GPIO_EVT_CONFST_Pos      (8U) /*Event Conflict Status*/
#define GPIO_EVT_CONFST_Msk      (0x1UL << GPIO_EVT_CONFST_Pos)
#define GPIO_EVT_CONFST          GPIO_EVT_CONFST_Msk
#define GPIO_EVT_ACK_Pos         (4U) /*GPIO Output Selection Due to Event ACK*/
#define GPIO_EVT_ACK_Msk         (0x3UL << GPIO_EVT_ACK_Pos)
#define GPIO_EVT_ACK             GPIO_EVT_ACK_Msk
#define GPIO_EVT_EN_Pos          (0U) /*GPIO Event Enable*/
#define GPIO_EVT_EN_Msk          (0x1UL << GPIO_EVT_EN_Pos)
#define GPIO_EVT_EN              GPIO_EVT_EN_Msk

/******************************************************************************/
/*                                                                            */
/*                                     SPI                                    */
/*                                                                            */
/******************************************************************************/

/************************** SPI Configuration Register  ***********************/

#define SPI_CFG_FLTSEL_Pos       (12U) /* SPI Filter Selection */
#define SPI_CFG_FLTSEL_Msk       (0x7UL << SPI_CFG_FLTSEL_Pos)
#define SPI_CFG_FLTSEL           SPI_CFG_FLTSEL_Msk
#define SPI_CFG_FLTEN_Pos        (11U) /* SPI Filter Enable */
#define SPI_CFG_FLTEN_Msk        (0x1UL << SPI_CFG_FLTEN_Pos)
#define SPI_CFG_FLTEN            SPI_CFG_FLTEN_Msk
#define SPI_CFG_TRMOD_Pos        (8U) /* SPI Transmission Mode */
#define SPI_CFG_TRMOD_Msk        (0x3UL << SPI_CFG_TRMOD_Pos)
#define SPI_CFG_TRMOD            SPI_CFG_TRMOD_Msk
#define SPI_CFG_FULLDUPLEX       (0x0UL << SPI_CFG_TRMOD_Pos)
#define SPI_CFG_SINGLETXONLY     (0x2UL << SPI_CFG_TRMOD_Pos)
#define SPI_CFG_SINGLERXONLY     (0x3UL << SPI_CFG_TRMOD_Pos)
#define SPI_CFG_SCKCPOL_Pos      (6U) /* SPI SCK Polarity */
#define SPI_CFG_SCKCPOL_Msk      (0x1UL << SPI_CFG_SCKCPOL_Pos)
#define SPI_CFG_SCKCPOL          SPI_CFG_SCKCPOL_Msk
#define SPI_CFG_SCKCPOL_LOW      (0x0UL << SPI_CFG_SCKCPOL_Pos)
#define SPI_CFG_SCKCPOL_HIGH     (0x1UL << SPI_CFG_SCKCPOL_Pos)
#define SPI_CFG_SCKCPHA_Pos      (5U) /* SPI SCK phase */
#define SPI_CFG_SCKCPHA_Msk      (0x1UL << SPI_CFG_SCKCPHA_Pos)
#define SPI_CFG_SCKCPHA          SPI_CFG_SCKCPHA_Msk
#define SPI_CFG_SCKCPHA_LOW      (0x0UL << SPI_CFG_SCKCPHA_Pos)
#define SPI_CFG_SCKCPHA_HIGH     (0x1UL << SPI_CFG_SCKCPHA_Pos)
#define SPI_CFG_MLSB_Pos         (4U) /* MSB Or LSB first */
#define SPI_CFG_MLSB_Msk         (0x1UL << SPI_CFG_MLSB_Pos)
#define SPI_CFG_MLSB             SPI_CFG_MLSB_Msk
#define SPI_CFG_MLSB_LSB         (0x1UL << SPI_CFG_MLSB_Pos)
#define SPI_CFG_MLSB_MSB         (0x0UL << SPI_CFG_MLSB_Pos)
#define SPI_CFG_WKMOD_Pos        (3U) /* SPI Work Mode */
#define SPI_CFG_WKMOD_Msk        (0x1UL << SPI_CFG_WKMOD_Pos)
#define SPI_CFG_WKMOD            SPI_CFG_WKMOD_Msk
#define SPI_CFG_MASTER           (0x1UL << SPI_CFG_WKMOD_Pos)
#define SPI_CFG_SLAVE            (0x0UL << SPI_CFG_WKMOD_Pos)
#define SPI_CFG_EN_Pos           (0U) /* SPI Module Enable */
#define SPI_CFG_EN_Msk           (0x1UL << SPI_CFG_EN_Pos)
#define SPI_CFG_EN               SPI_CFG_EN_Msk

/************************** SPI Frame Format Register  ***********************/
#define SPI_FRT_CNT_Pos          (16U) /* Data Frame Count */
#define SPI_FRT_CNT_Msk          (0xFFFFUL << SPI_FRT_CNT_Pos)
#define SPI_FRT_CNT              SPI_FRT_CNT_Msk
#define SPI_FRT_MULDT_Pos        (0U) /* Data Multi-Frame Enable */
#define SPI_FRT_MULDT_Msk        (0x1L << SPI_FRT_MULDT_Pos)
#define SPI_FRT_MULDT            SPI_FRT_MULDT_Msk

/**************** SPI Frame Length (Data-Length) Register  ********************/
#define SPI_FRW_WIDTH_Pos        (0U) /* Data Frame Bit Width */
#define SPI_FRW_WIDTH_Msk        (0x1FUL << SPI_FRW_WIDTH_Pos)
#define SPI_FRW_WIDTH            SPI_FRW_WIDTH_Msk

/************************** SPI Sample Delay Register  ************************/
#define SPI_DLY_CNT_Pos          (0U) /* Sample Delay Count of Frame */
#define SPI_DLY_CNT_Msk          (0xFFFFUL << SPI_DLY_CNT_Pos)
#define SPI_DLY_CNT              SPI_DLY_CNT_Msk

/************************** CS Pin Control Register  *************************/

#define SPI_CSCTL_SS1OUT_Pos     (17U) /* SS1 Output */
#define SPI_CSCTL_SS1OUT_Msk     (0x1UL << SPI_CSCTL_SS1OUT_Pos)
#define SPI_CSCTL_SS1OUT         SPI_CSCTL_SS1OUT_Msk
#define SPI_CSCTL_SS0OUT_Pos     (16U) /* SS0 Output */
#define SPI_CSCTL_SS0OUT_Msk     (0x1UL << SPI_CSCTL_SS0OUT_Pos)
#define SPI_CSCTL_SS0OUT         SPI_CSCTL_SS0OUT_Msk
#define SPI_CSCTL_SS1SFTEN_Pos   (9U) /* SS1 Controlled by Software Enable */
#define SPI_CSCTL_SS1SFTEN_Msk   (0x1UL << SPI_CSCTL_SS1SFTEN_Pos)
#define SPI_CSCTL_SS1SFTEN       SPI_CSCTL_SS1SFTEN_Msk
#define SPI_CSCTL_SS0SFTEN_Pos   (8U) /* SS0 Controlled by Software Enable */
#define SPI_CSCTL_SS0SFTEN_Msk   (0x1UL << SPI_CSCTL_SS0SFTEN_Pos)
#define SPI_CSCTL_SS0SFTEN       SPI_CSCTL_SS0SFTEN_Msk
#define SPI_CSCTL_SS1HWEN_Pos    (1U) /* SS1 Controlled by Hardware Enable */
#define SPI_CSCTL_SS1HWEN_Msk    (0x1UL << SPI_CSCTL_SS1HWEN_Pos)
#define SPI_CSCTL_SS1HWEN        SPI_CSCTL_SS1HWEN_Msk
#define SPI_CSCTL_SS0HWEN_Pos    (0U) /* SS0 Controlled by Hardware Enable */
#define SPI_CSCTL_SS0HWEN_Msk    (0x1UL << SPI_CSCTL_SS0HWEN_Pos)
#define SPI_CSCTL_SS0HWEN        SPI_CSCTL_SS0HWEN_Msk

/************************** CS Pin Timing Register  *************************/
#define SPI_CSTIM_SSDST_Pos      (24U)
#define SPI_CSTIM_SSDST_Msk      (0xFFUL << SPI_CSTIM_SSDST_Pos)
#define SPI_CSTIM_SSDST          SPI_CSTIM_SSDST_Msk
#define SPI_CSTIM_SDLY_Pos       (16U)
#define SPI_CSTIM_SDLY_Msk       (0xFFUL << SPI_CSTIM_SDLY_Pos)
#define SPI_CSTIM_SDLY           SPI_CSTIM_SDLY_Msk
#define SPI_CSTIM_HLDT_Pos       (8U)
#define SPI_CSTIM_HLDT_Msk       (0xFFUL << SPI_CSTIM_HLDT_Pos)
#define SPI_CSTIM_HLDT           SPI_CSTIM_HLDT_Msk
#define SPI_CSTIM_STPT_Pos       (0U)
#define SPI_CSTIM_STPT_Msk       (0xFFUL << SPI_CSTIM_STPT_Pos)
#define SPI_CSTIM_STPT           SPI_CSTIM_STPT_Msk

/************************** SPI FIFO Control Register  *************************/
#define SPI_FFCTL_RXFFCLR_Pos    (24U)  /* RX FIFO Clear */
#define SPI_FFCTL_RXFFCLR_Msk    (0x1UL << SPI_FFCTL_RXFFCLR_Pos)
#define SPI_FFCTL_RXFFCLR        SPI_FFCTL_RXFFCLR_Msk
#define SPI_FFCTL_RXTH_Pos       (16U) /* RX FIFO Threshold */
#define SPI_FFCTL_RXTH_Msk       (0xFUL << SPI_FFCTL_RXTH_Pos)
#define SPI_FFCTL_RXTH           SPI_FFCTL_RXTH_Msk
#define SPI_FFCTL_TXFFCLR_Pos    (8U)  /* TX FIFO Clear */
#define SPI_FFCTL_TXFFCLR_Msk    (0x1UL << SPI_FFCTL_TXFFCLR_Pos)
#define SPI_FFCTL_TXFFCLR        SPI_FFCTL_TXFFCLR_Msk
#define SPI_FFCTL_TXTH_Pos       (0U)  /* TX FIFO Threshold */
#define SPI_FFCTL_TXTH_Msk       (0xFUL << SPI_FFCTL_TXTH_Pos)
#define SPI_FFCTL_TXTH           SPI_FFCTL_TXTH_Msk

/************************** SPI FIFO Status Register  *************************/
#define SPI_FFST_RXCNT_Pos       (16U) /* RX FIFO Count */
#define SPI_FFST_RXCNT_Msk       (0xFUL << SPI_FFST_RXCNT_Pos)
#define SPI_FFST_RXCNT           SPI_FFST_RXCNT_Msk
#define SPI_FFST_TXCNT_Pos       (0U)  /* TX FIFO Count */
#define SPI_FFST_TXCNT_Msk       (0xFUL << SPI_FFST_TXCNT_Pos)
#define SPI_FFST_TXCNT           SPI_FFST_TXCNT_Msk

/************************** SPI Status Register  *************************/
#define SPI_ST_RXFFTHRWRN_Pos    (18U) /* RX Threshold Warning */
#define SPI_ST_RXFFTHRWRN_Msk    (0x1UL << SPI_ST_RXFFTHRWRN_Pos)
#define SPI_ST_RXFFTHRWRN        SPI_ST_RXFFTHRWRN_Msk
#define SPI_ST_RXFFNEMPT_Pos     (17U)  /* RX FIFO Is Not Empty */
#define SPI_ST_RXFFNEMPT_Msk     (0x1UL << SPI_ST_RXFFNEMPT_Pos)
#define SPI_ST_RXFFNEMPT         SPI_ST_RXFFNEMPT_Msk
#define SPI_ST_RXFFFUL_Pos       (16U)  /* RX FIFO Is Full */
#define SPI_ST_RXFFFUL_Msk       (0x1UL << SPI_ST_RXFFFUL_Pos)
#define SPI_ST_RXFFFUL           SPI_ST_RXFFFUL_Msk
#define SPI_ST_TXFFTHRWRN_Pos    (10U) /* TX Threshold Warning */
#define SPI_ST_TXFFTHRWRN_Msk    (0x1UL << SPI_ST_TXFFTHRWRN_Pos)
#define SPI_ST_TXFFTHRWRN        SPI_ST_TXFFTHRWRN_Msk
#define SPI_ST_TXFFNFULL_Pos     (9U)  /* TX FIFO Is Not Full */
#define SPI_ST_TXFFNFULL_Msk     (0x1UL << SPI_ST_TXFFNFULL_Pos)
#define SPI_ST_TXFFNFULL         SPI_ST_TXFFNFULL_Msk
#define SPI_ST_TXFFEMPT_Pos      (8U)  /* TX FIFO Is Empty */
#define SPI_ST_TXFFEMPT_Msk      (0x1UL << SPI_ST_TXFFEMPT_Pos)
#define SPI_ST_TXFFEMPT          SPI_ST_TXFFEMPT_Msk
#define SPI_ST_BUSY_Pos          (0U)  /* SPI Is Busy */
#define SPI_ST_BUSY_Msk          (0x1UL << SPI_ST_BUSY_Pos)
#define SPI_ST_BUSY              SPI_ST_BUSY_Msk

/************************** SPI DMA TX Register  *************************/
#define SPI_DMATX_LVL_Pos        (16U)  /* SPI DMA TX Level */
#define SPI_DMATX_LVL_Msk        (0xFUL << SPI_DMATX_LVL_Pos)
#define SPI_DMATX_LVL            SPI_DMATX_LVL_Msk
#define SPI_DMATX_EN_Pos         (0U)  /* SPI DMA TX Enable */
#define SPI_DMATX_EN_Msk         (0x1UL << SPI_DMATX_EN_Pos)
#define SPI_DMATX_EN             SPI_DMATX_EN_Msk

/************************** SPI DMA TX Register  *************************/
#define SPI_DMARX_LVL_Pos        (16U)  /* SPI DMA RX Level */
#define SPI_DMARX_LVL_Msk        (0xFUL << SPI_DMARX_LVL_Pos)
#define SPI_DMARX_LVL            SPI_DMARX_LVL_Msk
#define SPI_DMARX_EN_Pos         (0U)  /* SPI DMA RX Enable */
#define SPI_DMARX_EN_Msk         (0x1UL << SPI_DMARX_EN_Pos)
#define SPI_DMARX_EN             SPI_DMARX_EN_Msk

/************************** SPI DMA TX Register  *************************/
#define SPI_DMARX_LVL_Pos        (16U)  /* SPI DMA RX Level */
#define SPI_DMARX_LVL_Msk        (0xFUL << SPI_DMARX_LVL_Pos)
#define SPI_DMARX_LVL            SPI_DMARX_LVL_Msk
#define SPI_DMARX_EN_Pos         (0U)  /* SPI DMA RX Enable */
#define SPI_DMARX_EN_Msk         (0x1UL << SPI_DMARX_EN_Pos)
#define SPI_DMARX_EN             SPI_DMARX_EN_Msk

/********************* SPI Interrupt Control Register  *******************/
#define SPI_INTR_SNDCMPLT_Pos     (15U)  /* SPI Counter reach 0 interrupt */
#define SPI_INTR_SNDCMPLT_Msk     (0x1UL << SPI_INTR_SNDCMPLT_Pos)
#define SPI_INTR_SNDCMPLT         SPI_INTR_SNDCMPLT_Msk
#define SPI_INTR_SLVTXUNDRUN_Pos  (14U)
#define SPI_INTR_SLVTXUNDRUN_Msk  (0x1UL << SPI_INTR_SLVTXUNDRUN_Pos)
#define SPI_INTR_SLVTXUNDRUN      SPI_INTR_SLVTXUNDRUN_Msk
#define SPI_INTR_SLVRXOVRRUN_Pos  (13U)
#define SPI_INTR_SLVRXOVRRUN_Msk  (0x1UL << SPI_INTR_SLVRXOVRRUN_Pos)
#define SPI_INTR_SLVRXOVRRUN      SPI_INTR_SLVRXOVRRUN_Msk
#define SPI_INTR_RXFFTHRWRN_Pos   (10U)  /* SPI RX FIFO Threshold Warning */
#define SPI_INTR_RXFFTHRWRN_Msk   (0x1UL << SPI_INTR_RXFFTHRWRN_Pos)
#define SPI_INTR_RXFFTHRWRN       SPI_INTR_RXFFTHRWRN_Msk
#define SPI_INTR_RXFFOVR_Pos      (9U)  /* SPI RX FIFO Overflow */
#define SPI_INTR_RXFFOVR_Msk      (0x1UL << SPI_INTR_RXFFOVR_Pos)
#define SPI_INTR_RXFFOVR          SPI_INTR_RXFFOVR_Msk
#define SPI_INTR_RXFFUDF_Pos      (8U)  /* SPI RX FIFO underflow */
#define SPI_INTR_RXFFUDF_Msk      (0x1UL << SPI_INTR_RXFFUDF_Pos)
#define SPI_INTR_RXFFUDF          SPI_INTR_RXFFUDF_Msk
#define SPI_INTR_TXFFTHRWRN_Pos   (6U)  /* SPI TX FIFO Threshold Warning */
#define SPI_INTR_TXFFTHRWRN_Msk   (0x1UL << SPI_INTR_TXFFTHRWRN_Pos)
#define SPI_INTR_TXFFTHRWRN       SPI_INTR_TXFFTHRWRN_Msk
#define SPI_INTR_TXFFOVR_Pos      (5U)  /* SPI TX FIFO Overflow */
#define SPI_INTR_TXFFOVR_Msk      (0x1UL << SPI_INTR_TXFFOVR_Pos)
#define SPI_INTR_TXFFOVR          SPI_INTR_TXFFOVR_Msk
#define SPI_INTR_EN_Pos           (0U)  /* SPI Global Interrupt Enable */
#define SPI_INTR_EN_Msk           (0x1UL << SPI_INTR_EN_Pos)
#define SPI_INTR_EN               SPI_INTR_EN_Msk

/********************* SPI Interrupt Status Register  *********************/
#define SPI_INTST_SNDCMPLT_Pos     (15U)  /* SPI Counter reach 0 interrupt */
#define SPI_INTST_SNDCMPLT_Msk     (0x1UL << SPI_INTST_SNDCMPLT_Pos)
#define SPI_INTST_SNDCMPLT         SPI_INTST_SNDCMPLT_Msk
#define SPI_INTST_SLVTXUNDRUN_Pos  (14U)
#define SPI_INTST_SLVTXUNDRUN_Msk  (0x1UL << SPI_INTST_SLVTXUNDRUN_Pos)
#define SPI_INTST_SLVTXUNDRUN      SPI_INTST_SLVTXUNDRUN_Msk
#define SPI_INTST_SLVRXOVRRUN_Pos  (13U)
#define SPI_INTST_SLVRXOVRRUN_Msk  (0x1UL << SPI_INTST_SLVRXOVRRUN_Pos)
#define SPI_INTST_SLVRXOVRRUN      SPI_INTST_SLVRXOVRRUN_Msk
#define SPI_INTST_RXFFTHRWRN_Pos   (10U)  /* SPI RX FIFO Threshold Warning */
#define SPI_INTST_RXFFTHRWRN_Msk   (0x1UL << SPI_INTST_RXFFTHRWRN_Pos)
#define SPI_INTST_RXFFTHRWRN       SPI_INTST_RXFFTHRWRN_Msk
#define SPI_INTST_RXFFOVR_Pos      (9U)  /* SPI RX FIFO Overflow */
#define SPI_INTST_RXFFOVR_Msk      (0x1UL << SPI_INTST_RXFFOVR_Pos)
#define SPI_INTST_RXFFOVR          SPI_INTST_RXFFOVR_Msk
#define SPI_INTST_RXFFUDF_Pos      (8U)  /* SPI RX FIFO underflow */
#define SPI_INTST_RXFFUDF_Msk      (0x1UL << SPI_INTST_RXFFUDF_Pos)
#define SPI_INTST_RXFFUDF          SPI_INTST_RXFFUDF_Msk
#define SPI_INTST_TXFFTHRWRN_Pos   (6U)  /* SPI TX FIFO Threshold Warning */
#define SPI_INTST_TXFFTHRWRN_Msk   (0x1UL << SPI_INTST_TXFFTHRWRN_Pos)
#define SPI_INTST_TXFFTHRWRN       SPI_INTST_TXFFTHRWRN_Msk
#define SPI_INTST_TXFFOVR_Pos      (5U)  /* SPI TX FIFO Overflow */
#define SPI_INTST_TXFFOVR_Msk      (0x1UL << SPI_INTST_TXFFOVR_Pos)
#define SPI_INTST_TXFFOVR          SPI_INTST_TXFFOVR_Msk
#define SPI_INTST_EN_Pos           (0U)  /* SPI Global Interrupt Enable */
#define SPI_INTST_EN_Msk           (0x1UL << SPI_INTST_EN_Pos)
#define SPI_INTST_EN               SPI_INTST_EN_Msk


/******************************************************************************/
/*                                                                            */
/*                                     IOW                                    */
/*                                                                            */
/******************************************************************************/

/***************************** IOW Control Register  **************************/
#define IOW_CTRL_PIN7EN_Pos        (7U)  /* IOW Pin7 Interrupt Enable */
#define IOW_CTRL_PIN7EN_Msk        (0x1UL << IOW_CTRL_PIN7EN_Pos)
#define IOW_CTRL_PIN7EN            IOW_CTRL_PIN7EN_Msk
#define IOW_CTRL_PIN6EN_Pos        (6U)  /* IOW Pin6 Interrupt Enable */
#define IOW_CTRL_PIN6EN_Msk        (0x1UL << IOW_CTRL_PIN6EN_Pos)
#define IOW_CTRL_PIN6EN            IOW_CTRL_PIN6EN_Msk
#define IOW_CTRL_PIN5EN_Pos        (5U)  /* IOW Pin5 Interrupt Enable */
#define IOW_CTRL_PIN5EN_Msk        (0x1UL << IOW_CTRL_PIN5EN_Pos)
#define IOW_CTRL_PIN5EN            IOW_CTRL_PIN5EN_Msk
#define IOW_CTRL_PIN4EN_Pos        (4U)  /* IOW Pin4 Interrupt Enable */
#define IOW_CTRL_PIN4EN_Msk        (0x1UL << IOW_CTRL_PIN4EN_Pos)
#define IOW_CTRL_PIN4EN            IOW_CTRL_PIN4EN_Msk
#define IOW_CTRL_PIN3EN_Pos        (3U)  /* IOW Pin3 Interrupt Enable */
#define IOW_CTRL_PIN3EN_Msk        (0x1UL << IOW_CTRL_PIN3EN_Pos)
#define IOW_CTRL_PIN3EN            IOW_CTRL_PIN3EN_Msk
#define IOW_CTRL_PIN2EN_Pos        (2U)  /* IOW Pin2 Interrupt Enable */
#define IOW_CTRL_PIN2EN_Msk        (0x1UL << IOW_CTRL_PIN2EN_Pos)
#define IOW_CTRL_PIN2EN            IOW_CTRL_PIN2EN_Msk
#define IOW_CTRL_PIN1EN_Pos        (1U)  /* IOW Pin1 Interrupt Enable */
#define IOW_CTRL_PIN1EN_Msk        (0x1UL << IOW_CTRL_PIN1EN_Pos)
#define IOW_CTRL_PIN1EN            IOW_CTRL_PIN1EN_Msk
#define IOW_CTRL_PIN0EN_Pos        (0U)  /* IOW Pin0 Interrupt Enable */
#define IOW_CTRL_PIN0EN_Msk        (0x1UL << IOW_CTRL_PIN0EN_Pos)
#define IOW_CTRL_PIN0EN            IOW_CTRL_PIN0EN_Msk

/********************* IOW Interrupt Selection Register  ***********************/
#define IOW_INS_PIN7SEL_Pos        (28U)  /* IOW Pin7 Interrupt Selection */
#define IOW_INS_PIN7SEL_Msk        (0x1UL << IOW_INS_PIN7SEL_Pos)
#define IOW_INS_PIN7SEL            IOW_INS_PIN7SEL_Msk
#define IOW_INS_PIN6SEL_Pos        (24U)  /* IOW Pin6 Interrupt Selection */
#define IOW_INS_PIN6SEL_Msk        (0x1UL << IOW_INS_PIN6SEL_Pos)
#define IOW_INS_PIN6SEL            IOW_INS_PIN6SEL_Msk
#define IOW_INS_PIN5SEL_Pos        (20U)  /* IOW Pin5 Interrupt Selection */
#define IOW_INS_PIN5SEL_Msk        (0x1UL << IOW_INS_PIN5SEL_Pos)
#define IOW_INS_PIN5SEL            IOW_INS_PIN5SEL_Msk
#define IOW_INS_PIN4SEL_Pos        (16U)  /* IOW Pin4 Interrupt Selection */
#define IOW_INS_PIN4SEL_Msk        (0x1UL << IOW_INS_PIN4SEL_Pos)
#define IOW_INS_PIN4SEL            IOW_INS_PIN4SEL_Msk
#define IOW_INS_PIN3SEL_Pos        (12U)  /* IOW Pin3 Interrupt Selection */
#define IOW_INS_PIN3SEL_Msk        (0x1UL << IOW_INS_PIN3SEL_Pos)
#define IOW_INS_PIN3SEL            IOW_INS_PIN3SEL_Msk
#define IOW_INS_PIN2SEL_Pos        (8U)  /* IOW Pin2 Interrupt Selection */
#define IOW_INS_PIN2SEL_Msk        (0x1UL << IOW_INS_PIN2SEL_Pos)
#define IOW_INS_PIN2SEL            IOW_INS_PIN2SEL_Msk
#define IOW_INS_PIN1SEL_Pos        (4U)  /* IOW Pin1 Interrupt Selection */
#define IOW_INS_PIN1SEL_Msk        (0x1UL << IOW_INS_PIN1SEL_Pos)
#define IOW_INS_PIN1SEL            IOW_INS_PIN1SEL_Msk
#define IOW_INS_PIN0SEL_Pos        (0U)  /* IOW Pin0 Interrupt Selection */
#define IOW_INS_PIN0SEL_Msk        (0x1UL << IOW_INS_PIN0SEL_Pos)
#define IOW_INS_PIN0SEL            IOW_INS_PIN0SEL_Msk

/********************* IOW Interrupt Edge Register  ***********************/
#define IOW_EDG_PIN7SEL_Pos        (7U)  /* IOW Pin7 Interrupt Edge */
#define IOW_EDG_PIN7SEL_Msk        (0x1UL << IOW_EDG_PIN7SEL_Pos)
#define IOW_EDG_PIN7SEL            IOW_EDG_PIN7SEL_Msk
#define IOW_EDG_PIN6SEL_Pos        (6U)  /* IOW Pin6 Interrupt Edge */
#define IOW_EDG_PIN6SEL_Msk        (0x1UL << IOW_EDG_PIN6SEL_Pos)
#define IOW_EDG_PIN6SEL            IOW_EDG_PIN6SEL_Msk
#define IOW_EDG_PIN5SEL_Pos        (5U)  /* IOW Pin5 Interrupt Edge */
#define IOW_EDG_PIN5SEL_Msk        (0x1UL << IOW_EDG_PIN5SEL_Pos)
#define IOW_EDG_PIN5SEL            IOW_EDG_PIN5SEL_Msk
#define IOW_EDG_PIN4SEL_Pos        (4U)  /* IOW Pin4 Interrupt Edge */
#define IOW_EDG_PIN4SEL_Msk        (0x1UL << IOW_EDG_PIN4SEL_Pos)
#define IOW_EDG_PIN4SEL            IOW_EDG_PIN4SEL_Msk
#define IOW_EDG_PIN3SEL_Pos        (3U)  /* IOW Pin3 Interrupt Edge */
#define IOW_EDG_PIN3SEL_Msk        (0x1UL << IOW_EDG_PIN3SEL_Pos)
#define IOW_EDG_PIN3SEL            IOW_EDG_PIN3SEL_Msk
#define IOW_EDG_PIN2SEL_Pos        (2U)  /* IOW Pin2 Interrupt Edge */
#define IOW_EDG_PIN2SEL_Msk        (0x1UL << IOW_EDG_PIN2SEL_Pos)
#define IOW_EDG_PIN2SEL            IOW_EDG_PIN2SEL_Msk
#define IOW_EDG_PIN1SEL_Pos        (1U)  /* IOW Pin1 Interrupt Edge */
#define IOW_EDG_PIN1SEL_Msk        (0x1UL << IOW_EDG_PIN1SEL_Pos)
#define IOW_EDG_PIN1SEL            IOW_EDG_PIN1SEL_Msk
#define IOW_EDG_PIN0SEL_Pos        (0U)  /* IOW Pin0 Interrupt Edge */
#define IOW_EDG_PIN0SEL_Msk        (0x1UL << IOW_EDG_PIN0SEL_Pos)
#define IOW_EDG_PIN0SEL            IOW_EDG_PIN0SEL_Msk

/********************* IOW Interrupt Enable Register  ***********************/
#define IOW_INTEN_EN_Pos           (0U)  /* IOW Interrupt Enable */
#define IOW_INTEN_EN_Msk           (0x1UL << IOW_INTEN_EN_Pos)
#define IOW_INTEN_EN               IOW_INTEN_EN_Msk

/********************* IOW Interrupt Status Register  ***********************/
#define IOW_INTSR_PIN7_Pos         (7U)  /* IOW Pin7 Interrupt Status */
#define IOW_INTSR_PIN7_Msk         (0x1UL << IOW_INTSR_PIN7_Pos)
#define IOW_INTSR_PIN7             IOW_INTSR_PIN7_Msk
#define IOW_INTSR_PIN6_Pos         (6U)  /* IOW Pin6 Interrupt Status */
#define IOW_INTSR_PIN6_Msk         (0x1UL << IOW_INTSR_PIN6_Pos)
#define IOW_INTSR_PIN6             IOW_INTSR_PIN6_Msk
#define IOW_INTSR_PIN5_Pos         (5U)  /* IOW Pin5 Interrupt Status */
#define IOW_INTSR_PIN5_Msk         (0x1UL << IOW_INTSR_PIN5_Pos)
#define IOW_INTSR_PIN5             IOW_INTSR_PIN5_Msk
#define IOW_INTSR_PIN4_Pos         (4U)  /* IOW Pin4 Interrupt Status */
#define IOW_INTSR_PIN4_Msk         (0x1UL << IOW_INTSR_PIN4_Pos)
#define IOW_INTSR_PIN4             IOW_INTSR_PIN4_Msk
#define IOW_INTSR_PIN3_Pos         (3U)  /* IOW Pin3 Interrupt Status */
#define IOW_INTSR_PIN3_Msk         (0x1UL << IOW_INTSR_PIN3_Pos)
#define IOW_INTSR_PIN3             IOW_INTSR_PIN3_Msk
#define IOW_INTSR_PIN2_Pos         (2U)  /* IOW Pin2 Interrupt Status */
#define IOW_INTSR_PIN2_Msk         (0x1UL << IOW_INTSR_PIN2_Pos)
#define IOW_INTSR_PIN2             IOW_INTSR_PIN2_Msk
#define IOW_INTSR_PIN1_Pos         (1U)  /* IOW Pin1 Interrupt Status */
#define IOW_INTSR_PIN1_Msk         (0x1UL << IOW_INTSR_PIN1_Pos)
#define IOW_INTSR_PIN1             IOW_INTSR_PIN1_Msk
#define IOW_INTSR_PIN0_Pos         (0U)  /* IOW Pin0 Interrupt Status */
#define IOW_INTSR_PIN0_Msk         (0x1UL << IOW_INTSR_PIN0_Pos)
#define IOW_INTSR_PIN0             IOW_INTSR_PIN0_Msk



/******************************************************************************/
/*                                                                            */
/*                                   IO Port                                  */
/*                                                                            */
/******************************************************************************/

/************************** EBUS IO Detection  *************************/
#define IOPORT_IOD_EDG11_Pos        (22U)  /* IO11 Detection */
#define IOPORT_IOD_EDG11_Msk        (0x3UL << IOPORT_IOD_EDG11_Pos)
#define IOPORT_IOD_EDG11            IOPORT_IOD_EDG11_Msk
#define IOPORT_IOD_EDG10_Pos        (20U)  /* IO10 Detection */
#define IOPORT_IOD_EDG10_Msk        (0x3UL << IOPORT_IOD_EDG10_Pos)
#define IOPORT_IOD_EDG10            IOPORT_IOD_EDG10_Msk
#define IOPORT_IOD_EDG09_Pos        (18U)  /* IO9 Detection */
#define IOPORT_IOD_EDG09_Msk        (0x3UL << IOPORT_IOD_EDG09_Pos)
#define IOPORT_IOD_EDG09            IOPORT_IOD_EDG09_Msk
#define IOPORT_IOD_EDG08_Pos        (16U)  /* IO8 Detection */
#define IOPORT_IOD_EDG08_Msk        (0x3UL << IOPORT_IOD_EDG08_Pos)
#define IOPORT_IOD_EDG08            IOPORT_IOD_EDG08_Msk
#define IOPORT_IOD_EDG07_Pos        (14U)  /* IO7 Detection */
#define IOPORT_IOD_EDG07_Msk        (0x3UL << IOPORT_IOD_EDG07_Pos)
#define IOPORT_IOD_EDG07            IOPORT_IOD_EDG07_Msk
#define IOPORT_IOD_EDG06_Pos        (12U)  /* IO6 Detection */
#define IOPORT_IOD_EDG06_Msk        (0x3UL << IOPORT_IOD_EDG06_Pos)
#define IOPORT_IOD_EDG06            IOPORT_IOD_EDG06_Msk
#define IOPORT_IOD_EDG05_Pos        (10U)  /* IO5 Detection */
#define IOPORT_IOD_EDG05_Msk        (0x3UL << IOPORT_IOD_EDG05_Pos)
#define IOPORT_IOD_EDG05            IOPORT_IOD_EDG05_Msk
#define IOPORT_IOD_EDG04_Pos        (8U)  /* IO4 Detection */
#define IOPORT_IOD_EDG04_Msk        (0x3UL << IOPORT_IOD_EDG04_Pos)
#define IOPORT_IOD_EDG04            IOPORT_IOD_EDG04_Msk
#define IOPORT_IOD_EDG03_Pos        (6U)  /* IO3 Detection */
#define IOPORT_IOD_EDG03_Msk        (0x3UL << IOPORT_IOD_EDG03_Pos)
#define IOPORT_IOD_EDG03            IOPORT_IOD_EDG03_Msk
#define IOPORT_IOD_EDG02_Pos        (4U)  /* IO2 Detection */
#define IOPORT_IOD_EDG02_Msk        (0x3UL << IOPORT_IOD_EDG02_Pos)
#define IOPORT_IOD_EDG02            IOPORT_IOD_EDG02_Msk
#define IOPORT_IOD_EDG01_Pos        (2U)  /* IO1 Detection */
#define IOPORT_IOD_EDG01_Msk        (0x3UL << IOPORT_IOD_EDG01_Pos)
#define IOPORT_IOD_EDG01            IOPORT_IOD_EDG01_Msk
#define IOPORT_IOD_EDG00_Pos        (0U)  /* IO0 Detection */
#define IOPORT_IOD_EDG00_Msk        (0x3UL << IOPORT_IOD_EDG00_Pos)
#define IOPORT_IOD_EDG00            IOPORT_IOD_EDG00_Msk

/**************************** EBUS Interrupt Status ***************************/
#define IOPORT_STA_EDG11_Pos        (11U)  /* IO11 Interrupt Status */
#define IOPORT_STA_EDG11_Msk        (0x1UL << IOPORT_STA_EDG11_Pos)
#define IOPORT_STA_EDG11            IOPORT_STA_EDG11_Msk
#define IOPORT_STA_EDG10_Pos        (10U)  /* IO10 Interrupt Status */
#define IOPORT_STA_EDG10_Msk        (0x1UL << IOPORT_STA_EDG10_Pos)
#define IOPORT_STA_EDG10            IOPORT_STA_EDG10_Msk
#define IOPORT_STA_EDG09_Pos        (9U)  /* IO9 Interrupt Status */
#define IOPORT_STA_EDG09_Msk        (0x1UL << IOPORT_STA_EDG09_Pos)
#define IOPORT_STA_EDG09            IOPORT_STA_EDG09_Msk
#define IOPORT_STA_EDG08_Pos        (8U)  /* IO8 Interrupt Status */
#define IOPORT_STA_EDG08_Msk        (0x1UL << IOPORT_STA_EDG08_Pos)
#define IOPORT_STA_EDG08            IOPORT_STA_EDG08_Msk
#define IOPORT_STA_EDG07_Pos        (7U)  /* IO7 Interrupt Status */
#define IOPORT_STA_EDG07_Msk        (0x1UL << IOPORT_STA_EDG07_Pos)
#define IOPORT_STA_EDG07            IOPORT_STA_EDG07_Msk
#define IOPORT_STA_EDG06_Pos        (6U)  /* IO6 Interrupt Status */
#define IOPORT_STA_EDG06_Msk        (0x1UL << IOPORT_STA_EDG06_Pos)
#define IOPORT_STA_EDG06            IOPORT_STA_EDG06_Msk
#define IOPORT_STA_EDG05_Pos        (5U)  /* IO5 Interrupt Status */
#define IOPORT_STA_EDG05_Msk        (0x1UL << IOPORT_STA_EDG05_Pos)
#define IOPORT_STA_EDG05            IOPORT_STA_EDG05_Msk
#define IOPORT_STA_EDG04_Pos        (4U)  /* IO4 Interrupt Status */
#define IOPORT_STA_EDG04_Msk        (0x1UL << IOPORT_STA_EDG04_Pos)
#define IOPORT_STA_EDG04            IOPORT_STA_EDG04_Msk
#define IOPORT_STA_EDG03_Pos        (3U)  /* IO3 Interrupt Status */
#define IOPORT_STA_EDG03_Msk        (0x1UL << IOPORT_STA_EDG03_Pos)
#define IOPORT_STA_EDG03            IOPORT_STA_EDG03_Msk
#define IOPORT_STA_EDG02_Pos        (2U)  /* IO2 Interrupt Status */
#define IOPORT_STA_EDG02_Msk        (0x1UL << IOPORT_STA_EDG02_Pos)
#define IOPORT_STA_EDG02            IOPORT_STA_EDG02_Msk
#define IOPORT_STA_EDG01_Pos        (1U)  /* IO1 Interrupt Status */
#define IOPORT_STA_EDG01_Msk        (0x1UL << IOPORT_STA_EDG01_Pos)
#define IOPORT_STA_EDG01            IOPORT_STA_EDG01_Msk
#define IOPORT_STA_EDG00_Pos        (0U)  /* IO0 Interrupt Status */
#define IOPORT_STA_EDG00_Msk        (0x1UL << IOPORT_STA_EDG00_Pos)
#define IOPORT_STA_EDG00            IOPORT_STA_EDG00_Msk


/*********************** EBUS Interrupt Status Clear **********************/
#define IOPORT_CLR_EDG11_Pos        (11U)  /* IO11 Interrupt Status Clear */
#define IOPORT_CLR_EDG11_Msk        (0x1UL << IOPORT_CLR_EDG11_Pos)
#define IOPORT_CLR_EDG11            IOPORT_CLR_EDG11_Msk
#define IOPORT_CLR_EDG10_Pos        (10U)  /* IO10 Interrupt Status Clear */
#define IOPORT_CLR_EDG10_Msk        (0x1UL << IOPORT_CLR_EDG10_Pos)
#define IOPORT_CLR_EDG10            IOPORT_CLR_EDG10_Msk
#define IOPORT_CLR_EDG09_Pos        (9U)  /* IO9 Interrupt Status Clear */
#define IOPORT_CLR_EDG09_Msk        (0x1UL << IOPORT_CLR_EDG09_Pos)
#define IOPORT_CLR_EDG09            IOPORT_CLR_EDG09_Msk
#define IOPORT_CLR_EDG08_Pos        (8U)  /* IO8 Interrupt Status Clear */
#define IOPORT_CLR_EDG08_Msk        (0x1UL << IOPORT_CLR_EDG08_Pos)
#define IOPORT_CLR_EDG08            IOPORT_CLR_EDG08_Msk
#define IOPORT_CLR_EDG07_Pos        (7U)  /* IO7 Interrupt Status Clear */
#define IOPORT_CLR_EDG07_Msk        (0x1UL << IOPORT_CLR_EDG07_Pos)
#define IOPORT_CLR_EDG07            IOPORT_CLR_EDG07_Msk
#define IOPORT_CLR_EDG06_Pos        (6U)  /* IO6 Interrupt Status Clear */
#define IOPORT_CLR_EDG06_Msk        (0x1UL << IOPORT_CLR_EDG06_Pos)
#define IOPORT_CLR_EDG06            IOPORT_CLR_EDG06_Msk
#define IOPORT_CLR_EDG05_Pos        (5U)  /* IO5 Interrupt Status Clear */
#define IOPORT_CLR_EDG05_Msk        (0x1UL << IOPORT_CLR_EDG05_Pos)
#define IOPORT_CLR_EDG05            IOPORT_CLR_EDG05_Msk
#define IOPORT_CLR_EDG04_Pos        (4U)  /* IO4 Interrupt Status Clear */
#define IOPORT_CLR_EDG04_Msk        (0x1UL << IOPORT_CLR_EDG04_Pos)
#define IOPORT_CLR_EDG04            IOPORT_CLR_EDG04_Msk
#define IOPORT_CLR_EDG03_Pos        (3U)  /* IO3 Interrupt Status Clear */
#define IOPORT_CLR_EDG03_Msk        (0x1UL << IOPORT_CLR_EDG03_Pos)
#define IOPORT_CLR_EDG03            IOPORT_CLR_EDG03_Msk
#define IOPORT_CLR_EDG02_Pos        (2U)  /* IO2 Interrupt Status Clear */
#define IOPORT_CLR_EDG02_Msk        (0x1UL << IOPORT_CLR_EDG02_Pos)
#define IOPORT_CLR_EDG02            IOPORT_CLR_EDG02_Msk
#define IOPORT_CLR_EDG01_Pos        (1U)  /* IO1 Interrupt Status Clear */
#define IOPORT_CLR_EDG01_Msk        (0x1UL << IOPORT_CLR_EDG01_Pos)
#define IOPORT_CLR_EDG01            IOPORT_CLR_EDG01_Msk
#define IOPORT_CLR_EDG00_Pos        (0U)  /* IO0 Interrupt Status Clear */
#define IOPORT_CLR_EDG00_Msk        (0x1UL << IOPORT_CLR_EDG00_Pos)
#define IOPORT_CLR_EDG00            IOPORT_CLR_EDG00_Msk


/******************************************************************************/
/*                                                                            */
/*                            Window WATCHDOG                                 */
/*                                                                            */
/******************************************************************************/
/*******************  Bit definition for WWDG_CTRL register  ********************/
#define WWDG_CTRL_ENABLE_Pos               (31U)
#define WWDG_CTRL_ENABLE_Mask              (0x1UL << WWDG_CTRL_ENABLE_Pos)
#define WWDG_CTRL_ENABLE                   WWDG_CTRL_ENABLE_Mask

#define WWDG_CTRL_WINLEN_Pos               (3U)
#define WWDG_CTRL_WINLEN_Mask              (0x7UL << WWDG_CTRL_WINLEN_Pos)
#define WWDG_CTRL_WINLEN_80H               (0x0UL << WWDG_CTRL_WINLEN_Pos)
#define WWDG_CTRL_WINLEN_100H              (0x1UL << WWDG_CTRL_WINLEN_Pos)
#define WWDG_CTRL_WINLEN_200H              (0x2UL << WWDG_CTRL_WINLEN_Pos)
#define WWDG_CTRL_WINLEN_400H              (0x3UL << WWDG_CTRL_WINLEN_Pos)
#define WWDG_CTRL_WINLEN_800H              (0x4UL << WWDG_CTRL_WINLEN_Pos)
#define WWDG_CTRL_WINLEN_1000H             (0x5UL << WWDG_CTRL_WINLEN_Pos)
#define WWDG_CTRL_WINLEN_2000H             (0x6UL << WWDG_CTRL_WINLEN_Pos)
#define WWDG_CTRL_WINLEN_4000H             (0x7UL << WWDG_CTRL_WINLEN_Pos)

#define WWDG_CTRL_WMCLOSEFLAG_Pos          (2U)
#define WWDG_CTRL_WMCLOSEFLAG_Mask         (0x1UL << WWDG_CTRL_WMCLOSEFLAG_Pos)
#define WWDG_CTRL_WMCLOSEFLAG              WWDG_CTRL_WMCLOSEFLAG_Mask

#define WWDG_CTRL_WMEN_Pos                 (1U)
#define WWDG_CTRL_WMEN_Mask                (0x1UL << WWDG_CTRL_WMEN_Pos)
#define WWDG_CTRL_WINMODE                  WWDG_CTRL_WMEN_Mask
#define WWDG_CTRL_NORMALMODE               (0x0UL << WWDG_CTRL_WMEN_Pos)

#define WWDG_CTRL_EWIEN_Pos                (0U)
#define WWDG_CTRL_EWIEN_Mask               (0x1UL << WWDG_CTRL_EWIEN_Pos)
#define WWDG_CTRL_EWIEN                    WWDG_CTRL_EWIEN_Mask

/*******************  Bit definition for WWDG_ST register  *******************/
#define WWDG_ST_ENABLE_Pos                 (31U)
#define WWDG_ST_ENABLE_Mask                (0x1UL << WWDG_ST_ENABLE_Pos)
#define WWDG_ST_ENABLE                      WWDG_ST_ENABLE_Mask

#define WWDG_ST_REFRESH_Pos                (16U)
#define WWDG_ST_REFRESH_Mask               (0x1UL << WWDG_ST_REFRESH_Pos)
#define WWDG_ST_REFRESH                     WWDG_ST_REFRESH_Mask

#define WWDG_ST_EWI_Pos                    (0U)
#define WWDG_ST_EWI_Mask                   (0x1UL << WWDG_ST_EWI_Pos)
#define WWDG_ST_EWI                        WWDG_ST_EWI_Mask


/******************************************************************************/
/*                                                                            */
/*                            CAN                                */
/*                                                                            */
/******************************************************************************/
/*******************  Bit definition for CAN_CTRL register  ********************/
#define CAN_CTRL_CLRDATAOV_Pos               (17U)
#define CAN_CTRL_CLRDATAOV_Mask              (0x1UL << CAN_CTRL_CLRDATAOV_Pos)
#define CAN_CTRL_CLRDATAOV                   CAN_CTRL_CLRDATAOV_Mask

#define CAN_CTRL_RLSERXBUF_Pos               (16U)
#define CAN_CTRL_RLSERXBUF_Mask              (0x1UL << CAN_CTRL_RLSERXBUF_Pos)
#define CAN_CTRL_RLSERXBUF                   CAN_CTRL_RLSERXBUF_Mask

#define CAN_CTRL_SELFRXREQ_Pos               (8U)
#define CAN_CTRL_SELFRXREQ_Mask              (0x1UL << CAN_CTRL_SELFRXREQ_Pos)
#define CAN_CTRL_SELFRXREQ                   CAN_CTRL_SELFRXREQ_Mask

#define CAN_CTRL_ABORTTX_Pos                 (1U)
#define CAN_CTRL_ABORTTX_Mask                (0x1UL << CAN_CTRL_ABORTTX_Pos)
#define CAN_CTRL_ABORTTX                     CAN_CTRL_ABORTTX_Mask

#define CAN_CTRL_TXREQ_Pos                   (0U)
#define CAN_CTRL_TXREQ_Mask                  (0x1UL << CAN_CTRL_TXREQ_Pos)
#define CAN_CTRL_TXREQ                       CAN_CTRL_TXREQ_Mask

/*******************  Bit definition for CAN_CFG register  *******************/
#define CAN_CFG_SLEEPMODE_Pos                (8U)
#define CAN_CFG_SLEEPMODE_Mask               (0x1UL << CAN_CFG_SLEEPMODE_Pos)
#define CAN_CFG_SLEEPMODE                    CAN_CFG_SLEEPMODE_Mask

#define CAN_CFG_WORKMODE_Pos                 (0U)
#define CAN_CFG_WORKMODE_Mask                (0x3UL << CAN_CFG_WORKMODE_Pos)
#define CAN_CFG_WORKMODE                     CAN_CFG_WORKMODE_Mask
#define CAN_CFG_WORKMODE_INIT                (0x0UL << CAN_CFG_WORKMODE_Pos)
#define CAN_CFG_WORKMODE_NORMAL              (0x1UL << CAN_CFG_WORKMODE_Pos)
#define CAN_CFG_WORKMODE_LISTEN              (0x2UL << CAN_CFG_WORKMODE_Pos)
#define CAN_CFG_WORKMODE_SELFTEST            (0x3UL << CAN_CFG_WORKMODE_Pos)

/*******************  Bit definition for CAN_IE register  *******************/
#define CAN_IE_WAKEUP_Pos                    (24U)
#define CAN_IE_WAKEUP_Mask                   (0x1UL << CAN_IE_WAKEUP_Pos)
#define CAN_IE_WAKEUP                        CAN_IE_WAKEUP_Mask

#define CAN_IE_BUSERR_Pos                    (19U)
#define CAN_IE_BUSERR_Mask                   (0x1UL << CAN_IE_BUSERR_Pos)
#define CAN_IE_BUSERR                        CAN_IE_BUSERR_Mask

#define CAN_IE_ERRPASSIVE_Pos                (18U)
#define CAN_IE_ERRPASSIVE_Mask               (0x1UL << CAN_IE_ERRPASSIVE_Pos)
#define CAN_IE_ERRPASSIVE                    CAN_IE_ERRPASSIVE_Mask

#define CAN_IE_ERRWARNING_Pos                (17U)
#define CAN_IE_ERRWARNING_Mask               (0x1UL << CAN_IE_ERRWARNING_Pos)
#define CAN_IE_ERRWARNING                    CAN_IE_ERRWARNING_Mask

#define CAN_IE_ARBITLOST_Pos                 (16U)
#define CAN_IE_ARBITLOST_Mask                (0x1UL << CAN_IE_ARBITLOST_Pos)
#define CAN_IE_ARBITLOST                     CAN_IE_ARBITLOST_Mask

#define CAN_IE_DATAOVERRUN_Pos               (10U)
#define CAN_IE_DATAOVERRUN_Mask              (0x1UL << CAN_IE_DATAOVERRUN_Pos)
#define CAN_IE_DATAOVERRUN                   CAN_IE_DATAOVERRUN_Mask

#define CAN_IE_TRANSMIT_Pos                  (9U)
#define CAN_IE_TRANSMIT_Mask                 (0x1UL << CAN_IE_TRANSMIT_Pos)
#define CAN_IE_TRANSMIT                      CAN_IE_TRANSMIT_Mask

#define CAN_IE_RECEIVE_Pos                   (8U)
#define CAN_IE_RECEIVE_Mask                  (0x1UL << CAN_IE_RECEIVE_Pos)
#define CAN_IE_RECEIVE                       CAN_IE_RECEIVE_Mask

#define CAN_IE_INTENABLE_Pos                 (0U)
#define CAN_IE_INTENABLE_Mask                (0x1UL << CAN_IE_INTENABLE_Pos)
#define CAN_IE_INTENABLE                     CAN_IE_INTENABLE_Mask

/*******************  Bit definition for CAN_BUSTIM register  *******************/
#define CAN_BUSTIM_SAMPLING_Pos              (26U)
#define CAN_BUSTIM_SAMPLING_Mask             (0x1UL << CAN_BUSTIM_SAMPLING_Pos)
#define CAN_BUSTIM_SAMPLING_ONE              (0x0UL << CAN_BUSTIM_SAMPLING_Pos)
#define CAN_BUSTIM_SAMPLING_THREE            CAN_BUSTIM_SAMPLING_Mask

#define CAN_BUSTIM_SJW_Pos                   (24U)
#define CAN_BUSTIM_SJW_Mask                  (0x3UL << CAN_BUSTIM_SJW_Pos)

#define CAN_BUSTIM_TSEG2_Pos                 (20U)
#define CAN_BUSTIM_TSEG2_Mask                (0x7UL << CAN_BUSTIM_TSEG2_Pos)

#define CAN_BUSTIM_TSEG1_Pos                 (16U)
#define CAN_BUSTIM_TSEG1_Mask                (0xFUL << CAN_BUSTIM_TSEG1_Pos)

#define CAN_BUSTIM_BRP_Pos                   (0U)
#define CAN_BUSTIM_BRP_Mask                  (0xFFFUL << CAN_BUSTIM_BRP_Pos)

/*******************  Bit definition for CAN_OUTCTRL register  *******************/
#define CAN_OUTCTRL_OCMODE_Pos               (0U)
#define CAN_OUTCTRL_OCMODE_Mask              (0x3UL << CAN_OUTCTRL_OCMODE_Pos)
#define CAN_OUTCTRL_OCMODE_NORMAL            (0x2UL << CAN_OUTCTRL_OCMODE_Pos)
#define CAN_OUTCTRL_OCMODE_CLOCK             (0x3UL << CAN_OUTCTRL_OCMODE_Pos)

/*******************  Bit definition for CAN_ST register  *******************/
#define CAN_ST_BUSOFF_Pos                    (17U)
#define CAN_ST_BUSOFF_Mask                   (0x1UL << CAN_ST_BUSOFF_Pos)
#define CAN_ST_BUSOFF                        CAN_ST_BUSOFF_Mask

#define CAN_ST_ERRCNT_Pos                    (16U)
#define CAN_ST_ERRCNT_Mask                   (0x1UL << CAN_ST_ERRCNT_Pos)
#define CAN_ST_ERRCNT                        CAN_ST_ERRCNT_Mask

#define CAN_ST_RXDATAOV_Pos                  (10U)
#define CAN_ST_RXDATAOV_Mask                 (0x1UL << CAN_ST_RXDATAOV_Pos)
#define CAN_ST_RXDATAOV                      CAN_ST_RXDATAOV_Mask

#define CAN_ST_RXFIFONOTEMPTY_Pos            (9U)
#define CAN_ST_RXFIFONOTEMPTY_Mask           (0x1UL << CAN_ST_RXFIFONOTEMPTY_Pos)
#define CAN_ST_RXFIFONOTEMPTY                CAN_ST_RXFIFONOTEMPTY_Mask

#define CAN_ST_RXBUSY_Pos                    (8U)
#define CAN_ST_RXBUSY_Mask                   (0x1UL << CAN_ST_RXBUSY_Pos)
#define CAN_ST_RXBUSY                        CAN_ST_RXBUSY_Mask

#define CAN_ST_TXBUFLOCK_Pos                 (2U)
#define CAN_ST_TXBUFLOCK_Mask                (0x1UL << CAN_ST_TXBUFLOCK_Pos)
#define CAN_ST_TXBUFLOCK                     CAN_ST_TXBUFLOCK_Mask

#define CAN_ST_TXDONE_Pos                    (1U)
#define CAN_ST_TXDONE_Mask                   (0x1UL << CAN_ST_TXDONE_Pos)
#define CAN_ST_TXDONE                        CAN_ST_TXDONE_Mask

#define CAN_ST_TXBUSY_Pos                    (0U)
#define CAN_ST_TXBUSY_Mask                   (0x1UL << CAN_ST_TXBUSY_Pos)
#define CAN_ST_TXBUSY                        CAN_ST_TXBUSY_Mask

/*******************  Bit definition for CAN_INTST register  *******************/
#define CAN_INTST_WAKEUP_Pos                 (24U)
#define CAN_INTST_WAKEUP_Mask                (0x1UL << CAN_INTST_WAKEUP_Pos)
#define CAN_INTST_WAKEUP                     CAN_INTST_WAKEUP_Mask

#define CAN_INTST_BUSERR_Pos                 (19U)
#define CAN_INTST_BUSERR_Mask                (0x1UL << CAN_INTST_BUSERR_Pos)
#define CAN_INTST_BUSERR                     CAN_INTST_BUSERR_Mask

#define CAN_INTST_ERRPASSIVE_Pos             (18U)
#define CAN_INTST_ERRPASSIVE_Mask            (0x1UL << CAN_INTST_ERRPASSIVE_Pos)
#define CAN_INTST_ERRPASSIVE                 CAN_INTST_ERRPASSIVE_Mask

#define CAN_INTST_ERRWARNING_Pos             (17U)
#define CAN_INTST_ERRWARNING_Mask            (0x1UL << CAN_INTST_ERRWARNING_Pos)
#define CAN_INTST_ERRWARNING                 CAN_INTST_ERRWARNING_Mask

#define CAN_INTST_ARBITLOST_Pos              (16U)
#define CAN_INTST_ARBITLOST_Mask             (0x1UL << CAN_INTST_ARBITLOST_Pos)
#define CAN_INTST_ARBITLOST                  CAN_INTST_ARBITLOST_Mask

#define CAN_INTST_DATAOVERFLOW_Pos           (10U)
#define CAN_INTST_DATAOVERFLOW_Mask          (0x1UL << CAN_INTST_DATAOVERFLOW_Pos)
#define CAN_INTST_DATAOVERFLOW               CAN_INTST_DATAOVERFLOW_Mask

#define CAN_INTST_TRANSMIT_Pos               (9U)
#define CAN_INTST_TRANSMIT_Mask              (0x1UL << CAN_INTST_TRANSMIT_Pos)
#define CAN_INTST_TRANSMIT                   CAN_INTST_TRANSMIT_Mask

#define CAN_INTST_RECEIVE_Pos                (8U)
#define CAN_INTST_RECEIVE_Mask               (0x1UL << CAN_INTST_RECEIVE_Pos)
#define CAN_INTST_RECEIVE                    CAN_INTST_RECEIVE_Mask

#define CAN_INTST_INT_Pos                    (0U)
#define CAN_INTST_INT_Mask                   (0x1UL << CAN_INTST_INT_Pos)
#define CAN_INTST_INT                        CAN_INTST_INT_Mask

/*******************  Bit definition for CAN_RXBUFST register  *******************/
#define CAN_RXBUFST_STARTADDR_Pos            (16U)
#define CAN_RXBUFST_STARTADDR_Mask           (0x3FUL << CAN_RXBUFST_STARTADDR_Pos)
#define CAN_RXBUFST_STARTADDR                CAN_RXBUFST_STARTADDR_Mask

#define CAN_RXBUFST_MSGCNT_Pos               (0U)
#define CAN_RXBUFST_MSGCNT_Mask              (0x1FUL << CAN_RXBUFST_MSGCNT_Pos)
#define CAN_RXBUFST_MSGCNT                   CAN_RXBUFST_MSGCNT_Mask

/*******************  Bit definition for CAN_TXFRMINFO register  *******************/
#define CAN_TXFRMINFO_FF_Pos                 (7U)
#define CAN_TXFRMINFO_FF_Mask                (0x1UL << CAN_TXFRMINFO_FF_Pos)
#define CAN_TXFRMINFO_SFF                    (0x0UL << CAN_TXFRMINFO_FF_Pos)
#define CAN_TXFRMINFO_EFF                    (0x1UL << CAN_TXFRMINFO_FF_Pos)

#define CAN_TXFRMINFO_RTR_Pos                (6U)
#define CAN_TXFRMINFO_RTR_Mask               (0x1UL << CAN_TXFRMINFO_RTR_Pos)
#define CAN_TXFRMINFO_RTR                    CAN_TXFRMINFO_RTR_Mask

#define CAN_TXFRMINFO_DLC_Pos                (0U)
#define CAN_TXFRMINFO_DLC_Mask               (0xFUL << CAN_TXFRMINFO_DLC_Pos)
#define CAN_TXFRMINFO_DLC                    CAN_TXFRMINFO_DLC_Mask

/*******************  Bit definition for CAN_TXID register  *******************/
#define CAN_TXID_SFF_Pos                     (18U)
#define CAN_TXID_SFF_Mask                    (0x7FFUL << CAN_TXID_SFF_Pos)

#define CAN_TXID_EFF_Pos                     (0U)
#define CAN_TXID_EFF_Mask                    (0x1FFFFFFFUL << CAN_TXID_EFF_Pos)

/*******************  Bit definition for CAN_RXFRMINFO register  *******************/
#define CAN_RXFRMINFO_FF_Pos                 (7U)
#define CAN_RXFRMINFO_FF_Mask                (0x1UL << CAN_RXFRMINFO_FF_Pos)
#define CAN_RXFRMINFO_SFF                    (0x0UL << CAN_RXFRMINFO_FF_Pos)
#define CAN_RXFRMINFO_EFF                    (0x1UL << CAN_RXFRMINFO_FF_Pos)

#define CAN_RXFRMINFO_RTR_Pos                (6U)
#define CAN_RXFRMINFO_RTR_Mask               (0x1UL << CAN_RXFRMINFO_RTR_Pos)
#define CAN_RXFRMINFO_RTR                    CAN_RXFRMINFO_RTR_Mask

#define CAN_RXFRMINFO_DLC_Pos                (0U)
#define CAN_RXFRMINFO_DLC_Mask               (0xFUL << CAN_RXFRMINFO_DLC_Pos)
#define CAN_RXFRMINFO_DLC                    CAN_RXFRMINFO_DLC_Mask

/*******************  Bit definition for CAN_RXID register  *******************/
#define CAN_RXID_SFF_Pos                     (18U)
#define CAN_RXID_SFF_Mask                    (0x7FFUL << CAN_RXID_SFF_Pos)

#define CAN_RXID_EFF_Pos                     (0U)
#define CAN_RXID_EFF_Mask                    (0x1FFFFFFFUL << CAN_RXID_EFF_Pos)

/*******************  Bit definition for CAN_ERRINFO register  *******************/
#define CAN_ERRINFO_ERRCODE_Pos              (6U)
#define CAN_ERRINFO_ERRCODE_Mask             (0x3UL << CAN_ERRINFO_ERRCODE_Pos)
#define CAN_ERRINFO_ERRCODE_BIT              (0x0UL << CAN_ERRINFO_ERRCODE_Pos)
#define CAN_ERRINFO_ERRCODE_FORM             (0x1UL << CAN_ERRINFO_ERRCODE_Pos)
#define CAN_ERRINFO_ERRCODE_STUFF            (0x2UL << CAN_ERRINFO_ERRCODE_Pos)
#define CAN_ERRINFO_ERRCODE_OTHER            (0x3UL << CAN_ERRINFO_ERRCODE_Pos)

#define CAN_ERRINFO_ERRDIR_Pos               (5U)
#define CAN_ERRINFO_ERRDIR_Mask              (0x1UL << CAN_ERRINFO_ERRDIR_Pos)
#define CAN_ERRINFO_ERRDIR_TX                (0x0UL << CAN_ERRINFO_ERRDIR_Pos)
#define CAN_ERRINFO_ERRDIR_RX                (0x1UL << CAN_ERRINFO_ERRDIR_Pos)

#define CAN_ERRINFO_SEGCODE_Pos              (0U)
#define CAN_ERRINFO_SEGCODE_Mask             (0x1FUL << CAN_ERRINFO_SEGCODE_Pos)

/*******************  Bit definition for CAN_ERRCNT register  *******************/
#define CAN_ERRCNT_REC_Pos                   (16U)
#define CAN_ERRCNT_REC_Mask                  (0xFFUL << CAN_ERRCNT_REC_Pos)
#define CAN_ERRCNT_REC_BIT                   CAN_ERRCNT_REC_Mask

#define CAN_ERRCNT_TEC_Pos                   (0U)
#define CAN_ERRCNT_TEC_Mask                  (0xFFUL << CAN_ERRCNT_TEC_Pos)
#define CAN_ERRCNT_TEC_BIT                   CAN_ERRCNT_TEC_Mask


/******************************************************************************/
/*                                                                            */
/*                                   ACMP                                      */
/*                                                                            */
/******************************************************************************/
/**************************** ACMP CTRL Register ****************************/
#define ACMP_CMPx_CTRL_EBUS_SEL_Pos         (16U)
#define ACMP_CMPx_CTRL_EBUS_SEL_Mask        (0xFUL << ACMP_CMPx_CTRL_EBUS_SEL_Pos)
#define ACMP_CMPx_CTRL_EBUS_SEL             ACMP_CMPx_CTRL_EBUS_SEL_Mask
#define ACMP_CMPx_CTRL_EOUT_SEL_Pos         (14U)
#define ACMP_CMPx_CTRL_EOUT_SEL_Mask        (0x3UL << ACMP_CMPx_CTRL_EOUT_SEL_Pos)
#define ACMP_CMPx_CTRL_EOUT_SEL             ACMP_CMPx_CTRL_EOUT_SEL_Mask
#define ACMP_CMPx_CTRL_POUT_SEL_Pos         (12U)
#define ACMP_CMPx_CTRL_POUT_SEL_Mask        (0x3UL << ACMP_CMPx_CTRL_POUT_SEL_Pos)
#define ACMP_CMPx_CTRL_POUT_SEL             ACMP_CMPx_CTRL_POUT_SEL_Mask
#define ACMP_CMPx_CTRL_TOUT_SEL_Pos         (10U)
#define ACMP_CMPx_CTRL_TOUT_SEL_Mask        (0x3UL << ACMP_CMPx_CTRL_TOUT_SEL_Pos)
#define ACMP_CMPx_CTRL_TOUT_SEL             ACMP_CMPx_CTRL_TOUT_SEL_Mask
#define ACMP_CMPx_CTRL_HYSEL_Pos         	(8U)
#define ACMP_CMPx_CTRL_HYSEL_Mask        	(0x3UL << ACMP_CMPx_CTRL_HYSEL_Pos)
#define ACMP_CMPx_CTRL_HYSEL             	ACMP_CMPx_CTRL_HYSEL_Mask
#define ACMP_CMPx_CTRL_FLT_EN_Pos         	(7U)
#define ACMP_CMPx_CTRL_FLT_EN_Mask        	(0x1UL << ACMP_CMPx_CTRL_FLT_EN_Pos)
#define ACMP_CMPx_CTRL_FLT_EN             	ACMP_CMPx_CTRL_FLT_EN_Mask
#define ACMP_CMPx_CTRL_EDG_DET_EN_Pos       (6U)
#define ACMP_CMPx_CTRL_EDG_DET_EN_Mask      (0x1UL << ACMP_CMPx_CTRL_EDG_DET_EN_Pos)
#define ACMP_CMPx_CTRL_EDG_DET_EN           ACMP_CMPx_CTRL_EDG_DET_EN_Mask
#define ACMP_CMPx_CTRL_EDG_POL_EN_Pos       (5U)
#define ACMP_CMPx_CTRL_EDG_POL_EN_Mask      (0x1UL << ACMP_CMPx_CTRL_EDG_POL_EN_Pos)
#define ACMP_CMPx_CTRL_EDG_POL_EN           ACMP_CMPx_CTRL_EDG_POL_EN_Mask
#define ACMP_CMPx_CTRL_CMPx_IE_Pos       	(4U)
#define ACMP_CMPx_CTRL_CMPx_IE_Mask      	(0x1UL << ACMP_CMPx_CTRL_CMPx_IE_Pos)
#define ACMP_CMPx_CTRL_CMPx_IE           	ACMP_CMPx_CTRL_CMPx_IE_Mask
#define ACMP_CMPx_CTRL_BLK_POL_Pos       	(3U)
#define ACMP_CMPx_CTRL_BLK_POL_Mask      	(0x1UL << ACMP_CMPx_CTRL_BLK_POL_Pos)
#define ACMP_CMPx_CTRL_BLK_POL           	ACMP_CMPx_CTRL_BLK_POL_Mask
#define ACMP_CMPx_CTRL_BLK_EN_Pos       	(2U)
#define ACMP_CMPx_CTRL_BLK_EN_Mask      	(0x1UL << ACMP_CMPx_CTRL_BLK_EN_Pos)
#define ACMP_CMPx_CTRL_BLK_EN           	ACMP_CMPx_CTRL_BLK_EN_Mask
#define ACMP_CMPx_CTRL_OUT_POL_Pos       	(1U)
#define ACMP_CMPx_CTRL_OUT_POL_Mask      	(0x1UL << ACMP_CMPx_CTRL_OUT_POL_Pos)
#define ACMP_CMPx_CTRL_OUT_POL           	ACMP_CMPx_CTRL_OUT_POL_Mask
#define ACMP_CMPx_CTRL_EN_Pos       		(0U)
#define ACMP_CMPx_CTRL_EN_Mask      		(0x1UL << ACMP_CMPx_CTRL_EN_Pos)
#define ACMP_CMPx_CTRL_EN           		ACMP_CMPx_CTRL_EN_Mask

#define ACMP_CMPx_PN_PSEL_Pos       		(16U)
#define ACMP_CMPx_PN_PSEL_Mask      		(0x3UL << ACMP_CMPx_PN_PSEL_Pos)
#define ACMP_CMPx_PN_PSEL           		ACMP_CMPx_PN_PSEL_Mask
#define ACMP_CMPx_PN_NSEL_Pos       		(0U)
#define ACMP_CMPx_PN_NSEL_Mask      		(0x3UL << ACMP_CMPx_PN_NSEL_Pos)
#define ACMP_CMPx_PN_NSEL           		ACMP_CMPx_PN_NSEL_Mask

#define ACMP_CMPx_FLT_CTRL_CLR_Pos       	(31U)
#define ACMP_CMPx_FLT_CTRL_CLR_Mask      	(0x1UL << ACMP_CMPx_FLT_CTRL_CLR_Pos)
#define ACMP_CMPx_FLT_CTRL_CLR           	ACMP_CMPx_FLT_CTRL_CLR_Mask
#define ACMP_CMPx_FLT_CTRL_WIND_Pos       	(24U)
#define ACMP_CMPx_FLT_CTRL_WIND_Mask      	(0x1FUL << ACMP_CMPx_FLT_CTRL_WIND_Pos)
#define ACMP_CMPx_FLT_CTRL_WIND           	ACMP_CMPx_FLT_CTRL_WIND_Mask
#define ACMP_CMPx_FLT_CTRL_NUM_Pos       	(16U)
#define ACMP_CMPx_FLT_CTRL_NUM_Mask      	(0x1FUL << ACMP_CMPx_FLT_CTRL_NUM_Pos)
#define ACMP_CMPx_FLT_CTRL_NUM           	ACMP_CMPx_FLT_CTRL_NUM_Mask
#define ACMP_CMPx_FLT_CTRL_DIV_Pos       	(0U)
#define ACMP_CMPx_FLT_CTRL_DIV_Mask      	(0x3FFUL << ACMP_CMPx_FLT_CTRL_DIV_Pos)
#define ACMP_CMPx_FLT_CTRL_DIV           	ACMP_CMPx_FLT_CTRL_DIV_Mask

#define ACMP_CMPx_ST_NEG_Pos       			(3U)
#define ACMP_CMPx_ST_NEG_Mask      			(0x1UL << ACMP_CMPx_ST_NEG_Pos)
#define ACMP_CMPx_ST_NEG           			ACMP_CMPx_ST_NEG_Mask
#define ACMP_CMPx_ST_POS_Pos       			(2U)
#define ACMP_CMPx_ST_POS_Mask      			(0x1UL << ACMP_CMPx_ST_POS_Pos)
#define ACMP_CMPx_ST_POS           			ACMP_CMPx_ST_POS_Mask
#define ACMP_CMPx_ST_INT_Pos       			(1U)
#define ACMP_CMPx_ST_INT_Mask      			(0x1UL << ACMP_CMPx_ST_INT_Pos)
#define ACMP_CMPx_ST_INT           			ACMP_CMPx_ST_INT_Mask
#define ACMP_CMPx_ST_OUT_Pos       			(0U)
#define ACMP_CMPx_ST_OUT_Mask      			(0x1UL << ACMP_CMPx_ST_OUT_Pos)
#define ACMP_CMPx_ST_OUT           			ACMP_CMPx_ST_OUT_Mask

#define DAC_CTRL_EBUS_SEL_Pos       		(16U)
#define DAC_CTRL_EBUS_SEL_Mask      		(0xFUL << DAC_CTRL_EBUS_SEL_Pos)
#define DAC_CTRL_EBUS_SEL        			DAC_CTRL_EBUS_SEL_Mask
#define DAC_CTRL_RAMP_TRG_Pos       		(4U)
#define DAC_CTRL_RAMP_TRG_Mask      		(0x1UL << DAC_CTRL_RAMP_TRG_Pos)
#define DAC_CTRL_RAMP_TRG        			DAC_CTRL_RAMP_TRG_Mask
#define DAC_CTRL_RAMP_MD_Pos       			(3U)
#define DAC_CTRL_RAMP_MD_Mask      			(0x1UL << DAC_CTRL_RAMP_MD_Pos)
#define DAC_CTRL_RAMP_MD        			DAC_CTRL_RAMP_MD_Mask
#define DAC_CTRL_RAMP_EN_Pos       			(2U)
#define DAC_CTRL_RAMP_EN_Mask      			(0x1UL << DAC_CTRL_RAMP_EN_Pos)
#define DAC_CTRL_RAMP_EN        			DAC_CTRL_RAMP_EN_Mask
#define DAC_CTRL_DACx_MD_Pos       			(1U)
#define DAC_CTRL_DACx_MD_Mask      			(0x1UL << DAC_CTRL_DACx_MD_Pos)
#define DAC_CTRL_DACx_MD        			DAC_CTRL_DACx_MD_Mask
#define DAC_CTRL_DACx_EN_Pos       			(0U)
#define DAC_CTRL_DACx_EN_Mask      			(0x1UL << DAC_CTRL_DACx_EN_Pos)
#define DAC_CTRL_DACx_EN        			DAC_CTRL_DACx_EN_Mask

#define DAC_VAL_B_Pos       				(16U)
#define DAC_VAL_B_Mask      				(0xFFFUL << DAC_VAL_B_Pos)
#define DAC_VAL_B        					DAC_VAL_B_Mask
#define DAC_VAL_Pos       					(0U)
#define DAC_VAL_Mask      					(0xFFFUL << DAC_VAL_Pos)
#define DAC_VAL        						DAC_VAL_Mask

#define DAC_SDAT_Pos       					(0U)
#define DAC_SDAT_Mask      					(0xFFFUL << DAC_SDAT_Pos)
#define DAC_SDAT        					DAC_SDAT_Mask

#define DAC_RAMP_DLY_B_Pos       			(16U)
#define DAC_RAMP_DLY_B_Mask      			(0x3FFFUL << DAC_RAMP_DLY_B_Pos)
#define DAC_RAMP_DLY_B        				DAC_RAMP_DLY_B_Mask
#define DAC_RAMP_DLY_Pos       				(0U)
#define DAC_RAMP_DLY_Mask      				(0x3FFFUL << DAC_RAMP_DLY_Pos)
#define DAC_RAMP_DLY        				DAC_RAMP_DLY_Mask

#define DAC_RAMP_STEP_B_Pos       			(16U)
#define DAC_RAMP_STEP_B_Mask      			(0x3FFFUL << DAC_RAMP_STEP_B_Pos)
#define DAC_RAMP_STEP_B        				DAC_RAMP_STEP_B_Mask
#define DAC_RAMP_STEP_Pos       			(0U)
#define DAC_RAMP_STEP_Mask      			(0xFFFUL << DAC_RAMP_STEP_Pos)
#define DAC_RAMP_STEP        				DAC_RAMP_STEP_Mask

#define DAC_RAMP_SLOPE_B_Pos       			(16U)
#define DAC_RAMP_SLOPE_B_Mask      			(0x3FFFUL << DAC_RAMP_SLOPE_B_Pos)
#define DAC_RAMP_SLOPE_B        			DAC_RAMP_SLOPE_B_Mask
#define DAC_RAMP_SLOPE_Pos       			(0U)
#define DAC_RAMP_SLOPE_Mask      			(0x3FFFUL << DAC_RAMP_SLOPE_Pos)
#define DAC_RAMP_SLOPE        				DAC_RAMP_SLOPE_Mask

#define DAC_CMPx_ASEL_INP_Pos       		(16U)
#define DAC_CMPx_ASEL_INP_Mask      		(0xFUL << DAC_CMPx_ASEL_INP_Pos)
#define DAC_CMPx_ASEL_INP        			DAC_CMPx_ASEL_INP_Mask
#define DAC_CMPx_ASEL_Pos       			(0U)
#define DAC_CMPx_ASEL_Mask      			(0x7UL << DAC_CMPx_ASEL_Pos)
#define DAC_CMPx_ASEL        				DAC_CMPx_ASEL_Mask


/******************************* ADC Instances ********************************/
#define IS_ADC_ALL_INSTANCE(INSTANCE) (((INSTANCE) == ADC0) || ((INSTANCE) == ADC1))

/******************************* SPI Instances ********************************/
#define IS_SPI_ALL_INSTANCE(INSTANCE) ((INSTANCE) == SPI)

/******************************* CRC Instances ********************************/
#define IS_CRC_ALL_INSTANCE(INSTANCE) ((INSTANCE) == CRC)

/******************************* GPIO Instances *******************************/
#define IS_GPIO_ALL_INSTANCE(INSTANCE) (((INSTANCE) == GPIOA) || \
                                        ((INSTANCE) == GPIOB) || \
                                        ((INSTANCE) == GPIOC) || \
                                        ((INSTANCE) == GPIOD) || \
                                        ((INSTANCE) == GPIOE))

/******************************* IOW Instances *******************************/
#define IS_IOW_ALL_INSTANCE(INSTANCE) (((INSTANCE) == IOW))


/******************************* GPIO AF Instances ****************************/
#define IS_GPIO_AF_INSTANCE(INSTANCE)   IS_GPIO_ALL_INSTANCE(INSTANCE)



/****************** Base Timer Instances  **************************************/
#define IS_BTIM_16BITINSTANCE(INSTANCE)    (((INSTANCE) == BTIM00) || ((INSTANCE) == BTIM01) || ((INSTANCE) == BTIM10) || ((INSTANCE) == BTIM11))
#define IS_BTIM_32BITINSTANCE(INSTANCE)    (((INSTANCE) == BTIM00) || ((INSTANCE) == BTIM10))


/****************************** WWDG Instances ********************************/
#define IS_WWDG_INSTANCE(INSTANCE)  ((INSTANCE) == WWDG)



/******* GPIO Alternative Function Definitions *******/

/*********************** Alternative Function 0 *****************/
#define GPIOA10_AF0_SCI0_RTS_CS_CTS                         0x00
#define GPIOA11_AF0_SCI0_RX_MISO                            0x00
#define GPIOA12_AF0_SCI0_SCK_CTS                            0x00
#define GPIOA13_AF0_SCI0_TX_SDA_MOSI                        0x00
#define GPIOA14_AF0_SCI1_SCK_CTS                            0x00
#define GPIOA15_AF0_SCI1_RX_MISO                            0x00
#define GPIOB0_AF0_SCI1_TX_SDA_MOSI                         0x00
#define GPIOB1_AF0_SCI1_RTS_CS_CTS                          0x00
#define GPIOB2_AF0_SCI2_SCK_CTS                             0x00
#define GPIOB3_AF0_SCI2_TX_SDA_MOSI                         0x00
#define GPIOB4_AF0_SCI2_RX_MISO                             0x00
#define GPIOB5_AF0_SCI2_TX_SDA_MOSI                         0x00
#define GPIOB6_AF0_SCI2_RX_MISO                             0x00
#define GPIOC3_AF0_CAN_RX                                   0x00
#define GPIOC4_AF0_CAN_TX                                   0x00
#define GPIOC6_AF0_PCLBUZ                                   0x00
#define GPIOC7_AF0_ACMP3_OUT                                0x00
#define GPIOC8_AF0_ACMP0_OUT                                0x00
#define GPIOC9_AF0_ACMP1_OUT                                0x00
#define GPIOC10_AF0_ACMP2_OUT                               0x00
#define GPIOD2_AF0_I2C0_SCL                                 0x00
#define GPIOD3_AF0_I2C0_SDA                                 0x00
#define GPIOD13_AF0_SCI2_RTS_CS_CTS                         0x00
#define GPIOA0_AF0_SCI0_RTS_CS_CTS                          0x00


/*********************** Alternative Function 1 *****************/
#define GPIOA2_AF1_SCI0_RX_MISO                             0x01
#define GPIOA6_AF1_SCI1_RX_MISO                             0x01
#define GPIOA7_AF1_SCI1_TX_SDA_MOSI                         0x01
#define GPIOA8_AF1_NMI                                      0x01
#define GPIOA11_AF1_SCI0_SCK_CTS                            0x01
#define GPIOA13_AF1_SCI0_TX_SDA_MOSI                        0x01
#define GPIOA15_AF1_SCI1_SCK_CTS                            0x01
#define GPIOB0_AF1_SCI1_TX_SDA_MOSI                         0x01
#define GPIOB3_AF1_SCI2_TX_SDA_MOSI                         0x01
#define GPIOB4_AF1_SCI2_SCK_CTS                             0x01
#define GPIOB5_AF1_SCI2_TX_SDA_MOSI                         0x01
#define GPIOB6_AF1_SCI2_SCK_CTS                             0x01


/*********************** Alternative Function 2 *****************/
#define GPIOA6_AF2_CAN_RX                                   0x02
#define GPIOA7_AF2_CAN_TX                                   0x02
#define GPIOA8_AF2_CAN_DBG                                  0x02
#define GPIOA9_AF2_SPI_SS1                                  0x02
#define GPIOA10_AF2_SPI_SS0                                 0x02
#define GPIOA11_AF2_SWD_CLK                                 0x02
#define GPIOA13_AF2_SWD_DIO                                 0x02
#define GPIOA14_AF2_SPI_MOSI                                0x02
#define GPIOB2_AF2_SPI_SCK                                  0x02
#define GPIOB3_AF2_I2C0_SDA                                 0x02
#define GPIOB4_AF2_I2C0_SCL                                 0x02
#define GPIOB5_AF2_SPI_MOSI                                 0x02
#define GPIOB6_AF2_SPI_MISO                                 0x02
#define GPIOB7_AF2_SPI_SS0                                  0x02
#define GPIOC6_AF2_SPI_SS1                                  0x02
#define GPIOC7_AF2_SPI_SS0                                  0x02
#define GPIOC8_AF2_SPI_SCK                                  0x02
#define GPIOC9_AF2_SPI_MOSI                                 0x02
#define GPIOC10_AF2_SPI_MISO                                0x02
#define GPIOC11_AF2_CAN_RX                                  0x02
#define GPIOC12_AF2_CAN_TX                                  0x02
#define GPIOD13_AF2_I2C0_SCL                                0x02
#define GPIOA1_AF2_I2C0_SDA                                 0x02


/*********************** Alternative Function 3 *****************/
#define GPIOA2_AF3_SCI0_TX_SDA_MOSI                         0x03
#define GPIOA6_AF3_I2C1_SDA                                 0x03
#define GPIOA7_AF3_I2C1_SCL                                 0x03
#define GPIOA10_AF3_SCI0_TX_SDA_MOSI                        0x03
#define GPIOA11_AF3_SCI0_SCK_CTS                            0x03
#define GPIOA12_AF3_SCI1_TX_SDA_MOSI                        0x03
#define GPIOA13_AF3_SCI1_RX_MISO                            0x03
#define GPIOA15_AF3_CAN_RX                                  0x03
#define GPIOB0_AF3_CAN_TX                                   0x03
#define GPIOB2_AF3_REF_CLK                                  0x03
#define GPIOC3_AF3_I2C0_SCL                                 0x03
#define GPIOC4_AF3_I2C0_SDA                                 0x03
#define GPIOC5_AF3_TIMH_CH0                                 0x03
#define GPIOC6_AF3_TIMH_CH1                                 0x03
#define GPIOC7_AF3_TIMH_CH2                                 0x03
#define GPIOC8_AF3_TIMH_CH0N                                0x03
#define GPIOC9_AF3_TIMH_CH1N                                0x03
#define GPIOC10_AF3_TIMH_CH2N                               0x03


/*********************** Alternative Function 4 *****************/
#define GPIOA3_AF4_TIMS0_BKIN                               0x04
#define GPIOA4_AF4_TIMG2_ETR                                0x04
#define GPIOA5_AF4_TIMG1_ETR                                0x04
#define GPIOA9_AF4_TIMG2_CH0                                0x04
#define GPIOA10_AF4_TIMG2_CH1                               0x04
#define GPIOA12_AF4_TIMS0_CH1                               0x04
#define GPIOA13_AF4_TIMS0_CH2                               0x04
#define GPIOB2_AF4_TIMG1_CH0                                0x04
#define GPIOB3_AF4_TIMG1_CH1                                0x04
#define GPIOB4_AF4_TIMG1_CH2                                0x04
#define GPIOB5_AF4_TIMG1_CH3                                0x04
#define GPIOB6_AF4_TIMG0_CH3                                0x04
#define GPIOB7_AF4_TIMA1_CH3                                0x04
#define GPIOB8_AF4_TIMA1_BKIN                               0x04
#define GPIOB9_AF4_TIMA1_CH0                                0x04
#define GPIOB10_AF4_TIMA1_CH1                               0x04
#define GPIOB11_AF4_TIMA1_CH2                               0x04
#define GPIOB12_AF4_TIMA1_CH0N                              0x04
#define GPIOB13_AF4_TIMA1_CH1N                              0x04
#define GPIOB14_AF4_TIMA1_CH2N                              0x04
#define GPIOB15_AF4_TIMA0_CH2N                              0x04
#define GPIOC0_AF4_TIMA0_CH1N                               0x04
#define GPIOC1_AF4_TIMA0_CH0N                               0x04
#define GPIOC2_AF4_TIMA0_CH2                                0x04
#define GPIOC3_AF4_TIMA0_CH1                                0x04
#define GPIOC4_AF4_TIMA0_CH0                                0x04
#define GPIOC5_AF4_TIMA0_BKIN                               0x04
#define GPIOC6_AF4_TIMG1_CH0                                0x04
#define GPIOC7_AF4_TIMG1_CH1                                0x04
#define GPIOC8_AF4_TIMG0_CH0                                0x04
#define GPIOC9_AF4_TIMG0_CH1                                0x04
#define GPIOC10_AF4_TIMG0_CH2                               0x04
#define GPIOC11_AF4_TIMG2_CH0                               0x04
#define GPIOC12_AF4_TIMG2_CH1                               0x04
#define GPIOD12_AF4_TIMA0_CH3                               0x04
#define GPIOD13_AF4_TIMG2_CH2                               0x04
#define GPIOA0_AF4_TIMG2_CH3                                0x04


/*********************** Alternative Function 5 *****************/
#define GPIOA4_AF5_TIMG0_CH0                                0x05
#define GPIOA5_AF5_TIMG0_CH1                                0x05
#define GPIOA8_AF5_TIMS1_BKIN                               0x05
#define GPIOA9_AF5_TIMS1_CH1                                0x05
#define GPIOA10_AF5_TIMS1_CH2                               0x05
#define GPIOA11_AF5_TIMS0_CH1                               0x05
#define GPIOA12_AF5_TIMS0_CH1N                              0x05
#define GPIOA13_AF5_TIMS1_CH1N                              0x05
#define GPIOA14_AF5_TIMS1_CH1                               0x05
#define GPIOA15_AF5_TIMS2_BKIN                              0x05
#define GPIOB0_AF5_TIMS3_BKIN                               0x05
#define GPIOB3_AF5_TIMS0_CH1                                0x05
#define GPIOB4_AF5_TIMS0_CH1N                               0x05
#define GPIOB5_AF5_TIMS0_CH2                                0x05
#define GPIOB6_AF5_TIMS3_CH1                                0x05
#define GPIOB7_AF5_TIMS3_CH2                                0x05
#define GPIOC6_AF5_TIMA0_ETR                                0x05
#define GPIOC7_AF5_TIMA1_ETR                                0x05
#define GPIOC8_AF5_TIMS2_CH1                                0x05
#define GPIOC9_AF5_TIMS2_CH2                                0x05
#define GPIOC10_AF5_TIMS2_CH1                               0x05
#define GPIOC11_AF5_TIMS0_CH1                               0x05
#define GPIOC12_AF5_TIMS0_CH1                               0x05
#define GPIOD13_AF5_TIMG0_ETR                               0x05
#define GPIOA0_AF5_TIMS3_CH1N                               0x05
#define GPIOD12_AF5_TIMH_ETR					            0x05

/*********************** Alternative Function 6 *****************/
#define GPIOA6_AF6_EVT0                                     0x06
#define GPIOA7_AF6_EVT1                                     0x06
#define GPIOA8_AF6_EVT2                                     0x06
#define GPIOA10_AF6_EVT3                                    0x06
#define GPIOA12_AF6_EVT4                                    0x06
#define GPIOB2_AF6_TIMH_CH0                                 0x06
#define GPIOB3_AF6_TIMH_CH1                                 0x06
#define GPIOB4_AF6_TIMH_CH2                                 0x06
#define GPIOB5_AF6_TIMH_CH0N                                0x06
#define GPIOB6_AF6_TIMH_CH1N                                0x06
#define GPIOB7_AF6_TIMH_CH2N                                0x06
#define GPIOB8_AF6_EVT5                                     0x06
#define GPIOB11_AF6_EVT6                                    0x06
#define GPIOB15_AF6_EVT7                                    0x06
#define GPIOC3_AF6_EVT8                                     0x06
#define GPIOC6_AF6_EVT9                                     0x06
#define GPIOC7_AF6_EVT10                                    0x06
#define GPIOC10_AF6_TIMS2_CH1N                              0x06
#define GPIOC12_AF6_TIMS0_CH1N                              0x06
#define GPIOC15_AF6_EVT11                                   0x06
#define GPIOD7_AF6_EVT12                                    0x06
#define GPIOD11_AF6_EVT13                                   0x06
#define GPIOD12_AF6_TIMS3_CH1                               0x06
#define GPIOD13_AF6_TIMS3_CH2                               0x06
#define GPIOA0_AF6_EVT14                                    0x06
#define GPIOA1_AF6_EVT15                                    0x06


/*********************** Alternative Function 7 *****************/
#define GPIOA6_AF7_KR0                                      0x07
#define GPIOA7_AF7_KR1                                      0x07
#define GPIOA8_AF7_KR2                                      0x07
#define GPIOA10_AF7_KR3                                     0x07
#define GPIOA12_AF7_KR4                                     0x07
#define GPIOB3_AF7_TIMS0_CH1N                               0x07
#define GPIOB7_AF7_TIMS3_CH1N                               0x07
#define GPIOB8_AF7_KR5                                      0x07
#define GPIOB9_AF7_KR6                                      0x07
#define GPIOC5_AF7_KR7                                      0x07




#ifdef __cplusplus
}
#endif /* __cplusplus */

#endif /* __KPM32x14Txx_H */
