/**
 ******************************************************************************
  * @file    kpm32xx_ddl_basetimer.c
  * @author  Kiwi Software Team
  * @brief   BaseTimer(BTM) DDL module driver.
  *          This file provides firmware functions to manage the following
  *          functionalities of the basetimer peripheral:
  *          1.peripheral initializes
  *          2.counter
  *          3.deal with IRQ
  * @note
  *          V1.0.0, 2024/12/20.
  *
  * Copyright (c) 2024, Kiwi Instruments Co,. Ltd.
  *
  * Redistribution and use in source and binary forms, with or without modification,
  * are permitted provided that the following conditions are met:
  *
  *   1. Redistributions of source code must retain the above copyright notice,
  *      this list of conditions and the following disclaimer.
  *
  *   2. Redistributions in binary form must reproduce the above copyright notice,
  *      this list of conditions and the following disclaimer in the documentation
  *      and/or other materials provided with the distribution.
  *
  *   3. Neither the name of the copyright holder nor the names of its contributors
  *      may be used to endorse or promote products derived from this software without
  *      specific prior written permission.
  *
  * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
  * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
  * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
  * THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
  * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE
  * GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
  * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
  * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
  * THE POSSIBILITY OF SUCH DAMAGE.
  ******************************************************************************
  */

#include "kpm32xx_ddl.h"


#ifdef DDL_BASETIMER_MODULE_ENABLED

/**
  * @brief  Make BTM peripheral active.
  * @param  None.
  * @retval None.
  */
void DDL_BTM_Instance_Active(void)
{
	/* Do Clock and Reset BTM peripheral */
	__DDL_RCC_BTIM_ACTIVE();
	__DDL_RCC_BTIM_CLK_RELEASE();
}


/**
  * @brief  Configure the structure of BTM_BaseInit_T to default values.
  *
  * @param  BTM_BaseInit_T Pointer to a TIMS_BaseInit_T structure that contains
  *                        the configuration information for the given BTM module.
  *
  * @retval None
  */
void DDL_BTM_StructInit(BTM_BaseInit_T *pBtmStruct)
{
	pBtmStruct->bitMode    = MODE_16BIT;
	pBtmStruct->clkDivsion = DIV_16;
	pBtmStruct->eventOuput = EVT_DISABLE;
	pBtmStruct->loadValue  = 0;
	pBtmStruct->clkSource  = BTM_LIRC;
}


/**
  * @brief  Initializes the BTM by the specified parameters in the BTM_BaseInit_T.
  *			
  * @param  BTMx     BTM instance.
  *			pInitBTM Pointer to a BTM_BaseInit_T structure that contains
  *                  the configuration information for the given BTM module.
  *
  * @retval NONE
  */
void DDL_BTM_Init(BTIM_Type *BTMx, BTM_BaseInit_T *pInitBTM)
{
	uint32_t tmp = 0;
	
	tmp = BTMx->CTRL;
	
	/* config clock source */
	MODIFY_REG(SYSCFG->CSEL2, SYSCONF_CSEL2_BTIM_Msk, pInitBTM->clkSource);
	
	/* config
	 * 1. bitMode
	 * 2. event output Enable or Disable
	 * 3. clock division
	 */
	MODIFY_REG(tmp, (0x1UL << 31), (pInitBTM->bitMode    << 31));
	MODIFY_REG(tmp, (0x1UL << 6 ), (pInitBTM->eventOuput << 6 ));
	MODIFY_REG(tmp, (0x3UL << 4 ), (pInitBTM->clkDivsion << 4 ));
	
	BTMx->CTRL = tmp; 
	
	/* config load value */
	BTMx->LD   = pInitBTM->loadValue;
}


/**
  * @brief  use for BTM interrupt enable.
  * @param  BTMx BTM instance.		
  * @retval None
  */
void DDL_BTM_IntEnable(BTIM_Type *BTMx)
{
	SET_BIT(BTMx->CTRL, (0x1 << 7));
}


/**
  * @brief  use for BTM interrupt disable.
  * @param  BTMx BTM instance.		
  * @retval None
  */
void DDL_BTM_IntDisable(BTIM_Type *BTMx)
{
	CLEAR_BIT(BTMx->CTRL, (0x1 << 7));
}


/**
  * @brief  use for BTM event output enable.
  * @param  BTMx BTM instance.		
  * @retval None
  */
void DDL_BTM_EvtOutEnable(BTIM_Type *BTMx)
{
	SET_BIT(BTMx->CTRL, (0x1 << 6));
}


/**
  * @brief  use for BTM event output disable.
  * @param  BTMx BTM instance.		
  * @retval None
  */
void DDL_BTM_EvtOutDisable(BTIM_Type *BTMx)
{
	CLEAR_BIT(BTMx->CTRL, (0x1 << 6));
}


/**
  * @brief  use for timer start.
  * @param  BTMx BTM instance.
  * @retval None
  */
void DDL_BTM_Start(BTIM_Type *BTMx)
{
	SET_BIT(BTMx->CTRL, 0x1);
}


/**
  * @brief  use for timer stop.
  * @param  BTMx  BTM instance.
  * @retval None
  */
void DDL_BTM_Stop(BTIM_Type *BTMx)
{
	CLEAR_BIT(BTMx->CTRL, 0x1);
}

#endif /* DDL_BASETIMER_MODULE_ENABLED */
