/**
 ******************************************************************************
 * @file    kpm32xx_ddl_iow.h
 * @author  Kiwi Software Team
 * @brief   Header file of  IOW DDL module.
 * @note
 *		   V1.0.0, 2022/08/03.
 *
 * Copyright (c) 2024, Kiwi Instruments Co,. Ltd.
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 *	1. Redistributions of source code must retain the above copyright notice,
 *	   this list of conditions and the following disclaimer.
 *
 *	2. Redistributions in binary form must reproduce the above copyright notice,
 *	   this list of conditions and the following disclaimer in the documentation
 *	   and/or other materials provided with the distribution.
 *
 *	3. Neither the name of the copyright holder nor the names of its contributors
 *	   may be used to endorse or promote products derived from this software without
 *	   specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
 * THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE
 * GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 ******************************************************************************
 */


#ifndef __KPM32xx_DDL_IOW_H
#define __KPM32xx_DDL_IOW_H

#ifdef __cplusplus
extern "C" {
#endif

#include "kpm32xx_ddl_def.h"

#define IOW_KRX_NUMBER     8


typedef struct
{
	uint32_t krx;                            /* IOW channel: 0 ~ 7*/
	uint32_t edges;                          /* Rising or Falling Edge on pin*/
} IOW_Init_T;

/* IOW edge detection */
#define IOW_EDGE_FALLING                     0x0UL
#define IOW_EDGE_RISING                      0x1UL

/* IOW channels */
#define IOW_KR0                        0x00000001UL
#define IOW_KR1                        0x00000002UL
#define IOW_KR2                        0x00000004UL
#define IOW_KR3                        0x00000008UL
#define IOW_KR4                        0x00000010UL
#define IOW_KR5                        0x00000020UL
#define IOW_KR6                        0x00000040UL
#define IOW_KR7                        0x00000080UL

#define __DDL_IOW_INTRCHN_ENABLE(__INSTANCE__, __CHANNEL__)    SET_BIT((__INSTANCE__)->CTRL, __CHANNEL__)
#define __DDL_IOW_INTRCHN_DISABLE(__INSTANCE__, __CHANNEL__)   CLEAR_BIT((__INSTANCE__)->CTRL, __CHANNEL__)
#define __DDL_IOW_INTR_ENABLE(__INSTANCE__)                    SET_BIT((__INSTANCE__)->INTEN, IOW_INTEN_EN)
#define __DDL_IOW_INTR_DISABLE(__INSTANCE__)                   CLEAR_BIT((__INSTANCE__)->INTEN, IOW_INTEN_EN)
#define __DDL_IOW_CLEAR_CHNINTR(__INSTANCE__, __CHANNEL__)     WRITE_REG((__INSTANCE__)->INTSR, __CHANNEL__)
#define __DDL_IOW_GET_CHNEDGE(__INSTANCE__, __CHANNEL__)       READ_BIT((__INSTANCE__)->EDG, __CHANNEL__)



/**
  * @brief  Makes IOW active.
  * @param  none
  *
  * @retval none
  */
void DDL_IOW_Instance_Active(void);

/**
  * @brief  IOW interrupt enable.
  * @param  IOWx IOW instance.
  * @param  iowInit  pointer to a IOW_Init_T structure.
  *
  * @retval none
  */
void DDL_IOW_IntEnable(IOW_Type *IOWx, IOW_Init_T *iowInit);

/**
  * @brief  IOW interrupt disable.
  * @param  IOWx IOW instance.
  * @param  iowInit  pointer to a IOW_Init_T structure.
  *
  * @retval none
  */
void DDL_IOW_IntDisable(IOW_Type *IOWx, IOW_Init_T *iowInit);

/**
  * @brief  Initialize IOW_Init_T structure.
  * @param  iowInit  pointer to a IOW_Init_T structure.
  *
  * @retval none
  */
void DDL_IOW_StructInit(IOW_Init_T *iowInit);


/**
  * @brief  configure IOW parameters.
  * @param  IOWx     IO PORT instance.
  * @param  iowInit  pointer to a IOW_Init_T structure.
  *
  * @retval none
  */
DDL_Status_T DDL_IOW_Init(IOW_Type *IOWx, IOW_Init_T *iowInit);

/**
  * @brief  IOW interrupt handler.
  * @param  IOWx IOW instance
  *
  * @retval none
  */
void DDL_IOW_IRQHandler(IOW_Type *IOWx);



#ifdef __cplusplus
}
#endif
#endif /* __KPM32xx_DDL_IOW_H */
