/**
  *****************************************************************************************************
  * @file    uart_user_drv.c
  * @author  Kiwi Software Team
  * @brief   This file provides simplified functions to manage the following
  *          functionalities of SCI UART:
  *          + Transmit and receive data with interrupt mode
  * @note
  *          V1.0.0, 2024/12/20.
  *
  * Copyright (c) 2024, Kiwi Instruments Co,. Ltd.
  *
  * Redistribution and use in source and binary forms, with or without modification,
  * are permitted provided that the following conditions are met:
  *
  *   1. Redistributions of source code must retain the above copyright notice,
  *      this list of conditions and the following disclaimer.
  *
  *   2. Redistributions in binary form must reproduce the above copyright notice,
  *      this list of conditions and the following disclaimer in the documentation
  *      and/or other materials provided with the distribution.
  *
  *   3. Neither the name of the copyright holder nor the names of its contributors
  *      may be used to endorse or promote products derived from this software without
  *      specific prior written permission.
  *
  * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
  * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
  * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
  * THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
  * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE
  * GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
  * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
  * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
  * THE POSSIBILITY OF SUCH DAMAGE.
  *****************************************************************************************************
  */

#include "main.h"
#include "string.h"
#include <stdlib.h>
#include "uart_user_drv.h"



/**
  * @brief  receives an amount of data with interrupt.
  * @param  SCIx Pointer to SCI UART instance.
  *
  * @retval None
  */
void DDL_SCIUART_User_Receive_IT(SCI_Type *SCIx)
{
	ASSERT_PARAM(SCIx != NULL);

    __DDL_SCIUART_RX_ENABLE(SCIx);

    DDL_SCIUART_IntEnable(SCIx, SCI_IE_FRMERR | SCI_IE_PRTERR | SCI_IE_OVRERR | SCI_IE_RXEND);

    NVIC_EnableIRQ(DDL_Get_SCIModule_IRQn(SCIx));
}


/**
  * @brief  transmits an amount of data with interrupt.
  * @param  SCIx Pointer to SCI UART instance.
  * @param  pData    pointer to data buffer (uint8_t or uint16_t data frame).
  * @param  size     the amount of data frame (uint8_t or uint16_t) to be sent.
  *
  * @retval None
  */
void DDL_SCIUART_User_Transmit_IT(SCI_Type *SCIx, uint8_t *pData, uint16_t size)
{
	ASSERT_PARAM(pData != NULL);
	ASSERT_PARAM(SCIx != NULL);
	ASSERT_PARAM(size != 0);

    __DDL_SCIUART_TX_ENABLE(SCIx);
    
    if (((SCIx->MUART&0x03) == 0x02)&&((SCIx->MUART&0x0C) == 0x00))
    {
        MODIFY_REG(SCIx->MUART,(1<<10),(((*(pData) & 0xFFUL)>>7)<<10));
    }

	SCIx->TXD = *(pData) & 0xFFUL;

    DDL_SCIUART_IntEnable(SCIx, SCI_IE_FRMERR | SCI_IE_PRTERR | SCI_IE_OVRERR | SCI_IE_TXEND);
	/*Till now, SCI is ready to transaction data-frame */
	NVIC_EnableIRQ(DDL_Get_SCIModule_IRQn(SCIx));
}


/**
  * @brief  Mutually configure UART Tx and Rx .
  * @param  SCIx Pointer to SCI UART instance.
  * @param  uartTxRx UART Tx and Rx configuration.
  *
  * @retval None
  */
void Uart_TransData_RxTx_Mutual(SCI_Type *SCIx, UartTxRx_T uartTxRx)
{
	if (uartTxRx == UART_TX)
	{
        CLEAR_BIT(SCIx->MC, SCI_MC_RXEN);
        SET_BIT(SCIx->MC, SCI_MC_TXEN);
	}
	if (uartTxRx == UART_RX)
	{
        CLEAR_BIT(SCIx->MC, SCI_MC_TXEN);
        SET_BIT(SCIx->MC, SCI_MC_RXEN);
	}
}

