/**
  *****************************************************************************************************
  * @file    debug_console_ddl.c
  * @author  Kiwi Software Team
  * @brief   Serial Unit UART print log inforamtion.
  * @note
  *          V1.0.0, 2024/12/20.
  *
  * Copyright (c) 2024, Kiwi Instruments Co,. Ltd.
  *
  * Redistribution and use in source and binary forms, with or without modification,
  * are permitted provided that the following conditions are met:
  *
  *   1. Redistributions of source code must retain the above copyright notice,
  *      this list of conditions and the following disclaimer.
  *
  *   2. Redistributions in binary form must reproduce the above copyright notice,
  *      this list of conditions and the following disclaimer in the documentation
  *      and/or other materials provided with the distribution.
  *
  *   3. Neither the name of the copyright holder nor the names of its contributors
  *      may be used to endorse or promote products derived from this software without
  *      specific prior written permission.
  *
  * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
  * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
  * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
  * THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
  * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE
  * GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
  * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
  * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
  * THE POSSIBILITY OF SUCH DAMAGE.
  *****************************************************************************************************
  */

#include "stdio.h"
#include "debug_console_ddl.h"

#ifdef DDL_SCIUART_MODULE_ENABLED

#if !defined(__DEBUG_CONSOLE_PRINT__)
#define printf(...)  do{}while(0);
#endif

#if defined (__DEBUG_CONSOLE_PRINT__)

static void *pPrtInstance = NULL;

/**
  * @brief  This function is implemented for UART debug console.
  * @param  sciUart  pointer to a SCI_Type structure.
  * @retval DDL status
  */
DDL_Status_T DDL_SCIUart_DebugConsole_Init(SCI_Type *sciUart)
{
    SCIUART_Init_T sciUartStruct;
    
	DDL_SCIUART_StructInit(&sciUartStruct);
    
    if (DDL_RCC_GetSysClockFreq() == 100000000)
    {
        sciUartStruct.clkPres     = 2;
        sciUartStruct.brDiv1And2  = 0x144 << 16;
    }
    else if (DDL_RCC_GetSysClockFreq() == 200000000)
    {
        sciUartStruct.clkPres     = 3;
        sciUartStruct.brDiv1And2  = 0x144 << 16;
    }

    pPrtInstance = (void *)sciUart;

    DDL_SCIUART_Init(sciUart, &sciUartStruct);
	/* Enable SCI UART TX */
	__DDL_SCIUART_TX_ENABLE(sciUart);

    return DDL_OK;
}

#ifdef __ICCARM__
int putchar(int ch)
#else
int fputc(int ch, FILE *f)
#endif
{
    if (ch == '\n')
    {
        DDL_SCIUART_WaitPutCharCmplt((SCI_Type *)pPrtInstance, ch);
        DDL_SCIUART_WaitPutCharCmplt((SCI_Type *)pPrtInstance, '\r');
    }
    else
    {
        DDL_SCIUART_WaitPutCharCmplt((SCI_Type *)pPrtInstance, ch);
    }
#ifdef __ICCARM__
    return ch;
#else
    return 0;
#endif
}
#else

#ifdef __ICCARM__
int putchar(int ch)
#else
int fputc(int ch, FILE *f)
#endif
{
    return 0;
}
#endif /* __DEBUG_CONSOLE_PRINT__ */


#if defined(USE_ASSERT)

void Assert_Failed(uint8_t *file, uint32_t line)
{
    while (1)
    {

    }
}

#endif /*USE_ASSERT*/

#endif /* DDL_SCIUART_MODULE_ENABLED */
