/**
  ******************************************************************************
  * @file    emulated_eeprom.h
  * @author  Kiwi Software Team
  * @brief   Header file of emualted eeprom module.
  * @note
  *          V1.0.0, 2021/11/15, use data flash to emulate EEPROM.
  *
  * Copyright (c) 2024, Kiwi Instruments Co,. Ltd.
  *
  * Redistribution and use in source and binary forms, with or without modification,
  * are permitted provided that the following conditions are met:
  *
  *   1. Redistributions of source code must retain the above copyright notice,
  *      this list of conditions and the following disclaimer.
  *
  *   2. Redistributions in binary form must reproduce the above copyright notice,
  *      this list of conditions and the following disclaimer in the documentation
  *      and/or other materials provided with the distribution.
  *
  *   3. Neither the name of the copyright holder nor the names of its contributors
  *      may be used to endorse or promote products derived from this software without
  *      specific prior written permission.
  *
  * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
  * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
  * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
  * THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
  * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE
  * GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
  * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
  * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
  * THE POSSIBILITY OF SUCH DAMAGE.
  ******************************************************************************
  */


#ifndef __EMULATED_EEPROM_H
#define __EMULATED_EEPROM_H

#ifdef __cplusplus
extern "C" {
#endif

#include <stdio.h>
#include "kpm32xx_ddl.h"

/* Variables' number for each bank, user defined */
#define VAR_NUM               ((uint8_t)0x05)

/* Define the size of the sectors to be used */
#define PAGE_SIZE               FLASH_CODE_PAGESIZE

/* EEPROM start address in Flash */
#define EEPROM_PAGE_START_INDEX ((uint16_t)(FLASH_CODE_TOTAL_PAGES - 4))      /* Specify which page used for EEPROM */
#define EEPROM_START_ADDRESS    (FLASH_CODE_BASE + EEPROM_PAGE_START_INDEX * PAGE_SIZE)

/* Specify the data index of Data flash to use */
#define PAGE0_ID              ((uint16_t)(EEPROM_PAGE_START_INDEX))
#define PAGE1_ID              ((uint16_t)(EEPROM_PAGE_START_INDEX + 1))
#define PAGE2_ID              ((uint16_t)(EEPROM_PAGE_START_INDEX + 2))
#define PAGE3_ID              ((uint16_t)(EEPROM_PAGE_START_INDEX + 3))


/* Pages 0,1,2,3 base and end addresses */
#define PAGE0_BASE_ADDRESS    ((uint32_t)(EEPROM_START_ADDRESS + 0x0000))
#define PAGE0_END_ADDRESS     ((uint32_t)(EEPROM_START_ADDRESS + (PAGE_SIZE - 1)))

#define PAGE1_BASE_ADDRESS    ((uint32_t)(EEPROM_START_ADDRESS + PAGE_SIZE))
#define PAGE1_END_ADDRESS     ((uint32_t)(EEPROM_START_ADDRESS + (2 * PAGE_SIZE - 1)))

#define PAGE2_BASE_ADDRESS    ((uint32_t)(EEPROM_START_ADDRESS + 2 * PAGE_SIZE))
#define PAGE2_END_ADDRESS     ((uint32_t)(EEPROM_START_ADDRESS + (3 * PAGE_SIZE - 1)))

#define PAGE3_BASE_ADDRESS    ((uint32_t)(EEPROM_START_ADDRESS + 3 * PAGE_SIZE))
#define PAGE3_END_ADDRESS     ((uint32_t)(EEPROM_START_ADDRESS + (4 * PAGE_SIZE - 1)))

#define PAGE0                 ((uint16_t)0x0000)
#define PAGE1                 ((uint16_t)0x0001)
#define PAGE2                 ((uint16_t)0x0002)
#define PAGE3                 ((uint16_t)0x0003)

/*
 * bank 0 contains page 0 and page 1, bank 1 contains page 2 and page 3
 * bank 0 and Bank1 can work simultaneously.
 */
#define BANK0                 ((uint16_t)0x0000)
#define BANK1                 ((uint16_t)0x0001)



/**
  * @brief  Restore the pages to a known good state in case of page's status
  *         corruption after a power loss.
  * @param  bank  this parameter can be one of the following values:
  *               @arg BANK0: data flash sector 0 and sector 1
  *               @arg BANK1: data flash sector 2 and sector 3
  * @retval - Flash error code: on write Flash error
  *         - FLASH_COMPLETE: on success
  */
uint16_t EEPROM_Init(uint16_t bank);

/**
  * @brief  Returns the last stored variable data, if found, which correspond to
  *         the passed virtual address
  * @param  bank         Specify which bank to use
  * @param  virtAddress  Variable virtual address
  * @param  data         Global variable contains the read variable value
  * @retval Success or error status:
  *           - 0: if variable was found
  *           - 1: if the variable was not found
  *           - NO_VALID_PAGE: if no valid page was found.
  */
uint16_t EEPROM_Read(uint16_t bank, uint16_t VirtAddress, uint16_t* Data);

/**
  * @brief  Writes/updates variable data in EEPROM.
  * @param  bank         BANK0 or BANK1
  * @param  virtAddress  Variable virtual address
  * @param  data         16 bit data to be written
  * @retval Success or error status:
  *           - FLASH_COMPLETE: on success
  *           - PAGE_FULL: if valid page is full
  *           - NO_VALID_PAGE: if no valid page was found
  *           - Flash error code: on write Flash error
  */
uint16_t EEPROM_Write(uint16_t bank, uint16_t VirtAddress, uint16_t Data);

#ifdef __cplusplus
}
#endif

#endif

