/**
  *****************************************************************************************************
  * @file    kpm32xx_ddl_rcc.c
  * @author  Kiwi Software Team
  * @brief   RCC DDL module driver, and RCC means Reset and Clock Control module.
  *          This file provides firmware functions to manage the following
  *          functionalities:
  *          1. system clock configuration
  *          2. clock configurations of all peripherals.
  *
  * @note
  *          V1.0.0, 2024/12/20.
  *
  * Copyright (c) 2024, Kiwi Instruments Co,. Ltd.
  *
  * Redistribution and use in source and binary forms, with or without modification,
  * are permitted provided that the following conditions are met:
  *
  *   1. Redistributions of source code must retain the above copyright notice,
  *      this list of conditions and the following disclaimer.
  *
  *   2. Redistributions in binary form must reproduce the above copyright notice,
  *      this list of conditions and the following disclaimer in the documentation
  *      and/or other materials provided with the distribution.
  *
  *   3. Neither the name of the copyright holder nor the names of its contributors
  *      may be used to endorse or promote products derived from this software without
  *      specific prior written permission.
  *
  * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
  * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
  * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
  * THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
  * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE
  * GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
  * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
  * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
  * THE POSSIBILITY OF SUCH DAMAGE.
  *****************************************************************************************************
  */


#include "kpm32xx_ddl.h"


#ifdef DDL_RCC_MODULE_ENABLED

#define CLOCKSWITCH_TIMEOUT_VALUE         (5000U) /* 5 s    */

#define __MAXIMUM_FCLK_LEVEL1__            50000000
#define __MAXIMUM_FCLK_LEVEL3__            100000000
#define __MAXIMUM_FCLK_LEVEL6__            200000000

#define MAXIMUM_LOOPCNT_WAIT_CLKSET_DONE   1000
#define MAXIMUM_LOOPCNT_WAIT_HOSC_STABLE   4000 /*minimum: 300us*/
#define MAXIMUM_LOOPCNT_WAIT_LOSC_STABLE   4000 /*minimum: 300us*/
#define MAXIMUM_LOOPCNT_WAIT_HIRC_STABLE   1000 /*minimum: 1000 cycle */
#define MAXIMUM_LOOPCNT_WAIT_LIRC_STABLE   4000 /*minimum: 300us*/
#define MAXIMUM_LOOPCNT_WAIT_PLL_STABLE    4000 /*minimum: 300us*/

/*clkStable: RCC_STR_LOSCSTABLE, RCC_STR_HOSCSTABLE, RCC_STR_LIRCSTABLE, RCC_STR_HIRCSTABLE, RCC_STR_PLLSTABLE*/
static DDL_Status_T Wait_Clock_Till_Stable(uint32_t clkStable, uint32_t waitCycles)
{
	uint32_t loopCnt = waitCycles;

    while (loopCnt > 0)
    {
		if ((SYSCFG->RSTS & clkStable) == clkStable)
        {
            break;
        }

        loopCnt --;
    }

    if (loopCnt == 0)
    {
        return DDL_TIMEOUT;
    }

	return DDL_OK;
}


/**
  * @brief  Initialize the HOSC oscillator for system clock
  * @param  void
  * @retval DDL status
  */
DDL_Status_T DDL_RCC_HoscEnable(void)
{
	__DDL_RCC_HOSC_ENABLE();
	__DDL_RCC_HOSC_HINOUT();

	if (Wait_Clock_Till_Stable(RCC_STR_HOSCSTABLE, MAXIMUM_LOOPCNT_WAIT_HOSC_STABLE) != DDL_OK)
	{
		return DDL_TIMEOUT;
	}

	return DDL_OK;
}


/**
  * @brief  Disable the HOSC oscillator for system clock
  * @param  void
  * @retval DDL status
  */
DDL_Status_T DDL_RCC_HoscDisable(void)
{
	uint32_t tempSysClkSrc = 0;
    uint32_t tempPllClkSrc = 0;
	uint32_t tempMClkSrc   = 0;

	/* Get current system clock source and PLL source */
    tempSysClkSrc = __DDL_RCC_GET_SYSCLK_SOURCE();
    tempPllClkSrc = __DDL_RCC_GET_PLLCLK_SOURCE();
	tempMClkSrc   = __DDL_RCC_GET_MCLK_SOURCE();

	/* It's not allowed to disable HOSC while HOSC is as direct or indirect system clock source */
	if (((tempSysClkSrc == RCC_SYSCLKSOURCE_STATUS_MCLK) && (tempMClkSrc ==  RCC_MCLCK_SOURCE_HOSC)) || ((tempSysClkSrc == RCC_SYSCLKSOURCE_STATUS_PLLCLK) && (tempPllClkSrc == RCC_PLLSOURCE_HOSC)))
	{
		return DDL_ERROR;
	}
	else
	{
		/*Now, disable it and release HIN and HOUT function */
		__DDL_RCC_HOSC_DISABLE();
		__DDL_RCC_HOSC_NORM();
	}

	return DDL_OK;
}


/**
  * @brief  Initialize the HIRC oscillator for system clock
  * @param  void
  * @retval DDL status
  */
DDL_Status_T DDL_RCC_HircEnable(void)
{
	__DDL_RCC_HIRC_ENABLE();

	if (Wait_Clock_Till_Stable(RCC_STR_HIRCSTABLE, MAXIMUM_LOOPCNT_WAIT_HIRC_STABLE) != DDL_OK)
	{
		return DDL_TIMEOUT;
	}

	return DDL_OK;
}


/**
  * @brief  Disable the HIRC oscillator for system clock
  * @param  void
  * @retval DDL status
  */
DDL_Status_T DDL_RCC_HircDisable(void)
{
	uint32_t tempSysClkSrc = 0;
    uint32_t tempPllClkSrc = 0;
	uint32_t tempMClkSrc   = 0;

	/* Get current system clock source and PLL source */
    tempSysClkSrc = __DDL_RCC_GET_SYSCLK_SOURCE();
    tempPllClkSrc = __DDL_RCC_GET_PLLCLK_SOURCE();
	tempMClkSrc   = __DDL_RCC_GET_MCLK_SOURCE();

	if (((tempSysClkSrc == RCC_SYSCLKSOURCE_STATUS_MCLK) && (tempMClkSrc == RCC_MCLCK_SOURCE_HIRC)) || ((tempSysClkSrc == RCC_SYSCLKSOURCE_STATUS_PLLCLK) && (tempPllClkSrc == RCC_PLLSOURCE_HIRC)))
	{
		return DDL_ERROR;
	}
	else
	{
		/*Now, disable it */
		__DDL_RCC_HIRC_DISABLE();
	}

	return DDL_OK;
}


/**
  * @brief  Initialize the LIRC oscillator for system clock
  * @param  void
  * @retval DDL status
  */
DDL_Status_T DDL_RCC_LircEnable(void)
{
	__DDL_RCC_LIRC_ENABLE();

	if (Wait_Clock_Till_Stable(RCC_STR_LIRCSTABLE, MAXIMUM_LOOPCNT_WAIT_LIRC_STABLE) != DDL_OK)
	{
		return DDL_TIMEOUT;
	}

	return DDL_OK;
}


/**
  * @brief  Disable the LIRC oscillator for system clock
  * @param  void
  * @retval DDL status
  */
DDL_Status_T DDL_RCC_LircDisable(void)
{
	__DDL_RCC_LIRC_DISABLE();

	return DDL_OK;
}


/**
  * @brief  Configure MCLK source
  * @param  void
  * @retval DDL status
  */
DDL_Status_T  DDL_RCC_MclkSrcConfig(uint32_t mClkSource)
{
	if (mClkSource == RCC_MCLKSOURCE_HOSC)
	{
		if (!(IS_RCC_HOSC_STABLE()))
		{
			return DDL_ERROR;
		}
	}

	if (mClkSource == RCC_MCLKSOURCE_HIRC)
	{
		if (!(IS_RCC_HIRC_STABLE()))
		{
			return DDL_ERROR;
		}
	}

	/* Configure MCLK source */
	__DDL_RCC_MCLKSRC_CONFIG(mClkSource);

	return DDL_OK;
}


/**
  * @brief  Configure the elements of structure RCC_PLLInit_T to default values.
  * @param  pllInitStruct  pointer to a RCC_PLLInit_T structure that contains
  *                        the configuration information PLL.
  * @retval None.
  */
void DDL_RCC_PllStructInit(RCC_PLLInit_T  *pllInitStruct)
{
	pllInitStruct->pllSource = RCC_PLLSOURCE_HIRC;

	/*Caution: PLL prescaler: from 0 to 31*/
	pllInitStruct->pllPrescaler  = 0;
	pllInitStruct->pllDIVM	     = 4;
	pllInitStruct->pllMULN	     = 0x48;
}


/**
  * @brief  initialize the PLL oscillator for system clock
  * @param  pll  pointer to a pll structure that contains
  *                           the configuration information for oscillator.
  * @retval DDL status
  */
static DDL_Status_T DDL_RCC_PllEnable(RCC_PLLInit_T *pll)
{
	__DDL_SYSREG_UNLOCK();
	/* Configure pll Clock Source */
	MODIFY_REG(SYSCFG->PLL, SYSCONF_PLL_CLKSRC_Msk, pll->pllSource);
	/* Configure pll M */
	/* Configure pll N */
	/* Configure pll Prescaler */
	MODIFY_REG(SYSCFG->PLL, (SYSCONF_PLL_DIV_Msk | SYSCONF_PLL_PLLN_Msk | SYSCONF_PLL_PLLM), \
							(pll->pllDIVM      << SYSCONF_PLL_PLLM_Pos  | \
							 pll->pllMULN      << SYSCONF_PLL_PLLN_Pos  | \
							 pll->pllPrescaler << SYSCONF_PLL_DIV_Pos));
	/* Enable pll */
	SET_BIT(SYSCFG->PLL, SYSCONF_PLL_EN);
	__DDL_SYSREG_LOCK();

	if (Wait_Clock_Till_Stable(RCC_STR_PLLSTABLE, MAXIMUM_LOOPCNT_WAIT_PLL_STABLE) != DDL_OK)
	{
		return DDL_TIMEOUT;
	}

	return DDL_OK;
}


/**
  * @brief  Disable the PLL for system clock
  * @param  void
  * @retval DDL status
  */
DDL_Status_T DDL_RCC_PLLDisable(void)
{
	uint32_t tempSysClkSrc = 0;

	/* Get current system clock source and pll source */
    tempSysClkSrc = __DDL_RCC_GET_SYSCLK_SOURCE();

    /* It's not allowed to disable pll while pll is as direct or indirect system clock source */
	if (tempSysClkSrc == RCC_SYSCLKSOURCE_STATUS_PLLCLK)
	{
		return DDL_ERROR;
	}
	else
	{
		/*Now, disable it */
		__DDL_RCC_PLL_DISABLE();
	}

	return DDL_OK;
}


/**
  * @brief  Initializes the PLL by the specified parameters in the RCC_PLLInit_T.
  *
  * @param  pllInitStruct  pointer to a RCC_PLLInit_T structure that contains
  *                        the configuration information PLL.
  *
  * @retval DDL status
  */
DDL_Status_T DDL_RCC_PllConfig(RCC_PLLInit_T  *pllInitStruct)
{
	ASSERT_PARAM(IS_RCC_PLLSOURCE(pllInitStruct->pllSource));

	DDL_Status_T retval = DDL_OK;

	if (pllInitStruct->pllSource == RCC_PLLSOURCE_HOSC)
	{
		/* First, enable HOSC */
		retval = DDL_RCC_HoscEnable();
		if (retval != DDL_OK)
		{
			return retval;
		}
	}

	if (pllInitStruct->pllSource == RCC_PLLSOURCE_HIRC)
	{
		/* First, enable HIRC */
		retval = DDL_RCC_HircEnable();
		if (retval != DDL_OK)
		{
			return retval;
		}
	}

	retval = DDL_RCC_PllEnable(pllInitStruct);

    return retval;
}


/**
  * @brief  configure system clock to the expected working frequency.
  * @param  rccClkInitStruct  pointer to a RCC_OscInit_T structure that contains
  *                           the configuration information for system clock.
  * @retval DDL status
  */
DDL_Status_T DDL_RCC_ClkSrcConfig(RCC_ClkInit_T  *rccClkInitStruct)
{
	CacheInit_T cacheInit;
    uint32_t loopCnt = MAXIMUM_LOOPCNT_WAIT_CLKSET_DONE;

    ASSERT_PARAM(rccClkInitStruct != NULL);
    /* Check the parameters */
	ASSERT_PARAM(IS_SYSCLK_DEVIDER(rccClkInitStruct->sysClkDivider));
	ASSERT_PARAM(IS_RCC_SYSCLKSOURCE(rccClkInitStruct->sysClkSource));

	/* From the perspective of safety, Configure flash latency as 6 before system clock is set */
	DDL_FLASH_Unlock();
	DDL_FLASH_SetLatency(FLASH_CTRL_LATENCY_6);
	DDL_FLASH_Lock();

    /* Now, configure system-clock */
	__DDL_RCC_SYSCLKDIV_CONFIG(rccClkInitStruct->sysClkDivider);
    __DDL_RCC_SYSCLKSRC_CONFIG(rccClkInitStruct->sysClkSource);

    /* Read-out to check whether configuration has really completed */
    loopCnt = MAXIMUM_LOOPCNT_WAIT_CLKSET_DONE;
    while (loopCnt > 0)
    {
        if ((((SYSCFG->CSEL0 & SYSCONF_CSEL0_FCLKST_Msk) >> SYSCONF_CSEL0_FCLKST_Pos) << SYSCONF_CSEL0_FCLK_Pos) == rccClkInitStruct->sysClkSource)
        {
            break;
        }
        loopCnt --;
    }

    if (loopCnt == 0)
    {
        return DDL_TIMEOUT;
    }

    /* Now, we can get the correct system clock frequency */
    systemCoreClock = DDL_RCC_GetSysClockFreq();

	DDL_FLASH_Unlock();
	if (systemCoreClock <= __MAXIMUM_FCLK_LEVEL1__)
	{
		DDL_FLASH_SetLatency(FLASH_CTRL_LATENCY_1);
	}
	else if (systemCoreClock <= __MAXIMUM_FCLK_LEVEL3__)
	{
		DDL_FLASH_SetLatency(FLASH_CTRL_LATENCY_3);
	}
	else if (systemCoreClock <= __MAXIMUM_FCLK_LEVEL6__)
	{
		DDL_FLASH_SetLatency(FLASH_CTRL_LATENCY_6);
	}
	DDL_FLASH_Lock();

	/* Enable ICACHE, DCACHE, prefetch feature. */
	cacheInit.prefetchEnable  = SET;
	cacheInit.statisticEnable = SET;

	DDL_ICACHE_Init(&cacheInit);
	DDL_DCACHE_Init(&cacheInit);

	return DDL_InitTick(uwTickPriority);
}


/**
  * @brief  Get PLL clock frequency.
  *
  * @retval PLL frequency.
  */
static uint32_t RCC_Get_PLL_ClockFreq(void)
{
	uint32_t pllSrcClk, pllM, pllN, pllPrescaler, pllClkFreq;

	pllSrcClk = __DDL_RCC_GET_PLLCLK_SOURCE();

	if (pllSrcClk == RCC_PLL_SOURCE_HIRC)
	{
		pllSrcClk = HIRC_VALUE;
	}

	if (pllSrcClk == RCC_PLL_SOURCE_HOSC)
	{
		pllSrcClk = HOSC_VALUE;
	}

	pllM         = __DDL_RCC_GET_PLL_M();
	pllN         = __DDL_RCC_GET_PLL_N();
	pllPrescaler = __DDL_RCC_GET_PLL_P();

	if (pllM == 0)
	{
		pllM = 1;
	}

	if ((pllPrescaler == 0) || (pllPrescaler == 1))
	{
		pllPrescaler = 2;
	}

	pllClkFreq = pllSrcClk / pllM * (pllN + 128) / pllPrescaler;

	return pllClkFreq;
}


/**
  * @brief  Get system working frequency.
  *
  * @retval system clock frequency.
  */
uint32_t DDL_RCC_GetSysClockFreq(void)
{
	/*in unit of Hz*/
	uint32_t sysClkFreq = 0;
	uint32_t sysClkDiv  = 0;

	/* System clock's divider coefficient */
	sysClkDiv = __DDL_RCC_GET_SYSCLK_DIVIDER();

	if (__DDL_RCC_GET_SYSCLK_SOURCE() == RCC_SYSCLKSOURCE_STATUS_MCLK)
	{
		/* MCLK used as system clock source */
		sysClkFreq = MCLK_VALUE;
	}
	else if (__DDL_RCC_GET_SYSCLK_SOURCE() == RCC_SYSCLKSOURCE_STATUS_PLLCLK)
	{
        sysClkFreq = RCC_Get_PLL_ClockFreq();
	}

	sysClkFreq = sysClkFreq / (1 << sysClkDiv);

	return sysClkFreq;
}


/**
  * @brief  Configure Base Timer clock.
  * @param  btimClkInit  pointer to a RCC_BTIMInit_T structure that contains
  *                      the configuration information for Base Timer.
  * @retval DDL status
  */
DDL_Status_T DDL_RCC_BTIM_PeriphCLKConfig(RCC_BTIMInit_T *btimClkInit)
{
	ASSERT_PARAM(IS_BTIM_PERIPHCLOCK(btimClkInit->btimClkSRC));

	if (btimClkInit->btimClkSRC == RCC_BTIM_CLKSRC_LIRC)
	{
		/* Check LIRC is stable */
		if (!IS_RCC_LIRC_STABLE())
		{
			return DDL_ERROR;
		}
	}

	/* Configure Base-Timer Clock Source */
	MODIFY_REG(SYSCFG->CSEL2, SYSCONF_CSEL2_BTIM_Msk, btimClkInit->btimClkSRC);

	return DDL_OK;
}


/**
  * @brief  Configure Buzzer clock.
  * @param  buzzClkInit  pointer to a RCC_BUZInit_T structure that contains
  *                      the configuration information for Buzzer.
  * @retval DDL status
  */
DDL_Status_T DDL_RCC_BUZZ_PeriphCLKConfig(RCC_BUZInit_T *buzzClkInit)
{
	ASSERT_PARAM(IS_BUZ1_PERIPHCLOCK(buzzClkInit->buzClkSRC));
	ASSERT_PARAM(buzzClkInit->buzPrescalser <= __MAX_BUZ1_CLK_PRESCALER__);

	if (buzzClkInit->buzClkSRC == RCC_BUZ1_CLKSRC_HIRC)
	{
		if (!IS_RCC_HIRC_STABLE())
		{
			return DDL_ERROR;
		}
	}

	if (buzzClkInit->buzClkSRC == RCC_BUZ1_CLKSRC_HOSC)
	{
		if (!IS_RCC_HOSC_STABLE())
		{
			return DDL_ERROR;
		}
	}

	if (buzzClkInit->buzClkSRC == RCC_BUZ1_CLKSRC_LIRC)
	{
		if (!IS_RCC_LIRC_STABLE())
		{
			return DDL_ERROR;
		}
	}

	if (buzzClkInit->buzClkSRC == RCC_BUZ1_CLKSRC_PLL)
	{
		if (!IS_RCC_PLL_STABLE())
		{
			return DDL_ERROR;
		}
	}

	/* Configure BUZ1 Clock Source */
	MODIFY_REG(SYSCFG->CSEL2, SYSCONF_CSEL2_BUZ1_Msk, buzzClkInit->buzClkSRC);
	MODIFY_REG(SYSCFG->CSEL1, SYSCONF_CSEL1_BUZ1DIV_Msk, (uint32_t)((buzzClkInit->buzPrescalser) << SYSCONF_CSEL1_BUZ1DIV_Pos));

	return DDL_OK;
}


/**
  * @brief  Configure Noise Filter clock.
  * @param  nfClkInit  pointer to a RCC_NFInit_T structure that contains
  *                    the configuration information for Noise Filter.
  * @retval DDL status
  */
DDL_Status_T DDL_RCC_NF_PeriphCLKConfig(RCC_NFInit_T *nfClkInit)
{
	ASSERT_PARAM(IS_NF_PERIPHCLOCK(nfClkInit->nfClkSRC));

	if (nfClkInit->nfClkSRC == RCC_NF_CLKSRC_HIRC)
	{
		/* Check HIRC is stable */
		if (!IS_RCC_HIRC_STABLE())
		{
			return DDL_ERROR;
		}
		CLEAR_BIT(SYSCFG->CSEL2, RCC_NF_CLKSRC_FCLK);
	}
	else
	{
		SET_BIT(SYSCFG->CSEL2, RCC_NF_CLKSRC_FCLK);
	}

	/* Now, Set NF Clock 0 ~ 2 configuration */
	MODIFY_REG(SYSCFG->CSEL2, SYSCONF_CSEL2_NFCLK0_Msk, nfClkInit->nfClk0Prescaler << SYSCONF_CSEL2_NFCLK0_Pos);
	MODIFY_REG(SYSCFG->CSEL2, SYSCONF_CSEL2_NFCLK1_Msk, nfClkInit->nfClk1Prescaler << SYSCONF_CSEL2_NFCLK1_Pos);
	MODIFY_REG(SYSCFG->CSEL2, SYSCONF_CSEL2_NFCLK2_Msk, nfClkInit->nfClk2Prescaler << SYSCONF_CSEL2_NFCLK2_Pos);

	return DDL_OK;
}


/**
  * @brief  Configure ADC's clock.
  * @param  adcClkInit  pointer to a RCC_ADCInit_T structure that contains
  *                     the configuration information for ADC.
  * @retval DDL status
  */
DDL_Status_T DDL_RCC_ADC_PeriphCLKConfig(RCC_ADCInit_T *adcClkInit)
{
	if ((adcClkInit->periphClkSrcSelection & RCC_PERIPHCLK_ADC0) == RCC_PERIPHCLK_ADC0)
    {
		ASSERT_PARAM((adcClkInit->adcPrescalser) <= __MAX_ADC_CLK_PRESCALER__);

		/* Configure ADC Clock Prescaler */
		MODIFY_REG(SYSCFG->CSEL2, SYSCONF_CSEL2_ADC0DIV_Msk, (uint32_t)((adcClkInit->adcPrescalser) << SYSCONF_CSEL2_ADC0DIV_Pos));
    }

	if ((adcClkInit->periphClkSrcSelection & RCC_PERIPHCLK_ADC1) == RCC_PERIPHCLK_ADC1)
	{
		ASSERT_PARAM((adcClkInit->adcPrescalser) <= __MAX_ADC_CLK_PRESCALER__);

		/* Configure ADC Clock Prescaler */
		MODIFY_REG(SYSCFG->CSEL2, SYSCONF_CSEL2_ADC1DIV_Msk, (uint32_t)((adcClkInit->adcPrescalser) << SYSCONF_CSEL2_ADC1DIV_Pos));
	}

	return DDL_OK;
}


/**
  * @brief  Configure WatchDog's clock.
  * @param  wdtClkInit  pointer to a RCC_WDTInit_T structure that contains
  *                     the configuration information for WDT.
  * @retval DDL status
  */
DDL_Status_T DDL_RCC_WDT_PeriphCLKConfig(RCC_WDTInit_T *wdtClkInit)
{
	ASSERT_PARAM(IS_WDT_PERIPHCLOCK(wdtClkInit->wdtClkSrc));

	if (wdtClkInit->wdtClkSrc == RCC_WDT_CLKSRC_LIRC)
	{
		/* Check LIRC is stable */
		if (!IS_RCC_LIRC_STABLE())
		{
			return DDL_ERROR;
		}
	}

	__DDL_SYSREG_UNLOCK();
	MODIFY_REG(SYSCFG->CSEL2, SYSCONF_CSEL2_WDT_Msk, wdtClkInit->wdtClkSrc);
	__DDL_SYSREG_LOCK();

	return DDL_OK;
}

#endif /* DDL_RCC_MODULE_ENABLED */

