/**
 ******************************************************************************
 * @file    kpm32xx_ddl_timh.h
 * @author  Kiwi Software Team
 * @brief   Header file of  TIMH DDL module.
 * @note
 *		   V1.0.0, 2024/12/20.
 *
  * Copyright (c) 2024, Kiwi Instruments Co,. Ltd.
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 *	1. Redistributions of source code must retain the above copyright notice,
 *	   this list of conditions and the following disclaimer.
 *
 *	2. Redistributions in binary form must reproduce the above copyright notice,
 *	   this list of conditions and the following disclaimer in the documentation
 *	   and/or other materials provided with the distribution.
 *
 *	3. Neither the name of the copyright holder nor the names of its contributors
 *	   may be used to endorse or promote products derived from this software without
 *	   specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
 * THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE
 * GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 ******************************************************************************
 */


#ifndef __KPM32xx_DDL_TIMH_H
#define __KPM32xx_DDL_TIMH_H

#ifdef __cplusplus
extern "C" {
#endif


#include "kpm32xx_ddl_def.h"


/* thresould value for timeout*/
#define TIMH_HR_MODE_STARTUP_TIMEOUT    ((uint32_t)0x000FFFF0U)	/* timeout for hr mode strtup*/
#define TIMH_CAP_NOT_READY_TIMEOUT      ((uint32_t)0x000FFFF0U)	/* timeout for capture       */

/* combine PWM source select */
typedef enum
{
	COMB_SRC_PWM0H,
	COMB_SRC_PWM0L,
	COMB_SRC_PWM1H,
	COMB_SRC_PWM1L,
	COMB_SRC_PWM2H,
	COMB_SRC_PWM2L
}TIMH_Combine_Src_T;

/* combine PWM source polarity */
typedef enum
{
	COMB_Pol_MAIN,
	COMB_Pol_REVERSE
}TIMH_Combine_Pol_T;

/* combine PWM logic select */
typedef enum
{
	COMB_LOGIC_OR,
	COMB_LOGIC_AND,
	RESERVED,
	COMB_LOGIC_XOR
}TIMH_Combine_Logic_T;

/* combine PWM target output select */
typedef enum
{
	COMB_TARGET_CLOSE,
	COMB_TARGET_PWM1,
	COMB_TARGET_PWM2
}TIMH_Combine_Output_T;

/* PWM event select */
typedef enum
{
	TIMH_PGTRGx_SEL,
	TIMH_PWM_OCCUR_OUTPUT,
	TIMH_PCI_SYNC_OUTPUT,
	TIMH_PCI_FEEDFORWARD_OUTPUT,
	TIMH_PCI_CLD_OUTPUT,
	TIMH_PCI_FLT_OUTPUT,
	TIMH_CAHALF,
	TIMH_STEER,
	TIMH_ADC_TRIGGER1,
	TIMH_ADC_TRIGGER2,
	TIMH_HR_PRECISION_ERR        = 0xf
}TIMH_EBUS_EVENT_T;

/* PWM source event select */
typedef enum
{
	EBUS_SRC_TIMH0,
	EBUS_SRC_TIMH1,
	EBUS_SRC_TIMH2
}TIMH_EBUS_SOURCE_T;

/* PWM event output control register */
typedef enum
{
	TIMH_EVENT_A,
	TIMH_EVENT_B,
	TIMH_EVENT_C
}TIMH_PWM_EVENT_CTRL_T;

/* ADC triger1/2 source status */
typedef enum
{
	ADC_trg1_TRIGA_EN            = 1,
	ADC_trg1_TRIGA_DIS           = 0,
	ADC_trg1_TRIGB_EN            = 1,
	ADC_trg1_TRIGB_DIS           = 0,
	ADC_trg1_TRIGC_EN            = 1,
	ADC_trg1_TRIGC_DIS           = 0,	
	ADC_trg2_TRIGA_EN            = 1,
	ADC_trg2_TRIGA_DIS           = 0,
	ADC_trg2_TRIGB_EN            = 1,
	ADC_trg2_TRIGB_DIS           = 0,
	ADC_trg2_TRIGC_EN            = 1,
	ADC_trg2_TRIGC_DIS           = 0	
}TIMH_ADC_Trg_EN_T;

/* TIMH ouput mode */
typedef enum
{
	TIMH_OUTPUT_COMPLEMENTARY    = 0x0 << TIMH_PGxIOCON_PMOD_Pos,  /* Complementary output */
	TIMH_OUTPUT_INDEPENDENT      = 0x1 << TIMH_PGxIOCON_PMOD_Pos,  /* Independent output   */
	TIMH_OUTPUT_PUSH_PULL        = 0x2 << TIMH_PGxIOCON_PMOD_Pos   /* Push-pull output     */	
}TIMH_OUTPUT_T;

/* PWM_H/L ouput polarity */
typedef enum
{
	PWM_H_POL_HIGH ,                           /* PWM_H output polarity high */
	PWM_H_POL_LOW  ,                           /* PWM_H output polarity low  */
	PWM_L_POL_HIGH               = 0,          /* PWM_L output polarity high */	
	PWM_L_POL_LOW                = 1           /* PWM_L output polarity low  */	
}TIMH_POL_OUTPUT_T;

/* LEB trige source status */
typedef enum
{
	LEB_trige_DIS                = 0,
	LEB_trige_EN                 = 1,
}LEB_Trg_EN_T;

/* PCI signal accept mode definition */
typedef enum
{
	PCI_ACP_LEVEL_SENSITIVE,		
	PCI_ACP_RISING_EDGE,
	PCI_ACP_ANY_EDGE,  
	PCI_ACP_LATCHED,
	PCI_ACP_LATCHED_RISING_EDGE,
	PCI_ACP_LATCHED_ANY_EDGE       
}TIMH_PCI_ACP_T;

/* PCI signale source (PSS) definition */
typedef enum
{
	PCI_PSS_SOFEWARE,
	PCI_PSS_PWMPCI,
	PCI_PSS_COMBINE_TRIGER_A,
	PCI_PSS_COMBINE_TRIGER_B,
	PCI_PSS_EBUS_CHANNEL_8      = 0x08,
	PCI_PSS_EBUS_CHANNEL_9,
	PCI_PSS_EBUS_CHANNEL_10,
	PCI_PSS_EBUS_CHANNEL_11,
	PCI_PSS_EBUS_CHANNEL_12,
	PCI_PSS_EBUS_CHANNEL_13,
	PCI_PSS_EBUS_CHANNEL_14,
	PCI_PSS_EBUS_CHANNEL_15
}TIMH_PCI_PSS_T;

/* PCI signale Acceptance Qualifier Source (AQSS) definition */
typedef enum
{
	PCI_AQSS_NO_AQ,		
	PCI_AQSS_PWM_PERIOD_H,
	PCI_AQSS_LEB_ACTIVE,  
	PCI_AQSS_PG_TRIGGERED,
	PCI_AQSS_PCI_SOURCE_1,
	PCI_AQSS_PCI_SOURCE_8,
	PCI_AQSS_PCI_SOURCE_9,
	PCI_AQSS_SWPCI       
}TIMH_PCI_AQSS_T;

/* PCI termination signal source (TERM) definition */
typedef enum
{
	PCI_TERM_MANUAL_TERMINATE,
	PCI_TERM_AUTO_TERMINATE,  
	PCI_TERM_TRIGA,           
	PCI_TERM_TRIGB,           
	PCI_TERM_TRIGC,           
	PCI_TERM_PCI_SOURCE_1,   
	PCI_TERM_PCI_SOURCE_8,   
	PCI_TERM_PCI_SOURCE_9   
}TIMH_PCI_TERM_T;

/* PCI termination Qualifier Source (TQSS) definition */
typedef enum
{
	PCI_TQSS_NO_TQ,		
	PCI_TQSS_PWM_PERIOD_H,
	PCI_TQSS_LEB_ACTIVE,  
	PCI_TQSS_PG_TRIGGERED,
	PCI_TQSS_PCI_SOURCE_1,
	PCI_TQSS_PCI_SOURCE_8,
	PCI_TQSS_PCI_SOURCE_9,
	PCI_TQSS_SWPCI       
}TIMH_PCI_TQSS_T;

/* PCI bypass source definition */
typedef enum
{
	PCI_BPSEL_PG0,	
	PCI_BPSEL_PG1,
	PCI_BPSEL_PG2
}TIMH_PCI_BPSEL_T;

/* PCI bypass enable definition */
typedef enum
{
	PCI_BYPASS_DISABLE,	
	PCI_BYPASS_ENABLE
}TIMH_PCI_BP_EN_T;

/* LEB setting value */
typedef struct
{
	uint32_t LEB_Period          : 16;         /** Leading-Edge Blanking Period
                                                  This parameter can be a value of Min_Data = 0x0040 and Max_Date = 0xFFF8 */
	uint32_t PWM_L_Fall          : 1;          /** switch for PWM_Low_Falling to triger LEB
                                                  This parameter can be a value of @enum LEB_Trg_EN_T */
	uint32_t PWM_L_Rise          : 1;          /** switch for PWM_Low_Rasing to triger LEB
                                                  This parameter can be a value of @enum LEB_Trg_EN_T */
	uint32_t PWM_H_Fall          : 1;          /** switch for PWM_High_Falling to triger LEB
                                                  This parameter can be a value of @enum LEB_Trg_EN_T */
	uint32_t PWM_H_Rise          : 1;          /** switch for PWM_High_Rasing to triger LEB
                                                  This parameter can be a value of @enum LEB_Trg_EN_T */
}TIMH_LEB_T;

/* ADC trig1/2 congfig handler */
typedef struct 
{
	uint32_t ADTR1EN1            : 1;    	   /* TIMHxTRIGA as source of ADC_Triger_1   */
	uint32_t ADTR1EN2            : 1;    	   /* TIMHxTRIGB as source of ADC_Triger_1   */
	uint32_t ADTR1EN3            : 1;    	   /* TIMHxTRIGC as source of ADC_Triger_1   */
	uint32_t ADTR1PS             : 5;		   /* Offset value of ADC_Triger_1           */
	uint32_t ADTR1OFS            : 5;          /* Frequency division of ADC_Triger_1     */
	uint32_t ADTR2EN1            : 1;    	   /* TIMHxTRIGA as source of ADC_Triger_2   */
	uint32_t ADTR2EN2            : 1;    	   /* TIMHxTRIGB as source of ADC_Triger_2   */
	uint32_t ADTR2EN3            : 1;    	   /* TIMHxTRIGC as source of ADC_Triger_2   */
}TIMH_ADC_Trige_T;                            
											  
/* TIMH Overwrite Structure definition */     
typedef struct                                
{                                             
	uint32_t debugPwm_L          : 1;          /* PWML pin level in debug mode
                                                  This parameter can be a value of 0 and 1 */
	uint32_t debugPwm_H          : 1;          /* PWMH pin level in debug mode
                                                  This parameter can be a value of 0 and 1 */           
	uint32_t feedFowardPwm_L     : 1;          /* PWML pin level when FFPCI active
                                                  This parameter can be a value of 0 and 1 */
	uint32_t feedFowardPwm_H     : 1;          /* PWMH pin level when FFPCI active
                                                  This parameter can be a value of 0 and 1 */
	uint32_t currentLimitPwm_L   : 1;          /* PWML pin level when CLPCI active
                                                  This parameter can be a value of 0 and 1 */
	uint32_t currentLimitPwm_H   : 1;          /* PWMH pin level when CLPCI active
                                                  This parameter can be a value of 0 and 1 */
	uint32_t faultPwm_L          : 1;          /* PWML pin level when FPCI active
                                                  This parameter can be a value of 0 and 1 */
	uint32_t faultPwm_H          : 1;          /* PWMH pin level when FPCI active
                                                  This parameter can be a value of 0 and 1 */
	uint32_t osync               : 2;          /* config timing of overwrite value executed in PWM pin
                                                  This parameter can be a value of @ref TIMH_Overwrite_Sync */
	uint32_t overWriteDatePwm_L  : 1;          /* PWML pin level when sofeware overwrite executed
                                                  This parameter can be a value of 0 and 1 */
	uint32_t overWriteDatePwm_H  : 1;          /* PWMH pin level when sofeware overwrite executed
                                                  This parameter can be a value of 0 and 1 */
	uint32_t ovrDateEnable_L     : 1;          /* PWML pin sofeware overwrite enable status
                                                  This parameter can be a value of @ref TIMH_Sofeware_Overwrite_EN */
	uint32_t ovrDateEnable_H     : 1;          /* PWMH pin sofeware overwrite enable status
                                                  This parameter can be a value of @ref TIMH_Sofeware_Overwrite_EN */
	uint32_t swap                : 1;          /* PWMH/L pin level swap
                                                  This parameter can be a value of @ref TIMH_PWMH_L_Swap */
	uint32_t RESERVED            : 7;            
	uint32_t outputShutdown_L    : 1;          /* PWML pin level when output shutdown
                                                  This parameter can be a value of 0 and 1 */
	uint32_t outputShutdown_H    : 1;          /* PWMH pin level when output shutdown
                                                  This parameter can be a value of 0 and 1 */
}TIMH_Overwrite_T;
                                                   
/* TIMHx_yPCI Handle Structure definition */     
typedef struct
{                                                  
	uint32_t pciSource          : 5;           /* TIMH pci signal source(PCI[PSS])
                                                  This parameter can be a value of @enum TIMH_PCI_PSS_T */
	uint32_t srcPolarity        : 1;           /* TIMH pci signal polarity(PCI[PPS])
                                                  This parameter can be a value of Min_Data = 0 and Max_Date = 1 */
	uint32_t sync               : 1;           /* TIMH pci sync control(PCI[PSYNC])
                                                  This parameter can be a value of Min_Data = 0 and Max_Date = 1 */
	uint32_t sofewareTERM       : 1;           /* TIMH pci sofeware termination(PCI[SWTERM])
                                                  PCI[SWTERM] bit will be reset to 0 automatic if set 1 */
	uint32_t pciAQSS            : 3;           /* TIMH pci AQSS signal source(PCI[AQSS])
                                                  This parameter can be a value of @enum TIMH_PCI_AQSS_T */
	uint32_t aqssPolarity       : 1;           /* TIMH pci aqss signal polarity(PCI[AQPS])
                                                  This parameter can be a value of Min_Data = 0 and Max_Date = 1 */
	uint32_t pciTERM            : 3;           /* TIMH pci TERM signal source(PCI[TERM])
                                                  This parameter can be a value of @enum TIMH_PCI_TERM_T */
	uint32_t TSYNCDIS           : 1;           /* TIMH pci Tsync control(PCI[TSYNCDIS])
                                                  This parameter can be a value of Min_Data = 0 and Max_Date = 1 */
	uint32_t pciTQSS            : 3;           /* TIMH pci TQSS signal source(PCI[TQSS])
                                                  This parameter can be a value of @enum TIMH_PCI_TQSS_T */
	uint32_t tqssPolarity       : 1;           /* TIMH pci TQSS signal polarity(PCI[TQPS])
                                                  This parameter can be a value of Min_Data = 0 and Max_Date = 1 */
	uint32_t latchMode          : 1;           /* TIMH pci active signal latch mode(PCI[LATMOD])
                                                  This parameter can be a value of Min_Data = 0 and Max_Date = 1 */
	uint32_t sofewarePCI_Mode   : 2;           /* TIMH sofeware pci control mode(PCI[SWPCIM])
                                                  This parameter can be a value of @ref PCI_Sofeware_Control_Mode */
	uint32_t sofewarePCI        : 1;           /* TIMH sofeware pci control bit(PCI[SWPCI])
                                                  This parameter can be a value of Min_Data = 0 and Max_Date = 1 */
	uint32_t acceptMode         : 3;           /* TIMH sofeware pci signal accept mode(PCI[ACP])
                                                  This parameter can be a value of @enum TIMH_PCI_ACP_T */
	uint32_t RESERVED           : 1;
	uint32_t bypassSourceSel    : 3;           /* TIMH sofeware pci bypass source(PCI[BPSEL])
                                                  This parameter can be a value of @enum TIMH_PCI_BPSEL_T */
	uint32_t bypassEnable       : 1;           /* TIMH sofeware pci bypass enable bit(PCI[BPEN])
                                                  This parameter can be a value of @enum TIMH_PCI_BP_EN_T */
}TIMH_yPCI_Init_T;

/* TIMH common clock config struct */
typedef struct
{
	uint32_t clockDivision;
	uint32_t frequencyScale;
	uint32_t FSMINPER;
} TIMH_ClkInit_T;

/* TIMH trigA/B/C vaule Structure definition */
typedef struct
{
	uint32_t trgiA;
	uint32_t trgiB;
	uint32_t trgiC;
} TIMH_TRIGx_T;

/* TIMH deadtime_H/L vaule */
typedef struct
{
	uint16_t DTH;
	uint16_t DTL;
} TIMH_DEADTIME_T;

/* TIMH Base Init struct */
typedef struct
{
	uint32_t			        period;                 /* Specifies the period value to be loaded into the Period Register.
						        				           This parameter can be a number between Min_Data = 0x0020 and Max_Data = 0xFFFE in standard mode.
                                                           Min_Data = 0x00000400 and Max_Data = 0xFFFFFFFE in HR mode. */
	uint32_t			        phase;			        /* pwm phase */
	uint32_t			        dutyCycle;		        /* pwm dutycycle */
	uint32_t                    dutyCycleAdj;	        /* pwm dutyCycle Adjustment */
						        				        
	uint32_t			        clockSource;	        /* TIMH clock source 
	                                                       This parameter can be a value of @ref TIMH_Clock_Source */
	uint32_t    		        counterMode;   	        /* TIMH counter mode.
	                                                       This parameter can be a value of @ref TIMH_Count_Mode */
	uint32_t			        hrModeEnable;	        /* enable or disable hr_mode */
	TIMH_TRIGx_T		        trig_xValue;	        /* congfig value of trigA/B/C register */
	TIMH_DEADTIME_T		        deadTime;		        /* Specifies deadtime H/L value */
	TIMH_OUTPUT_T               outputMode;		        /* Specifies output mode */							            
	TIMH_POL_OUTPUT_T           pwm_hPolarity;          /* Pwm high output polarity. This parameter can be a value of @enum TIMH_POL_OUTPUT_T */
	TIMH_POL_OUTPUT_T           pwm_lPolarity;          /* Pwm low output  polarity. This parameter can be a value of @enum TIMH_POL_OUTPUT_T */
	uint32_t                    updateMode;             /* update mode, PGxCON[UPDMOD]. can be a value of @REF TIMH_Data_Updata_Mode */
} TIMH_BaseInit_T;
										
/** @defgroup TIMH_Overwrite_Sync : TIMH Overwrite Sync */                  
#define OVERWRITE_IN_NEXT_SOC                               0UL
#define OVERWRITE_IMMEDIATELY                               1UL
#define OVERWRITE_AS_UPDMOD                                 2UL

/** @defgroup TIMH_PWMH_L_Swap : Option for PWMH PWML Swap */
#define NO_SWAP_PWMH_PWML                                   0UL
#define SWAP_PWMH_PWML                                      1UL

/** @defgroup  TIMH_Sofeware_Overwrite_EN : Overwrite by sofeware enable */
#define SOFEWARE_OVERWRITE_DISABLE                          0UL
#define SOFEWARE_OVERWRITE_ENABLE                           1UL

/** @defgroup TIMH_Data_Updata_Mode : TIMH Data Updata Mode */
#define MASTER_SOC_MODE                                     0
#define MASTER_IMMEDIATELT_MODE                             1
#define SLAVE_SOC_MODE                                      2
#define SLAVE_IMMEDIATELT_MODE                              3

/** @defgroup Clock_Divison : TIMH Clock Division */
#define TIMH_CLOCKDIVISION_DIV2								0UL << TIMH_PCLKCON_DIVSEL_Pos
#define TIMH_CLOCKDIVISION_DIV4								1UL << TIMH_PCLKCON_DIVSEL_Pos
#define TIMH_CLOCKDIVISION_DIV8								2UL << TIMH_PCLKCON_DIVSEL_Pos
#define TIMH_CLOCKDIVISION_DIV16							3UL << TIMH_PCLKCON_DIVSEL_Pos

/** @defgroup TIMH_Clock_Source : TIMH Clock Source */
#define TIMHx_CLOCKSOURCE_NONE          					0UL << TIMH_PGxCON_CLKSEL_Pos	
#define TIMHx_CLOCKSOURCE_CLK_IN							1UL << TIMH_PGxCON_CLKSEL_Pos	
#define TIMHx_CLOCKSOURCE_CLK_DIV    						2UL << TIMH_PGxCON_CLKSEL_Pos
#define TIMHx_CLOCKSOURCE_CLK_FRQ_SCA    					3UL << TIMH_PGxCON_CLKSEL_Pos
															
/** @defgroup TIMH_Count_Mode : TIMH Counter Mode */
#define TIMH_CNT_MODE_INDEPENDENT_EDGE          			0UL << TIMH_PGxCON_MODSEL_Pos	
#define TIMH_CNT_MODE_VARIABLE_PHASE						1UL << TIMH_PGxCON_MODSEL_Pos	
#define TIMH_CNT_MODE_INDT_EDGE_DAUL_OUTPUT					2UL << TIMH_PGxCON_MODSEL_Pos	
#define TIMH_CNT_MODE_CENTER_ALIGNED						4UL << TIMH_PGxCON_MODSEL_Pos	
#define TIMH_CNT_MODE_CENTER_ALIGNED_DAUL_UPDATE			5UL << TIMH_PGxCON_MODSEL_Pos	
#define TIMH_CNT_MODE_CENTER_ALIGNED_DAUL_EDGE				6UL << TIMH_PGxCON_MODSEL_Pos	
#define TIMH_CNT_MODE_CENTER_ALIGNED_DAUL_EDGE_UPDATE		7UL << TIMH_PGxCON_MODSEL_Pos	

/** 
  * Config update mode, PGxCON[UPDMOD] bit.
  * 'MODE' can be one of the values below :  
  * @defgroup TIMH_Data_Updata_Mode
  */
#define __TIMH_PGxOCN_UPDMOD(__HANDLE__,MODE)           	MODIFY_REG((__HANDLE__)->CON , TIMH_PGxCON_UPDMOD ,MODE << TIMH_PGxCON_UPDMOD_Pos)

/**
  * Set or reset TIMHx ON bits(CON[ON]). Enable or disable the specific TIMH.
  */
#define __DDL_TIMH_ENABLE(__HANDLE__)                  		((__HANDLE__)->CON |=  (TIMH_PGxCON_ON))
#define __DDL_TIMH_DISABLE(__HANDLE__)                 		((__HANDLE__)->CON &= ~(TIMH_PGxCON_ON))

/**
  * Set TRSET bits(PGxSTAT[TRSET]). Start a period of pwm.
  */
#define __DDL_TIMH_TRSET_SET(__HANDLE__)					SET_BIT((__HANDLE__)->STAT,TIMH_PGxSTAT_TRSET)

/** 
  * Set TRCLR bits(PGxSTAT[TRCLR]). Stop pwm waveform.
  */
#define __DDL_TIMH_TRCLR_CLER(__HANDLE__)					SET_BIT((__HANDLE__)->STAT,TIMH_PGxSTAT_TRCLR)

/** 
  * Open broadcast enable control
  */
#define __TIMH_PGxOCN_MSTEN(__HANDLE__)           			MODIFY_REG((__HANDLE__)->CON, TIMH_PGxCON_MSTEN, TIMH_PGxCON_MSTEN)

/**  @defgroup TIMH_IEVT_Intr_Selection : IEVT type interrupt */
#define TIMH_IT_EOC											0UL << TIMH_PGxEVT_IEVTSEL_Pos
#define TIMH_IT_TRIGA                                       1UL << TIMH_PGxEVT_IEVTSEL_Pos
#define TIMH_IT_TRIGB                                       2UL << TIMH_PGxEVT_IEVTSEL_Pos
#define TIMH_IT_TRIGC                                       3UL << TIMH_PGxEVT_IEVTSEL_Pos
#define TIMH_IT_ADC_TRIG1                                   4UL << TIMH_PGxEVT_IEVTSEL_Pos
#define TIMH_IT_ADC_TRIG2                                   5UL << TIMH_PGxEVT_IEVTSEL_Pos
#define TIMH_IT_UPDATE                                 		6UL << TIMH_PGxEVT_IEVTSEL_Pos

/** @defgroup TIMH_yPCI_Intr_Selection : yPCI type interrupt */
#define TIMH_IT_FPCI										1UL << TIMH_PGxEVT_FLTIEN_Pos
#define TIMH_IT_CLTPCI                                      1UL << TIMH_PGxEVT_CLEN_Pos
#define TIMH_IT_FFPCI                                       1UL << TIMH_PGxEVT_FFEN_Pos
#define TIMH_IT_SPCI                                        1UL << TIMH_PGxEVT_SIEN_Pos

/**
  * Enbaled or disabled interrupt.
  * 'SEL' can be one of the values below :  
  * @defgroup TIMH_IEVT_Intr_Selection
  * @defgroup TIMH_yPCI_Intr_Selection
  */
#define __DDL_TIMHx_IT_EN(TIMHx,SEL)                        SET_BITS((TIMHx)->EVT, (SEL))
#define __DDL_TIMHx_IT_DIS(TIMHx,SEL)                       CLEAR_BITS((TIMHx)->EVT, (SEL))

/**
  * Get witch type interruption be enabled.
  * Return a 32bit result(PGxEVT[24:31]), you can compare with :
  * @defgroup TIMH_IEVT_Intr_Selection
  * @defgroup TIMH_yPCI_Intr_Selection
  */
#define __DDL_TIMHx_GET_IT_SOURCE(TIMHx)                    GET_BITS((TIMHx)->EVT, (TIMH_PGxEVT_IEVTSEL| \
                                                                                    TIMH_PGxEVT_FFIEN  | \
                                                                                    TIMH_PGxEVT_CLIEN  | \
                                                                                    TIMH_PGxEVT_FLTIEN))

/** @defgroup TIMH_IEVT_Status TIMH_IEVT : Flag bit Status */
#define TIMH_FLAG_IEVT                                      TIMH_PGxSTAT_IEVT
#define TIMH_FLAG_TRIGA                                     TIMH_PGxSTAT_TRIGA
#define TIMH_FLAG_TRIGB                                     TIMH_PGxSTAT_TRIGB
#define TIMH_FLAG_TRIGC                                     TIMH_PGxSTAT_TRIGC
#define TIMH_FLAG_ADTRG1                                    TIMH_PGxSTAT_ADTRG1
#define TIMH_FLAG_ADTRG2                                    TIMH_PGxSTAT_ADTRG2

/** @defgroup TIMH_PCI_Event_Status : TIMH PCI Event Flag Status */
#define TIMH_FLAG_SPCIEVT                                   TIMH_PGxSTAT_SEVT
#define TIMH_FLAG_FPCIEVT                                   TIMH_PGxSTAT_FLTEVT
#define TIMH_FLAG_CLPCIEVT                                  TIMH_PGxSTAT_CLEVT
#define TIMH_FLAG_FFPCIEVT                                  TIMH_PGxSTAT_FFEVT

/**
  * Judge the IEVT status bit.Two value(0/1) will be returned.
  * @retval 1 : IEVT is generated
  *         0 : IEVT not generated
  */
#define __DDL_TIMH_IEVT_GETFLAG(__HANDLE__)					((__HANDLE__)->STAT & (TIMH_FLAG_IEVT))

/* Clean the IEVT flag, reset IEVT status bit */
#define __DDL_TIMH_IEVT_CLEARFLAG(__HANDLE__)               SET_BIT((__HANDLE__)->STAT,TIMH_FLAG_IEVT)

/** 
  * Clean the TRIGA flag, reset TRIGA status bit 
  */
#define __DDL_TIMH_TRIGA_CLEARFLAG(__HANDLE__)				SET_BIT((__HANDLE__)->STAT,TIMH_FLAG_TRIGA)

/**
  * Clean the TRIGB flag, reset TRIGB status bit.
  */
#define __DDL_TIMH_TRIGB_CLEARFLAG(__HANDLE__)				SET_BIT((__HANDLE__)->STAT,TIMH_FLAG_TRIGB)

/**
  * Clean the TRIGC flag, reset TRIGB status bit.
  */
#define __DDL_TIMH_TRIGC_CLEARFLAG(__HANDLE__)				SET_BIT((__HANDLE__)->STAT,TIMH_FLAG_TRIGC)

/**
  * Clean the ADTRG1 flag, reset ADTRG1 status bit.
  */
#define __DDL_TIMH_ADTRG1_CLEARFLAG(__HANDLE__)				SET_BIT((__HANDLE__)->STAT,TIMH_FLAG_ADTRG1)

/**
  * Clean the ADTRG2 flag, reset ADTRG2 status bit.
  */
#define __DDL_TIMH_ADTRG2_CLEARFLAG(__HANDLE__)				SET_BIT((__HANDLE__)->STAT,TIMH_FLAG_ADTRG2)

/**
  * Get the status of PCI type interrupt.
  * Two value(0/1) will be returned. 
  * @retval 1 : PCI type interrupt is generated.
  *         0 : PCI type interrupt not generated.
  */
#define __DDL_PCI_SEVT_GETFLAG(__HANDLE__)					((__HANDLE__)->STAT & (TIMH_FLAG_SPCIEVT))
#define __DDL_PCI_FLTEVT_GETFLAG(__HANDLE__)				((__HANDLE__)->STAT & (TIMH_FLAG_FPCIEVT))
#define __DDL_PCI_CLEVT_GETFLAG(__HANDLE__)				    ((__HANDLE__)->STAT & (TIMH_FLAG_CLPCIEVT))
#define __DDL_PCI_FFEVT_GETFLAG(__HANDLE__)				    ((__HANDLE__)->STAT & (TIMH_FLAG_FFPCIEVT))

/**
  * Clean PCI type interrupt flag, reset status bit.
  */
#define __DDL_PCI_SEVT_CLEARFLAG(__HANDLE__)				SET_BIT((__HANDLE__)->STAT,TIMH_FLAG_SPCIEVT)
#define __DDL_PCI_FLTEVT_CLEARFLAG(__HANDLE__)				SET_BIT((__HANDLE__)->STAT,TIMH_FLAG_FPCIEVT)
#define __DDL_PCI_CLEVT_CLEARFLAG(__HANDLE__)				SET_BIT((__HANDLE__)->STAT,TIMH_FLAG_CLPCIEVT)
#define __DDL_PCI_FFEVT_CLEARFLAG(__HANDLE__)				SET_BIT((__HANDLE__)->STAT,TIMH_FLAG_FFPCIEVT)

/** @defgroup TIMH_PCI_IT_Enable : TIMH PCI Type Interrupr Enable. */
#define TIMH_IT_FLTIEN									    TIMH_PGxEVT_FLTIEN		
#define TIMH_IT_CLIEN                                       TIMH_PGxEVT_CLIEN      
#define TIMH_IT_FFIEN                                       TIMH_PGxEVT_FFIEN      
#define TIMH_IT_SIEN                                        TIMH_PGxEVT_SIEN      

/** 
  * Enable or disable the specific PCI module interrupt.
  * 'IT_SRC' can be one of the values below : 
  * @defgroup TIMH_PCI_IT_Enable. 
  */
#define __DDL_TIMH_PCI_IT_EN(__HANDLE__,IT_SRC)             SET_BIT((__HANDLE__)->instance->EVT,(IT_SRC))
#define __DDL_TIMH_PCI_IT_DIS(__HANDLE__,IT_SRC)            CLEAR_BIT((__HANDLE__)->instance->EVT,(IT_SRC))

/**
  * Judge if PCI generated interrupt or not.
  * Two value(0/1) will be returned. 
  * @retval 1 : interrupt is generated by specific PCI module
  *         0 : interrupt is not generated by specific PCI module
  */
#define __DDL_TIMH_GET_FLTIEN_IT_SOURCE(__HANDLE__)         ((__HANDLE__)->EVT & (TIMH_IT_FLTIEN))
#define __DDL_TIMH_GET_CLIEN_IT_SOURCE(__HANDLE__)          ((__HANDLE__)->EVT & (TIMH_IT_CLIEN))      
#define __DDL_TIMH_GET_FFIEN_IT_SOURCE(__HANDLE__)          ((__HANDLE__)->EVT & (TIMH_IT_FFIEN))     
#define __DDL_TIMH_GET_SIEN_IT_SOURCE(__HANDLE__)           ((__HANDLE__)->EVT & (TIMH_IT_SIEN))      

/** @defgpoup TIMH_SOC_Source : TIMH SOC Source */
#define TIMH_SOC_SOURCE_LOCAL_EOC							0UL << TIMH_PGxCON_SOCS_Pos	
#define TIMH_SOC_SOURCE_PG0   								1UL << TIMH_PGxCON_SOCS_Pos	
#define TIMH_SOC_SOURCE_PG1    								2UL << TIMH_PGxCON_SOCS_Pos	
#define TIMH_SOC_SOURCE_PG2    								3UL << TIMH_PGxCON_SOCS_Pos	
#define TIMH_SOC_SOURCE_TRIG_or_PCI							15UL <<TIMH_PGxCON_SOCS_Pos 

/** 
  * Select the SOC signal source.
  * 'SOCS' can be one of the values below :
  * @defgpoup TIMH_SOC_Source
  */
#define __DDL_TIMHx_SOC_SOURCE_SEL(__HANDLE__,SOCS)			SET_BIT((__HANDLE__)->CON, (SOCS))

/** @defgroup TIMH_Triger_Output : TIMH Triger Output */
#define TIMHx_TRIGER_OUT_SEL_EOC							0UL 	
#define TIMHx_TRIGER_OUT_SEL_TRIGA							1UL 	
#define TIMHx_TRIGER_OUT_SEL_TRIGB							2UL 	
#define TIMHx_TRIGER_OUT_SEL_TRIGC							3UL 	

/**
  * Select the triger output signal source.
  * 'SEL' can be one of the values :
  * @defgroup TIMH_Triger_Output
  */
#define __DDL_TIMHx_TRIGER_OUT_SEL(__HANDLE__,SEL)			SET_BIT((__HANDLE__)->EVT, (SEL<< TIMH_PGxEVT_PGTRGSEL_Pos))

/** @defgroup TIMH_HR_Mode_Enable : TIMH HR Mode Enable */
#define TIMH_HR_MODE_DISABLE          						0UL << TIMH_PGxCON_HREN_Pos	
#define TIMH_HR_MODE_ENABLE									1UL << TIMH_PGxCON_HREN_Pos	

/**
  * Update pwm data
  */
#define __DDL_TIMH_UPDREQ(__HANDLE__)                  		SET_BIT((__HANDLE__)->STAT,TIMH_PGxSTAT_UPDREQ)

/**
  * Set pwm triger mode that one triger signal will generate one or multiple period.
  */
#define __DDL_TIMH_TRGMODE_REPEAT(__HANDLE__)				((__HANDLE__)->CON |= (TIMH_PGxCON_TRGMOD))
#define __DDL_TIMH_TRGMODE_SIGLE(__HANDLE__)          		((__HANDLE__)->CON &= ~(TIMH_PGxCON_TRGMOD))

/** @defgroup TIMH_Trigger_Count : TIMH Trigger Count */
#define TIMHx_TRIG_COUNT_1T									0UL << TIMH_PGxCON_TRGCNT_Pos	
#define TIMHx_TRIG_COUNT_2T									1UL << TIMH_PGxCON_TRGCNT_Pos	
#define TIMHx_TRIG_COUNT_3T									2UL << TIMH_PGxCON_TRGCNT_Pos	
#define TIMHx_TRIG_COUNT_4T									3UL << TIMH_PGxCON_TRGCNT_Pos		
#define TIMHx_TRIG_COUNT_5T									4UL << TIMH_PGxCON_TRGCNT_Pos
#define TIMHx_TRIG_COUNT_6T                     			5UL << TIMH_PGxCON_TRGCNT_Pos
#define TIMHx_TRIG_COUNT_7T                     			6UL << TIMH_PGxCON_TRGCNT_Pos
#define TIMHx_TRIG_COUNT_8T									7UL << TIMH_PGxCON_TRGCNT_Pos

/** 
  * Set number of pwm period be generated by triger signal in repeat mode.
  * 'xT' can be one of the values below : 
  * @defgroup TIMH_Trigger_Count
  */
#define __DDL_TRIG_COUNT_xT(__HANDLE__,xT)					SET_BIT((__HANDLE__)->CON, (xT))

/** @defgroup TIMH_Caputer_Source : TIMH Caputer Source */
#define TIMHx_CAP_SRC_SOFEWARE                              0UL	
#define TIMHx_CAP_SRC_SPCI                                  1UL	
#define TIMHx_CAP_SRC_FFPCI                                 2UL	
#define TIMHx_CAP_SRC_CLPCI                                 3UL		
#define TIMHx_CAP_SRC_FPCI                                  4UL

/** @defgroup PCI_PWM_Source_Select : PCI PWM Source Select. This output link to PCI source_0001(PGxyPCI[PSS]) */
#define PWMPCI_SRC_PG0										0UL
#define PWMPCI_SRC_PG1										1UL
#define PWMPCI_SRC_PG2										2UL

/** 
  * Select PCI PWM Source.
  * 'SRC' This paramerter can be one of the values :
  * @defgroup PCI_PWM_Source_Select
  */
#define __DDL_TIMHx_PWMPCI_SRC(__HANDLE__,SRC)				SET_BIT((__HANDLE__)->LEB, ((SRC)<<TIMH_PGxLEB_PWMPCI_Pos))

/** @defgroup PCI_Sofeware_Control_Mode : PCI Sofeware Control Mode. */
#define	SWPCIM_TO_RECV_LOGIC                                0UL
#define SWPCIM_TO_AQSS                                      1UL
#define SWPCIM_TO_TQSS                                      2UL

/** 
  * Switch witch input connect to yPCI's SWPCI signal.
  * 'SEL' This paramerter can be one of the values :
  * @defgroup PCI_Sofeware_Control_Mode
  */
#define __DDL_F_PCI_SWPCIM_SEL(__HANDLE__,SEL)			    SET_BIT((__HANDLE__)->FPCI,  (SEL << TIMH_PGxyPCI_SWPCIM_Pos))				
#define __DDL_CL_PCI_SWPCIM_SEL(__HANDLE__,SEL)			    SET_BIT((__HANDLE__)->CLPCI, (SEL << TIMH_PGxyPCI_SWPCIM_Pos))				
#define __DDL_FF_PCI_SWPCIM_SEL(__HANDLE__,SEL)			    SET_BIT((__HANDLE__)->FFPCI, (SEL << TIMH_PGxyPCI_SWPCIM_Pos))				
#define __DDL_S_PCI_SWPCIM_SEL(__HANDLE__,SEL)			    SET_BIT((__HANDLE__)->SPCI,  (SEL << TIMH_PGxyPCI_SWPCIM_Pos))				

/**
  * Drive signal 1 or 0 to PCI. PGxyPCI[SWPCI] bit = 'SET'.
  */
#define __DDL_F_PCI_SWPCI_SET(__HANDLE__, SET)              MODIFY_REG((__HANDLE__)->FPCI,  TIMH_PGxyPCI_SWPCI_Msk, ((SET) << TIMH_PGxyPCI_SWPCI_Pos))				
#define __DDL_CL_PCI_SWPCI_SET(__HANDLE__,SET)              MODIFY_REG((__HANDLE__)->CLPCI, TIMH_PGxyPCI_SWPCI_Msk, ((SET) << TIMH_PGxyPCI_SWPCI_Pos))				
#define __DDL_FF_PCI_SWPCI_SET(__HANDLE__,SET)              MODIFY_REG((__HANDLE__)->FFPCI, TIMH_PGxyPCI_SWPCI_Msk, ((SET) << TIMH_PGxyPCI_SWPCI_Pos))		
#define __DDL_S_PCI_SWPCI_SET(__HANDLE__, SET)              MODIFY_REG((__HANDLE__)->SPCI,  TIMH_PGxyPCI_SWPCI_Msk, ((SET) << TIMH_PGxyPCI_SWPCI_Pos))				

/** 
  * Config the pci signal accept mode.
  * 'SEL' can be one of the values below :
  * @enum TIMH_PCI_ACP_T
  */
#define __DDL_F_PCI_ACP_MODE_SEL(__HANDLE__, SEL)           SET_BIT((__HANDLE__)->FPCI,  ((SEL) << TIMH_PGxyPCI_ACP_Pos))
#define __DDL_CL_PCI_ACP_MODE_SEL(__HANDLE__,SEL)           SET_BIT((__HANDLE__)->CLPCI, ((SEL) << TIMH_PGxyPCI_ACP_Pos))
#define __DDL_FF_PCI_ACP_MODE_SEL(__HANDLE__,SEL)           SET_BIT((__HANDLE__)->FFPCI, ((SEL) << TIMH_PGxyPCI_ACP_Pos))			
#define __DDL_S_PCI_ACP_MODE_SEL(__HANDLE__, SEL)           SET_BIT((__HANDLE__)->SPCI,  ((SEL) << TIMH_PGxyPCI_ACP_Pos))				

/** 
  * Config the pci signal source.
  * 'SEL' can be one of the values below :
  * @enum TIMH_PCI_PSS_T
  */
#define __DDL_F_PCI_SOURCE(__HANDLE__, SRC)                 MODIFY_REG((__HANDLE__)->FPCI,  TIMH_PGxyPCI_PSS, ((SRC) << TIMH_PGxyPCI_PSS_Pos))
#define __DDL_CL_PCI_SOURCE(__HANDLE__,SRC)                 MODIFY_REG((__HANDLE__)->CLPCI, TIMH_PGxyPCI_PSS, ((SRC) << TIMH_PGxyPCI_PSS_Pos))
#define __DDL_FF_PCI_SOURCE(__HANDLE__,SRC)                 MODIFY_REG((__HANDLE__)->FFPCI, TIMH_PGxyPCI_PSS, ((SRC) << TIMH_PGxyPCI_PSS_Pos))
#define __DDL_S_PCI_SOURCE(__HANDLE__, SRC)                 MODIFY_REG((__HANDLE__)->SPCI,  TIMH_PGxyPCI_PSS, ((SRC) << TIMH_PGxyPCI_PSS_Pos))

/** 
  * Config pci Source signal polarity.
  * 'SET' can be 1 or 0
  */
#define __DDL_F_PCI_PPS(__HANDLE__,SET)                     SET_BIT((__HANDLE__)->FPCI,  ((SET) << TIMH_PGxyPCI_PPS_Pos))
#define __DDL_CL_PCI_PPS(__HANDLE__,SET)	                SET_BIT((__HANDLE__)->CLPCI, ((SET) << TIMH_PGxyPCI_PPS_Pos))
#define __DDL_FF_PCI_PPS(__HANDLE__,SET)	                SET_BIT((__HANDLE__)->FFPCI, ((SET) << TIMH_PGxyPCI_PPS_Pos))
#define __DDL_S_PCI_PPS(__HANDLE__,SET)	                    SET_BIT((__HANDLE__)->SPCI,  ((SET) << TIMH_PGxyPCI_PPS_Pos))

/** 
  * Config pci AQSS signal source. 
  * 'SRC' can be one of the values :
  * @enum TIMH_PCI_AQSS_T
  */
#define __DDL_F_AQSS_SOURCE(__HANDLE__,SRC)					MODIFY_REG((__HANDLE__)->FPCI, TIMH_PGxyPCI_AQSS, ((SRC)<< TIMH_PGxyPCI_AQSS_Pos))
#define __DDL_CL_AQSS_SOURCE(__HANDLE__,SRC)				MODIFY_REG((__HANDLE__)->CLPCI,TIMH_PGxyPCI_AQSS, ((SRC)<< TIMH_PGxyPCI_AQSS_Pos))
#define __DDL_FF_AQSS_SOURCE(__HANDLE__,SRC)				MODIFY_REG((__HANDLE__)->FFPCI,TIMH_PGxyPCI_AQSS, ((SRC)<< TIMH_PGxyPCI_AQSS_Pos))
#define __DDL_S_AQSS_SOURCE(__HANDLE__,SRC)					MODIFY_REG((__HANDLE__)->SPCI, TIMH_PGxyPCI_AQSS, ((SRC)<< TIMH_PGxyPCI_AQSS_Pos))

/** 
  * Config pci AQSS signal polarity. 
  * 'SET' can be 1 or 0 
  */
#define __DDL_F_PCI_AQPS(__HANDLE__,SET)                    MODIFY_REG((__HANDLE__)->FPCI,  TIMH_PGxyPCI_AQPS, ((SET)<<TIMH_PGxyPCI_AQPS_Pos))
#define __DDL_CL_PCI_AQPS(__HANDLE__,SET)	                MODIFY_REG((__HANDLE__)->CLPCI, TIMH_PGxyPCI_AQPS, ((SET)<<TIMH_PGxyPCI_AQPS_Pos))
#define __DDL_FF_PCI_AQPS(__HANDLE__,SET)	                MODIFY_REG((__HANDLE__)->FFPCI, TIMH_PGxyPCI_AQPS, ((SET)<<TIMH_PGxyPCI_AQPS_Pos))
#define __DDL_S_PCI_AQPS(__HANDLE__,SET)	                MODIFY_REG((__HANDLE__)->SPCI,  TIMH_PGxyPCI_AQPS, ((SET)<<TIMH_PGxyPCI_AQPS_Pos))

/** 
  * Config pci TERM signal source.
  * 'EVT' can be one of the values below :
  * @defgroup TIMH_PCI_TERM_T
  */
#define __DDL_F_TERM_EVENT(__HANDLE__,EVT)					MODIFY_REG((__HANDLE__)->FPCI,  TIMH_PGxyPCI_TERM, ((EVT)<< TIMH_PGxyPCI_TERM_Pos))
#define __DDL_CL_TERM_EVENT(__HANDLE__,EVT)					MODIFY_REG((__HANDLE__)->CLPCI, TIMH_PGxyPCI_TERM, ((EVT)<< TIMH_PGxyPCI_TERM_Pos))
#define __DDL_FF_TERM_EVENT(__HANDLE__,EVT)					MODIFY_REG((__HANDLE__)->FFPCI, TIMH_PGxyPCI_TERM, ((EVT)<< TIMH_PGxyPCI_TERM_Pos))
#define __DDL_S_TERM_EVENT(__HANDLE__,EVT)					MODIFY_REG((__HANDLE__)->SPCI,  TIMH_PGxyPCI_TERM, ((EVT)<< TIMH_PGxyPCI_TERM_Pos))

/** 
  * Config pci termination Qualifier Source (TQSS) signal source.
  * 'SEL' can be one of the values below :
  * @defgroup TIMH_PCI_TQSS_T
  */
#define __DDL_F_TQSS_SEL(__HANDLE__,SEL)					MODIFY_REG((__HANDLE__)->FPCI, TIMH_PGxyPCI_TQSS, ((SEL) << TIMH_PGxyPCI_TQSS_Pos))
#define __DDL_CL_TQSS_SEL(__HANDLE__,SEL)					MODIFY_REG((__HANDLE__)->CLPCI, TIMH_PGxyPCI_TQSS, ((SEL) << TIMH_PGxyPCI_TQSS_Pos))
#define __DDL_FF_TQSS_SEL(__HANDLE__,SEL)					MODIFY_REG((__HANDLE__)->FFPCI, TIMH_PGxyPCI_TQSS, ((SEL) << TIMH_PGxyPCI_TQSS_Pos))
#define __DDL_S_TQSS_SEL(__HANDLE__,SEL)					MODIFY_REG((__HANDLE__)->SPCI, TIMH_PGxyPCI_TQSS, ((SEL) << TIMH_PGxyPCI_TQSS_Pos))

/**
  * Set sofeware terminate signal as 1
  */
#define __DDL_F_SET_SWTERM(__HANDLE__)                      SET_BIT((__HANDLE__)->FPCI, TIMH_PGxyPCI_SWTERM)
#define __DDL_CL_SET_SWTERM(__HANDLE__)                     SET_BIT((__HANDLE__)->CLPCI, TIMH_PGxyPCI_SWTERM)
#define __DDL_FF_SET_SWTERM(__HANDLE__)                     SET_BIT((__HANDLE__)->FFPCI, TIMH_PGxyPCI_SWTERM)
#define __DDL_S_SET_SWTERM(__HANDLE__)                      SET_BIT((__HANDLE__)->SPCI, TIMH_PGxyPCI_SWTERM)

/**
  * @brief  use to activate TIMH module
  * @param  None
  * @retval None
  */
void DDL_TIMH_Instance_Activate(void);

/**
  * @brief  Configure the structure of TIMH_ClkInit_T to default values.
  * @param  pTimhClk : Pointer to a TIMH_ClkInit_T structure.
  * @retval None
  */
void DDL_TIMH_ClkStructInit(TIMH_ClkInit_T *pTimhClk);

/**
  * @brief  use to config TIMH common clock
  * @param  clkConfig : pointer to a TIMH_Clk_Config_T structure.
  * @retval None
  */
void DDL_TIMH_Clock_Init(TIMH_ClkInit_T *clkConfig);

/**
  * @brief  Configure the structure of TIMH_BaseInit_T to default values.
  * @param  pTimhStruct : Pointer to a TIMH_BaseInit_T structure.
  * @retval None
  */
void DDL_TIMH_BaseStructInit(TIMH_BaseInit_T *pTimhStruct);

/**
  * @brief  Initializes TIMH by the specified parameters in the TIMH_BaseInit_T.
  * @param  TIMHx       : TIMH instance.
  *         pTimhStruct : Pointer to a TIMH_BaseInit_T structure.
  * @retval None
  */
void DDL_TIMH_Init(TIMH_Type *TIMHx, TIMH_BaseInit_T *pTimhStruct);

/**
  * @brief  Start pwm
  * @param  TIMHx : TIMH instance.
  * @retval None
  */
void DDL_TIMH_PWM_Start(TIMH_Type *TIMHx);

/**
  * @brief  Stop pwm
  * @param  TIMHx : TIMH instance.
  * @retval None
  */
void DDL_TIMH_PWM_Stop(TIMH_Type *TIMHx);

/**
  * @brief  TIMH interrupt enable
  * @param  TIMHx    : TIMH instance.
  *         intrMask : search
  *                    @defgroup TIMH_IEVT_Intr_Selection
  *                    @defgroup TIMH_yPCI_Intr_Selection.
  * @retval None
  */
void DDL_TIMH_IntEnable(TIMH_Type *TIMHx, uint32_t intrMask);

/**
  * @brief  TIMH interrupt disable
  * @param  TIMHx    : TIMH instance.
  *         intrMask : search
  *                    @defgroup TIMH_IEVT_Intr_Selection
  *                    @defgroup TIMH_yPCI_Intr_Selection.
  * @retval None
  */
void DDL_TIMH_IntDisable(TIMH_Type *TIMHx, uint32_t intrMask);

/**
  * @brief  Configure the structure of TIMH_Overwrite_T to default values.
  * @param  overWrite : Pointer to a TIMH_Overwrite_T structure.
  * @retval None
  */
void DDL_TIMH_Overwrite_StructInit(TIMH_Overwrite_T *overWrite);

/**
  * @brief  Config overwirte value and behave mode for xPCI and sofeware control
  * @param  TIMHx     : TIMH instance.
  *         overWrite : pointer to a TIMH_Overwrite_T structure.
  * @retval None
  */
void DDL_TIMH_Overwrite_Config(TIMH_Type *TIMHx, TIMH_Overwrite_T *overWrite);

/**
  * @brief  Configure the structure of TIMH_yPCI_Init_T to default values.
  * @param  pTimhStruct : Pointer to a TIMH_BaseInit_T structure.
  * @retval None
  */
void DDL_TIMH_yPCI_StructInit(TIMH_yPCI_Init_T *yPCI);

/**
  * @brief  use to stepup SPCI module.(SPCI is used for start a PWM period)
  * @param  TIMHx : TIMH instance.
  *         yPCI  : pointer to a TIMH_yPCI_Init_T structure for SPCI setup.
  * @retval None
  */
void DDL_TIMH_SPCI_Setup(TIMH_Type *TIMHx, TIMH_yPCI_Init_T *yPCI);

/**
  * @brief  use to stepup SPCI module with bypass source
  * @param  TIMHx         : TIMH instance.
  *         pci_init      : pointer to a TIMH_yPCI_Init_T structure for SPCI setup.
  *         bypass_source : pointer to a TIMH_PCI_BPSEL_T structure for bypass source.
  * @retval None
  */
void DDL_TIMH_SPCI_Bypass_Setup(TIMH_Type        *TIMHx,
                                TIMH_yPCI_Init_T *yPCI,
                                TIMH_PCI_BPSEL_T bypass_source);

/**
  * @brief  use to stepup FPCI module
  * @param  TIMHx     : TIMH instance.
  *         yPCI      : pointer to a TIMH_yPCI_Init_T structure for FPCI setup.
  *         overWrite : pointer to a TIMH_Overwrite_T structure for FPCI overwrite value.
  * @retval None
  */
void DDL_TIMH_FPCI_Setup(TIMH_Type        *TIMHx,
                         TIMH_yPCI_Init_T *yPCI,
                         TIMH_Overwrite_T *overWrite);

/**
  * @brief  use to stepup FPCI module with bypass source
  * @param  TIMHx         : TIMH instance.
  *         yPCI          : pointer to a TIMH_yPCI_Init_T structure for FPCI setup.
  *         overWrite     : pointer to a TIMH_Overwrite_T structure for FPCI overwrite value.
  *         bypass_source : pointer to a TIMH_PCI_BPSEL_T structure for bypass source.
  * @retval None
  */
void DDL_TIMH_FPCI_Bypass_Setup(TIMH_Type        *TIMHx,
                                TIMH_yPCI_Init_T *yPCI,
                                TIMH_Overwrite_T *overWrite,
                                TIMH_PCI_BPSEL_T bypass_source);

/**
  * @brief  use to stepup CLPCI module
  * @param  TIMHx     : TIMH instance.
  *         yPCI      : pointer to a TIMH_yPCI_Init_T structure for CLPCI setup.
  *         overWrite : pointer to a TIMH_Overwrite_T structure for CLPCI overwrite value.
  * @retval None
  */
void DDL_TIMH_CLPCI_Setup(TIMH_Type        *TIMHx,
                          TIMH_yPCI_Init_T *yPCI,
                          TIMH_Overwrite_T *overWrite);

/**
  * @brief  use to stepup CLPCI module with bypass source
  * @param  TIMHx         : TIMH instance.
  *         yPCI          : pointer to a TIMH_yPCI_Init_T structure for CLPCI setup.
  *         overWrite     : pointer to a TIMH_Overwrite_T structure for CLPCI overwrite value.
  *         bypass_source : pointer to a TIMH_PCI_BPSEL_T structure for bypass source.
  * @retval None
  */
void DDL_TIMH_CLPCI_Bypass_Setup(TIMH_Type        *TIMHx,
                                 TIMH_yPCI_Init_T *yPCI,
                                 TIMH_Overwrite_T *overWrite,
                                 TIMH_PCI_BPSEL_T bypass_source);

/**
  * @brief  use to stepup FFPCI module
  * @param  TIMHx     : TIMH instance.
  *         yPCI      : pointer to a TIMH_yPCI_Init_T structure for FFPCI setup.
  *         overWrite : pointer to a TIMH_Overwrite_T structure for FFPCI overwrite value.
  * @retval None
  */
void DDL_TIMH_FFPCI_Setup(TIMH_Type        *TIMHx,
                          TIMH_yPCI_Init_T *yPCI,
                          TIMH_Overwrite_T *overWrite);

/**
  * @brief  use to stepup FFPCI module with bypass source
  * @param  TIMHx         : TIMH instance.
  *         yPCI          : pointer to a TIMH_yPCI_Init_T structure for FFPCI setup.
  *         overWrite     : pointer to a TIMH_Overwrite_T structure for FFPCI overwrite value.
  *         bypass_source : pointer to a TIMH_PCI_BPSEL_T structure for bypass source.
  * @retval None
  */
void DDL_TIMH_FFPCI_Bypass_Setup(TIMH_Type        *TIMHx,
                                 TIMH_yPCI_Init_T *yPCI,
                                 TIMH_Overwrite_T *overWrite,
                                 TIMH_PCI_BPSEL_T bypass_source);

/**
  * @brief  use to config LEB funtion
  * @param  TIMHx        : TIMH instance.
  *         lebPeriod    : Leading-Edge Blanking Period
  *                        if TIMHx clk is 48Mhz, LEB time length = ( ((LEBP[15:0]>>3)+1) * 8 ) / 48000000
  *         lebTrigerSel : point to a TIMH_LEB_T for config LEB trige source
  * @retval None
  */
void DDL_TIMH_LEB_Config(TIMH_Type  *TIMHx,
	                     uint16_t   lebPeriod,
	                     TIMH_LEB_T *lebTrigerSel);


/**
  * @brief  This API is used to config adc triger funtion.
  * @param  TIMHx                 : TIMH instance.
  *         adcTrig1_FreqDiv      : ADC_Triger1 frequncey division
  *         adcTrig1_Offset       : ADC_Triger1 start offset
  *         htim adtr1TRIGA       : TRIGA as ADC_Triger1's source
  *         htim adtr1TRIGB       : TRIGB as ADC_Triger1's source
  *         htim adtr1TRIGC       : TRIGC as ADC_Triger1's source
  *         htim adtr2TRIGA       : TRIGA as ADC_Triger2's source
  *         htim adtr2TRIGB       : TRIGB as ADC_Triger2's source
  *         htim adtr2TRIGC       : TRIGC as ADC_Triger2's source
  * @retval None
  */
void DDL_TIMH_ADC_Triger_Config(TIMH_Type            *TIMHx,
	                            uint8_t              adcTrig1_FreqDiv,
	                            uint8_t              adcTrig1_Offset,
	                            TIMH_ADC_Trg_EN_T    adtr1TRIGA,
	                            TIMH_ADC_Trg_EN_T    adtr1TRIGB,
	                            TIMH_ADC_Trg_EN_T    adtr1TRIGC,
	                            TIMH_ADC_Trg_EN_T    adtr2TRIGA,
	                            TIMH_ADC_Trg_EN_T    adtr2TRIGB,
	                            TIMH_ADC_Trg_EN_T    adtr2TRIGC );

/**
  * @brief  TIMH Capture Config
  * @param  TIMHx      : TIMH instance.
  *         captureSrc : value from 0x00 to 0x04, refer to 
  *                      @defgroup TIMH_Caputer_Source
  * @retval None
  */
void DDL_TIMH_Capture_Config(TIMH_Type *TIMHx, uint8_t captureSrc);

/**
  * @brief  Get TIMHx Capture value
  * @param  TIMHx        : TIMH instance.
  *         captureValue : value of PGxCAP
  * @retval None
  */
void DDL_TIMH_Get_Capture_Value(TIMH_Type *TIMHx, uint32_t *captureValue);

/**
  * @brief  combine output config
  * @param  combSource1Sel   : combine  source1 select
  *         combSource0Sel   : combine  source0 select
  *         combSource1Pol   : combine  source1 polarity config
  *         combSource0Pol   : combine  source0 polarity config
  *         logicFunctionSel : logic function select
  *         logicOutputSel   : logic output select
  * @retval None
  *
  */
void DDL_TIMH_COMB_OUTPUT_CONFIG(uint8_t combSource1Sel,   uint8_t combSource0Sel,
                                 uint8_t combSource1Pol,   uint8_t combSource0Pol,
                                 uint8_t logicFunctionSel, uint8_t logicOutputSel);

/**
  * @brief  event output register config
  * @param  pwmEventCtrl     : select PWM event output control register
  *         eventOutputDelay : PWM event output delay control
  *                            0 (delay 8 clock)
  *                            1 (no delay     )
  *         pwmEventSel      : PWM event select
  *         pwmSourceSel     : PWM event source select
  * @retval None
  */
void DDL_Event_Output_Config(TIMH_PWM_EVENT_CTRL_T pwmEventCtrl,
                             uint8_t               eventOutputDelay,
                             TIMH_EBUS_EVENT_T     pwmEventSel,
                             TIMH_EBUS_SOURCE_T    pwmSourceSel);

/**
  * @brief  deadtime config, clk = 200M(IC), dtime = xx us
  * @param  TIMHx  : TIMH instance.
  *         dtimeF : failing
  *         dtimeR : rising
  * @retval None
  */
void DDL_TIMH_DeadTime_Init(TIMH_Type *TIMHx,
	                        uint16_t  dtimeF,
	                        uint16_t  dtimeR);

#ifdef __cplusplus
}
#endif
#endif /* __KPM32xx_DDL_TIMH_H */


