/**
 ******************************************************************************
 * @file    kpm32xx_ddl_ebus.h
 * @author  Kiwi Software Team
 * @brief   Header file of EBUS DDL module.
 * @note
 *          V1.0.0, 2024/12/20.
 *
 * Copyright (c) 2024, Kiwi Instruments Co., Ltd.
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 *	1. Redistributions of source code must retain the above copyright notice,
 *	   this list of conditions and the following disclaimer.
 *
 *	2. Redistributions in binary form must reproduce the above copyright notice,
 *	   this list of conditions and the following disclaimer in the documentation
 *	   and/or other materials provided with the distribution.
 *
 *	3. Neither the name of the copyright holder nor the names of its contributors
 *	   may be used to endorse or promote products derived from this software without
 *	   specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
 * THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE
 * GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 ******************************************************************************
 */


#ifndef __KPM32xx_DDL_EBUS_H
#define __KPM32xx_DDL_EBUS_H

#ifdef __cplusplus
extern "C" {
#endif

#include "kpm32xx_ddl_def.h"



/**
 * EBUS is a event-Bridge which connects Master and Slave on each side.
 * For better understanding, refer to the following event flow:
 * EBUS Masters --> EBUS --> EBUS Slaves.
 * Caution: event can be triggered only by Master and slave only responses to the
 * event once when one event happens.
 */

#ifdef DDL_EBUS_MODULE_ENABLED
#define __EBUS_CHANNEL_MAXIMUM__              16U

/* two adjacent channels 'AND' or 'OR' operation */
#define EBUS_CHN_ADJ2CHN_OREN                 (0x01U << EBUS_CHN_OREN_Pos)
#define EBUS_CHN_ADJ2CHN_ORDIS                (0x00U << EBUS_CHN_OREN_Pos)
#define EBUS_CHN_ADJ2CHN_ANDEN                (0x01U << EBUS_CHN_ANDEN_Pos)
#define EBUS_CHN_ADJ2CHN_ANDDIS               (0x00U << EBUS_CHN_ANDEN_Pos)

/* Enable EBUS output to PAD */
#define EBUS_CHN_OUT2PAD_ENABLE               (0x01U << EBUS_CHN_OE_Pos)
#define EBUS_CHN_OUT2PAD_DISABLE              (0x00U << EBUS_CHN_OE_Pos)

/* 4 edge detection ways for each EBUS channel */
#define EBUS_CHN_DETSRC_SYNC                  (0x00U << EBUS_CHN_DETSRC_Pos)
#define EBUS_CHN_DETSRC_RISING                (0x01U << EBUS_CHN_DETSRC_Pos)
#define EBUS_CHN_DETSRC_FALLING               (0x02U << EBUS_CHN_DETSRC_Pos)
#define EBUS_CHN_DETSRC_RSORFL                (0x03U << EBUS_CHN_DETSRC_Pos)

/* Async/Sync mode for each EBUS channel */
#define	EBUS_CHN_ASYNC_MODE                   (0x01U << EBUS_CHN_ASYNC_Pos)
#define	EBUS_CHN_SYNC_MODE                    (0x00U << EBUS_CHN_ASYNC_Pos)

/* EBUS channel output inverse or not */
#define	EBUS_CHN_INVOUT                       (0x01U << EBUS_CHN_INVEN_Pos)
#define	EBUS_CHN_NOROUT                       (0x00U << EBUS_CHN_INVEN_Pos)

/* EBUS master sources */
#define	EBUS_CHN_MSTR_CPU                       		(0x00U << EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_DMA                       		(0x04U << EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_TIMA0_OC0                 		(0x08U << EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_TIMA0_OC1                 		(0x09U << EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_TIMA0_OC2                 		(10U << EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_TIMA0_OC3                 		(11U << EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_TIMA0_EVT0                		(12U<< EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_TIMA0_EVT1                		(13U<< EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_TIMA0_TRGO1               		(14U<< EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_TIMA0_TRGO                		(15U<< EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_TIMA1_OC0                 		(16U<< EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_TIMA1_OC1                 		(17U<< EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_TIMA1_OC2                 		(18U<< EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_TIMA1_OC3                 		(19U<< EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_TIMA1_EVT0                		(20U<< EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_TIMA1_EVT1                		(21U<< EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_TIMA1_TRGO1               		(22U<< EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_TIMA1_TRGO                		(23U<< EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_TIMH_EVENTA                		(24U << EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_TIMH_EVENTB                 		(25U<< EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_TIMH_EVENTC                 		(26U<< EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_TIMG0_OC0                 		(32U<< EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_TIMG0_OC1                 		(33U<< EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_TIMG0_OC2                 		(34U<< EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_TIMG0_OC3                 		(35U<< EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_TIMG0_TRGO                		(36U<< EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_TIMG1_OC0                 		(40U<< EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_TIMG1_OC1                 		(41U<< EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_TIMG1_OC2                 		(42U<< EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_TIMG1_OC3                 		(43U<< EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_TIMG1_TRGO                		(44U<< EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_TIMG2_OC0                 		(48U<< EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_TIMG2_OC1                 		(49U<< EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_TIMG2_OC2                 		(50U<< EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_TIMG2_OC3                 		(51U<< EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_TIMG2_TRGO                		(52U<< EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_TIMS0_OC0                 		(56U<< EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_TIMS0_OC1                 		(57U<< EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_TIMS0_TRGO                		(58U<< EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_TIMS1_OC0                 		(60U<< EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_TIMS1_OC1                 		(61U<< EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_TIMS1_TRGO                		(62U<< EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_TIMS2_OC0                 		(64U<< EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_TIMS2_OC1                 		(65U<< EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_TIMS2_TRGO                		(66U<< EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_TIMS3_OC0                 		(68U<< EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_TIMS3_OC1                 		(69U<< EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_TIMS3_TRGO                		(70U<< EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_BTM1                      		(72U<< EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_BTM2                      		(73U<< EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_BTM3                      		(74U<< EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_BTM4                      		(75U<< EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_ACMP0                     		(88U<< EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_ACMP1                     		(89U<< EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_ACMP2                     		(90U<< EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_ACMP3                     		(91U<< EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_ADC0_CPLTEVT0	                    (96U<< EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_ADC0_CPLTEVT1                     (97U<< EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_ADC0_CPLTEVT2                     (98U<< EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_ADC0_CPLTEVT3                     (99U<< EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_ADC0_CPLTEVT4                     (100U<< EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_ADC0_CPLTEVT5                     (101U<< EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_ADC0_CPLTEVT6                     (102U<< EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_ADC0_CPLTEVT7                     (103U<< EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_ADC0_CPLTEVT8                     (104U<< EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_ADC0_CPLTEVT9                     (105U<< EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_ADC0_CPLTEVT10                    (106U<< EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_ADC0_CPLTEVT11                    (107U<< EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_ADC0_CPLTEVT12                    (108U<< EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_ADC0_CPLTEVT13                    (109U<< EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_ADC0_CPLTEVT14                    (110U<< EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_ADC0_CPLTEVT15                    (111U<< EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_ADC0_PPB0_ULEVT                   (112U<< EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_ADC0_PPB0_LLEVT                   (113U<< EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_ADC0_PPB0_CROSS0                  (114U<< EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_ADC0_PPB1_ULEVT                   (115U<< EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_ADC0_PPB1_LLEVT                   (116U<< EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_ADC0_PPB1_CROSS0                  (117U<< EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_ADC0_PPB2_ULEVT                   (118U<< EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_ADC0_PPB2_LLEVT                   (119U<< EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_ADC0_PPB2_CROSS0                  (120U<< EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_ADC0_PPB3_ULEVT                   (121U<< EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_ADC0_PPB3_LLEVT                   (122U<< EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_ADC0_PPB3_CROSS0                  (123U<< EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_ADC0_ANYSOC_CPLTEVT               (124U<< EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_ADC0_ANYSOC_OVF_EVENT             (125U<< EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_ADC1_CPLTEVT0               		(128U<< EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_ADC1_CPLTEVT1               		(129U<< EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_ADC1_CPLTEVT2               		(130U<< EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_ADC1_CPLTEVT3               		(131U<< EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_ADC1_CPLTEVT4               		(132U<< EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_ADC1_CPLTEVT5               		(133U<< EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_ADC1_CPLTEVT6               		(134U<< EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_ADC1_CPLTEVT7               		(135U<< EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_ADC1_CPLTEVT8               		(136U<< EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_ADC1_CPLTEVT9               		(137U<< EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_ADC1_CPLTEVT10              		(138U<< EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_ADC1_CPLTEVT11             		(139U<< EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_ADC1_CPLTEVT12             		(140U<< EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_ADC1_CPLTEVT13             		(141U<< EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_ADC1_CPLTEVT14             		(142U<< EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_ADC1_CPLTEVT15             		(143U<< EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_ADC1_PPB0_ULEVT                   (144U<< EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_ADC1_PPB0_LLEVT                   (145U<< EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_ADC1_PPB0_CROSS0                  (146U<< EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_ADC1_PPB1_ULEVT                   (147U<< EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_ADC1_PPB1_LLEVT                   (148U<< EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_ADC1_PPB1_CROSS0                  (149U<< EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_ADC1_PPB2_ULEVT                   (150U<< EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_ADC1_PPB2_LLEVT                   (151U<< EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_ADC1_PPB2_CROSS0                  (152U<< EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_ADC1_PPB3_ULEVT                   (153U<< EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_ADC1_PPB3_LLEVT                   (154U<< EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_ADC1_PPB3_CROSS0                  (155U<< EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_ADC1_ANYSOC_CPLTEVT               (156U<< EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_ADC1_ANYSOC_OVF_EVENT             (157U<< EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_IO0_SOURCE                        (240U<< EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_IO1_SOURCE                        (241U<< EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_IO2_SOURCE                        (242U<< EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_IO3_SOURCE                        (243U<< EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_IO4_SOURCE                        (244U<< EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_IO5_SOURCE                        (245U<< EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_IO6_SOURCE                        (246U<< EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_IO7_SOURCE                        (247U<< EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_IO8_SOURCE                        (248U<< EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_IO9_SOURCE                        (249U<< EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_IO10_SOURCE                       (250U<< EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_IO11_SOURCE                       (251U<< EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_IO12_SOURCE                       (252U<< EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_IO13_SOURCE                       (253U<< EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_IO14_SOURCE                       (254U<< EBUS_CHN_SRCSEL_Pos)
#define EBUS_CHN_MSTR_IO15_SOURCE                       (255U<< EBUS_CHN_SRCSEL_Pos)
/* EBUS channels */
typedef enum
{
	EBUS_CHN_0                                = 0x00U,
	EBUS_CHN_1                                = 0x01U,
	EBUS_CHN_2                                = 0x02U,
	EBUS_CHN_3                                = 0x03U,
	EBUS_CHN_4                                = 0x04U,
	EBUS_CHN_5                                = 0x05U,
	EBUS_CHN_6                                = 0x06U,
	EBUS_CHN_7                                = 0x07U,
	EBUS_CHN_8								  = 0x08U,
	EBUS_CHN_9								  = 0x09U,
	EBUS_CHN_10								  = 10U,
	EBUS_CHN_11								  = 11U,
	EBUS_CHN_12								  = 12U,
	EBUS_CHN_13								  = 13U,
	EBUS_CHN_14								  = 14U,
	EBUS_CHN_15								  = 15U,
	EBUS_CHN_INVALID,
}DDL_EBUSChn_T;
typedef struct
{
	uint32_t      syncMode;         /* EBUS sync or async mode */
	uint32_t      mstrSrc;          /* EBUS master Sources */
	uint32_t      adj2ChansAnd;     /* two adjacent channels 'AND' operation */
	uint32_t      adj2ChansOr;      /* two adjacent channels 'OR' operation */
	uint32_t      invOut;           /* inversion before outputing to PAD */
	uint32_t      detSrc;           /* rising or falling detection for each EBUS channel */
	uint32_t      out2Pad;          /* Out to PAD */
    DDL_EBUSChn_T                 channel;
}EBUS_Init_T;



/* EBUS channel is free or occupied */
typedef enum
{
	EBUS_CHN_FREE                             = 0x00U,
	EBUS_CHN_OCCUPIED                         = 0x01U,
}EBUSFree_T;

/* CPU event types */
typedef enum
{
	EBUS_CPUEVT_LEVEL                         = 0x00U,
	EBUS_CPUEVT_PULSE                         = 0x01U,
}EBUSCPUEvtLvl_T;


/* Route event to CPU that's generate a WFE event to wakeup CPU from sleep mode */
#define __DDL_EBUS_EVENT2CPU_ENABLE(__INSTANCE__)            SET_BIT((__INSTANCE__)->CPU, EBUS_CPU_EN)
#define __DDL_EBUS_EVENT2CPU_DISABLE(__INSTANCE__)           CLEAR_BIT((__INSTANCE__)->CPU, EBUS_CPU_EN)


/**
  * @brief  Make EBUS peripheral active.
  * @param  None.
  * @retval None.
  */
void DDL_EBUS_Instance_Active(void);


/**
  * @brief  Configure the structure of EBUS_Init_T to default values.
  *
  * @param  eBusStruct Pointer to a EBUS_Init_T structure that contains
  *                        the configuration information for the given EBUS module.
  *
  * @retval None
  */
void DDL_EBUS_StructInit(EBUS_Init_T *eBusStruct);


/**
  * @brief  Initializes the EBUS by the specified parameters in
  *         the EBUS_Handle_T and initialize the associated handle.
  * @param  EBUSx Pointer to DMA instance.
  * @param  eBusStruct Pointer to a EBUS_Init_T structure that contains
  *                  the configuration information for the given DMA module.
  * @retval None
  */
void DDL_EBUS_Init(EBUS_Type *EBUSx, EBUS_Init_T *eBusStruct);


/**
  * @brief  DeInitializes the EBUS peripheral.
  * @param  EBUSx Pointer to DMA instance.
  * @param  eBusStruct Pointer to a EBUS_Init_T structure that contains
  *                  the configuration information for the given DMA module.
  * @retval None
  */
void DDL_EBUS_DeInit(EBUS_Type *EBUSx, EBUS_Init_T *eBusStruct);


/**
  * @brief  CPU triggers an event to Peripheral in form of Level or Pulse
  *         TrigOrClr makes sense:
  *         1. only when CPUEvtType is EBUS_CPUEVT_LEVEL;
  *         2. when TrigOrClr is 1, EBUS_CPUEVT_LEVEL asserts to given channel of chnIndex;
  *         3. when TrigOrClr is 0, EBUS_CPUEVT_LEVEL desserts.
  *
  * @param  EBUSx Pointer to DMA instance.
  * @param  eBusStruct Pointer to a EBUS_Init_T structure that contains
  *                  the configuration information for the given DMA module.
  * @param  CPUEvtType  CPU event types.
  * @param  TrigOrClr   refer to the brief description.
  *
  */
void DDL_EBUS_CPUEvtRoute2Periph(EBUS_Type *EBUSx, EBUS_Init_T *eBusStruct, EBUSCPUEvtLvl_T CPUEvtType, uint32_t TrigOrClr);


/**
  * @brief  periapherals trigger EBUS event to CPU.
  * @param  EBUSx Pointer to DMA instance.
  * @param  eBusStruct Pointer to a EBUS_Init_T structure that contains
  *                  the configuration information for the given DMA module.
  *
  * @retval none
  */
void DDL_EBUS_PeriphEvtRoute2CPU(EBUS_Type *EBUSx, EBUS_Init_T *eBusStruct);


#endif

#ifdef __cplusplus
}
#endif
#endif /** __KPM32xx_DDL_EBUS_H */
