/**
  *****************************************************************************************************
  * @file    main.c
  * @author  Kiwi Software Team
  * @brief   This example code demonstrates how WWDG works.
  *
  * Copyright (c) 2024, Kiwi Instruments Co,. Ltd.
  *
  * Redistribution and use in source and binary forms, with or without modification,
  * are permitted provided that the following conditions are met:
  *
  *   1. Redistributions of source code must retain the above copyright notice,
  *      this list of conditions and the following disclaimer.
  *
  *   2. Redistributions in binary form must reproduce the above copyright notice,
  *      this list of conditions and the following disclaimer in the documentation
  *      and/or other materials provided with the distribution.
  *
  *   3. Neither the name of the copyright holder nor the names of its contributors
  *      may be used to endorse or promote products derived from this software without
  *      specific prior written permission.
  *
  * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
  * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
  * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
  * THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
  * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE
  * GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
  * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
  * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
  * THE POSSIBILITY OF SUCH DAMAGE.
  *****************************************************************************************************
  */

#include "main.h"
#include "kpm32xx_demoboard_ddl.h"


/**
  * @brief  Handle WWDG interrupt request.
  * @note   The Early warning Interrupt (EWI) can be used if specific safety operations
  *         or data logging must be performed before the actual reset is generated.
  *         The EWI interrupt is enabled by calling.
  *         When the downcounter reaches the threshold, and EWI interrupt is
  *         generated and the corresponding Interrupt Service Routine (ISR) can
  *         be used to trigger specific actions (such as communications or data
  *         logging), before resetting the device.
  * @param  WWDGx WWDG instance.
  * @retval None
  */
void DDL_WWDG_Callback(WWDG_Type *WWDGx)
{
	if (__DDL_WWDG_EWI_IS_ENABLE(WWDGx) != RESET)
	{
		if (__DDL_WWDG_GET_EWI_STATUS(WWDGx) != RESET)
		{
			__DDL_WWDG_REGS_UNLOCK__(WWDGx);

            /* open can feed dog */
//			__DDL_WWDG_CLEAR_EWI_STATUS(WWDGx);

			__DDL_WWDG_REGS_LOCK__(WWDGx);

            /* do some thing */
		}
	}
}


#if defined (__DEBUG_CONSOLE_PRINT__)
/**
  * @brief  This function is implemented for UART print PinMux configuration.
  * @param  None
  * @retval None
  */
void DebugConsole_PinMux_Config(void)
{
	/* SCI1 UART Tx */
	DDL_GPIO_Config2AltFunc(GPIOA, GPIO_PIN_7, GPIOA7_AF1_SCI1_TX_SDA_MOSI);
}
#endif


/**
  * @brief  This function is implemented for system clock configuration.
  *         where:
  *         System Clock source            = PLL (HIRC)
  *         System Clock frequency         = 100MHz
  * @param  None
  * @retval None
  */
void SystemClock_Config(void)
{
	RCC_PLLInit_T pllInitStruct    = {0};
	RCC_ClkInit_T rccClkInitStruct = {0};

	DDL_RCC_PllStructInit(&pllInitStruct);
	/* FCLK : 100MHz */
	pllInitStruct.pllPrescaler  = 4;
	if (DDL_RCC_PllConfig(&pllInitStruct) != DDL_OK)
	{
		while(1) ;
	}

	/* Select PLL as system clock source */
	rccClkInitStruct.sysClkSource  = RCC_SYSCLKSOURCE_PLLCLK;
	rccClkInitStruct.sysClkDivider = 0;
	if (DDL_RCC_ClkSrcConfig(&rccClkInitStruct) != DDL_OK)
	{
		while(1) ;
	}
}



/**
  * @brief  The application entry point.
  * @retval int
  */
int main(void)
{
    static uint32_t rebootTime = 1000;
	WWDG_Init_T wwdgInt;
	uint8_t loopCnt = 0;

	/* Reset of all peripherals, Initializes the Flash interface and the Systick. */
	DDL_Init();
	SystemClock_Config();

    /* To make GPIO active */
	DDL_GPIO_Instance_Active();

	/* To make GPIO active */
	DDL_GPIO_Instance_Active();

#if defined (__DEBUG_CONSOLE_PRINT__)
	DebugConsole_PinMux_Config();
	DDL_SCI_Instance_Active(SCI1);
	DDL_SCIUart_DebugConsole_Init(SCI1);
#endif

    DDL_WWDG_Instance_Active();

    if(READ_BIT(SYSCFG->RSTS, (1<<1)))
	{
        printf(" WWDG already reboot, test success \n");
		/* clear wwdg reboot flag */
		SET_BIT(SYSCFG->RSTS, (1<<1));
	}
    printf(" WWDG test \n");

    /* Init for WWDG : 1s */
	DDL_WWDG_StructInit(&wwdgInt);
	wwdgInt.counter = (uint32_t)(rebootTime * (8000000 / 1000));
	DDL_WWDG_Init(WWDG,&wwdgInt);

    NVIC_SetPriority(WWDG_IRQn, __PREEMPT_INTRERRUPT_PRIORITY_0__);
	NVIC_EnableIRQ(WWDG_IRQn);

    BSP_LEDx_Init(LED_S1);

    while(1)
	{
		BSP_LEDx_Twinkle(LED_S1);
		DDL_Delay(300);
		/* watch dog refresh */
		DDL_WWDG_Refresh(WWDG);

		if(loopCnt++ > 20)
		{
			printf("Wait for system reboot by WWDG\n");
			while(1);
		}
	}

}

