/**
  *****************************************************************************************************
  * @file    iic_user_drv.c
  * @author  Kiwi Software Team
  * @brief   This file provides simplified functions to manage the following
  *          functionalities of SCI IIC:
  *          + Transmit and receive data with interrupt mode
  * @note
  *          V1.0.0, 2022/08/03.
  *
  * Copyright (c) 2024, Kiwi Instruments Co,. Ltd.
  *
  * Redistribution and use in source and binary forms, with or without modification,
  * are permitted provided that the following conditions are met:
  *
  *   1. Redistributions of source code must retain the above copyright notice,
  *      this list of conditions and the following disclaimer.
  *
  *   2. Redistributions in binary form must reproduce the above copyright notice,
  *      this list of conditions and the following disclaimer in the documentation
  *      and/or other materials provided with the distribution.
  *
  *   3. Neither the name of the copyright holder nor the names of its contributors
  *      may be used to endorse or promote products derived from this software without
  *      specific prior written permission.
  *
  * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
  * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
  * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
  * THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
  * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE
  * GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
  * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
  * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
  * THE POSSIBILITY OF SUCH DAMAGE.
  *****************************************************************************************************
  */

#include "main.h"
#include "string.h"
#include <stdlib.h>
#include "iic_user_drv.h"



/**
  * @brief  I2C as master transmits an amount of data with interrupt.
  * @param  I2Cx     Pointer to IIC instance.
  *         slvAddr  slave address.
  *         pData    pointer to data buffer.
  *         size     the amount of data to be sent.
  *
  * @retval DDL status
  */
DDL_Status_T DDL_I2C_Master_Transmit_IT(I2C_Type *I2Cx, uint16_t slvAddr, uint8_t *pData, uint16_t size)
{
	__DDL_I2C_WRITE_CONFIG(I2Cx);
	I2Cx->DATACNT = size;
	I2Cx->FIFOCTL = 2;
	/* Enable all possible interrupts:
	 * Interrupt rises on 8th or 9th Clock and when STOP signal is detected.
	 */
	__DDL_I2C_CLEAR_TCNTDONE_INTR(I2Cx);
	__DDL_I2C_CLEAR_NACK_INTR(I2Cx);
	__DDL_I2C_CLEAR_ADDR_INTR(I2Cx);
	__DDL_I2C_CLEAR_DATA_INTR(I2Cx);

	__DDL_I2C_CLEAR_STOP_INTR(I2Cx);

	__DDL_I2C_INTENABLE(I2Cx, I2C_INTEN_STOPINT | I2C_INTEN_WTIMEN | I2C_INTEN_ADDRINT | I2C_INTEN_NACK| I2C_INTEN_TCNTEN);
	/* Configure ACK Interrupt (That's DATA_INT and ADDR_INT) on the 9th clock */
	__DDL_I2C_WTIM_8OR9THCLK_CONFIG(I2Cx, I2C_INTOCUR_ON9TH_CLK);
    I2Cx->SLAD = slvAddr;

	/* Generate One I2C Start-up Signal*/
	__DDL_I2C_STT_GENERATE(I2Cx);

	return DDL_OK;

}


/**
  * @brief  I2C as master receives an amount of data with interrupt.
  * @param  I2Cx     Pointer to IIC instance.
  *         slvAddr  slave address.
  *         pData    pointer to data buffer.
  *         size     the amount of data to receive.
  *
  * @retval DDL status
  */
DDL_Status_T DDL_SCII2C_Master_Receive_IT(I2C_Type *I2Cx, uint16_t slvAddr, uint8_t *pData, uint16_t size)
{
    ASSERT_PARAM(pData != NULL);
	ASSERT_PARAM(size  != 0U);

	__DDL_I2C_READ_CONFIG(I2Cx);
	I2Cx->DATACNT = size;
	I2Cx->FIFOCTL = 6;
	/* Configure ACK Interrupt (That's DATA_INT and ADDR_INT) on the 9th clock */
	__DDL_I2C_WTIM_8OR9THCLK_CONFIG(I2Cx, I2C_INTOCUR_ON8TH_CLK);

	__DDL_I2C_CLEAR_TCNTDONE_INTR(I2Cx);
	__DDL_I2C_CLEAR_NACK_INTR(I2Cx);
	__DDL_I2C_CLEAR_ADDR_INTR(I2Cx);
	__DDL_I2C_CLEAR_DATA_INTR(I2Cx);

	__DDL_I2C_CLEAR_STOP_INTR(I2Cx);
	/* Enable all possible interrupts:
	 * Interrupt rises on 8th or 9th Clock and when STOP signal is detected.
	 */
	__DDL_I2C_INTENABLE(I2Cx, I2C_INTEN_STOPINT | I2C_INTEN_WTIMEN | I2C_INTEN_ADDRINT | I2C_INTEN_FIFOTHRE | I2C_INTEN_TCNTEN);

	I2Cx->SLAD = slvAddr;

	/* Generate One I2C Start-Up Signal*/
	__DDL_I2C_STT_GENERATE(I2Cx);

	return DDL_OK;
}

