/**
  *****************************************************************************************************
  * @file    main.c
  * @author  Kiwi Software Team
  * @brief   This example code demonstrates how EBUS works.
  *
  * Copyright (c) 2024, Kiwi Instruments Co,. Ltd.
  *
  * Redistribution and use in source and binary forms, with or without modification,
  * are permitted provided that the following conditions are met:
  *
  *   1. Redistributions of source code must retain the above copyright notice,
  *      this list of conditions and the following disclaimer.
  *
  *   2. Redistributions in binary form must reproduce the above copyright notice,
  *      this list of conditions and the following disclaimer in the documentation
  *      and/or other materials provided with the distribution.
  *
  *   3. Neither the name of the copyright holder nor the names of its contributors
  *      may be used to endorse or promote products derived from this software without
  *      specific prior written permission.
  *
  * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
  * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
  * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
  * THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
  * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE
  * GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
  * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
  * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
  * THE POSSIBILITY OF SUCH DAMAGE.
  *****************************************************************************************************
  */

#include "main.h"
#include "kpm32xx_demoboard_ddl.h"


EBUS_Init_T eBusStruct;


#if defined (__DEBUG_CONSOLE_PRINT__)
/**
  * @brief  This function is implemented for UART print PinMux configuration.
  * @param  None
  * @retval None
  */
void DebugConsole_PinMux_Config(void)
{
	/* SCI1 UART Tx */
	DDL_GPIO_Config2AltFunc(GPIOA, GPIO_PIN_7, GPIOA7_AF1_SCI1_TX_SDA_MOSI);
}
#endif


/**
  * @brief  This function is implemented for system clock configuration.
  *         where:
  *         System Clock source            = PLL (HIRC)
  *         System Clock frequency         = 100MHz
  * @param  None
  * @retval None
  */
void SystemClock_Config(void)
{
	RCC_PLLInit_T pllInitStruct    = {0};
	RCC_ClkInit_T rccClkInitStruct = {0};

	DDL_RCC_PllStructInit(&pllInitStruct);
	/* FCLK : 100MHz */
	pllInitStruct.pllPrescaler  = 4;
	if (DDL_RCC_PllConfig(&pllInitStruct) != DDL_OK)
	{
		while(1) ;
	}

	/* Select PLL as system clock source */
	rccClkInitStruct.sysClkSource  = RCC_SYSCLKSOURCE_PLLCLK;
	rccClkInitStruct.sysClkDivider = 0;
	if (DDL_RCC_ClkSrcConfig(&rccClkInitStruct) != DDL_OK)
	{
		while(1) ;
	}
}

/**
  * @brief  16bit BTM01 initialize
  *
  * @retval None
  *
  * @note   Clock source as default : LIRC.
  *         Works as 16bit_Mode with interrupt enable.
  */
void BTM01_16bit_Init(void)
{
	BTM_BaseInit_T btm01Struct;

	DDL_BTM_Stop(BTIM01);
	DDL_BTM_StructInit(&btm01Struct);

	btm01Struct.loadValue  = 0x800;
	DDL_BTM_Init(BTIM01, &btm01Struct);
}



/**
  * @brief  The application entry point.
  * @retval int
  */
int main(void)
{
	/* Reset of all peripherals, Initializes the Flash interface and the Systick. */
	DDL_Init();
	SystemClock_Config();
    DDL_Delay(2000);

	/* To make GPIO active */
	DDL_GPIO_Instance_Active();

#if defined (__DEBUG_CONSOLE_PRINT__)
	DebugConsole_PinMux_Config();
	DDL_SCI_Instance_Active(SCI1);
	DDL_SCIUart_DebugConsole_Init(SCI1);
#endif
    printf("This demo demonstrate how EBUS and Base-Timer were used to Wake-up CPU sleeped caused by WFE instrtion \n");
    DDL_EBUS_Instance_Active();

    DDL_EBUS_StructInit(&eBusStruct);

    eBusStruct.channel      = EBUS_CHN_0;
	DDL_EBUS_Init(EBUS, &eBusStruct);
	/** CPU waken-up due to event cauded by BTM 0 interrupt */
	DDL_EBUS_PeriphEvtRoute2CPU(EBUS, &eBusStruct);

    DDL_BTM_Instance_Active();
    BTM01_16bit_Init();
    NVIC_SetPriority(BASETIMER0_IRQn, 0);
	NVIC_EnableIRQ(BASETIMER0_IRQn);

	/** Now, start Base Timer couting */
    WRITE_REG(BTIM0_COMM->ST, (BTIM_ST_TM0INT | BTIM_ST_TM1INT));
    DDL_BTM_EvtOutEnable(BTIM01);
    DDL_BTM_Start(BTIM01);

	CLEAR_BIT(SCB->SCR, 1UL << 2);

	while(1)
	{
		printf("CPU is ready to sleep \n");
		DDL_SuspendTick();
	   /* Request Wait For event */
		__SEV();
		__WFE();
		__WFE();
		DDL_ResumeTick();
		DDL_Delay(2000);
		printf("CPU Waken up \n");
	}

}

