/**
  *****************************************************************************************************
  * @file    main.c
  * @author  Kiwi Software Team
  * @brief   This example code demonstrates how DMA works.
  *
  * Copyright (c) 2024, Kiwi Instruments Co,. Ltd.
  *
  * Redistribution and use in source and binary forms, with or without modification,
  * are permitted provided that the following conditions are met:
  *
  *   1. Redistributions of source code must retain the above copyright notice,
  *      this list of conditions and the following disclaimer.
  *
  *   2. Redistributions in binary form must reproduce the above copyright notice,
  *      this list of conditions and the following disclaimer in the documentation
  *      and/or other materials provided with the distribution.
  *
  *   3. Neither the name of the copyright holder nor the names of its contributors
  *      may be used to endorse or promote products derived from this software without
  *      specific prior written permission.
  *
  * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
  * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
  * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
  * THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
  * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE
  * GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
  * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
  * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
  * THE POSSIBILITY OF SUCH DAMAGE.
  *****************************************************************************************************
  */

#include "main.h"
#include "kpm32xx_demoboard_ddl.h"


#define BUFFER_SIZE              114

volatile uint32_t transComplete = 0;
volatile uint32_t transError = 0;

static uint32_t srcBuffer[BUFFER_SIZE] =
{
  0x00001021, 0x20423063, 0x408450a5, 0x60c670e7, 0x9129a14a, 0xb16bc18c,
  0xd1ade1ce, 0xf1ef1231, 0x32732252, 0x52b54294, 0x72f762d6, 0x93398318,
  0xa35ad3bd, 0xc39cf3ff, 0xe3de2462, 0x34430420, 0x64e674c7, 0x44a45485,
  0xa56ab54b, 0x85289509, 0xf5cfc5ac, 0xd58d3653, 0x26721611, 0x063076d7,
  0x569546b4, 0xb75ba77a, 0x97198738, 0xf7dfe7fe, 0xc7bc48c4, 0x58e56886,
  0x78a70840, 0x18612802, 0xc9ccd9ed, 0xe98ef9af, 0x89489969, 0xa90ab92b,
  0x4ad47ab7, 0x6a961a71, 0x0a503a33, 0x2a12dbfd, 0xfbbfeb9e, 0x9b798b58,
  0xbb3bab1a, 0x6ca67c87, 0x5cc52c22, 0x3c030c60, 0x1c41edae, 0xfd8fcdec,
  0xad2abd0b, 0x8d689d49, 0x7e976eb6, 0x5ed54ef4, 0x2e321e51, 0x0e70ff9f,
  0xefbedfdd, 0xcffcbf1b, 0x9f598f78, 0x918881a9, 0xb1caa1eb, 0xd10cc12d,
  0xe16f1080, 0x00a130c2, 0x20e35004, 0x40257046, 0x83b99398, 0xa3fbb3da,
  0xc33dd31c, 0xe37ff35e, 0x129022f3, 0x32d24235, 0x52146277, 0x7256b5ea,
  0x95a88589, 0xf56ee54f, 0xd52cc50d, 0x34e224c3, 0x04817466, 0x64475424,
  0x4405a7db, 0xb7fa8799, 0xe75ff77e, 0xc71dd73c, 0x26d336f2, 0x069116b0,
  0x76764615, 0x5634d94c, 0xc96df90e, 0xe92f99c8, 0xb98aa9ab, 0x58444865,
  0x78066827, 0x18c008e1, 0x28a3cb7d, 0xdb5ceb3f, 0xfb1e8bf9, 0x9bd8abbb,
  0x4a755a54, 0x6a377a16, 0x0af11ad0, 0x2ab33a92, 0xed0fdd6c, 0xcd4dbdaa,
  0xad8b9de8, 0x8dc97c26, 0x5c644c45, 0x3ca22c83, 0x1ce00cc1, 0xef1fff3e,
  0xdf7caf9b, 0xbfba8fd9, 0x9ff86e17, 0x7e364e55, 0x2e933eb2, 0x0ed11ef0
};

static uint32_t dstBuffer[BUFFER_SIZE];

#if defined (__DEBUG_CONSOLE_PRINT__)
/**
  * @brief  This function is implemented for UART print PinMux configuration.
  * @param  None
  * @retval None
  */
void DebugConsole_PinMux_Config(void)
{
	/* SCI1 UART Tx */
	DDL_GPIO_Config2AltFunc(GPIOA, GPIO_PIN_7, GPIOA7_AF1_SCI1_TX_SDA_MOSI);
}
#endif


/**
  * @brief  Handle DMA interrupt request.
  * @param  DMAx  DMAx Pointer to DMA instance.
  * @retval None
  */
void DDL_DMA_Callback(DMA_Type *DMAx, DMA_ChannelConfig_T *pDMAStruct, DMA_ChannelIndex_T channel)
{
	uint8_t event = 0;

	if (DMAx->INT_FLAG & (0x1UL << channel))
	{
		__DDL_DMA_CLEAR_IT_FLAG(DMAx, (0x1UL << channel));

		__DDL_DMA_DISABLE_IT(DMAx, (0x1UL << channel));
		if (pDMAStruct->ebusEnable != RESET)
		{
			SET_BIT(DMAx->EBUS_CTRL, 0x1U << (channel * 4 + 3));
		}
		__DDL_DMA_CHANNEL_DISABLE(DMAx, (0x1UL << channel));

		if (__DDL_DMA_GET_ERR_STATUS(DMAx))
		{
			__DDL_DMA_CLEAR_ERR_STATUS(DMAx);
			event = (uint8_t)DMA_EVENT_TRANS_ERROR;
		}
		else
		{
			event = (uint8_t)DMA_EVENT_TRANS_COMPLETE;
		}
        if (event == DMA_EVENT_TRANS_ERROR)
        {
            transError = 1;
        }
        else
        {
            transComplete = 1;
        }
	}
}


/**
  * @brief  This function is implemented for system clock configuration.
  *         where:
  *         System Clock source            = PLL (HIRC)
  *         System Clock frequency         = 100MHz
  * @param  None
  * @retval None
  */
void SystemClock_Config(void)
{
	RCC_PLLInit_T pllInitStruct    = {0};
	RCC_ClkInit_T rccClkInitStruct = {0};

	DDL_RCC_PllStructInit(&pllInitStruct);
	/* FCLK : 100MHz */
	pllInitStruct.pllPrescaler  = 4;
	if (DDL_RCC_PllConfig(&pllInitStruct) != DDL_OK)
	{
		while(1) ;
	}

	/* Select PLL as system clock source */
	rccClkInitStruct.sysClkSource  = RCC_SYSCLKSOURCE_PLLCLK;
	rccClkInitStruct.sysClkDivider = 0;
	if (DDL_RCC_ClkSrcConfig(&rccClkInitStruct) != DDL_OK)
	{
		while(1) ;
	}
}


/**
  * @brief  DMA single demo .
  * @retval None
  */
int M2M_Single_Channel_Demo()
{
    DMA_ChannelConfig_T pDMAStruct;
	int ret = 0;
	uint32_t transLen = 0;

	DDL_DMA_ChannelStructInit(&pDMAStruct);
	DDL_DMA_Init(DMA);
    NVIC_EnableIRQ(DMA_IRQn);
    NVIC_SetPriority(DMA_IRQn, __PREEMPT_INTRERRUPT_PRIORITY_0__);
	memset(dstBuffer, 0, sizeof(dstBuffer));
	transLen = BUFFER_SIZE;//0x1;//0x10000;//BUFFER_SIZE;
    /* Configure the source, destination address and the data length */
    pDMAStruct.srcAddress = (uint32_t)srcBuffer;/*0x40010000,SRAM_BASE+0x20,FLASH_BASE*/
    pDMAStruct.dstAddress = (uint32_t)dstBuffer;/*0x40030004*/
    pDMAStruct.dataLength = transLen;
	DDL_DMA_SetChannelConfig(DMA, DMA_CHANNEL_0, &pDMAStruct);
	DDL_DMA_ChannelStart(DMA, DMA_CHANNEL_0, DMA_DEVICE_3, &pDMAStruct);
    DDL_DMA_IntEnable(DMA, DMA_CHANNEL_0);

	/* this while here only for test case, for real application,  please process in the callback. */
	while ((transComplete == 0) && (transError == 0))
	{
		;
	}
	if (transError == 1)
	{
		transError = 0;
		return -2;
	}
	transComplete = 0;

    if (memcmp(dstBuffer, srcBuffer, sizeof(srcBuffer)))
	{
		ret = -3;
	}

	return ret;
}


/**
  * @brief  The application entry point.
  * @retval int
  */
int main(void)
{
	/* Reset of all peripherals, Initializes the Flash interface and the Systick. */
	DDL_Init();
	SystemClock_Config();

	/* To make GPIO active */
	DDL_GPIO_Instance_Active();

#if defined (__DEBUG_CONSOLE_PRINT__)
	DebugConsole_PinMux_Config();
	DDL_SCI_Instance_Active(SCI1);
	DDL_SCIUart_DebugConsole_Init(SCI1);
#endif
    __DDL_RCC_DMA_ACTIVE();
    __DDL_RCC_DMA_CLK_RELEASE();
    printf("DMA test start:\n");

	/* different data alignment test */
	printf("	DMA_DATAALIGN_BYTE test start:\n");
	if (M2M_Single_Channel_Demo() != 0)
	{
		printf("	DMA_DATAALIGN_BYTE test fail.\n");
	}
	else
	{
		printf("	DMA_DATAALIGN_BYTE test success.\n");
	}

	printf("DMA test end.\n");
    while (1)
    {}

}

